package com.floreantpos.extension.cardconnect;

import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.json.JSONObject;

import com.floreantpos.PosException;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

public class SecuredProcessor implements CardProcessor {
	private static final String STR_NEWLINE = "\n"; //$NON-NLS-1$
	private String username;
	private String password;
	private String merchantId;
	private String authString;
	private String encodedAuthString;
	private GsonBuilder gsonBuilder;

	public SecuredProcessor() {
		Store store = DataProvider.get().getStore();
		this.merchantId = store.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_MERCHANT_ID);
		this.username = store.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_USERNAME);
		this.password = store.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_PASSWORD);

		this.authString = this.username + ":" + this.password; //$NON-NLS-1$
		this.encodedAuthString = Base64.getEncoder().encodeToString(this.authString.getBytes());
		this.gsonBuilder = new GsonBuilder();
	}

	public SecuredProcessor(String username2, String password2, String merchantId2) {
		this.username = username2;
		this.merchantId = merchantId2;
		this.password = password2;
		
		this.authString = this.username + ":" + this.password; //$NON-NLS-1$
		this.encodedAuthString = Base64.getEncoder().encodeToString(this.authString.getBytes());
		this.gsonBuilder = new GsonBuilder();
	}

	@Override
	public void preAuth(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeAuthRequest(transaction, false);
		transaction.setAuthorizable(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeCaptureRequest(transaction);
		transaction.setCaptured(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void chargeAmount(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeAuthRequest(transaction, true);
		transaction.setAuthorizable(false);
		transaction.setCaptured(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeVoidRequest(transaction.getCardTransactionId(), transaction);
		transaction.setVoided(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeRefundRequest(transaction.getCardTransactionId(), refundAmount, transaction);
		transaction.setRefunded(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	@Override
	public void cancelTransaction() {
	}

	private void saveTransaction(PosTransaction transaction, String responseJSON) throws Exception {
		JSONObject jsonObject = new JSONObject(responseJSON);
		Map<String, Object> properties = jsonObject.toMap();
		String retref = (String) properties.get(SecuredConstants.PROP_IPP320_RETREF);
		String authCode = (String) properties.get(SecuredConstants.PROP_IPP320_AUTHCODE);
		String accountNumber = (String) properties.get(SecuredConstants.PROP_IPP320_ACCOUNT);
		transaction.setCardTransactionId(retref);
		if (StringUtils.isNotEmpty(authCode)) {
			transaction.setCardAuthCode(authCode);
		}
		if (StringUtils.isNotEmpty(accountNumber)) {
			accountNumber = accountNumber.substring(accountNumber.length() - 4, accountNumber.length());
			transaction.setCardNumber(accountNumber);
		}
		this.setTransactionProperties(transaction, properties);
	}

	private void setTransactionProperties(PosTransaction transaction, Map<String, Object> properties) throws Exception {
		if (transaction == null || properties == null) {
			return;
		}
		List<Entry<String, Object>> entries = new ArrayList<>(properties.entrySet());
		for (Entry<String, Object> entry : entries) {
			transaction.addExtraProperty(entry.getKey(), String.valueOf(entry.getValue()));
		}
	}

	private boolean isCardInfoValid(PosTransaction transaction) throws Exception {
		if (transaction.getCardReader().equals(CardReader.SWIPE.name()) && StringUtils.isNotBlank(transaction.getCardTrack())) {
			return Boolean.TRUE;
		}
		else if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			if (StringUtils.isBlank(transaction.getCardNumber())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.0")); //$NON-NLS-1$
			}
			if (!(StringUtils.isNotBlank(transaction.getCardExpYear()) && transaction.getCardExpYear().length() >= 2)) {
				throw new PosException(IppMessages.getString("SecuredProcessor.1")); //$NON-NLS-1$
			}
			if (StringUtils.isBlank(transaction.getCardExpMonth())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.2")); //$NON-NLS-1$
			}
			//			if (StringUtils.isBlank(transaction.getCardCVV())) {
			//				return Boolean.FALSE;
			//			}
		}
		return Boolean.TRUE;
	}

	private String makeAuthRequest(PosTransaction transaction, boolean capture) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_ORDERID, transaction.getTicketId());
		requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, transaction.getAmount());
		requestParams.put(SecuredConstants.PROP_IPP320_CURRENCY, DataProvider.get().getMainCurrency().getCode());
		requestParams.put(SecuredConstants.PROP_IPP320_CAPTURE, capture ? "Y" : "N"); //$NON-NLS-1$ //$NON-NLS-2$
		requestParams.put(SecuredConstants.PROP_IPP320_TOKENIZE, "Y"); //$NON-NLS-1$

		this.isCardInfoValid(transaction);

		if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			requestParams.put(SecuredConstants.PROP_IPP320_ACCOUNT, transaction.getCardNumber());
			String expiry = transaction.getCardExpMonth() + transaction.getCardExpYear().substring(transaction.getCardExpYear().length() - 2);
			requestParams.put(SecuredConstants.PROP_IPP320_EXPIRY, expiry);
			requestParams.put(SecuredConstants.PROP_IPP320_CVV2, transaction.getCardCVV());
			requestParams.put(SecuredConstants.PROP_IPP320_NAME, transaction.getCardHolderName());
		}
		else if (transaction.getCardReader().equals(CardReader.SWIPE.name())) {
			requestParams.put(SecuredConstants.PROP_IPP320_TRACK, transaction.getCardTrack());
		}
		return this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_AUTH, requestParams, transaction);
	}

	private String makeCaptureRequest(PosTransaction posTransaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, posTransaction.getCardTransactionId());
		requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, posTransaction.getAmount());

		String responseJSON = this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_CAPTURE, requestParams, posTransaction);
		//		Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
		//		Object amountStr = responseObjects.get(SecuredConstants.PROP_IPP320_AMOUNT);
		//		if (amountStr != null) {
		//			Double amount = Double.parseDouble((String) amountStr);
		//			posTransaction.setAmount(amount);
		//		}

		return responseJSON;
	}

	private String makeVoidRequest(String retref, PosTransaction transaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, retref);
		if (transaction.getAmount() != null) {
			requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, transaction.getAmount());
		}
		return this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_VOID, requestParams, transaction);
	}

	private String makeRefundRequest(String retref, Double amount, PosTransaction transaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, retref);
		if (amount != null) {
			requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, amount);
		}
		return this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_REFUND, requestParams, transaction);
	}

	@SuppressWarnings("resource")
	private String makeRequest(String requestURL, Map<String, Object> requestParams, PosTransaction posTransaction) throws Exception {
		HttpClient httpClient = HttpClientBuilder.create().build();
		HttpPut putRequest = new HttpPut(requestURL);
		Gson gson = this.gsonBuilder.create();
		String paramsJSON = gson.toJson(requestParams);
		StringEntity requestParamsEntity = new StringEntity(paramsJSON);
		putRequest.addHeader("content-type", "application/json"); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setHeader("Authorization", "Basic " + this.encodedAuthString); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setEntity(requestParamsEntity);
		HttpResponse response = httpClient.execute(putRequest);
		int statusCode = response.getStatusLine().getStatusCode();
		String responseJSON = ""; //$NON-NLS-1$

		if (statusCode == 200) {
			HttpEntity entity = response.getEntity();
			responseJSON = EntityUtils.toString(entity);
			this.postProcessingLog(statusCode, responseJSON);
			Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
			SecuredResponseStatus responseStatus = SecuredResponseStatus.fromStatus((String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPSTAT));
			if (responseStatus != SecuredResponseStatus.APPROVED) {
				throw new PosException((String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPTEXT));
				//				if (responseStatus == SecuredResponseStatus.RETRY) {
				//					throw new SecuredException(IppMessages.getString("IPP320Processor.0")); //$NON-NLS-1$
				//				}
				//				else {
				//					throw new PosException(jsonObject.getString("resptext")); //$NON-NLS-1$
				//				}
			}
		}
		else if (statusCode == 401) {
			throw new PosException(IppMessages.getString("SecuredProcessor.3")); //$NON-NLS-1$
		}
		else {
			this.postProcessingLog(statusCode, responseJSON);
			throw new SecuredException(IppMessages.getString("IPP320Processor.1")); //$NON-NLS-1$
		}
		return responseJSON;
	}

	@SuppressWarnings("resource")
	public boolean makeTestRequest() throws Exception {
		HttpClient httpClient = HttpClientBuilder.create().build();
		String requestURL = SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_TEST_CONNECTION;
		HttpPut putRequest = new HttpPut(requestURL);
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		Gson gson = this.gsonBuilder.create();
		String paramsJSON = gson.toJson(requestParams);
		StringEntity requestParamsEntity = new StringEntity(paramsJSON);
		putRequest.addHeader("content-type", "application/json"); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setHeader("Authorization", "Basic " + this.encodedAuthString); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setEntity(requestParamsEntity);
		HttpResponse response = httpClient.execute(putRequest);
		int statusCode = response.getStatusLine().getStatusCode();
		if (statusCode == 200) {
			HttpEntity entity = response.getEntity();
			String responseHTML = EntityUtils.toString(entity);
			String successMessage = "CardConnect REST Servlet"; //$NON-NLS-1$
			return responseHTML.contains(successMessage);
		}
		return false;
	}

	private void postProcessingLog(int statusCode, String responseJSON) {
		StringBuilder logBuilder = new StringBuilder().append(STR_NEWLINE);
		logBuilder.append(IppMessages.getString("SecuredProcessor.5")).append(STR_NEWLINE); //$NON-NLS-1$
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.6"), statusCode)).append(STR_NEWLINE); //$NON-NLS-1$
		if (StringUtils.isNotBlank(responseJSON)) {
			Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
			SecuredResponseStatus responseStatus = SecuredResponseStatus.fromStatus((String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPSTAT));
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.7"), responseStatus.name())).append(STR_NEWLINE); //$NON-NLS-1$
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.8"), responseObjects.get(SecuredConstants.PROP_IPP320_AMOUNT))) //$NON-NLS-1$
					.append(STR_NEWLINE);
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.9"), responseObjects.get(SecuredConstants.PROP_IPP320_AUTHCODE))) //$NON-NLS-1$
					.append(STR_NEWLINE);
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.10"), responseJSON)).append(STR_NEWLINE); //$NON-NLS-1$
		}
	}

	private void preProcessingLog(PosTransaction transaction) {
		StringBuilder logBuilder = new StringBuilder().append(STR_NEWLINE);
		logBuilder.append(IppMessages.getString("SecuredProcessor.11")).append(STR_NEWLINE); //$NON-NLS-1$
		try {
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.12"), transaction.getCardReader())).append(STR_NEWLINE); //$NON-NLS-1$
			if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.13"), transaction.getCardHolderName())).append(STR_NEWLINE); //$NON-NLS-1$
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.14"), //$NON-NLS-1$
						transaction.getCardNumber().substring(transaction.getCardNo().length() - 4))).append(STR_NEWLINE);
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.15"), transaction.getCardExpMonth())).append(STR_NEWLINE); //$NON-NLS-1$
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.16"), transaction.getCardExpYear())).append(STR_NEWLINE); //$NON-NLS-1$
			}
		} catch (Exception e) {
		}
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.17"), transaction.getTicketId())).append(STR_NEWLINE); //$NON-NLS-1$
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.18"), transaction.getAmount())).append(STR_NEWLINE); //$NON-NLS-1$
	}
}