/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseTicketDiscount;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.util.DiscountUtil;
import com.floreantpos.util.NumberUtil;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "itemCode" })
public class TicketDiscount extends BaseTicketDiscount implements ITicketItem {
	private static final long serialVersionUID = 1L;

	public static String PROP_LOYALTY_POINT = "loyaltyPoint"; //$NON-NLS-1$
	public static String PROP_LOYALTY_CHARGED = "loyaltyCharged"; //$NON-NLS-1$

	private Integer loyaltyPoint;
	private Boolean loyaltyCharged;
	private int originalType;
	private double originalValue;
	

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public TicketDiscount() {
	}

	/**
	 * Constructor for primary key
	 */
	public TicketDiscount(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public TicketDiscount(java.lang.String id, String name, double quantity, double value, int type, Ticket ticket) {
		setId(id);
		setName(name);
		setCouponQuantity(quantity);
		setValue(value);
		setType(type);
		setTicket(ticket);
		setTicketId(ticket.getId());
	}

	@Override
	public Double getCouponQuantity() {
		Double couponQuantity = super.getCouponQuantity();
		if (couponQuantity == 0)
			return 1.0;
		return couponQuantity;
	}

	public String getNameDisplay() {
		double couponQuantity = getCouponQuantity();
		String display;
		if (couponQuantity > 1) {
			display = NumberUtil.trimDecilamIfNotNeeded(couponQuantity) + "x " + getName(); //$NON-NLS-1$
		}
		else {
			display = getName().trim();
		}
		return display; //$NON-NLS-1$

	}

	@Override
	public String getItemCode() {
		return null;
	}

	@Override
	public boolean canAddCookingInstruction() {
		return false;
	}

	@Override
	public boolean canAddDiscount() {
		return false;
	}

	@Override
	public boolean canVoid() {
		return false;
	}

	@Override
	public boolean canAddAdOn() {
		return false;
	}

	@Override
	public Boolean isPrintedToKitchen() {
		return false;
	}

	@Override
	public String getUnitPriceDisplay() {
		return null;
	}

	@Override
	public String getItemQuantityDisplay() {
		return null;
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		return null;
	}

	@Override
	public Double getSubtotalAmount() {
		return -getTotalDiscountAmount();
	}

	@Override
	public String getSubTotalAmountDisplay() {
		return String.valueOf(-getTotalDiscountAmount());
	}

	@Override
	public void setDiscountAmount(Double amount) {
	}

	@Override
	public Double getDiscountAmount() {
		return null;
	}

	@Override
	public KitchenStatus getKitchenStatusValue() {
		return null;
	}

	@Override
	public boolean isSaved() {
		return getId() != null;
	}

	public boolean isLoyaltyCharged() {
		if (getType() != Discount.DISCOUNT_TYPE_LOYALTY) {
			return true;
		}
		return loyaltyCharged != null && loyaltyCharged;
	}

	public JSONObject toJson() {
		JSONObject jsonObject = new JSONObject();
		jsonObject.put(PROP_ID, this.getId());
		jsonObject.put(PROP_LOYALTY_POINT, this.getLoyaltyPoint());
		jsonObject.put(PROP_LOYALTY_CHARGED, this.getLoyaltyCharged());
		jsonObject.put(PROP_DISCOUNT_ID, getDiscountId());
		jsonObject.put(PROP_NAME, getName());
		jsonObject.put(PROP_TYPE, getType());
		jsonObject.put("originalType", getOriginalType()); //NON-NLS-1$
		jsonObject.put("originalValue", getOriginalValue()); //NON-NLS-1$
		jsonObject.put(PROP_AUTO_APPLY, isAutoApply());
		jsonObject.put(PROP_COUPON_QUANTITY, getCouponQuantity());
		jsonObject.put(PROP_MINIMUM_AMOUNT, getMinimumAmount());
		jsonObject.put(PROP_VALUE, getValue());
		jsonObject.put(PROP_TOTAL_DISCOUNT_AMOUNT, getTotalDiscountAmount());

		return jsonObject;
	}

	public Integer getLoyaltyPoint() {
		return loyaltyPoint;
	}

	public void setLoyaltyPoint(Integer loyaltyPoint) {
		this.loyaltyPoint = loyaltyPoint;
	}

	public Boolean getLoyaltyCharged() {
		return loyaltyCharged;
	}

	public void setLoyaltyCharged(Boolean loyaltyCharged) {
		this.loyaltyCharged = loyaltyCharged;
	}

	public String getTicketId() {
		return ticketId;
	}

	public void setTicketId(String ticketId) {
		this.ticketId = ticketId;
	}

	private String ticketId;


	public int getOriginalType() {
		return originalType;
	}

	public void setOriginalType(int originalType) {
		this.originalType = originalType;
	}

	public double getOriginalValue() {
		return originalValue;
	}

	public void setOriginalValue(double originalValue) {
		this.originalValue = originalValue;
	}
	
	public void calculatePercentageForAmount() {
		Double value = getValue() * getCouponQuantity();
		setValue(DiscountUtil.calculateDiscountPercentageFromAmount(getSubtotalAmount(), value));
	}
}