package com.floreantpos.model;

import java.math.BigDecimal;
import java.math.RoundingMode;

import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseTicketItemDiscount;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.util.DiscountUtil;
import com.floreantpos.util.NumberUtil;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "subTotalAmountWithoutModifiersDisplay" })
public class TicketItemDiscount extends BaseTicketItemDiscount implements ITicketItem {
	private static final long serialVersionUID = 1L;
	
	private int originalType;
	private double originalValue;
	private int tableRowNum;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public TicketItemDiscount() {
	}

	/**
	 * Constructor for primary key
	 */
	public TicketItemDiscount(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public TicketItemDiscount(java.lang.String id, String name, double quantity, double value, int type) {
		setId(id);
		setDiscountId(id);
		setName(name);
		setCouponQuantity(quantity);
		setValue(value);
		setType(type);
	}

	private double discountPercentage;

	public TicketItemDiscount(TicketItemDiscount fromDiscount) {
		setDiscountId(fromDiscount.getDiscountId());
		setName(fromDiscount.getName());
		setType(fromDiscount.getType());
		setAutoApply(fromDiscount.isAutoApply());
		setCouponQuantity(fromDiscount.getCouponQuantity());
		setMinimumAmount(fromDiscount.getMinimumAmount());
		setValue(fromDiscount.getValue());
		setAmount(fromDiscount.getAmount());
	}

	public double calculateDiscount(final double subtotalAmount) {
		double quantity = 1;
		double couponQuantity = getCouponQuantity();
		if (getMinimumAmount() > 1) {
			quantity = Math.abs(Math.floor(getTicketItem().getQuantity() / getMinimumAmount()));
			if (couponQuantity >= quantity) {
				couponQuantity = quantity;
			}
		}
		double discountAmount = DiscountUtil.calculateDiscountAmount(this, subtotalAmount) * couponQuantity;
		if (discountAmount > 0 && discountAmount > Math.abs(subtotalAmount))
			discountAmount = subtotalAmount;
		setAmount(discountAmount);
		return discountAmount;
	}

	public BigDecimal calculateDiscount(BigDecimal subtotalAmount) {
		BigDecimal quantity = NumberUtil.convertToBigDecimal("1"); //$NON-NLS-1$
		BigDecimal couponQuantity = NumberUtil.convertToBigDecimal(getCouponQuantity());
		BigDecimal minAmount = NumberUtil.convertToBigDecimal(getMinimumAmount());

		if (minAmount.compareTo(BigDecimal.ONE) > 0) {
			quantity = NumberUtil.convertToBigDecimal(getTicketItem().getQuantity()).divide(minAmount, 4, RoundingMode.HALF_DOWN);
			if (couponQuantity.compareTo(quantity) >= 0) {
				couponQuantity = quantity;
			}
		}
		BigDecimal discountAmount = DiscountUtil.calculateDiscountAmount(this, subtotalAmount).multiply(couponQuantity);
		if (discountAmount.compareTo(BigDecimal.ZERO) > 0 && discountAmount.compareTo(subtotalAmount) > 0)
			discountAmount = subtotalAmount;
		setAmount(discountAmount.doubleValue());
		return discountAmount;
	}

	public void setTableRowNum(int tableRowNum) {
		this.tableRowNum = tableRowNum;
	}

	public int getTableRowNum() {
		return tableRowNum;
	}

	@Override
	public String getItemCode() {
		return ""; //$NON-NLS-1$
	}

	@Override
	public String toString() {
		return getName();
	}

	@Override
	public String getNameDisplay() {
		double couponQuantity = getCouponQuantity();
		String display;
		if (couponQuantity > 1) {
			display = NumberUtil.trimDecilamIfNotNeeded(couponQuantity) + "x " + getName(); //$NON-NLS-1$
		}
		else {
			display = getName().trim();
		}
		return "  *" + display; //$NON-NLS-1$

	}

	@Override
	public boolean canAddCookingInstruction() {
		return false;
	}

	@Override
	public boolean canAddDiscount() {
		return false;
	}

	@Override
	public boolean canVoid() {
		return false;
	}

	@Override
	public boolean canAddAdOn() {
		return false;
	}

	@Override
	public Boolean isPrintedToKitchen() {
		return false;
	}

	@Override
	public String getUnitPriceDisplay() {
		return null;
	}

	@Override
	public String getItemQuantityDisplay() {
		return null;
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		return null;
	}

	@Override
	public void setDiscountAmount(Double amount) {

	}

	@Override
	public Double getDiscountAmount() {
		return null;
	}

	@Override
	public KitchenStatus getKitchenStatusValue() {
		return null;
	}

	@Override
	public Double getSubtotalAmount() {
		TicketItem ticketItem = getTicketItem();
		if (ticketItem != null && ticketItem.getDiscounts() != null && ticketItem.getDiscounts().size() == 1) {
			return -ticketItem.getDiscountAmount();
		}
		double discount = getAmount();
		if (discount <= 0)
			return 0.0; //NON-NLS-1$
		return -discount;
	}

	@Override
	public String getSubTotalAmountDisplay() {
		return NumberUtil.formatAmount(getSubtotalAmount());
	}

	public double getDiscountPercentage() {
		return discountPercentage;
	}

	public void setDiscountPercentage(double discountPercentage) {
		this.discountPercentage = discountPercentage;
	}

	public JSONObject toJson() {
		JSONObject jsonObject = new JSONObject();
		jsonObject.put(PROP_DISCOUNT_ID, getDiscountId());
		jsonObject.put(PROP_NAME, getName());
		jsonObject.put(PROP_TYPE, getType());
		jsonObject.put("originalType", getOriginalType()); //NON-NLS-1$
		jsonObject.put("originalValue", getOriginalValue()); //NON-NLS-1$
		jsonObject.put(PROP_AUTO_APPLY, isAutoApply());
		jsonObject.put(PROP_COUPON_QUANTITY, getCouponQuantity());
		jsonObject.put(PROP_MINIMUM_AMOUNT, getMinimumAmount());
		jsonObject.put(PROP_VALUE, getValue());
		jsonObject.put(PROP_AMOUNT, getAmount());
		return jsonObject;
	}

	public String getTicketItemId() {
		return ticketItemId;
	}

	public void setTicketItemId(String ticketItemId) {
		this.ticketItemId = ticketItemId;
	}

	private String ticketItemId;

	@Override
	public boolean isSaved() {
		return getTicketItem() == null ? false : getTicketItem().isSaved();
	}

	public int getOriginalType() {
		return originalType;
	}

	public void setOriginalType(int originalType) {
		this.originalType = originalType;
	}

	public double getOriginalValue() {
		return originalValue;
	}

	public void setOriginalValue(double originalValue) {
		this.originalValue = originalValue;
	}
}