package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.Address;
import com.floreantpos.model.Outlet;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class OutletDAO extends BaseOutletDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OutletDAO() {

	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime((Outlet) obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime((Outlet) obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime((Outlet) obj);
		super.saveOrUpdate(obj, s);
	}

	public Outlet initialize(Outlet outlet) {
		if (outlet == null || outlet.getId() == null)
			return outlet;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(outlet);

			Hibernate.initialize(outlet.getDepartments());

			return outlet;
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateOutlets(List<Outlet> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				Outlet item = (Outlet) iterator.next();
				Outlet existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List getOutletsWithSameAddressId(String fieldValue) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Outlet.class);
			criteria.add(Restrictions.eq(Outlet.PROP_ADDRESS_ID, fieldValue).ignoreCase());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public boolean isOutletSyncable(String outletId) {

		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (outletId != null) {
				criteria.add(Restrictions.eq(Outlet.PROP_ID, outletId));
			}

			Criterion nullUpdateTime = Restrictions.isNull(Outlet.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(Outlet.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(Outlet.PROP_LAST_UPDATE_TIME, Outlet.PROP_LAST_SYNC_TIME);

			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

			List result = criteria.list();
			if (result.size() == 0) {
				return false;
			}
			return true;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateOutlet(Outlet bean, Address address) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			AddressDAO.getInstance().saveOrUpdate(address, session);
			bean.setAddress(address);
			saveOrUpdate(bean, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> findAllOutletIds() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.setProjection(Projections.property(Outlet.PROP_ID));
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

}