/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.StoreAlreadyCloseException;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Address;
import com.floreantpos.model.DataSyncInfo;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.StoreSessionControl;
import com.floreantpos.model.User;
import com.floreantpos.util.StoreUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;
import com.orocube.siiopa.common.model.SubscriptionProduct;

public class StoreDAO extends BaseStoreDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public StoreDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public static Store getRestaurant() {
		return getInstance().get("1"); //$NON-NLS-1$
	}

	@Deprecated
	public static Date geServerTimestamp() {
		return getServerTimestamp();
	}

	public static Date getServerTimestamp() {
		//		Query query = getInstance().getSession().createQuery("select current_timestamp() from Store");
		//		query.setFirstResult(0);
		//		query.setMaxResults(1);
		//		Timestamp timestamp = (Timestamp) query.list().get(0);
		return new Date();
	}

	//	public void refresh(Store store) {
	//		Session session = null;
	//		try {
	//			session = getSession();
	//			session.refresh(store);
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public void closeStore(User closedBy) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();
			StoreSessionControl currentStoreOperation = StoreUtil.getCurrentStoreOperation();
			if (currentStoreOperation.getCurrentData() == null) {
				throw new StoreAlreadyCloseException("Store is already closed."); //$NON-NLS-1$
			}
			StoreSession currentData = currentStoreOperation.getCurrentData();
			currentData.setClosedBy(closedBy);
			currentData.setCloseTime(new Date());
			currentStoreOperation.setCurrentData(null);

			session.saveOrUpdate(currentData);
			session.saveOrUpdate(currentStoreOperation);

			String sqlDeleteItem = "delete from KITCHEN_TICKET_ITEM"; //$NON-NLS-1$
			String sqlDeleteTicketTables = "delete from KIT_TICKET_TABLE_NUM"; //$NON-NLS-1$
			String sqlKitchenTicket = "delete from KITCHEN_TICKET"; //$NON-NLS-1$

			Query query = session.createSQLQuery(sqlDeleteItem);
			query.executeUpdate();
			Query query2 = session.createSQLQuery(sqlDeleteTicketTables);
			query2.executeUpdate();
			Query query3 = session.createSQLQuery(sqlKitchenTicket);
			query3.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public Store getCloudRestaurant(String uId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			criteria.add(Restrictions.eq(Store.PROP_UUID, uId));

			return (Store) criteria.uniqueResult();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}
	
	public Store getStoreByFriendlyId(String friendlyId) {
		try (Session session = createNewSession()){
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Store.PROP_FRIENDLY_UID, friendlyId));
			criteria.setMaxResults(1);
			
			return (Store) criteria.uniqueResult();
		}
	}

	public void saveOrUpdateStore(List<Store> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean isDownload) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator<Store> iterator = dataList.iterator(); iterator.hasNext();) {
				Store store = (Store) iterator.next();
				Store existingStore = get(store.getId());
				if (existingStore != null) {
					if (!BaseDataServiceDao.get().shouldSave(store.getLastUpdateTime(), existingStore.getLastUpdateTime())) {
						PosLog.info(getClass(), store.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
				}
				existingStore.setName(store.getName());
				existingStore.setEnableFloridaTaxRule(store.isEnableFloridaTaxRule());
				existingStore.setShowVoidedItemsOnTicket(store.isShowVoidedItemsOnTicket());
				existingStore.addProperty(AppConstants.SETUP_COMPLETE, store.getProperty(AppConstants.SETUP_COMPLETE));
				saveOrUpdate(existingStore, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void addOrUpdateStoreProperties(Session session, Store store, Store existingStore, Map<String, String> properties, boolean isDownload) {
		Map<String, String> existingProperties = null;

		List<String> subscriptionProductList = new ArrayList<>();
		SubscriptionProduct[] values = SubscriptionProduct.values();
		for (SubscriptionProduct subscriptionProduct : values) {
			subscriptionProductList.add(subscriptionProduct.getId());
		}

		if (existingStore != null) {
			existingProperties = existingStore.getProperties();
		}

		for (Map.Entry<String, String> propertyValue : properties.entrySet()) {
			String key = propertyValue.getKey();
			String existingPropertyValue = existingProperties.get(key);
			if (!isDownload) {
				if (StringUtils.isNotEmpty(existingPropertyValue)) {
					String startString = StringUtils.substringBefore(key, "."); //$NON-NLS-1$
					if (subscriptionProductList.contains(startString) || startString.contains("database")) {//$NON-NLS-1$
						continue;
					}
					addProperty(existingProperties, propertyValue, existingPropertyValue);
				}
				else {
					String extension = key.substring(key.lastIndexOf(".") + 1); //$NON-NLS-1$
					if (!extension.equals("lastUpdateTime")) { //$NON-NLS-1$
						addProperty(existingProperties, propertyValue, propertyValue.getValue());
					}
				}
			}
			else {
				String extension = key.substring(key.lastIndexOf(".") + 1); //$NON-NLS-1$
				if (!extension.equals("lastUpdateTime")) { //$NON-NLS-1$
					addProperty(existingProperties, propertyValue, existingPropertyValue);
				}
			}
		}
	}

	private void addProperty(Map<String, String> existingProperties, Map.Entry<String, String> propertyValue, String value) {
		if (StringUtils.isNotEmpty(value)) {
			existingProperties.put(propertyValue.getKey(), propertyValue.getValue());
		}
		else {
			existingProperties.putIfAbsent(propertyValue.getKey(), propertyValue.getValue());
		}
	}

	private void saveOrUpdateStoreAddress(Session session, Store item, Address address) {
		if (address == null) {
			return;
		}
		AddressDAO instance = AddressDAO.getInstance();

		Address existingAddress = instance.get(address.getId());
		if (existingAddress == null) {
			instance.save(address);
		}
		else {
			address.setVersion(existingAddress.getVersion());
		}
	}

	private void saveOrUpdateStore(Session session, Store store, Store existingStore) {
		if (existingStore == null) {
			//save(store, session);
		}
		else {
			store.setVersion(existingStore.getVersion());
			store.setUuid(existingStore.getUuid());
			store.setDefaultOutletId(existingStore.getDefaultOutletId());
		}
	}

	public void removeLastUpdateTime() {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			Store store = session.get(Store.class, "1"); //$NON-NLS-1$
			removeLastUpdateTime(store);
			transaction.commit();
		}
	}

	public void removeLastUpdateTime(Store store) {
		Map<String, String> properties = store.getProperties();
		List<String> keys = new ArrayList<>(properties.keySet());
		keys.forEach(key -> {
			if (key.endsWith("lastUpdateTime")) { //$NON-NLS-1$
				properties.remove(key);
			}
		});
		properties.remove(Store.PROP_SYNC_CONFLICT_RESOLVE_OPTION);
	}

	public void updateLastUpdateTimeProperty(Date currentDate, List<DataSyncInfo> syncInfos) {
		Store store = StoreDAO.getRestaurant();
		for (DataSyncInfo dataSyncInfo : syncInfos) {
			GenericDAO.getInstance().updateLastUpdateTimeProperty(store, dataSyncInfo.getBeanClass(), currentDate);
		}
		StoreDAO.getInstance().saveOrUpdate(store);
	}
}