package com.floreantpos.print;

import java.awt.print.PrinterAbortException;
import java.io.File;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.ext.ReciptPaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.KitchenTicketDataSource;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.report.TicketDataSource;
import com.floreantpos.report.TicketPrintProperties;
import com.floreantpos.ui.dialog.POSMessageDialog;
import com.floreantpos.ui.views.TicketReceiptView;
import com.floreantpos.util.PrintServiceUtil;

import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;
import net.sf.jasperreports.engine.export.JRPrintServiceExporter;
import net.sf.jasperreports.export.SimpleExporterInput;
import net.sf.jasperreports.export.SimplePrintServiceExporterConfiguration;

public class JasperPrintService extends ReceiptPrintServiceProvider {

	@Override
	public void printTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		JasperPrint jasperPrint = createPrint(ticket, map, null);
		jasperPrint.setName(ReceiptPrintService.ORDER_ + ticket.getId() + deviceName);
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void printTransaction(Ticket ticket, HashMap map, PosTransaction transaction, String deviceName) throws Exception {
		JasperPrint jasperPrint = createPrint(ticket, map, transaction);
		jasperPrint.setName(ReceiptPrintService.ORDER_ + ticket.getId() + deviceName);
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void printRefundTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		JasperPrint jasperPrint = createRefundPrint(ticket, map);
		jasperPrint.setName("REFUND_" + ticket.getId()); //$NON-NLS-1$
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void printVoidTicket(Ticket ticket, HashMap map, String deviceName) throws Exception {
		JasperPrint jasperPrint = createPrint(ticket, map, null);
		jasperPrint.setName("VOID_" + ticket.getId()); //$NON-NLS-1$
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void printKitchenTicket(KitchenTicket kitchenTicket, HashMap map, Ticket ticket, String printerName, String deviceName) throws Exception {
		JasperPrint jasperPrint = createKitchenPrint(map, printerName, kitchenTicket, deviceName);
		jasperPrint.setName("KitchenReceipt_" + ticket.getId() + "_" + kitchenTicket.getSequenceNumber()); //$NON-NLS-1$ //$NON-NLS-2$
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void printVoidKitchenTicket(KitchenTicket kitchenTicket, HashMap map, String printerName, String deviceName) throws Exception {
		KitchenTicketDataSource dataSource = new KitchenTicketDataSource(kitchenTicket);
		JasperPrint jasperPrint = createJasperPrint(ReportUtil.getReport("kitchen-receipt"), map, new JRTableModelDataSource(dataSource)); // $NON-NLS-1$
		jasperPrint.setName("Void_KitchenReceipt_" + kitchenTicket.getId() + "_" + kitchenTicket.getSequenceNumber()); //$NON-NLS-1$ //$NON-NLS-2$
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

	@Override
	public void rendererTicketReceipt(Ticket ticket, JPanel reportPanel) throws Exception {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); //$NON-NLS-1$ //$NON-NLS-2$
		HashMap map = ReceiptPrintService.populateTicketProperties(ticket, printProperties, null);
		map.put(JRParameter.IS_IGNORE_PAGINATION, true);
		JasperPrint jasperPrint = createPrint(ticket, map, null, false);
		TicketReceiptView receiptView = new TicketReceiptView(jasperPrint);
		reportPanel.add(receiptView.getReportPanel());
	}

	@Override
	public void rendererPosTransactionReceipt(PosTransaction transaction, JPanel reportPanel) throws Exception {
		JasperPrint jasperPrint = createTransactionReceipt(transaction);
		TicketReceiptView receiptView = new TicketReceiptView(jasperPrint);
		reportPanel.add(receiptView.getReportPanel());
	}

	@Override
	public void rendererKitchenReceipt(Ticket ticket, JPanel reportPanel) throws Exception {
		JasperPrint jasperPrint = getKitchenJasperPrintPreview(ticket);
		TicketReceiptView receiptView = new TicketReceiptView(jasperPrint);
		reportPanel.add(receiptView.getReportPanel());
	}

	public JasperPrint createJasperPrint(JasperReport report, Map<String, Object> properties, JRDataSource dataSource) throws Exception {
		JasperPrint jasperPrint = JasperFillManager.fillReport(report, properties, dataSource);
		return jasperPrint;
	}

	public JasperPrint createPrint(Ticket ticket, Map<String, Object> map, PosTransaction transaction) throws Exception {
		return createPrint(ticket, map, transaction, false);
	}

	public JasperPrint createPrint(Ticket ticket, Map<String, Object> map, PosTransaction transaction, boolean includeSeat) throws Exception {
		TicketDataSource dataSource = new TicketDataSource(ticket, includeSeat);

		// main recipt display which paper size was selected
		ReciptPaperSize paperSize = getReceiptPaperSize(AppConstants.MAIN_RECIPT_PAPER_SIZE_IN_MM);
		return createJasperPrint(ReportUtil.getReport(paperSize.getReportNameAccording2Size("ticket-recipt")), map, //$NON-NLS-1$
				new JRTableModelDataSource(dataSource));
	}

	public JasperPrint createRefundPrint(Ticket ticket, HashMap map) throws Exception {
		TicketDataSource dataSource = new TicketDataSource(ticket);

		ReciptPaperSize paperSize = getReceiptPaperSize(AppConstants.MAIN_RECIPT_PAPER_SIZE_IN_MM);
		return createJasperPrint(ReportUtil.getReport(paperSize.getReportNameAccording2Size("ticket-recipt")), map, //$NON-NLS-1$
				new JRTableModelDataSource(dataSource));
	}

	public JasperPrint createTransactionReceipt(PosTransaction transaction) {
		try {
			Ticket ticket = transaction.getTicket();

			TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.3"), true, true, true); //$NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = ReceiptPrintService.populateTicketProperties(ticket, printProperties, transaction);
			map.put(JRParameter.IS_IGNORE_PAGINATION, true);
			if (transaction != null && transaction.isCard()) {

				CardReader cardReader = CardReader.fromString(transaction.getCardReader());

				if (cardReader == CardReader.EXTERNAL_TERMINAL) {
					return null;
				}

				map.put(ReceiptPrintService.CARD_PAYMENT, true); //$NON-NLS-1$
				map.put(ReceiptPrintService.COPY_TYPE, Messages.getString("ReceiptPrintService.4")); //$NON-NLS-1$ //$NON-NLS-2$
				JasperPrint jasperPrint = createPrint(ticket, map, transaction);
				jasperPrint.setName("Ticket-" + ticket.getId() + "-CustomerCopy"); //$NON-NLS-1$ //$NON-NLS-2$
				jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());

				map.put(ReceiptPrintService.COPY_TYPE, Messages.getString("ReceiptPrintService.5")); //$NON-NLS-1$ //$NON-NLS-2$
				jasperPrint = createPrint(ticket, map, transaction);
				jasperPrint.setName("Ticket-" + ticket.getId() + "-MerchantCopy"); //$NON-NLS-1$ //$NON-NLS-2$
				jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());
				return jasperPrint;
			}
			else {
				JasperPrint jasperPrint = createPrint(ticket, map, transaction);
				jasperPrint.setName("Ticket-" + ticket.getId()); //$NON-NLS-1$
				jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());
				return jasperPrint;
			}
		} catch (Exception e) {
			ReceiptPrintService.getLogger().error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
		return null;
	}

	private static ReciptPaperSize getReceiptPaperSize(String reciptPaperSizeInMm) {
		return ReceiptPrintService.getReceiptPaperSize(reciptPaperSizeInMm);
	}

	public JasperPrint createKitchenPrint(KitchenTicket ticket) throws Exception {
		HashMap map = new HashMap();

		map.put(ReceiptPrintService.CARD_PAYMENT, true); //$NON-NLS-1$
		map.put(ReceiptPrintService.SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		map.put(ReceiptPrintService.SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		KitchenTicketDataSource dataSource = new KitchenTicketDataSource(ticket);

		// kitchen receipt display which paper size was selected
		ReciptPaperSize paperSize = getReceiptPaperSize(AppConstants.KITCHEN_RECIPT_PAPER_SIZE_IN_MM);

		return createJasperPrint(ReportUtil.getReport(paperSize.getReportNameAccording2Size("kitchen-receipt")), map, //$NON-NLS-1$
				new JRTableModelDataSource(dataSource));
	}

	public JasperPrint createKitchenPrint(HashMap map, String virtualPrinterName, KitchenTicket ticket, String deviceName) throws Exception {
		return createKitchenPrint(map, virtualPrinterName, ticket, deviceName, false);
	}

	public JasperPrint createKitchenPrint(HashMap map, String virtualPrinterName, KitchenTicket ticket, String deviceName, boolean ignorePagination)
			throws Exception {
		KitchenTicketDataSource dataSource = new KitchenTicketDataSource(ticket);
		// kitchen receipt display which paper size was selected
		String reciptName = "kitchen-receipt"; //$NON-NLS-1$
		ReciptPaperSize paperSize = getReceiptPaperSize(AppConstants.KITCHEN_RECIPT_PAPER_SIZE_IN_MM);
		if (Application.getInstance().getTerminal().isGroupByCatagoryKitReceipt()) {
			reciptName = "kitchen-receipt-with-group"; //$NON-NLS-1$
		}
		return createJasperPrint(ReportUtil.getReport(paperSize.getReportNameAccording2Size(reciptName)), map, new JRTableModelDataSource(dataSource)); // $NON-NLS-1$
	}

	public static void printQuitely(JasperPrint jasperPrint) throws JRException {
		try {
			DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss"); //$NON-NLS-1$
			Date date = new Date();
			String dateTime = dateFormat.format(date);

			String printerName = jasperPrint.getProperty(ReceiptPrintService.PROP_PRINTER_NAME);
			if (printerName != null && printerName.equals(ReceiptPrintService.OROPOS_PDF_PRINTER)) {

				String fileName = jasperPrint.getName().replace(printerName, "") + "-[" + dateTime + "]"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				File pdfFile = new File(ReceiptPrintService.getPdfPrinterDir(), fileName + ".pdf"); //$NON-NLS-1$
				JasperExportManager.exportReportToPdfFile(jasperPrint, pdfFile.getAbsolutePath());
			}
			else {
				SimplePrintServiceExporterConfiguration configuration = new SimplePrintServiceExporterConfiguration();
				configuration.setPrintService(PrintServiceUtil.getPrintServiceForPrinter(jasperPrint.getProperty(ReceiptPrintService.PROP_PRINTER_NAME)));

				JRPrintServiceExporter exporter = new JRPrintServiceExporter();
				exporter.setExporterInput(new SimpleExporterInput(jasperPrint));
				exporter.setConfiguration(configuration);
				exporter.exportReport();
			}
		} catch (Exception x) {
			if (x != null && x.getCause() instanceof PrinterAbortException) {
				// do nothing
			}
			else {
				String msg = "No print selected\n"; //$NON-NLS-1$
				ReceiptPrintService.getLogger().error(msg + x);
			}
		}
	}

	private JasperPrint getKitchenJasperPrintPreview(Ticket ticket) {
		try {
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, false);
			for (KitchenTicket kitchenTicket : kitchenTickets) {
				kitchenTicket.setParentTicket(ticket);
				HashMap map = ReceiptPrintService.populateKitchenTicketProperties(kitchenTicket, "", "", true);
				JasperPrint jasperPrint = createKitchenPrint(map, "", kitchenTicket, "", true); //$NON-NLS-1$ //$NON-NLS-2$
				jasperPrint.setName("FP_KitchenReceipt_" + ticket.getId() + "_" + kitchenTicket.getSequenceNumber()); //$NON-NLS-1$ //$NON-NLS-2$
				return jasperPrint;
			}
		} catch (Exception e) {
			throw new PosException(e);
		}
		return null;
	}

	public void printGenericReport(HashMap<String, Object> map) throws Exception {
		JasperPrint jasperPrint = createJasperPrint(ReportUtil.getReport("generic-receipt"), map, //$NON-NLS-1$
				new JREmptyDataSource());
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());
		printQuitely(jasperPrint);
	}

	public void testPrinter(HashMap<String, Object> map, String deviceName) throws Exception {
		JasperPrint jasperPrint = createJasperPrint(ReportUtil.getReport("test-printer"), map, new JREmptyDataSource()); //$NON-NLS-1$
		jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, deviceName);
		printQuitely(jasperPrint);
	}

}
