package com.orocube.rest.service;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.eclipse.persistence.jaxb.MarshallerProperties;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.PosLog;
import com.floreantpos.main.Application;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.ComboTicketItem;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.Department;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.ModifiableTicketItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.ReversalTransaction;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemCookingInstruction;
import com.floreantpos.model.TicketItemDiscount;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.User;
import com.floreantpos.model.VoidTransaction;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.webservice.RuntimeTypeAdapterFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.orocube.rest.service.mqtt.MessageType;
import com.orocube.rest.service.mqtt.MqttSender;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ServiceUtils {

	public static String convertToJson(Object element) throws Exception {
		if (element == null)
			return ""; //$NON-NLS-1$

		String resp = ""; //$NON-NLS-1$
		try {
			System.setProperty("javax.xml.bind.context.factory", "org.eclipse.persistence.jaxb.JAXBContextFactory"); //$NON-NLS-1$ //$NON-NLS-2$

			JAXBContext messageContext = JAXBContext.newInstance(element.getClass());
			Marshaller marshaller = messageContext.createMarshaller();

			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			marshaller.setProperty(MarshallerProperties.MEDIA_TYPE, "application/json"); //$NON-NLS-1$
			marshaller.setProperty(MarshallerProperties.JSON_INCLUDE_ROOT, true);

			StringWriter dataWriter = new StringWriter();
			marshaller.marshal(element, dataWriter);

			resp = dataWriter.toString();

			//String len = String.format("%05d", resp.length()); //$NON-NLS-1$
			//resp = len + resp;
			return resp;
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return ""; //$NON-NLS-1$
	}

	public static String getAsJsonString(PosResponse element) throws Exception {
		return getAsJsonString(element, StoreSession.class, MenuShift.class, TicketItemCookingInstruction.class, Department.class);
	}

	public static String getAsJsonString(PosResponse element, boolean excludeCashDrawer) throws Exception {
		if (excludeCashDrawer)
			return getAsJsonString(element, CashDrawer.class, StoreSession.class, MenuShift.class, TicketItemCookingInstruction.class, Department.class);
		else
			return getAsJsonString(element, StoreSession.class, MenuShift.class, TicketItemCookingInstruction.class, Department.class);
	}

	public static String getAsJsonString(PosResponse element, Class... exclusionClasses) throws Exception {
		if (element == null)
			return ""; //$NON-NLS-1$

		try {
			Gson gson = BaseDataServiceDao.getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(exclusionClasses)).create();
			String json = gson.toJson(element);
			return json;
		} catch (Exception ex) {
			PosLog.error(ServiceUtils.class, ex);
		}
		return ""; //$NON-NLS-1$
	}

	public static String convertKitTicketsToJsonString(List<KitchenTicket> kitchenTickets) throws Exception {
		List<KitchenTicket> exportKitchenTickets = new ArrayList<>();
		for (KitchenTicket kitchenTicket : kitchenTickets) {
			if (kitchenTicket.getId() == null) {
				continue;
			}
			kitchenTicket.setParentTicket(null);
			kitchenTicket.setPrinter(null);
			List<KitchenTicketItem> ticketItems = kitchenTicket.getTicketItems();
			if (ticketItems == null) {
				continue;
			}
			for (KitchenTicketItem kitchenTicketItem : ticketItems) {
				kitchenTicketItem.setKitchenTicket(null);
			}
			exportKitchenTickets.add(kitchenTicket);
		}
		JSONObject request = new JSONObject();
		request.put("deviceId", String.valueOf(DataProvider.get().getCurrentTerminal().getId())); //$NON-NLS-1$
		request.put("storeId", DataProvider.get().getStore().getId()); //$NON-NLS-1$
		//		request.put("outletId", Application.getInstance().getCurrentTerminalOutletId()); //$NON-NLS-1$
		request.put("request", MessageType.KITCHEN_TICKET); //$NON-NLS-1$
		request.put("responseCode", 1); //$NON-NLS-1$

		try {
			Gson gson = BaseDataServiceDao.getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(User.class, StoreSession.class, Terminal.class,
					MenuShift.class, TicketItemCookingInstruction.class, Department.class, MenuItem.class, Attribute.class)).create();
			String json = gson.toJson(exportKitchenTickets);
			request.put("data", "%data"); //$NON-NLS-1$ //$NON-NLS-2$
			return request.toString().replace("\"%data\"", json); //$NON-NLS-1$
		} catch (Exception ex) {
			PosLog.error(ServiceUtils.class, ex);
		}
		return null;
	}

	public static String convertTicketsToJson(Object tickets) throws Exception {
		if (tickets == null)
			return ""; //$NON-NLS-1$

		try {
			Gson gson = BaseDataServiceDao.getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(User.class, Outlet.class, StoreSession.class,
					Terminal.class, MenuShift.class, TicketItemCookingInstruction.class, Department.class)).create();
			String json = gson.toJson(tickets);
			return json;
		} catch (Exception ex) {
			PosLog.error(ServiceUtils.class, ex);
		}
		return ""; //$NON-NLS-1$
	}

	public static Ticket convertToObject(String element) throws Exception {
		if (element == null)
			return null;

		try {
			System.setProperty("javax.xml.bind.context.factory", "org.eclipse.persistence.jaxb.JAXBContextFactory"); //$NON-NLS-1$ //$NON-NLS-2$

			JAXBContext messageContext = JAXBContext.newInstance(element.getClass());
			Unmarshaller unmarshaller = messageContext.createUnmarshaller();

			//unmarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			//unmarshaller.setProperty(MarshallerProperties.MEDIA_TYPE, "application/json");
			//unmarshaller.setProperty(MarshallerProperties.JSON_INCLUDE_ROOT, true);

			StringReader reader = new StringReader(element);
			Ticket ticket = (Ticket) unmarshaller.unmarshal(reader);
			return ticket;
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return null;
	}

	public static String convertImageIconToString(ImageIcon imageIcon) {
		String imageString = ""; //$NON-NLS-1$
		if (imageIcon != null) {
			try {

				Image image = imageIcon.getImage();
				BufferedImage bufferedImage = new BufferedImage(image.getWidth(null), image.getHeight(null), BufferedImage.TYPE_INT_ARGB);

				Graphics2D bGr = bufferedImage.createGraphics();
				bGr.drawImage(image, 0, 0, null);
				bGr.dispose();
				File outputfile = new File("saved.png"); //$NON-NLS-1$
				ImageIO.write(bufferedImage, "png", outputfile); //$NON-NLS-1$

				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				ImageIO.write(bufferedImage, "png", baos); //$NON-NLS-1$
				baos.flush();
				byte[] byteArray = baos.toByteArray();
				byte[] encoded_data = Base64.encodeBase64(byteArray);
				imageString = new String(encoded_data);
				baos.close();
			} catch (IOException e) {
				System.out.println(e.getMessage());
			}
		}
		return imageString;
	}

	public static List<Integer> convertJsonToIntList(String request, String key) {
		try {
			JSONObject jsonObj = new JSONObject(request);

			String valueString = String.valueOf(jsonObj.get(key));
			return convertStringArrayToList(valueString);
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return null;
	}

	public static List<ShopTableStatus> convertJsonToShopTableStatusList(String request) {
		try {
			if (request == null)
				return null;
			List<ShopTableStatus> shopTableStatusList = new ArrayList<>();
			JSONArray jsonArray = new JSONArray(request);
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject json = jsonArray.getJSONObject(i);
				ShopTableStatus shopTableStatus = new GsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create()
						.fromJson(json.toString(), ShopTableStatus.class);
				if (shopTableStatus != null) {
					shopTableStatusList.add(shopTableStatus);
				}
			}
			return shopTableStatusList;
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return null;
	}

	public static List<CashDrawer> convertJsonToCashDrawerList(String request) {
		try {
			if (request == null)
				return null;
			List<CashDrawer> cashDrawers = new ArrayList<CashDrawer>();
			JSONArray jsonArray = new JSONArray(request);
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject json = jsonArray.getJSONObject(i);
				CashDrawer cashDrawer = new GsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create()
						.fromJson(json.toString(), CashDrawer.class);
				if (cashDrawer != null) {
					cashDrawers.add(cashDrawer);
				}
			}
			return cashDrawers;
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return null;
	}

	public static List<Integer> convertStringArrayToList(String stringArray) {
		List<Integer> list = new ArrayList<Integer>();
		if (stringArray != null) {
			stringArray = stringArray.replace("[", "").replace("]", "").replaceAll("\"", Matcher.quoteReplacement("")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
			String[] splitedTables = stringArray.split(","); //$NON-NLS-1$

			for (int i = 0; i < splitedTables.length; i++) {
				Integer integerValue = Integer.valueOf(splitedTables[i]);
				list.add(integerValue);
			}
		}
		return list;
	}

	public static String createPosResponseAsString(List<Ticket> tickets) throws Exception {
		return createPosResponseAsString(tickets, Application.getInstance().getMqttDeviceId());
	}

	public static String createPosResponseAsString(List<Ticket> tickets, String deviceId) throws Exception {
		PosResponse reponse = createPosResponse(tickets, deviceId, MqttSender.OROPOS);
		String jsonResponse = convertTicketsToJson(reponse);
		return jsonResponse;
	}

	public static String createPosResponseAsString(List<Ticket> tickets, String deviceId, boolean publishToMobileAndOtherDbClients, MqttSender sender)
			throws Exception {
		PosResponse reponse = createPosResponse(tickets, deviceId, sender);
		if (publishToMobileAndOtherDbClients) {
			Integer masterTerminal = TerminalDAO.getInstance().getMasterTerminalId();
			reponse.setBlockTerminalId(masterTerminal);
		}
		String jsonResponse = convertTicketsToJson(reponse);
		return jsonResponse;
	}

	public static PosResponse createPosResponse(List<Ticket> tickets, String deviceId, MqttSender sender) throws Exception {
		PosResponse reponse = new PosResponse();
		reponse.setResponseCode(1);
		reponse.setSource(sender.name());
		reponse.setRequest(MessageType.TICKET); //$NON-NLS-1$
		reponse.setMsg((tickets.size() == 1 ? Messages.getString("ServiceUtils.30") : tickets.size() + Messages.getString("ServiceUtils.31")) //$NON-NLS-1$//$NON-NLS-2$
				+ Messages.getString("ServiceUtils.32")); //$NON-NLS-1$
		reponse.setDeviceId(deviceId);
		Terminal currentTerminal = Application.getInstance().getTerminal();
		String currentOutletId = currentTerminal == null ? null : currentTerminal.getOutletId();
		reponse.setOutletId(StringUtils.isEmpty(currentOutletId) ? DataProvider.get().getStore().getDefaultOutletId() : currentOutletId);
		reponse.setData(tickets);
		for (Ticket ticket : tickets) {
			TicketDAO.getInstance().loadFullTicket(ticket);
			if (ticket.getTerminalId() == 0 && currentTerminal != null) {
				ticket.setTerminalId(currentTerminal.getId());
			}
			Set<PosTransaction> transactions = ticket.getTransactions();
			if (transactions != null && transactions.size() > 0) {
				for (PosTransaction posTransaction : transactions) {
					updatePosTransactionUsingClassType(posTransaction);
					posTransaction.setTicket(null);
					posTransaction.setTicketId(ticket.getId());
				}
			}

			OrderType orderType = ticket.getOrderType();
			orderType.setDepartments(null);
			orderType.setCategories(null);
			orderType.setTerminalTypes(null);

			Outlet outlet = ticket.getOutlet();
			if (outlet != null)
				outlet.setDepartments(null);
			Terminal terminal = ticket.getTerminal();
			if (terminal != null) {
				terminal.setOutlet(null);
				terminal.setCurrentCashDrawer(null);
				terminal.setTerminalType(null);
			}
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				doRemoveTicketItemsLink(ticketItem, Boolean.FALSE);
			}
		}
		return reponse;
	}

	private static void doRemoveTicketItemsLink(TicketItem ticketItem, boolean isComboItem) {
		if (isComboItem) {
			ticketItem.setParentTicketItem(null);
		}
		else {
			ticketItem.setTicket(null);
		}
		String menuItemId = ticketItem.getMenuItemId();
		ticketItem.setMenuItem(null);
		ticketItem.setMenuItemId(menuItemId);

		TicketItemModifier sizeModifier = ticketItem.getSizeModifier();
		if (sizeModifier != null) {
			sizeModifier.setTicketItem(null);
		}

		List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				ticketItemModifier.setTicketItem(null);
			}
		}

		List<TicketItemDiscount> discountList = ticketItem.getDiscounts();
		if (discountList != null && discountList.size() > 0) {
			for (TicketItemDiscount ticketDiscount : discountList) {
				ticketDiscount.setTicketItem(null);
			}
		}

		if (ticketItem.isComboItem()) {
			List<TicketItem> comboTicketItems = ticketItem.getComboItems();
			if (comboTicketItems != null) {
				for (TicketItem cTicketItem : comboTicketItems) {
					doRemoveTicketItemsLink(cTicketItem, Boolean.TRUE);
				}
			}
		}
	}

	public static void updatePosTransactionUsingClassType(PosTransaction t) throws Exception {

		if (t instanceof CashTransaction) {
			t.setClassType("CASH"); //$NON-NLS-1$
		}
		else if (t instanceof CustomPaymentTransaction) {
			t.setClassType("CUSTOM_PAYMENT"); //$NON-NLS-1$
		}
		else if (t instanceof GiftCertificateTransaction) {
			t.setClassType("GIFT_CERT"); //$NON-NLS-1$
		}
		else if (t instanceof CreditCardTransaction) {
			t.setClassType("CREDIT_CARD"); //$NON-NLS-1$
		}
		else if (t instanceof DebitCardTransaction) {
			t.setClassType("DEBIT_CARD"); //$NON-NLS-1$
		}
		else if (t instanceof CashDropTransaction) {
			t.setClassType("CASH_DROP"); //$NON-NLS-1$
		}
		else if (t instanceof RefundTransaction) {
			t.setClassType("REFUND"); //$NON-NLS-1$
		}
		else if (t instanceof PayOutTransaction) {
			t.setClassType("PAY_OUT"); //$NON-NLS-1$
		}
		else if (t instanceof VoidTransaction) {
			t.setClassType("VOID_TRANS"); //$NON-NLS-1$
		}
		else if (t instanceof CustomerAccountTransaction) {
			t.setClassType("CUSTOMER_ACCOUNT"); //$NON-NLS-1$
		}
		else if (t instanceof ReversalTransaction) {
			t.setClassType("REVERSAL"); //$NON-NLS-1$
		}
	}

	public static PosTransaction convertPosTransactionUsingClassType(PosTransaction sourceTrans) throws Exception {
		String classType = sourceTrans.getClassType();

		PosTransaction transaction = null;
		switch (classType) {
			case "CASH": //$NON-NLS-1$
				transaction = new CashTransaction();
				break;
			case "CUSTOM_PAYMENT": //$NON-NLS-1$
				transaction = new CustomPaymentTransaction();
				break;
			case "GIFT_CERT": //$NON-NLS-1$
				transaction = new GiftCertificateTransaction();
				break;
			case "CREDIT_CARD": //$NON-NLS-1$
				transaction = new CreditCardTransaction();
				break;
			case "DEBIT_CARD": //$NON-NLS-1$
				transaction = new DebitCardTransaction();
				break;
			case "CASH_DROP": //$NON-NLS-1$
				transaction = new CashDropTransaction();
				break;
			case "REFUND": //$NON-NLS-1$
				transaction = new RefundTransaction();
				break;
			case "PAY_OUT": //$NON-NLS-1$
				transaction = new PayOutTransaction();
				break;
			case "VOID_TRANS": //$NON-NLS-1$
				transaction = new VoidTransaction();
				break;
			case "CUSTOMER_ACCOUNT": //$NON-NLS-1$
				transaction = new CustomerAccountTransaction();
				break;
			case "REVERSAL": //$NON-NLS-1$
				transaction = new ReversalTransaction();
				break;
			default:
				break;
		}
		try {
			PropertyUtils.copyProperties(transaction, sourceTrans);
		} catch (Exception e) {
			transaction = (PosTransaction) copy(sourceTrans, transaction.getClass());
		}
		transaction.setVoided(sourceTrans.isVoided());
		transaction.setAuthorizable(sourceTrans.isAuthorizable());
		transaction.setCaptured(sourceTrans.isCaptured());
		transaction.setMarkedCaptured(sourceTrans.isMarkedCaptured());
		return transaction;
	}

	private static Object copy(Object source, Class targetClass) {
		try {
			Gson gson = new GsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create();
			return gson.fromJson(gson.toJson(source), targetClass);
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return source;
	}

	public static byte[] convertImageResourceToByteArray(ImageResource imageResource) {
		return Base64.encodeBase64(imageResource.getImageBytes());
	}

	/**
	 * @param ticket
	 * 
	 * removed ticket from ticket item then convert to Gson
	 * @return
	 * @throws Exception
	 */
	public static String convertTicketToJson(Ticket ticket) throws Exception {
		if (ticket == null)
			return ""; //$NON-NLS-1$

		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			String menuItemId = ticketItem.getMenuItemId();
			ticketItem.setMenuItem(null);
			ticketItem.setMenuItemId(menuItemId);

			if (ticketItem.isHasModifiers()) {
				List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
				for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
					ticketItemModifier.setTicketItem(null);
				}
			}
			ticketItem.setTicket(null);
		}

		Gson gson = getTicketGson();

		return gson.toJson(ticket);
	}

	/**
	 * @param ticketJson
	 * 
	 *  set ticket in ticket items after converted to Ticket
	 *  
	 * @return
	 * @throws Exception
	 */
	public static Ticket convertJsonToTicket(String ticketJson) throws Exception {
		if (StringUtils.isBlank(ticketJson))
			return null;

		Gson gson = getTicketGson();

		Ticket ticket = (Ticket) gson.fromJson(ticketJson, Ticket.class);
		if (ticket != null) {
			List<TicketItem> ticketItems = ticket.getTicketItems();
			for (TicketItem ticketItem : ticketItems) {
				ticketItem.setTicket(ticket);
				if (ticketItem.isHasModifiers()) {
					for (TicketItemModifier ticketItemModifier : ticketItem.getTicketItemModifiers()) {
						ticketItemModifier.setTicketItem(ticketItem);
					}
				}
			}
		}
		return ticket;
	}

	public static Gson getTicketGson() throws Exception {
		//@formatter:off
		RuntimeTypeAdapterFactory<TicketItem> ticketItemAdapterFactory = RuntimeTypeAdapterFactory.of(TicketItem.class, "type") //$NON-NLS-1$
				.registerSubtype(TicketItem.class, "TicketItem") //$NON-NLS-1$
				.registerSubtype(ModifiableTicketItem.class, "ModifiableTicketItem") //$NON-NLS-1$
				.registerSubtype(ComboTicketItem.class, "ComboTicketItem"); //$NON-NLS-1$
		
		RuntimeTypeAdapterFactory<PosTransaction> transactionAdapterFactory = RuntimeTypeAdapterFactory.of(PosTransaction.class, "type") //$NON-NLS-1$
				.registerSubtype(CashTransaction.class, "CashTransaction") //$NON-NLS-1$
				.registerSubtype(CustomPaymentTransaction.class, "CustomPaymentTransaction") //$NON-NLS-1$
				.registerSubtype(GiftCertificateTransaction.class, "GiftCertificateTransaction") //$NON-NLS-1$
				.registerSubtype(CreditCardTransaction.class, "CreditCardTransaction") //$NON-NLS-1$
				.registerSubtype(DebitCardTransaction.class, "DebitCardTransaction") //$NON-NLS-1$
				.registerSubtype(CashDropTransaction.class, "CashDropTransaction") //$NON-NLS-1$
				.registerSubtype(RefundTransaction.class, "RefundTransaction") //$NON-NLS-1$
				.registerSubtype(PayOutTransaction.class, "PayOutTransaction") //$NON-NLS-1$
				.registerSubtype(VoidTransaction.class, "VoidTransaction") //$NON-NLS-1$
				.registerSubtype(CustomerAccountTransaction.class, "CustomerAccountTransaction") //$NON-NLS-1$
				.registerSubtype(ReversalTransaction.class, "ReversalTransaction") //$NON-NLS-1$
				;
		//@formatter:on

		GsonBuilder gsonBuilder = BaseDataServiceDao.getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(User.class, Outlet.class,
				StoreSession.class, Terminal.class, MenuShift.class, TicketItemCookingInstruction.class, Department.class));
		gsonBuilder.registerTypeAdapterFactory(ticketItemAdapterFactory);
		gsonBuilder.registerTypeAdapterFactory(transactionAdapterFactory);

		return gsonBuilder.create();

	}
}
