/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.AttendenceHistory;
import com.floreantpos.model.Shift;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.AttendanceReportData;
import com.floreantpos.report.PayrollReportData;
import com.floreantpos.report.WeeklyPayrollReportData;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class AttendenceHistoryDAO extends BaseAttendenceHistoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AttendenceHistoryDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<User> findNumberOfClockedInUserAtHour(Date fromDay, Date toDay, int hour, Terminal terminal) {
		Session session = null;

		ArrayList<User> users = new ArrayList<User>();

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, fromDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, toDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_HOUR, new Short((short) hour)));

			if (terminal != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_TERMINAL_ID, terminal.getId()));
			}

			List list = criteria.list();
			for (Object object : list) {
				AttendenceHistory history = (AttendenceHistory) object;

				if (!history.isClockedOut()) {
					users.add(history.getUser());
				}
				else if (history.getClockOutHour() >= hour) {
					users.add(history.getUser());
				}
			}
			return users;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.2"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<User> findNumberOfClockedInUserAtShift(Date fromDay, Date toDay, Shift shift, Terminal terminal) {
		Session session = null;

		ArrayList<User> users = new ArrayList<User>();

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, fromDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, toDay));

			if (shift != null) {
				criteria.add(Restrictions.le(AttendenceHistory.PROP_SHIFT_ID, shift.getId()));
			}

			if (terminal != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_TERMINAL_ID, terminal.getId()));
			}

			List list = criteria.list();
			for (Object object : list) {
				AttendenceHistory history = (AttendenceHistory) object;

				//				if (!history.isClockedOut()) {
				//					users.add(history.getUser());
				//				}
				//				else if (history.getClockOutHour() >= hour) {
				//					users.add(history.getUser());
				//				}
				users.add(history.getUser());
			}
			return users;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.5"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public AttendenceHistory findByLastClockInTime(User user) {
		try (Session session = createNewSession()) {
			return findByLastClockInTime(user, session);
		}
	}

	@Deprecated
	public AttendenceHistory findHistoryByClockedInTime(User user) {
		Session session = null;

		try {
			session = createNewSession();
			return findHistoryByClockedInTime(user, session);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public AttendenceHistory findByLastClockInTime(User user, Session session) {
		Criteria criteria = session.createCriteria(AttendenceHistory.class);
		if (user != null) {
			criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
		}
		//criteria.add(Restrictions.isNull(AttendenceHistory.PROP_CLOCK_OUT_TIME));
		criteria.addOrder(Order.desc(AttendenceHistory.PROP_CLOCK_IN_TIME));
		criteria.setFirstResult(0);
		criteria.setMaxResults(1);

		List<AttendenceHistory> list = criteria.list();
		if (list.size() > 0) {
			return list.get(0);
		}

		return null;
	}

	@Deprecated
	public AttendenceHistory findHistoryByClockedInTime(User user, Session session) {
		Criteria criteria = session.createCriteria(AttendenceHistory.class);
		if (user != null) {
			criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
		}
		criteria.add(Restrictions.isNull(AttendenceHistory.PROP_CLOCK_OUT_TIME));
		criteria.addOrder(Order.desc(AttendenceHistory.PROP_CLOCK_IN_TIME));
		criteria.setFirstResult(0);
		criteria.setMaxResults(1);

		List<AttendenceHistory> list = criteria.list();
		if (list.size() > 0) {
			return list.get(0);
		}

		return null;
	}

	public List<PayrollReportData> findPayroll(Date from, Date to, User user) {
		ArrayList<PayrollReportData> reportDataList = new ArrayList<PayrollReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
			if (user != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			}

			populateReportDataList(reportDataList, criteria.list());
			if (user != null) {
				List<User> linkedUsers = user.getLinkedUser();
				for (User linkedUser : linkedUsers) {
					if (linkedUser.getId().equals(user.getId())) {
						continue;
					}
					criteria = session.createCriteria(AttendenceHistory.class);
					criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
					criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
					criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
					criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
					criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, linkedUser.getId()));
					populateReportDataList(reportDataList, criteria.list());
				}
			}
			return reportDataList;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.6"), e); //$NON-NLS-1$
		}
	}

	private void populateReportDataList(ArrayList<PayrollReportData> reportDataList, List searchResultList) {
		for (Iterator iterator = searchResultList.iterator(); iterator.hasNext();) {
			AttendenceHistory history = (AttendenceHistory) iterator.next();
			PayrollReportData data = new PayrollReportData();
			data.setFrom(history.getClockInTime());
			data.setTo(history.getClockOutTime());
			data.setDate(history.getClockInTime());
			data.setUser(history.getUser());
			data.calculate();

			reportDataList.add(data);
		}
	}

	public List<WeeklyPayrollReportData> findWeeklyPayroll(Date from, Date to, User user, int firstDayOfWeek) {
		Session session = null;
		List<WeeklyPayrollReportData> reportDataList = new ArrayList<WeeklyPayrollReportData>();
		List<Date[]> searchDateList = new ArrayList<>();
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);

			Calendar calendar = Calendar.getInstance();
			calendar.setTime(from);
			Date[] dateRange = new Date[2];
			dateRange[0] = from;
			while (!from.after(to)) {
				int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
				int difference = dayOfWeek - (firstDayOfWeek - 1);
				calendar.add(Calendar.DATE, 1);

				if (difference == 0 || difference == 7 || calendar.getTime().after(to)) {
					calendar.add(Calendar.DATE, -1);
					Date toDate = calendar.getTime();
					dateRange[1] = toDate;
					searchDateList.add(dateRange);
					dateRange = new Date[2];
					calendar.add(Calendar.DATE, 1);
					from = calendar.getTime();
					dateRange[0] = from;
				}
			}

			for (Date[] dateRangeArray : searchDateList) {
				criteria = session.createCriteria(AttendenceHistory.class);
				Date startOfDay = DateUtil.startOfDay(dateRangeArray[0]);
				Date endOfDay = DateUtil.endOfDay(dateRangeArray[1]);

				criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, startOfDay));
				criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, endOfDay));
				criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
				criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
				if (user != null) {
					criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
				}

				reportDataList.addAll(populateWeeklyPayrollReportDataList(criteria.list(), startOfDay, endOfDay, firstDayOfWeek));
				if (user != null) {
					List<User> linkedUsers = user.getLinkedUser();
					for (User linkedUser : linkedUsers) {
						if (linkedUser.getId().equals(user.getId())) {
							continue;
						}
						criteria = session.createCriteria(AttendenceHistory.class);
						criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
						criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
						criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
						criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
						criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, linkedUser.getId()));
						reportDataList.addAll(populateWeeklyPayrollReportDataList(criteria.list(), startOfDay, endOfDay, firstDayOfWeek));
					}
				}
			}
			return reportDataList;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private List<WeeklyPayrollReportData> populateWeeklyPayrollReportDataList(List searchResultList, Date from, Date to, int firstDayOfWeek) {
		Map<String, WeeklyPayrollReportData> reportDataMap = new HashMap<String, WeeklyPayrollReportData>();
		for (Iterator iterator = searchResultList.iterator(); iterator.hasNext();) {
			AttendenceHistory history = (AttendenceHistory) iterator.next();
			WeeklyPayrollReportData data = reportDataMap.get(history.getUser().getId());
			if (data == null) {
				data = new WeeklyPayrollReportData();
				data.setFirstDayOfWeek(firstDayOfWeek);
				reportDataMap.put(history.getUser().getId(), data);
			}
			data.setTotalWorkHourMs(data.getTotalWorkHourMs() + (history.getClockOutTime().getTime() - history.getClockInTime().getTime()));
			data.setFromDateOfWeek(from);
			data.setToDateOfWeek(to);
			data.setUser(history.getUser());
		}
		Collection<WeeklyPayrollReportData> values = reportDataMap.values();
		for (WeeklyPayrollReportData data : values) {
			data.calculateTotalHour();
		}
		return new ArrayList<WeeklyPayrollReportData>(values);
	}

	public List<AttendanceReportData> findAttendance(Date from, Date to, User user) {
		Session session = null;

		ArrayList<AttendanceReportData> list = new ArrayList<AttendanceReportData>();

		try {

			session = createNewSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.between(AttendenceHistory.PROP_CLOCK_IN_TIME, from, to));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
			if (user != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			}
			List list2 = criteria.list();

			for (Iterator iterator = list2.iterator(); iterator.hasNext();) {
				AttendenceHistory history = (AttendenceHistory) iterator.next();
				AttendanceReportData data = new AttendanceReportData();
				data.setClockIn(history.getClockInTime());
				data.setClockOut(history.getClockOutTime());
				data.setUser(history.getUser());
				data.setName(history.getUser().getFirstName());
				data.calculate();

				list.add(data);
			}

			return list;
		} catch (Exception e) {
			throw new PosException("Unable to find Attendance", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<AttendenceHistory> findHistory(Date from, Date toDate, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.between(AttendenceHistory.PROP_CLOCK_IN_TIME, from, toDate));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_ID));
			if (user != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			}

			List list2 = criteria.list();

			return list2;
		} catch (Exception e) {
			throw new PosException("Unable to find History", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<AttendenceHistory> findAttendanceHistory(Date from, Date to, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_ID));
			if (user != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			}

			List list2 = criteria.list();

			return list2;
		} catch (Exception e) {
			throw new PosException("Unable to find History", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateAttendenceHistoryList(List<AttendenceHistory> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<AttendenceHistory> iterator = dataList.iterator(); iterator.hasNext();) {
				AttendenceHistory item = (AttendenceHistory) iterator.next();
				AttendenceHistory existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public static AttendenceHistory createNewHistory(User user, Terminal terminal, Shift shift) {
		AttendenceHistory attendenceHistory = new AttendenceHistory();
		Date lastClockInTime = user.getLastClockInTime();
		Calendar c = Calendar.getInstance();
		if (lastClockInTime != null) {
			c.setTime(lastClockInTime);
		}

		attendenceHistory.setClockInTime(lastClockInTime);
		attendenceHistory.setClockInHour(Short.valueOf((short) c.get(Calendar.HOUR)));
		attendenceHistory.setUser(user);
		attendenceHistory.setTerminal(terminal);
		attendenceHistory.setShift(shift);

		return attendenceHistory;
	}
}