/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.ext.KitchenStatus;

public class KitchenTicketDAO extends BaseKitchenTicketDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public KitchenTicketDAO() {
	}

	public Date getLastUpdateDate() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.max(KitchenTicket.PROP_LAST_UPDATE_TIME));
			criteria.setMaxResults(1);
			return (Date) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	public List<KitchenTicket> findAllOpen() {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(KitchenTicket.PROP_STATUS, KitchenStatus.WAITING.name()));
			List list = criteria.list();

			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<KitchenTicket> findAllOpen(Date lastUpdateTime) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(KitchenTicket.PROP_STATUS, KitchenStatus.WAITING.name()));
			if (lastUpdateTime != null) {
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime)));
			}
			List list = criteria.list();

			return list;
		} finally {
			closeSession(session);
		}
	}

	public void dispatchAll() {
		Session session = null;
		Transaction tx = null;
		try {
			session = getSession();
			tx = session.beginTransaction();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(KitchenTicket.PROP_STATUS, KitchenStatus.BUMP.name()));
			List<KitchenTicket> KitchenTicketList = criteria.list();

			if (KitchenTicketList.isEmpty()) {
				tx.rollback();
				throw new PosException("No data to dispatch");
			}

			for (KitchenTicket kitchenTicket : KitchenTicketList) {
				delete(kitchenTicket, session);
			}
			tx.commit();

		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<KitchenTicket> findByParentId(String ticketId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(KitchenTicket.PROP_TICKET_ID, ticketId));
			List list = criteria.list();

			return list;
		} finally {
			closeSession(session);
		}
	}

	public KitchenTicket findByLastOrderParentId(String ticketId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(KitchenTicket.PROP_TICKET_ID, ticketId));
			criteria.addOrder(Order.desc(KitchenTicket.PROP_CREATE_DATE));
			List list = criteria.list();
			if (list.size() > 0) {
				return (KitchenTicket) list.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}
	}

	/**
	 * Save kitchen tickets and ticket after kitche print
	 * 
	 * @param ticket
	 * @param kitchenTickets
	 */
	public synchronized void save(List<KitchenTicket> kitchenTickets) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (KitchenTicket kitchenTicket : kitchenTickets) {
				session.saveOrUpdate(kitchenTicket);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public Date isDataUpdated(Date modifiedDate) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.max(KitchenTicket.PROP_CREATE_DATE));
			if (modifiedDate != null)
				criteria.add(Restrictions.gt(KitchenTicket.PROP_CREATE_DATE, modifiedDate));

			Object data = criteria.uniqueResult();
			if (data == null) {
				return null;
			}
			return (Date) data;
		} finally {
			closeSession(session);
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		Date currentTime = StoreDAO.getServerTimestamp();
		KitchenTicket kitchenTicket = (KitchenTicket) obj;
		kitchenTicket.setLastUpdateTime(currentTime);
		s.saveOrUpdate(kitchenTicket);
	}

	public void bumpOrUnbump(KitchenTicket kitchenTicket, KitchenStatus kitchenTicketStatus, KitchenStatus ticketItemStatus, boolean bump) {
		bumpOrUnbump(kitchenTicket, null, kitchenTicketStatus, ticketItemStatus, bump);
	}

	public void bumpOrUnbump(KitchenTicket kitchenTicket, Ticket parentTicket, KitchenStatus kitchenTicketStatus, KitchenStatus ticketItemStatus,
			boolean bump) {
		if (parentTicket == null) {
			parentTicket = TicketDAO.getInstance().loadFullTicket(kitchenTicket.getTicketId(), kitchenTicket.getOutletId());
		}
		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (KitchenTicketItem kitchenTicketItem : kitchenTicket.getTicketItems()) {
				if (kitchenTicketItem.isVoided()) {
					continue;
				}
				kitchenTicketItem.setStatus(kitchenTicketStatus.name());
				double itemQuantity = kitchenTicketItem.getQuantity();

				if (parentTicket != null) {
					for (TicketItem item : parentTicket.getTicketItems()) {
						if (StringUtils.isNotEmpty(kitchenTicketItem.getTicketItemId()) && kitchenTicketItem.getTicketItemId().equals(item.getId())) {
							if (item.getKitchenStatusValue() == ticketItemStatus) {
								continue;
							}
							if (itemQuantity == 0) {
								break;
							}
							item.setKitchenStatusValueWithChildren(ticketItemStatus);
							itemQuantity -= item.getQuantity();
						}
					}
				}
			}
			kitchenTicket.setStatus(kitchenTicketStatus.name());
			Date currentTime = StoreDAO.getServerTimestamp();
			if (bump)
				kitchenTicket.setClosingDate(currentTime);
			else
				kitchenTicket.setClosingDate(null);

			saveOrUpdate(kitchenTicket, session);
			if (parentTicket != null) {
				parentTicket.setStatus(bump ? Ticket.STATUS_READY : Ticket.STATUS_WAITING);
				TicketDAO.getInstance().saveOrUpdate(parentTicket, session);
			}
			tx.commit();
		} catch (Exception ex) {
			tx.rollback();
			throw ex;
		} finally {
			session.close();
		}
	}
}