/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.model.ShopFloor;
import com.floreantpos.model.ShopSeat;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableType;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ShopFloorDAO extends BaseShopFloorDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ShopFloorDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void updateTime(Object model) {
		ShopFloor shopFloor = (ShopFloor) model;
		if (shopFloor.getTables() != null) {
			for (ShopTable shopTable : shopFloor.getTables()) {
				super.updateTime(shopTable);
			}
		}
		super.updateTime(model);
	}

	@Override
	public List<ShopFloor> findAll() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ShopFloor.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public boolean hasFloor() {
		Number result = (Number) getSession().createCriteria(getReferenceClass()).setProjection(Projections.rowCount()).uniqueResult();
		return result.intValue() != 0;
	}

	public void initializeTables(ShopFloor shopFloor) {
		if (shopFloor == null || shopFloor.getId() == null) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(shopFloor);
			Hibernate.initialize(shopFloor.getTables());
		} finally {
			closeSession(session);
		}
	}

	@Override
	public void delete(ShopFloor shopFloor) throws HibernateException {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();
			refresh(shopFloor, session);
			shopFloor.setDeleted(Boolean.TRUE);
			Set<ShopTable> tables = shopFloor.getTables();

			if (tables != null && !tables.isEmpty()) {
				for (Iterator iterator = tables.iterator(); iterator.hasNext();) {
					ShopTable shopTable = (ShopTable) iterator.next();
					shopTable.setFloorId(null);
					ShopTableDAO.getInstance().update(shopTable, session);
				}
				shopFloor.getTables().removeAll(tables);
				//saveOrUpdate(shopFloor, session);
			}
			saveOrUpdate(shopFloor, session);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopFloorDAO.class).error(e);

			throw new HibernateException(e);
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateShopFloors(List<ShopFloor> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<ShopFloor> iterator = dataList.iterator(); iterator.hasNext();) {
				ShopFloor shopFloor = (ShopFloor) iterator.next();
				ShopFloor existingShopFloor = get(shopFloor.getId(), shopFloor.getOutletId());
				if (existingShopFloor == null) {
					shopFloor.setUpdateLastUpdateTime(updateLastUpdateTime);
					shopFloor.setUpdateSyncTime(updateSyncTime);
					save(shopFloor, session);
				}
				else {
					if (!BaseDataServiceDao.get().shouldSave(shopFloor.getLastUpdateTime(), existingShopFloor.getLastUpdateTime())) {
						PosLog.info(getClass(), shopFloor.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					shopFloor.setVersion(existingShopFloor.getVersion());
					shopFloor.setUpdateLastUpdateTime(updateLastUpdateTime);
					shopFloor.setUpdateSyncTime(updateSyncTime);
				}
				shopFloor.setTables(null);
				ShopFloorDAO.getInstance().initializeTables(existingShopFloor);
				if (existingShopFloor != null) {
					if (existingShopFloor.getTables() != null && existingShopFloor.getTables().size() > 0) {
						for (ShopTable shopTable : existingShopFloor.getTables()) {
							shopTable.setUpdateLastUpdateTime(false);
							shopTable.setUpdateSyncTime(false);
							shopFloor.addTotables(shopTable);
						}
					}
				}
				update(shopFloor, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public ShopFloor get(String id, String outletId) {
		return get(new ShopFloor(id, outletId));
	}

	private void saveShopTables(Session session, ShopFloor shopFloor, ShopFloor existingShopFloor, Set<ShopTable> shopTables, boolean updateLastUpdateTime,
			boolean updateSyncTime) {
		if (shopTables != null && shopTables.size() > 0) {
			for (ShopTable table : shopTables) {
				table.setFloor(shopFloor);
				Set<ShopSeat> shopSeats = table.getSeats();
				List<ShopTableType> shopTableTypes = table.getTypes();
				table.setSeats(null);
				table.setTypes(null);
				ShopTableDAO shopTableDao = ShopTableDAO.getInstance();
				ShopTable existingTable = shopTableDao.get(table.getId(), table.getOutletId());
				if (existingTable == null) {
					table.setUpdateLastUpdateTime(updateLastUpdateTime);
					table.setUpdateSyncTime(updateSyncTime);
					session.save(table);
				}
				else {
					table.setUpdateLastUpdateTime(updateLastUpdateTime);
					table.setUpdateSyncTime(updateSyncTime);
					table.setVersion(existingTable.getVersion());
					shopTableDao.initializeSeats(existingTable);
					shopTableDao.initializeTypes(existingTable);
				}
				shopTableDao.saveOrSetVersionShopSeat(session, table, existingTable, shopSeats);
				shopTableDao.saveOrSetVersionShopTableTypes(session, table, existingTable, shopTableTypes);
			}
		}
	}

	//	private void saveOrUpdateShopTable(ShopTable table, List<ShopTable> existingTables, Session session) {
	//		ShopTableDAO dao = new ShopTableDAO();
	//		if (existingTables == null || existingTables.size() == 0)
	//			return;
	//		ShopTable existingTable = null;
	//		int idx = existingTables.indexOf(table);
	//		if (idx != -1) {
	//			existingTable = existingTables.get(idx);
	//			if (existingTable == null) {
	//				dao.save(table, session);
	//			}
	//			else {
	//				table.setVersion(existingTable.getVersion());
	//			}
	//		}
	//		else {
	//			saveOrUpdateShopTable(table, session);
	//		}
	//	}
	//
	//	private void saveOrUpdateShopTable(ShopTable table, Session session) {
	//		Integer tableId = table.getId();
	//		if (tableId != null) {
	//			ShopTable existingShopTable = ShopTableDAO.getInstance().get(tableId);
	//			if (existingShopTable != null) {
	//				table.setVersion(existingShopTable.getVersion());
	//			}
	//		}
	//		else {
	//			ShopTableDAO.getInstance().save(table, session);
	//		}
	//	}

}