/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Interval;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.StoreDAO;

public class DateUtil {
	private final static DateFormat localeFormat = DateFormat.getDateInstance(DateFormat.SHORT);
	private static SimpleDateFormat monthDateFormat = new SimpleDateFormat("MMM dd");//$NON-NLS-1$

	public static String formatDateAsLocale(Date date) {
		String dateString = localeFormat.format(date);
		return dateString;
	}

	public static String formatReportDateAsString(Date date) {
		return formatDateWithTime(date);
	}

	public static String formatDateAsString(Date date) {
		return formatDateWithTime(date);
	}

	public static String formatDateAsUTCString(Date date) {
		//		dateFormat9.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		//		String dateString = dateFormat9.format(date);

		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static Date parseByFullDate(String date) throws Exception {
		return formatDateWithTimeAndSec(date);
	}

	public static String formatFullDateAsString(Date date) {
		return formatDate(date);
	}

	public static Date parseBySortDate(String date) throws Exception {
		return formatDate(date);
	}

	public static Date startOfDay(Date date) {
		//		Calendar cal = Calendar.getInstance();
		//		cal.setTime(date == null ? new Date() : date);
		//		cal.set(Calendar.HOUR_OF_DAY, 0);
		//		cal.set(Calendar.MINUTE, 0);
		//		cal.set(Calendar.SECOND, 0);
		//
		//		return new Date(cal.getTimeInMillis());

		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.systemDefault());
		return Date.from(startOfDay.toInstant());
	}

	public static Date getUTCStartOfDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), getClientZoneId());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(getClientZoneId());
		return Date.from(startOfDay.toInstant());
	}

	public static Date getUTCEndOfDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), getClientZoneId());
		dateTime = dateTime.plusDays(1);
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(getClientZoneId());
		return Date.from(startOfDay.toInstant());
	}

	private static ZoneId getClientZoneId() {
		int timeZoneOffset = DataProvider.get().getTimeZoneOffset();
		if (timeZoneOffset == 0) {
			return ZoneId.systemDefault();
		}
		return ZoneId.ofOffset("UTC", ZoneOffset.ofTotalSeconds(timeZoneOffset / 1000)); //$NON-NLS-1$
	}

	public static Date startOfNextDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		dateTime = dateTime.plusDays(1);

		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.systemDefault());
		return Date.from(startOfDay.toInstant());
	}

	public static Date endOfDay(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date == null ? new Date() : date);
		cal.set(Calendar.HOUR_OF_DAY, 23);
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.SECOND, 59);

		return new Date(cal.getTimeInMillis());
	}

	public static Date startOfDayInUTC(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.of("UTC")); //$NON-NLS-1$

		return Date.from(startOfDay.toInstant());
	}

	public static Date endOfDayInUTC(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime atStartOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.of("UTC")); //$NON-NLS-1$
		ZonedDateTime enDateTime = atStartOfDay.plusDays(1);

		return Date.from(enDateTime.toInstant());
	}

	public static Date toUTC(Date date) {
		int rawOffset = TimeZone.getDefault().getRawOffset();
		return new Date(date.getTime() - rawOffset);
	}

	public static Date startOfMonth(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.DAY_OF_MONTH, 1);

		return new Date(cal.getTimeInMillis());
	}

	public static Date endOfMonth(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.HOUR_OF_DAY, 23);
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.SECOND, 59);
		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));

		return new Date(cal.getTimeInMillis());
	}

	public static Date copyTime(Date copyTo, Date copyFrom) {

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(copyFrom);
		int hour = calendar.get(Calendar.HOUR_OF_DAY);
		int min = calendar.get(Calendar.MINUTE);

		calendar.setTime(copyTo);
		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, min);
		return calendar.getTime();

	}

	public static boolean between(Date startDate, Date endDate, Date guniping) {
		if (startDate == null || endDate == null) {
			return false;
		}

		return (guniping.equals(startDate) || guniping.after(startDate)) && (guniping.equals(endDate) || guniping.before(endDate));
	}

	public static String formatSmall(Date date) {
		return formatDate(date);
	}

	public static String getReportDate() {
		return formatDateWithTimeAndSec(new Date());
	}

	public static boolean isToday(Date date) {
		return isSameDay(date, Calendar.getInstance().getTime());
	}

	public static boolean isToday(Calendar cal) {
		return isSameDay(cal, Calendar.getInstance());
	}

	public static String formatFullDateAndTimeAsString(Date date) {
		String dateString = date == null ? "" : formatDateWithTimeAndSec(date); //$NON-NLS-1$
		return dateString;
	}

	public static String formatFullDateAndTimeWithoutYearAsString(Date date) {
		return formatDateWithTimeAndSec(date);
	}

	public static boolean isSameDay(Date date1, Date date2) {
		Calendar cal1 = Calendar.getInstance();
		cal1.setTime(date1);
		Calendar cal2 = Calendar.getInstance();
		cal2.setTime(date2);
		return isSameDay(cal1, cal2);
	}

	public static boolean isSameDay(Calendar cal1, Calendar cal2) {
		return (cal1.get(Calendar.ERA) == cal2.get(Calendar.ERA) && cal1.get(Calendar.YEAR) == cal2.get(Calendar.YEAR)
				&& cal1.get(Calendar.DAY_OF_YEAR) == cal2.get(Calendar.DAY_OF_YEAR));
	}

	public static String getOnlyFormattedDate(Date date) {
		return formatDateWithTime(date);
	}

	public static String getElapsedTime(Date oldTime, Date newTime) {
		DateTime startDate = new DateTime(oldTime);
		DateTime endDate = new DateTime(newTime);
		Interval interval = new Interval(startDate, endDate);
		long days = interval.toDuration().getStandardDays();
		long hours = interval.toDuration().getStandardHours();
		long minutes = interval.toDuration().getStandardMinutes();
		long seconds = interval.toDuration().getStandardSeconds();

		hours = hours % 24;
		minutes = minutes % 60;
		seconds = seconds % 60;

		String strDays = days + "d, ";//$NON-NLS-1$
		String strHours = hours + "hr, ";//$NON-NLS-1$
		String strMins = minutes + "mn";//$NON-NLS-1$
		String strSec = seconds + "secs";//$NON-NLS-1$
		String strAgo = "";//$NON-NLS-1$

		String fullTime = strDays + strHours + strMins + strAgo;
		String timeWithoutDay = strHours + strMins + strAgo;
		String timeWithoutHour = strMins + strAgo;
		String timeWithoutMin = strSec + strAgo;

		if (days != 0) {
			return fullTime;
		}
		else if (hours != 0) {
			return timeWithoutDay;
		}
		else if (minutes != 0) {
			return timeWithoutHour;
		}
		else if (seconds != 0) {
			return timeWithoutMin;
		}
		else {
			return "not printed yet";//$NON-NLS-1$
		}
	}

	public static Calendar getLocalTimeCalendar() {
		return Calendar.getInstance();
	}

	public static Calendar getServerTimeCalendar() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(StoreDAO.getServerTimestamp());
		return calendar;
	}

	public static Date getGmtTime() {
		return Calendar.getInstance(TimeZone.getTimeZone("GMT")).getTime();//$NON-NLS-1$
	}

	public static Calendar getGmtCalendar() {
		return Calendar.getInstance(TimeZone.getTimeZone("GMT"));//$NON-NLS-1$
	}

	//Dynamic Date Format
	public static String getStoreDateFormat() {
		Store store = DataProvider.get().getStore();
		return store.getDateFormat();
	}

	public static SimpleDateFormat getSimpleDateFormat() {
		Store store = DataProvider.get().getStore();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(store.getDateFormat());
		return simpleDateFormat;

	}

	public static String formatDate(Date date) {
		SimpleDateFormat simpleDateFormat = getSimpleDateFormat();
		return simpleDateFormat.format(date);

	}

	public static Date formatDate(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = getSimpleDateFormat();
		return simpleDateFormat.parse(date);

	}

	/**
	 * ************************************************************************
	 * * inputFormat Can't be null
	 * ************************************************************************
	 */
	public static String parseStringDateToString(String inputFormat, String date) throws Exception {
		String outputFormat = getStoreDateFormat();
		SimpleDateFormat simpleDateOutputFormat = new SimpleDateFormat(outputFormat);

		SimpleDateFormat simpleDateInputFormat = new SimpleDateFormat(inputFormat);

		Date parseDate = simpleDateInputFormat.parse(date);

		return simpleDateOutputFormat.format(parseDate);
	}

	public static String formatAsShortDate(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		return simpleDateFormat.format(date);

	}

	public static String formatDateWithTime(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static String tweentyFourHoursDateFormat(Date date) {
		return new SimpleDateFormat(getStoreDateFormat() + " HH:mm", Locale.UK).format(date); //$NON-NLS-1$
	}

	public static Date formatDateWithTime(String date) throws ParseException {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.parse(date);

	}

	public static String formatDateWithTimeAndSec(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static Date formatDateWithTimeAndSec(String date) throws ParseException {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.parse(date);

	}

	public static Date formatDateWithDefaultTimeAndSec(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy hh:mm:ss a"); //$NON-NLS-1$
		try {
			Date parse = simpleDateFormat.parse(date);
			return parse;
		} catch (Exception e) {
		}
		return null;
	}

	public static String formatDateWithDefaultTimeAndSec(Date date) {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static String formatSyncTime(Date date) {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy HH:mm:ss"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static Date parseSyncTime(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy HH:mm:ss"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		try {
			Date parse = simpleDateFormat.parse(date);
			return parse;
		} catch (Exception e) {

		}
		return DateUtil.formatDateWithDefaultTimeAndSec(date);
	}

	public static String formatAsDefaultMonthDate(Date date) {
		return monthDateFormat.format(date);
	}

	/**
	 * * This method is used to format date to string date.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static String formatDateWithBrowserTimeOffset(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	public static String formatDateWithBrowserTimeOffset(Date date, SimpleDateFormat simpleDateFormat) {
		int timezoneOffset = DataProvider.get().getTimeZoneOffset();
		if (timezoneOffset == 0) {
			return simpleDateFormat.format(date);
		}
		String[] availableIDs = TimeZone.getAvailableIDs(timezoneOffset);
		String timezone = null;
		if (availableIDs.length > 0) {
			timezone = availableIDs[0];
		}
		if (StringUtils.isNotEmpty(timezone)) {
			simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timezone));
		}
		return simpleDateFormat.format(date);
	}

	/**
	 * * This method is used to format string date to date.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static Date formatDateWithBrowserTimeOffset(String date) {
		try {
			String format = DataProvider.get().getStore().getDateFormat();
			SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$

			int timezoneOffset = DataProvider.get().getTimeZoneOffset();
			if (timezoneOffset == 0) {
				return simpleDateFormat.parse(date);
			}
			String[] availableIDs = TimeZone.getAvailableIDs(timezoneOffset);
			String timezone = null;
			if (availableIDs.length > 0) {
				timezone = availableIDs[0];
			}

			if (StringUtils.isNotEmpty(timezone)) {
				simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timezone));
			}
			return simpleDateFormat.parse(date);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return null;
		}
	}
//
//	public static String convertDateToBrowserTimeZone(Date date) {
//		if (date == null) {
//			return ""; //$NON-NLS-1$
//		}
//		String format = DataProvider.get().getStore().getDateFormat();
//		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a z"); //$NON-NLS-1$
//		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
//	}

	public static Date convertDateToBrowserTime(Date date) {
		try {
			int timezoneOffset = DataProvider.get().getTimeZoneOffset();
			return new Date(date.getTime() - timezoneOffset);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return date;
		}
	}

	/**
	 * * This method is used to format date to string date without second value.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static String formatDateWithBrowserTimeOffsetWithoutSec(Date date) {
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	//	public static Date startOfDayInUTC(Date date) {
	//		date = startOfDay(date);
	////		int rawOffset = TimeZone.getDefault().getRawOffset();
	////		date = new Date(date.getTime() + rawOffset);
	//
	//		return date;
	//	}
	//
	//	public static Date endOfDayInUTC(Date date) {
	//		date = endOfDay(date);
	////		int rawOffset = TimeZone.getDefault().getRawOffset();
	////		date = new Date(date.getTime() + rawOffset);
	//
	//		return date;
	//	}

	public static String formatReportDate(Date date) {
		String format = getStoreDateFormat();
		//SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a z"); //$NON-NLS-1$
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static String format(String pattern, Date date) {
		return new SimpleDateFormat(pattern).format(date);
	}

	public static void validateDate(Date fromDate, Date toDate) {
		if (fromDate == null) {
			throw new PosException(POSConstants.FROM_DATE_IS_NULL);
		}
		if (toDate == null) {
			throw new PosException(POSConstants.TO_DATE_IS_NULL);
		}

		if (fromDate.after(toDate)) {
			throw new PosException(POSConstants.FROM_DATE_CANNOT_BE_GREATER_THAN_TO_DATE_);
		}
	}

}
