/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.util.List;

import javax.swing.table.AbstractTableModel;

import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;

public class TicketReportModel extends AbstractTableModel {
	//private static DecimalFormat formatter = new DecimalFormat("#,##0.00"); //$NON-NLS-1$

	private String[] columnNames = { "id", "date", "orderType", "owner", "total", "due", "outletId" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$
	private List<Ticket> items;
	private double grandTotal;
	private double totalDue;
	private boolean showFormattedDate;

	public TicketReportModel() {
		super();
	}

	public int getRowCount() {
		if (items == null)
			return 0;

		return items.size();
	}

	public int getColumnCount() {
		return columnNames.length;
	}

	@Override
	public String getColumnName(int column) {
		return columnNames[column];
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		Ticket ticket = items.get(rowIndex);

		switch (columnIndex) {
			case 0:
				return String.valueOf(ticket.getId());

			case 1:
				if (showFormattedDate) {
					return DateUtil.formatDateWithBrowserTimeOffset(ticket.getCreateDate());
				}
				return DateUtil.formatFullDateAndTimeAsString(ticket.getCreateDate());

			case 2:
				return ticket.getOrderType() == null ? "" : ticket.getOrderType().getName(); //$NON-NLS-1$

			case 3:
				return ticket.getOwner() == null ? "" : ticket.getOwner().getFullName(); //$NON-NLS-1$

			case 4:
				return NumberUtil.formatNumber(ticket.getTotalAmountWithTips());

			case 5:
				return NumberUtil.formatNumber(ticket.getDueAmount());

			case 6:
				return ticket.getOutletId();
		}
		return null;
	}

	public List<Ticket> getItems() {
		return items;
	}

	public void setItems(List<Ticket> items) {
		this.items = items;
	}

	public String getGrandTotalAsString() {
		return NumberUtil.formatNumber(grandTotal);
	}

	public void setGrandTotal(double grandTotal) {
		this.grandTotal = grandTotal;
	}

	public void calculateGrandTotal() {
		grandTotal = 0;
		if (items == null) {
			return;
		}

		for (Ticket item : items) {
			grandTotal += item.getTotalAmount();
		}
	}

	public String getTotalDueAsString() {
		return NumberUtil.formatNumber(totalDue);
	}

	public void setTotalDue(double totalDue) {
		this.totalDue = totalDue;
	}

	public void calculateTotalDue() {
		totalDue = 0;
		if (items == null) {
			return;
		}
		for (Ticket item : items) {
			totalDue += item.getDueAmount();
		}
	}

	public boolean isShowFormattedDate() {
		return showFormattedDate;
	}

	public void setShowFormattedDate(boolean showFormattedDate) {
		this.showFormattedDate = showFormattedDate;
	}

}
