package com.orocube.rest.service.mqtt;

import java.util.UUID;

import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttMessageListener;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.main.Application;
import com.floreantpos.model.Store;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.MqttCommand;
import com.floreantpos.util.POSUtil;
import com.orocube.common.util.TerminalUtil;

public class OroMqttClient extends OroMqttMessageReceiver {
	private static final String MQTT_SERVER_URL = "tcp://siiopa.com:1883"; //$NON-NLS-1$
	private MqttClient oroMqttClient;
	private int qos = 0;

	private static OroMqttClient instance;
	//	private String storeUniqueId;
	private String deviceId;

	private OroMqttClient() {
	}

	public static OroMqttClient getInstance() {
		if (instance == null) {
			instance = new OroMqttClient();
		}
		return instance;
	}

	public MqttClient getMqttClient() {
		return getInstance().oroMqttClient;
	}

	public boolean isConnectedToServer() {
		if (oroMqttClient == null) {
			return false;
		}
		if (oroMqttClient.isConnected())
			return true;
		return false;
	}

	public void initMqttClient(String deviceId) throws Exception {
		if (Application.getInstance().isDevelopmentMode()) {
			initMqttClient(deviceId, new DebugCallbackHandler());
		}
		else {
			initMqttClient(deviceId, null);
		}
	}

	public void initMqttClient(String deviceIdStr, MqttCallback callBack) throws Exception {
		this.deviceId = deviceIdStr;

		String serverAddress = MQTT_SERVER_URL; //$NON-NLS-1$
		oroMqttClient = new MqttClient(serverAddress, deviceId, new MemoryPersistence());
		MqttConnectOptions connOpts = new MqttConnectOptions();
		connOpts.setCleanSession(true);
		connOpts.setConnectionTimeout(2000);
		connOpts.setAutomaticReconnect(true);
		connOpts.setKeepAliveInterval(120);
		oroMqttClient.setCallback(callBack);
		oroMqttClient.connect(connOpts);
		subscribe();
	}

	public void initCloudClient(String deviceIdStr) throws Exception {
		this.deviceId = deviceIdStr;

		String serverAddress = MQTT_SERVER_URL; //$NON-NLS-1$
		oroMqttClient = new MqttClient(serverAddress, deviceId, new MemoryPersistence());
		MqttConnectOptions connOpts = new MqttConnectOptions();
		connOpts.setCleanSession(true);
		connOpts.setConnectionTimeout(2000);
		connOpts.setAutomaticReconnect(true);
		connOpts.setKeepAliveInterval(120);
		oroMqttClient.connect(connOpts);
	}

	public void subscribe() throws Exception {
		//		String storeUUID = ""; //$NON-NLS-1$
		//		Store store = DataProvider.get().getStore();
		//		if (store != null) {
		//			storeUUID = store.getUuid();
		//		}
		//
		//		storeUniqueId = storeUUID; //AppConfig.getString("store_uuid", storeUUID) + "/"; //$NON-NLS-1$ //$NON-NLS-2$
		subscribe(getStoreUuid() + MqttTopics.PUBLIC, qos, this);
	}

	public void subscribeToStore(String topic, int qos, IMqttMessageListener messageListener) throws Exception {
		subscribe(getStoreUuid() + topic, qos, messageListener);
	}

	public void subscribe(String topic, int qos, IMqttMessageListener messageListener) throws Exception {
		if (oroMqttClient == null) {
			return;
		}
		if (!oroMqttClient.isConnected()) {
			throw new PosException(Messages.getString("OroMqttClient.4")); //$NON-NLS-1$
		}
		oroMqttClient.subscribe(topic, qos, messageListener);
		PosLog.debug(getClass(), "subscribed to topic: " + topic); //$NON-NLS-1$
	}

	public void unsubscribe(String topic) throws Exception {
		if (oroMqttClient == null) {
			return;
		}
		if (oroMqttClient != null && !oroMqttClient.isConnected()) {
			throw new PosException(Messages.getString("OroMqttClient.4")); //$NON-NLS-1$
		}
		oroMqttClient.unsubscribe(getStoreUuid() + topic);
		PosLog.debug(getClass(), "unsubscribed to topic: " + topic); //$NON-NLS-1$
	}

	protected void showMqttNotification(String msg) {
		PosLog.info(getClass(), msg);
	}

	public void publishOnThread(final String topic, final String msg) {
		//		new Thread(new Runnable() {
		//
		//			@Override
		//			public void run() {
		//				publishData(topic, msg);
		//			}
		//		}).start();
		publishData(topic, msg);
	}

	public void publishData(final String topic, final String msg) {
		publishData(topic, msg, false);
	}

	public void publishData(final String topic, final String msg, boolean compress) {
		try {
			if (oroMqttClient == null) {
				return;
			}
			if (!isConnectedToServer())
				return;

			PosLog.debug(getClass(), "publishing to topic: " + getStoreUuid() + topic); //$NON-NLS-1$
			oroMqttClient.publish(getStoreUuid() + topic, compress ? POSUtil.compress(msg) : msg.getBytes(), qos, false);
		} catch (Exception me) {
			PosLog.error(getClass(), "Notification service error. " + me.getMessage()); //$NON-NLS-1$
		}
	}

	public String getDeviceId() {
		return deviceId;
	}

	private String getStoreUuid() {
		Store store = DataProvider.get().getStore();
		if (store != null) {
			return store.getUuid() + "/"; //$NON-NLS-1$
		}

		return UUID.randomUUID().toString() + "/"; //$NON-NLS-1$
	}

	class DebugCallbackHandler implements MqttCallback {

		@Override
		public void connectionLost(Throwable arg0) {
			//			PosLog.debug(getClass(), "mqtt connection lost"); //$NON-NLS-1$
		}

		@Override
		public void deliveryComplete(IMqttDeliveryToken arg0) {
			PosLog.debug(getClass(), "mqtt delivery complete"); //$NON-NLS-1$
		}

		@Override
		public void messageArrived(String topic, MqttMessage message) throws Exception {
			PosLog.debug(getClass(), "message arrived. topic: " + topic + ", message: " + message); //$NON-NLS-1$ //$NON-NLS-2$
		}

	}

	public void notifyDataUpdated(Class<?> beanClass) {
		if (beanClass == null) {
			return;
		}
		if (DataProvider.get().isCaching(beanClass)) {
			try {
				JSONObject jsonObject = new JSONObject();
				jsonObject.put("terminalKey", TerminalUtil.getSystemUID()); //$NON-NLS-1$
				OroMqttClient.getInstance().publishData(MqttCommand.TOPIC_REFRESH_CACHE, jsonObject.toString());
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
	}

}