package com.orocube.rest.service.mqtt;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttMessageListener;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.json.JSONObject;

import com.floreantpos.PosLog;
import com.floreantpos.main.Application;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.util.MqttCommand;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.util.AsyncAction;
import com.floreantpos.util.POSUtil;
import com.orocube.common.util.TerminalUtil;
import com.orocube.rest.service.PosResponse;

public class OroMqttMessageReceiver implements IMqttMessageListener {
	private List<OroMqttDataListener> dataListeners = new ArrayList<>();

	@Override
	public void messageArrived(String topic, MqttMessage message) throws Exception {
		AsyncAction.execute(new Runnable() {

			@Override
			public void run() {
				try {
					String data = POSUtil.decompress(message.getPayload());
					if (StringUtils.isEmpty(data)) {
						return;
					}
					debug("Notification received: "); //$NON-NLS-1$
					JSONObject jsonObject = new JSONObject(data);
					if (!isValidMessage(jsonObject)) {
						return;
					}

					messageReceived(jsonObject);
				} catch (Exception e) {
					PosLog.error(getClass(), e);
				}
			}
		});
	}

	private void messageReceived(JSONObject jsonObject) throws Exception {
//		if (!isValidMqttSender(jsonObject)) {
//			return;
//		}
//		BaseDataServiceDao dataServiceDao = BaseDataServiceDao.getInstance();
//		String jsonData = jsonObject.toString();
//		String request = jsonObject.getString(PosResponse.REQUEST);
//		switch (request) { //$NON-NLS-1$
//			case MessageType.TICKET:
//				List<Ticket> tickets = dataServiceDao.saveOrUpdateTickets(jsonData, false);
//				if (tickets != null && tickets.size() > 0) {
//					for (Ticket ticket : tickets) {
//						//printToKitchen(ticket);
//					}
//				}
//				fireTicketReceived();
//				break;
//			//				
//			//			case MessageType.CASH_DRAWER:
//			//				dataServiceDao.saveOrUpdateCashDrawers(jsonString);
//			//				break;
//			//				
//			//			case MessageType.MENU_ITEM:
//			//				dataServiceDao.saveOrUpdateMenuItems(jsonString);
//			//				break;
//			//				
//			//			case MessageType.MENU_GROUP:
//			//				dataServiceDao.saveOrUpdateMenuGroups(jsonString);
//			//				break;
//			//				
//			//			case MessageType.MENU_CATEGORY:
//			//				dataServiceDao.saveOrUpdateMenuCategories(jsonString);
//			//				break;
//			//
//			//			case MessageType.TERMINAL:
//			//				dataServiceDao.saveOrUpdateTerminals(jsonString);
//			//				break;
//			//				
//			//			case MessageType.ORDER_TYPE:
//			//				dataServiceDao.saveOrUpdateDataList(jsonString, OrderType.class);
//			//				break;
//			//				
//			//			case MessageType.SHOP_TABLE:
//			//				dataServiceDao.saveOrUpdateDataList(jsonString, ShopTable.class);
//			//				break;
//			//				
//			//			case MessageType.KITCHEN_TICKET:
//			//				dataServiceDao.saveOrUpdateKitchenTickets(jsonString);
//			//				break;
//		}
	}

	private void printToKitchen(Ticket ticket) {
		if (ticket.getType() == TicketType.ONLINE.getTypeNo() && ticket.getStatus() != null && ticket.getStatus().equals(Ticket.STATUS_CONFIRMED)) {
			try {
				List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, true);
				ReceiptPrintService.doPrintToKitchenAndSaveStatus(ticket, kitchenTickets, true);
				sendUpdateNotificationToKitchen();
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
	}

	private void sendUpdateNotificationToKitchen() {
		try {
			JSONObject jsonObject = new JSONObject();
			jsonObject.put("terminalKey", TerminalUtil.getSystemUID()); //$NON-NLS-1$
			jsonObject.put("command", MqttCommand.CMD_REFRESH_KITCHEN_ORDER); //$NON-NLS-1$
			jsonObject.put("isKds", Boolean.FALSE); //$NON-NLS-1$

			OroMqttClient.getInstance().publishData(MqttCommand.TOPIC_KIT_DIS_UPDATE, jsonObject.toString());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	private boolean isValidMqttSender(JSONObject jsonObject) {
		String mqttSender = jsonObject.getString(PosResponse.SOURCE); //$NON-NLS-1$
		return StringUtils.isNotBlank(mqttSender) && MqttSender.valueOf(mqttSender) == MqttSender.ONLINE_ORDER;
	}

	private boolean isPosLogFileRequest(JSONObject jsonObject) {
		return jsonObject.getString(PosResponse.REQUEST).equals(MessageType.LOG_FILE_REQUEST);
	}

	private boolean isValidMessage(JSONObject jsonObject) {
		String deviceId = jsonObject.getString(PosResponse.DEVICE_ID); //$NON-NLS-1$
		if (deviceId != null && deviceId.equals(Application.getInstance().getMqttDeviceId())) {
			return false;
		}
		if (!jsonObject.has(PosResponse.REQUEST)) { //$NON-NLS-1$
			debug("Empty request"); //$NON-NLS-1$
			return false;
		}

		Terminal currentTerminal = Application.getInstance().getTerminal();
		if (currentTerminal == null) {
			return false;
		}

		boolean masterTerminal = currentTerminal.isMasterTerminal();
		if (!masterTerminal) {
			debug("Notification skipped. This is not master terminal."); //$NON-NLS-1$
		}

		String currentOutletId = null;//Application.getInstance().getCurrentTerminalOutletId();
		if (StringUtils.isEmpty(currentOutletId)) {
			debug("Notification skipped. No outlet is defined for this terminal."); //$NON-NLS-1$
			return false;
		}

		/*String requestedOutletId = jsonObject.has(PosResponse.OUTLET_ID) ? jsonObject.getString(PosResponse.OUTLET_ID) : null; //$NON-NLS-1$ //$NON-NLS-2$
		if (StringUtils.isNotEmpty(requestedOutletId) && !requestedOutletId.equals(currentOutletId)) {
			debug("Notification skipped. Outlet not match."); //$NON-NLS-1$
			return false;
		}*/

		if (masterTerminal) {
			Integer blockedTerminalId = jsonObject.has(PosResponse.BLOCK_TERMINAL_ID) ? jsonObject.getInt(PosResponse.BLOCK_TERMINAL_ID) : null; //$NON-NLS-1$ //$NON-NLS-2$
			if (blockedTerminalId != null && blockedTerminalId > 0 && currentTerminal.getId() == blockedTerminalId) {
				return false;
			}
		}
		return masterTerminal;
	}

	private void debug(String msg) {
		PosLog.debug(OroMqttClient.class, msg);
	}

	public void addDataListener(OroMqttDataListener dataListener) {
		dataListeners.add(dataListener);
	}

	public void removeDataListener(OroMqttDataListener dataListener) {
		dataListeners.remove(dataListener);
	}

	private void fireTicketReceived() {
		for (OroMqttDataListener oroMqttDataListener : dataListeners) {
			oroMqttDataListener.ticketReceived();
		}
	}
}
