/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import static com.floreantpos.util.NumberUtil.convertToBigDecimal;
import static com.floreantpos.util.NumberUtil.round;

import java.io.StringReader;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import com.floreantpos.Messages;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.base.BaseTicketItem;
import com.floreantpos.model.dao.ActionHistoryDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.CopyUtil;
import com.floreantpos.util.JsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.OrgJsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "inventoryAdjusted", "menuItem", "voidItem" })
@XmlSeeAlso({ ModifiableTicketItem.class, ComboTicketItem.class })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@JsonTypeInfo(use = Id.CLASS, include = JsonTypeInfo.As.PROPERTY, property = "classType")
@JsonSubTypes({ @Type(value = ModifiableTicketItem.class), @Type(value = ComboTicketItem.class) })
public class TicketItem extends BaseTicketItem implements ITicketItem, PropertyContainer {

	public static final String JSON_PROP_DISCOUNT_ON_SC = "discountOnSc";
	public static final String JSON_PROP_VOID_ID = "voidId"; //$NON-NLS-1$
	public static final String JSON_PROP_VOIDED_BY_USER = "voidedByUser";
	public static final String JSON_PROP_MODIIFIER_VOID_IDS = "modifierVoidIds"; //$NON-NLS-1$
	public static final String JSON_PROP_VOID_REASON = "voidReason"; //$NON-NLS-1$
	public static final String JSON_PROP_VOID_QUANTITY = "voidQuantity"; //$NON-NLS-1$
	public static final String JSON_PROP_WASTED = "wasted"; //$NON-NLS-1$
	public static final String JSON_PROP_PRINTED_TO_KITCHEN = "originalItemPrintedToKitchen"; //$NON-NLS-1$
	public static final String JSON_PROP_RETURNED = "returned"; //$NON-NLS-1$
	public static final String JSON_PROP_RETURNED_ITEM_VOIDED = "returnedVoided"; //$NON-NLS-1$
	public static final String PROPERTY_COOKING_INSTRUCTION = "COOKING_INSTRUCTION"; //$NON-NLS-1$
	private static final String JSON_PROP_GIFT_CARD_PAID_AMOUNT = "gift_card_paid_amount"; //$NON-NLS-1$
	private static final String JSON_PROP_SC_WITHOUT_MODOFIERS = "scWithoutModifiers"; //$NON-NLS-1$

	public static final String PROPERTY_TAX = "TAX"; //$NON-NLS-1$
	public static final String PROPERTY_DISCOUNT = "DISCOUNT"; //$NON-NLS-1$
	private static final long serialVersionUID = 1L;
	private transient boolean syncEdited;
	private transient com.google.gson.JsonObject propertiesContainer;
	private String classType;
	private MenuItem menuItem;
	private int tableRowNum;
	private Double quantityToShip;

	public static final int HALF = 0;
	public static final int FULL = 1;
	public static final int SHARED_FULL = 2;
	public static final int SHARED_CUSTOM = 3;

	private Boolean includeVoidQuantity;
	private VoidItem voidItem;
	private java.util.List<TicketItemCookingInstruction> cookingInstructions;
	private java.util.List<TicketItemTax> taxes;
	private java.util.List<TicketItemDiscount> discounts;
	private TicketItem parentTicketItem;
	private Integer sortOrder;

	public TicketItem() {
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public TicketItem(java.lang.String id) {
		super(id);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public TicketItem(Ticket ticket, String name, double unitPrice, double quantity, TicketItemTax tax) {
		setName(name);
		setUnitPrice(unitPrice);
		setQuantity(quantity);
		addTotaxes(tax);
		setTicket(ticket);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public TicketItem createNew() {
		if (this instanceof ModifiableTicketItem) {
			return new ModifiableTicketItem();
		}
		else {
			return new TicketItem();
		}
	}

	/**
	 * Return the value associated with the column: HAS_MODIIERS
	 */
	public java.lang.Boolean isHasModifiers() {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		return ticketItemModifiers != null && ticketItemModifiers.size() > 0;
	}

	/**
	 * Return the value associated with the column: SIZE_MODIFIER_ID
	 */
	public com.floreantpos.model.TicketItemModifier getSizeModifier() {
		return null;
	}

	/**
	 * Set the value related to the column: SIZE_MODIFIER_ID
	 * @param sizeModifier the SIZE_MODIFIER_ID value
	 */
	public void setSizeModifier(com.floreantpos.model.TicketItemModifier sizeModifier) {
	}

	/**
	 * Return the value associated with the column: ticketItemModifiers
	 */
	public java.util.List<com.floreantpos.model.TicketItemModifier> getTicketItemModifiers() {
		return null;
	}

	/**
	 * Set the value related to the column: ticketItemModifiers
	 * @param ticketItemModifiers the ticketItemModifiers value
	 */
	public void setTicketItemModifiers(java.util.List<com.floreantpos.model.TicketItemModifier> ticketItemModifiers) {
	}

	public void addToticketItemModifiers(com.floreantpos.model.TicketItemModifier ticketItemModifier) {
	}

	public TicketItem clone() {
		return (TicketItem) SerializationUtils.clone(this);
	}

	public TicketItem cloneAsNew() {
		return cloneAsNew(false);
	}

	public TicketItem cloneAsNew(boolean needToCheeckPrintStatus) {
		try {
			TicketItem newTicketItem = (TicketItem) CopyUtil.deepCopy(this);
			boolean isPrintedToKitchen = needToCheeckPrintStatus && newTicketItem.isPrintedToKitchen();
			newTicketItem.setId(null);
			newTicketItem.setCreateDate(StoreDAO.getServerTimestamp());
			newTicketItem.setVersion(0);
			newTicketItem.setPrintedToKitchen(isPrintedToKitchen);
			newTicketItem.setInventoryAdjustQty(0.0);
			newTicketItem.setCloudSynced(false);
			newTicketItem.setHasSyncError(false);
			newTicketItem.setKitchenStatusValue(KitchenStatus.WAITING);
			newTicketItem.setSeat(null);
			if (newTicketItem.getSizeModifier() != null) {
				newTicketItem.getSizeModifier().setId(null);
				newTicketItem.getSizeModifier().setPrintedToKitchen(isPrintedToKitchen);
			}

			cloneModifiers(this, newTicketItem, isPrintedToKitchen);

			List<TicketItemCookingInstruction> cookingInstructions = newTicketItem.getCookingInstructions();
			if (cookingInstructions != null) {
				for (TicketItemCookingInstruction cookingInstruction : cookingInstructions) {
					if (cookingInstruction == null) {
						continue;
					}
					cookingInstruction.setPrintedToKitchen(isPrintedToKitchen);
					cookingInstruction.setKitchenStatusValue(KitchenStatus.WAITING);
				}
			}

			newTicketItem.setComboItems(null);
			if (isComboItem()) {
				List<TicketItem> comboItems = this.getComboItems();
				if (comboItems != null) {
					for (TicketItem ticketItem : comboItems) {
						TicketItem comboItem = (TicketItem) SerializationUtils.clone(ticketItem);
						comboItem.setId(null);
						comboItem.setCreateDate(StoreDAO.getServerTimestamp());
						comboItem.setParentTicketItem(newTicketItem);
						cloneModifiers(ticketItem, comboItem, isPrintedToKitchen);
						newTicketItem.addTocomboItems(comboItem);
					}
				}
			}
			return newTicketItem;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private void cloneModifiers(TicketItem originalTicketItem, TicketItem newTicketItem, boolean isPrintedToKitchen) {
		newTicketItem.setTicketItemModifiers(null);
		List<TicketItemModifier> ticketItemModifiers = originalTicketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				TicketItemModifier modifier = (TicketItemModifier) SerializationUtils.clone(ticketItemModifier);
				modifier.setId(null);
				modifier.setTicketItem(newTicketItem);
				modifier.setPrintedToKitchen(isPrintedToKitchen);
				modifier.setKitchenStatusValue(KitchenStatus.WAITING);
				newTicketItem.addToticketItemModifiers(modifier);
			}
		}
	}

	public void setTaxes(List<TicketItemTax> taxes) {
		this.taxes = taxes;
		buildTaxes();
	}

	public List<TicketItemTax> getTaxes() {
		if (taxes == null) {
			taxes = new ArrayList<>();

			String property = super.getTaxesProperty();
			if (StringUtils.isNotEmpty(property)) {
				JsonReader jsonParser = Json.createReader(new StringReader(property));
				JsonArray jsonArray = jsonParser.readArray();
				jsonParser.close();
				for (int i = 0; i < jsonArray.size(); i++) {
					JsonObject jsonObject = (JsonObject) jsonArray.get(i);
					TicketItemTax tit = new TicketItemTax();
					tit.setId(JsonUtil.getString(jsonObject, TicketItemTax.PROP_ID));
					tit.setName(JsonUtil.getString(jsonObject, TicketItemTax.PROP_NAME));
					tit.setRate(JsonUtil.getDouble(jsonObject, TicketItemTax.PROP_RATE));
					tit.setTaxAmount(JsonUtil.getDouble(jsonObject, TicketItemTax.PROP_TAX_AMOUNT));
					taxes.add(tit);
				}
			}
		}
		return taxes;
	}

	public void addTotaxes(TicketItemTax tax) {
		taxes = getTaxes();
		if (taxes == null) {
			taxes = new ArrayList<TicketItemTax>();
		}
		taxes.add(tax);
	}

	public void buildTaxes() {
		if (taxes == null || taxes.isEmpty()) {
			setTaxesProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemTax ticketItemTax : taxes) {
			jsonArray.put(ticketItemTax.toJson());
		}
		setTaxesProperty(jsonArray.toString());
	}

	public void setDiscounts(List<TicketItemDiscount> discounts) {
		this.discounts = discounts;
	}

	public List<TicketItemDiscount> getDiscounts() {
		if (discounts == null) {
			discounts = new ArrayList<>();

			String property = super.getDiscountsProperty();
			if (StringUtils.isNotEmpty(property)) {
				JSONArray jsonArray = new JSONArray(property);
				for (int i = 0; i < jsonArray.length(); i++) {
					JSONObject jsonObject = jsonArray.getJSONObject(i);
					TicketItemDiscount tid = new TicketItemDiscount();
					tid.setTicketItem(this);
					tid.setDiscountId(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_DISCOUNT_ID));
					tid.setName(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_NAME));
					tid.setType(OrgJsonUtil.getInt(jsonObject, TicketItemDiscount.PROP_TYPE));
					tid.setAutoApply(OrgJsonUtil.getBoolean(jsonObject, TicketItemDiscount.PROP_AUTO_APPLY));
					tid.setCouponQuantity(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_COUPON_QUANTITY));
					tid.setMinimumAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_MINIMUM_AMOUNT));
					tid.setValue(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_VALUE));
					tid.setAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_AMOUNT));
					discounts.add(tid);
				}
			}
		}
		return discounts;
	}

	public void addTodiscounts(TicketItemDiscount discount) {
		discounts = getDiscounts();
		if (discounts == null) {
			discounts = new ArrayList<>();
		}
		discounts.add(discount);
		ActionHistoryDAO.addTicketItemDiscountAddedActionHistory(this.getTicket(), discount);
	}

	public void buildDiscounts() {
		if (discounts == null || discounts.isEmpty()) {
			setDiscountsProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemDiscount ticketItemDiscount : discounts) {
			jsonArray.put(ticketItemDiscount.toJson());
		}
		setDiscountsProperty(jsonArray.toString());
	}

	public int getTableRowNum() {
		return tableRowNum;
	}

	public void setTableRowNum(int tableRowNum) {
		this.tableRowNum = tableRowNum;
	}

	public boolean canAddCookingInstruction() {
		if (isPrintedToKitchen())
			return false;

		return true;
	}

	@Override
	public String toString() {
		return "name: " + getName() + " id: " + getId();
	}

	public void setCookingInstructions(java.util.List<TicketItemCookingInstruction> cookingInstructions) {
		this.cookingInstructions = cookingInstructions;
		buildCoookingInstructions();
	}

	public void addCookingInstructions(List<TicketItemCookingInstruction> instructions) {
		cookingInstructions = getCookingInstructions();
		if (cookingInstructions == null) {
			cookingInstructions = new ArrayList<TicketItemCookingInstruction>(2);
		}
		for (TicketItemCookingInstruction ticketItemCookingInstruction : instructions) {
			addCookingInstruction(ticketItemCookingInstruction);
		}
	}

	public void addCookingInstruction(TicketItemCookingInstruction cookingInstruction) {
		cookingInstructions = getCookingInstructions();
		if (cookingInstructions == null) {
			cookingInstructions = new ArrayList<>();
			return;
		}
		cookingInstructions.add(cookingInstruction);
		buildCoookingInstructions();
	}

	public void buildCoookingInstructions() {
		if (cookingInstructions == null || cookingInstructions.isEmpty()) {
			setCookingInstructionsProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemCookingInstruction cookingIns : cookingInstructions) {
			jsonArray.put(cookingIns.toJson());
		}
		setCookingInstructionsProperty(jsonArray.toString());
	}

	public void removeCookingInstruction(TicketItemCookingInstruction itemCookingInstruction) {
		List<TicketItemCookingInstruction> cookingInstructions2 = getCookingInstructions();
		if (cookingInstructions2 == null || cookingInstructions2.size() == 0) {
			return;
		}
		for (Iterator iterator = cookingInstructions2.iterator(); iterator.hasNext();) {
			TicketItemCookingInstruction ticketItemCookingInstruction = (TicketItemCookingInstruction) iterator.next();
			if (ticketItemCookingInstruction.getTableRowNum() == itemCookingInstruction.getTableRowNum()) {
				iterator.remove();
				buildCoookingInstructions();
				return;
			}
		}
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (itemId != null && itemId.equals(menuModifier.getId())) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, int modifierType, OrderType type, Multiplier multiplier) {
		return addTicketItemModifier(menuModifier, modifierType, type, multiplier, 1);
	}

	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, int modifierType, OrderType type, Multiplier multiplier, double quantity) {
		TicketItemModifier ticketItemModifier = new TicketItemModifier();
		ticketItemModifier.setTaxIncluded(Application.getInstance().isPriceIncludesTax());
		ticketItemModifier.setItemId(menuModifier.getId());
		ticketItemModifier.setPageItemId(menuModifier.getPageItemId());
		MenuItemModifierSpec menuItemModifierGroup = menuModifier.getMenuItemModifierGroup();
		if (menuItemModifierGroup != null) {
			ticketItemModifier.setGroupId(menuItemModifierGroup.getId());
		}
		ticketItemModifier.setItemQuantity(quantity);
		ticketItemModifier.setName(menuModifier.getDisplayName());
		double price = menuModifier.getPriceForMultiplier(multiplier);
		if (multiplier != null) {
			ticketItemModifier.setMultiplierName(multiplier.getId());
			ticketItemModifier.setName(multiplier.getTicketPrefix() + " " + menuModifier.getDisplayName()); //$NON-NLS-1$
		}
		ticketItemModifier.setUnitPrice(price);
		setModifierUnitCost(menuModifier, multiplier, quantity, ticketItemModifier);
		ticketItemModifier.setTaxes(menuModifier.getTaxByOrderType(type, this));

		ticketItemModifier.setModifierType(modifierType);
		ticketItemModifier.setShouldPrintToKitchen(menuModifier.isShouldPrintToKitchen());
		ticketItemModifier.setTicketItem(this);
		if (StringUtils.isNotEmpty(menuModifier.getTranslatedName())) {
			ticketItemModifier.addProperty(AppConstants.TRANSLATED_NAME, menuModifier.getTranslatedName());
		}
		addToticketItemModifiers(ticketItemModifier);

		return ticketItemModifier;
	}

	private void setModifierUnitCost(MenuModifier menuModifier, Multiplier multiplier, double quantity, TicketItemModifier ticketItemModifier) {
		double cost = menuModifier.getCost();
		double d = multiplier != null ? (multiplier.getRate() / 100.0) : 0;
		double unitCost = cost * d;
		ticketItemModifier.setUnitCost(unitCost);
	}

	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, boolean addOn) {
		TicketItemModifier ticketItemModifier = new TicketItemModifier();
		//TODO: SET TAX INCLUDE FIELD
		ticketItemModifier.setTaxIncluded(Application.getInstance().isPriceIncludesTax());
		ticketItemModifier.setItemId(menuModifier.getId());
		MenuItemModifierSpec menuItemModifierGroup = menuModifier.getMenuItemModifierGroup();
		if (menuItemModifierGroup != null) {
			ticketItemModifier.setGroupId(menuItemModifierGroup.getId());
		}
		ticketItemModifier.setItemCount(1);
		ticketItemModifier.setName(menuModifier.getDisplayName());

		if (addOn) {
			ticketItemModifier.setUnitPrice(menuModifier.getExtraPrice());
			ticketItemModifier.setModifierType(TicketItemModifier.EXTRA_MODIFIER);
		}
		else {
			ticketItemModifier.setUnitPrice(menuModifier.getPrice());
			ticketItemModifier.setModifierType(TicketItemModifier.NORMAL_MODIFIER);
		}
		ticketItemModifier.setTaxes(menuModifier.getTaxByOrderType(ticketItemModifier.getTicketItem().getTicket().getOrderType(), this));
		ticketItemModifier.setShouldPrintToKitchen(menuModifier.isShouldPrintToKitchen());
		ticketItemModifier.setTicketItem(this);

		addToticketItemModifiers(ticketItemModifier);

		return ticketItemModifier;
	}

	public void updateModifiersUnitPrice(double defaultSellPortion) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				if (!ticketItemModifier.isInfoOnly()) {
					ticketItemModifier.setUnitPrice(ticketItemModifier.getUnitPrice() * defaultSellPortion / 100);
				}
			}
		}
	}

	public void updateModifiersUnitPriceByGroup(MenuItemModifierSpec group) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		List<TicketItemModifier> groupModifiers = new ArrayList<>();
		if (ticketItemModifiers != null) {
			double numOfModifiers = 0;
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				if (!ticketItemModifier.isInfoOnly() && ticketItemModifier.getGroupId().equals(group.getId())) {
					groupModifiers.add(ticketItemModifier);
					numOfModifiers += ticketItemModifier.getItemQuantity();
				}
			}
			Double groupPrice = group.getPrice((int) numOfModifiers);
			if (groupPrice != null && numOfModifiers > 0) {
				for (TicketItemModifier ticketItemModifier : groupModifiers) {
					ticketItemModifier.setUnitPrice(ticketItemModifier.getMultiplierPrice(groupPrice / (int) numOfModifiers));
				}
			}
		}
	}

	public boolean contains(TicketItemModifier ticketItemModifier) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		int count = 0;
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier2 : ticketItemModifiers) {
				if (!ticketItemModifier2.isInfoOnly()) {
					if (ticketItemModifier.getName().trim().equals(ticketItemModifier2.getName().trim())) {
						count++;
					}
				}
			}
		}
		return (count > 1) ? true : false;
	}

	public TicketItemModifier removeTicketItemModifier(TicketItemModifier ticketItemModifier) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers == null)
			return ticketItemModifier;

		for (Iterator iter = ticketItemModifiers.iterator(); iter.hasNext();) {
			TicketItemModifier oldTicketItemModifier = (TicketItemModifier) iter.next();
			if (oldTicketItemModifier.getItemId().equals(ticketItemModifier.getItemId())
					&& oldTicketItemModifier.getModifierType() == ticketItemModifier.getModifierType()) {
				iter.remove();
				return oldTicketItemModifier;
			}
		}
		return ticketItemModifier;
	}

	public TicketItemModifier removeTicketItemModifier(int modifierIndex) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers == null) {
			return null;
		}
		TicketItemModifier ticketItemModifier = ticketItemModifiers.get(modifierIndex);
		ticketItemModifiers.remove(modifierIndex);
		return ticketItemModifier;
	}

	public void calculatePrice() {
		if (getDataVersion() == 2) {
			calculatePriceV2();
		}
		else {
			calculatePriceV1();
		}
	}

	private void calculatePriceV1() {
		if (isTreatAsSeat()) {
			return;
		}

		//priceIncludesTax = Application.getInstance().isPriceIncludesTax();
		final BigDecimal unitPrice = convertToBigDecimal(getUnitPrice());
		BigDecimal itemQuantity = convertToBigDecimal(getQuantity());
		BigDecimal subtotalWithoutModifiers = unitPrice.multiply(itemQuantity);
		BigDecimal discountWithoutModifiers = calculateDiscount(subtotalWithoutModifiers);
		BigDecimal serviceCharge = calculateServiceCharge(subtotalWithoutModifiers.subtract(discountWithoutModifiers));
		BigDecimal modifierSubtotal = convertToBigDecimal("0"); //$NON-NLS-1$
		BigDecimal modifierDiscount = convertToBigDecimal("0"); //$NON-NLS-1$
		BigDecimal modifierTax = convertToBigDecimal("0"); //$NON-NLS-1$
		double cost = getUnitCost() * getQuantity();
		double modifierCost = 0;

		BigDecimal taxWithoutModifiers = null;
		if (isTaxOnServiceCharge()) {
			taxWithoutModifiers = calculateTax(subtotalWithoutModifiers.subtract(discountWithoutModifiers).add(serviceCharge));
		}
		else {
			taxWithoutModifiers = calculateTax(subtotalWithoutModifiers.subtract(discountWithoutModifiers));
		}

		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			Set<String> averagePricedModifierList = new HashSet<String>();
			for (TicketItemModifier modifier : ticketItemModifiers) {
				if (modifier.isInfoOnly()) {
					continue;
				}
				modifier.calculatePrice();
				if (modifier.isShouldSectionWisePrice() == true && averagePricedModifierList.contains(modifier.getItemId())) {
					continue;
				}
				modifierSubtotal = modifierSubtotal.add(convertToBigDecimal(modifier.getSubTotalAmount()));
				modifierDiscount = modifierDiscount.add(convertToBigDecimal(modifier.getDiscountAmount()));
				modifierTax = modifierTax.add(convertToBigDecimal(modifier.getTaxAmount()));
				serviceCharge = serviceCharge.add(convertToBigDecimal(modifier.getServiceCharge()));
				modifierCost += modifier.getTotalCost();

				averagePricedModifierList.add(modifier.getItemId());
			}
		}

		BigDecimal subtotal = subtotalWithoutModifiers.add(modifierSubtotal);
		BigDecimal discount = calculateDiscount(subtotal);
		BigDecimal tax = taxWithoutModifiers.add(modifierTax);

		BigDecimal total = convertToBigDecimal("0"); //$NON-NLS-1$
		BigDecimal totalWithoutModifiers = convertToBigDecimal("0"); //$NON-NLS-1$

		if (isTaxIncluded()) {
			total = subtotal.subtract(discount).add(serviceCharge);
			totalWithoutModifiers = subtotalWithoutModifiers.subtract(discountWithoutModifiers).add(serviceCharge);
		}
		else {
			total = subtotal.subtract(discount).add(tax).add(serviceCharge);
			totalWithoutModifiers = subtotalWithoutModifiers.subtract(discountWithoutModifiers).add(taxWithoutModifiers).add(serviceCharge);
		}

		setTotalCost(cost + modifierCost);
		setTotalAmountWithoutModifiers(cost);

		setSubtotalAmount(round(subtotal.doubleValue()));
		setSubtotalAmountWithoutModifiers(round(subtotalWithoutModifiers.doubleValue()));
		setDiscountAmount(round(discount.doubleValue()));
		setDiscountWithoutModifiers(round(discountWithoutModifiers.doubleValue()));

		setTaxAmount(round(tax.doubleValue()));
		setTaxAmountWithoutModifiers(round(taxWithoutModifiers.doubleValue()));
		setServiceCharge(round(serviceCharge.doubleValue()));
		setTotalAmount(round(total.doubleValue()));
		setTotalAmountWithoutModifiers(round(totalWithoutModifiers.doubleValue()));

		setAdjustedUnitPrice(unitPrice.doubleValue());
		setAdjustedDiscount(round(discount.doubleValue()));
		setAdjustedDiscountWithoutModifiers(round(discountWithoutModifiers.doubleValue()));
		setAdjustedSubtotal(round(subtotal.doubleValue()));
		setAdjustedSubtotalWithoutModifiers(round(subtotalWithoutModifiers.doubleValue()));

		setAdjustedTax(round(tax.doubleValue()));
		setAdjustedTaxWithoutModifiers(round(taxWithoutModifiers.doubleValue()));

		setAdjustedTotal(round(total.doubleValue()));
		setAdjustedTotalWithoutModifiers(round(totalWithoutModifiers.doubleValue()));
	}

	@Deprecated
	public void calculateAdjustedPrice() {
		if (isTreatAsSeat()) {
			return;
		}
		Ticket ticket = getTicket();
		if (ticket == null) {
			return;
		}

		//@formatter:off
		BigDecimal ticketSubtotalAmount = convertToBigDecimal(ticket.getSubtotalAmountWithVoidItems());
		BigDecimal ticketDiscountAmount = convertToBigDecimal(ticket.getTicketDiscountAmount());
		BigDecimal itemDiscountAmount = convertToBigDecimal(ticket.getItemDiscountAmount());
		BigDecimal ticketSubtotalAfterItemDiscount = ticketSubtotalAmount.subtract(itemDiscountAmount);
		BigDecimal ticketSubtotalAfterAllDiscount = ticketSubtotalAfterItemDiscount.subtract(ticketDiscountAmount);
		BigDecimal subtotalAfterDiscount = convertToBigDecimal(getSubtotalAmountWithoutModifiers())
				.subtract(convertToBigDecimal(getDiscountWithoutModifiers()));
		BigDecimal adjustedSubtotalWithoutModifiers = subtotalAfterDiscount;
		if (ticketSubtotalAfterItemDiscount.compareTo(BigDecimal.ZERO) != 0) {
			adjustedSubtotalWithoutModifiers = subtotalAfterDiscount.multiply(ticketSubtotalAfterAllDiscount).divide(ticketSubtotalAfterItemDiscount, 4,
					RoundingMode.HALF_UP);
		}

		BigDecimal serviceCharge = calculateServiceCharge(adjustedSubtotalWithoutModifiers);
		BigDecimal adjustedTaxWithoutModifiers = calculateTax(adjustedSubtotalWithoutModifiers.add(serviceCharge));
		BigDecimal itemQuantity = convertToBigDecimal(getQuantity());
		BigDecimal adjustedUnitPrice = adjustedSubtotalWithoutModifiers.divide(itemQuantity, 4, RoundingMode.HALF_UP);

		BigDecimal modifierAdjustedSubtotal = convertToBigDecimal("0"); //$NON-NLS-1$
		BigDecimal modifierAdjustedDiscount = convertToBigDecimal("0"); //$NON-NLS-1$
		BigDecimal modifierAdjustedTax = convertToBigDecimal("0"); //$NON-NLS-1$

		if (adjustedSubtotalWithoutModifiers.doubleValue() < 0) {
			adjustedSubtotalWithoutModifiers = BigDecimal.ZERO;
		}

		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			Set<String> averagePricedModifierList = new HashSet<String>();
			for (TicketItemModifier modifier : ticketItemModifiers) {
				if (modifier.isInfoOnly()) {
					continue;
				}
				modifier.calculateAdjustedPrice();
				if (!modifier.isShouldSectionWisePrice() && averagePricedModifierList.contains(modifier.getItemId())) {
					continue;
				}
				modifierAdjustedSubtotal = modifierAdjustedSubtotal.add(convertToBigDecimal(modifier.getAdjustedSubtotal()).abs());
				modifierAdjustedDiscount = modifierAdjustedDiscount.add(convertToBigDecimal(modifier.getAdjustedDiscount()).abs());
				modifierAdjustedTax = modifierAdjustedTax.add(convertToBigDecimal(modifier.getAdjustedTax()).abs());

				averagePricedModifierList.add(modifier.getItemId());
			}
		}

		BigDecimal adjustedSubtotal = adjustedSubtotalWithoutModifiers.add(modifierAdjustedSubtotal);
		BigDecimal myAdjustedDiscountWithoutModifiers = convertToBigDecimal(getSubtotalAmountWithoutModifiers()).subtract(adjustedSubtotalWithoutModifiers);
		BigDecimal adjustedDiscount = myAdjustedDiscountWithoutModifiers.add(modifierAdjustedDiscount);
		BigDecimal adjustedTax = adjustedTaxWithoutModifiers.add(modifierAdjustedTax);
		BigDecimal adjustedTotal = convertToBigDecimal(0);
		BigDecimal adjustedTotalWithoutModifiers = convertToBigDecimal(0);
		if (isTaxIncluded()) {
			adjustedTotal = adjustedSubtotal.add(serviceCharge);
			adjustedTotalWithoutModifiers = adjustedSubtotalWithoutModifiers.add(serviceCharge);
		}
		else {
			adjustedTotal = adjustedSubtotal.add(adjustedTax).add(serviceCharge);
			adjustedTotalWithoutModifiers = adjustedSubtotalWithoutModifiers.add(adjustedTaxWithoutModifiers).add(serviceCharge);
		}

//		adjustedTotal = round(adjustedTotal);

		setAdjustedUnitPrice(round(adjustedUnitPrice.doubleValue()));
		setAdjustedDiscount(round(adjustedDiscount.doubleValue()));
		setAdjustedDiscountWithoutModifiers(round(myAdjustedDiscountWithoutModifiers.doubleValue()));
		setAdjustedSubtotal(adjustedSubtotal.doubleValue());
		setAdjustedSubtotalWithoutModifiers(round(adjustedSubtotalWithoutModifiers.doubleValue()));
		setServiceCharge(round(serviceCharge.doubleValue()));
		setAdjustedTax(round(adjustedTax.doubleValue()));
		setAdjustedTaxWithoutModifiers(round(adjustedTaxWithoutModifiers.doubleValue()));
		setAdjustedTotal(round(adjustedTotal.doubleValue()));
		setAdjustedTotalWithoutModifiers(round(adjustedTotalWithoutModifiers.doubleValue()));
		//@formatter:on
	}

	public void calculatePriceV2() {
		if (isTreatAsSeat()) {
			return;
		}
		Ticket ticket = getTicket();
		final double unitPrice = getUnitPrice();
		final double itemQuantity = getQuantity();
		final double subtotalWithoutModifiers = round(unitPrice * itemQuantity);
		final double discountWithoutModifiers = round(calculateDiscountV2(subtotalWithoutModifiers));
		double serviceChargeWithoutModifiers = 0;
		double modifierSubtotal = 0.0; //$NON-NLS-1$
		double modifierDiscount = 0.0; //$NON-NLS-1$
		double modifierTax = 0.0; //$NON-NLS-1$
		double modifierServiceCharge = 0;
		double taxWithoutModifiers = 0.0;
		double modifierCost = 0;
		double cost = getUnitCost() * itemQuantity;

		if (ticket != null && (ticket.isDiscountOnSerivceCharge() || ticket.hasRepriceDiscount())) {
			serviceChargeWithoutModifiers = round(calculateServiceChargeV2(subtotalWithoutModifiers - discountWithoutModifiers));
		}
		else {
			serviceChargeWithoutModifiers = round(calculateServiceChargeV2(subtotalWithoutModifiers));
		}

		if (isTaxOnServiceCharge()) {
			taxWithoutModifiers = round(calculateTaxV2(subtotalWithoutModifiers - discountWithoutModifiers + serviceChargeWithoutModifiers));
		}
		else {
			taxWithoutModifiers = round(calculateTaxV2(subtotalWithoutModifiers - discountWithoutModifiers));
		}

		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			Set<String> averagePricedModifierList = new HashSet<String>();
			for (TicketItemModifier modifier : ticketItemModifiers) {
				if (modifier.isInfoOnly()) {
					continue;
				}
				modifier.calculatePriceV2();
				if (modifier.isShouldSectionWisePrice() == true && averagePricedModifierList.contains(modifier.getItemId())) {
					continue;
				}
				modifierSubtotal += modifier.getSubTotalAmount();
				modifierDiscount += modifier.getDiscountAmount();
				modifierTax += modifier.getTaxAmount();
				modifierServiceCharge += modifier.getServiceCharge();
				modifierCost += modifier.getTotalCost();

				averagePricedModifierList.add(modifier.getItemId());
			}
		}

		final double totalCost = round(cost + modifierCost);
		final double subtotal = round(subtotalWithoutModifiers + modifierSubtotal);
		final double discount = round(discountWithoutModifiers + modifierDiscount);
		final double tax = round(taxWithoutModifiers + modifierTax);
		final double serviceCharge = round(serviceChargeWithoutModifiers + modifierServiceCharge);
		double total = 0.0; //$NON-NLS-1$
		double totalWithoutModifiers = 0.0; //$NON-NLS-1$

		if (isTaxIncluded()) {
			total = round(subtotal - discount + serviceCharge);
			totalWithoutModifiers = round(subtotalWithoutModifiers - discountWithoutModifiers + serviceChargeWithoutModifiers);
		}
		else {
			total = round(subtotal - discount + tax + serviceCharge);
			totalWithoutModifiers = round(subtotalWithoutModifiers - discountWithoutModifiers + taxWithoutModifiers + serviceChargeWithoutModifiers);
		}

		setTotalCost(totalCost);
		setTotalCostWithoutModifiers(cost);

		setSubtotalAmount(subtotal);
		setSubtotalAmountWithoutModifiers(subtotalWithoutModifiers);
		setDiscountAmount(discount);
		setDiscountWithoutModifiers(discountWithoutModifiers);

		setTaxAmount(tax);
		setTaxAmountWithoutModifiers(taxWithoutModifiers);
		setServiceCharge(round(serviceCharge));
		setServiceChargeWithoutModifiers(serviceChargeWithoutModifiers);
		setTotalAmount(total);
		setTotalAmountWithoutModifiers(totalWithoutModifiers);

		setAdjustedUnitPrice(unitPrice);
		setAdjustedDiscount(discount);
		setAdjustedDiscountWithoutModifiers(discountWithoutModifiers);
		setAdjustedSubtotal(subtotal);
		setAdjustedSubtotalWithoutModifiers(subtotalWithoutModifiers);

		setAdjustedTax(tax);
		setAdjustedTaxWithoutModifiers(taxWithoutModifiers);

		setAdjustedTotal(total);
		setAdjustedTotalWithoutModifiers(totalWithoutModifiers);
	}

	public void calculateAdjustedPriceV2() {
		if (isTreatAsSeat() || !isTicketDiscountApplicable()) {
			return;
		}
		Ticket ticket = getTicket();
		if (ticket == null) {
			return;
		}

		double itemQuantity = getQuantity();
		double ticketDiscountPercentage = ticket.getDiscountPercentageRate();
		double unitPriceAfterDiscount = ((getSubtotalAmountWithoutModifiers() - getDiscountWithoutModifiers()) / itemQuantity);
		double adjustedUnitPrice = unitPriceAfterDiscount - (unitPriceAfterDiscount * ticketDiscountPercentage);
		if (getUnitPrice() == 0) {
			adjustedUnitPrice = 0;
		}

		double adjustedSubtotalWithoutModifiers = round(adjustedUnitPrice * itemQuantity);
		double adjustedDiscountWithoutModifiers = round(getSubtotalAmountWithoutModifiers() - adjustedSubtotalWithoutModifiers);
		double adjustedScWithoutModifiers = 0;
		double adjustedTaxWithoutModifiers = 0.0;
		double modifierServiceCharge = 0;

		if (ticket != null && ticket.isDiscountOnSerivceCharge() || ticket.hasRepriceDiscount()) {
			adjustedScWithoutModifiers = round(calculateServiceChargeV2(adjustedSubtotalWithoutModifiers));
		}
		else {
			adjustedScWithoutModifiers = getServiceChargeWithoutModifiers();
		}

		if (isTaxOnServiceCharge()) {
			adjustedTaxWithoutModifiers = round(calculateTaxV2(adjustedSubtotalWithoutModifiers + adjustedScWithoutModifiers));
		}
		else {
			adjustedTaxWithoutModifiers = round(calculateTaxV2(adjustedSubtotalWithoutModifiers));
		}
		double modifierAdjustedSubtotal = 0.0; //$NON-NLS-1$
		double modifierAdjustedDiscount = 0.0; //$NON-NLS-1$
		double modifierAdjustedTax = 0.0; //$NON-NLS-1$

		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			Set<String> averagePricedModifierList = new HashSet<String>();
			for (TicketItemModifier modifier : ticketItemModifiers) {
				if (modifier.isInfoOnly()) {
					continue;
				}
				modifier.calculateAdjustedPriceV2();
				if (!modifier.isShouldSectionWisePrice() && averagePricedModifierList.contains(modifier.getItemId())) {
					continue;
				}
				modifierAdjustedSubtotal += modifier.getAdjustedSubtotal();
				modifierAdjustedDiscount += modifier.getAdjustedDiscount();
				modifierAdjustedTax += modifier.getAdjustedTax();
				modifierServiceCharge += modifier.getServiceCharge();

				averagePricedModifierList.add(modifier.getItemId());
			}
		}

		double adjustedSubtotal = round(adjustedSubtotalWithoutModifiers + modifierAdjustedSubtotal);
		double adjustedDiscount = round(adjustedDiscountWithoutModifiers + modifierAdjustedDiscount);
		double adjustedTax = round(adjustedTaxWithoutModifiers + modifierAdjustedTax);
		double adjustedTotal = 0.0;
		double adjustedTotalWithoutModifiers = 0.0;
		double adjustedServiceCharge = round(adjustedScWithoutModifiers + modifierServiceCharge);

		if (isTaxIncluded()) {
			adjustedTotal = round(adjustedSubtotal + adjustedScWithoutModifiers);
			adjustedTotalWithoutModifiers = round(adjustedSubtotalWithoutModifiers + adjustedScWithoutModifiers);
		}
		else {
			adjustedTotal = round(adjustedSubtotal + adjustedTax + adjustedServiceCharge);
			adjustedTotalWithoutModifiers = round(adjustedSubtotalWithoutModifiers + adjustedTaxWithoutModifiers + adjustedScWithoutModifiers);
		}

		setAdjustedUnitPrice(adjustedUnitPrice);

		setAdjustedDiscount(adjustedDiscount);
		setAdjustedDiscountWithoutModifiers(adjustedDiscountWithoutModifiers);

		setServiceCharge(adjustedServiceCharge);
		setServiceChargeWithoutModifiers(adjustedScWithoutModifiers);

		setAdjustedSubtotal(adjustedSubtotal);
		setAdjustedSubtotalWithoutModifiers(adjustedSubtotalWithoutModifiers);

		setAdjustedTax(adjustedTax);
		setAdjustedTaxWithoutModifiers(adjustedTaxWithoutModifiers);

		setAdjustedTotal(adjustedTotal);
		setAdjustedTotalWithoutModifiers(adjustedTotalWithoutModifiers);
		//@formatter:on
	}

	@Deprecated
	public BigDecimal calculateServiceCharge(BigDecimal subtotalAmount) {
		if (!isServiceChargeApplicable()) {
			return BigDecimal.ZERO;
		}
		Ticket ticket = getTicket();
		if (ticket == null || ticket.getOrderType() == null) {
			return new BigDecimal("0"); //$NON-NLS-1$
		}
		Double serviceChargeRate = getServiceChargeRate();
		if (serviceChargeRate.doubleValue() == 0) {
			serviceChargeRate = ticket.getOutletServiceChargeRate();
		}
		if (serviceChargeRate == 0.0 || (!ticket.isServiceChargeApplicable())) {
			return new BigDecimal("0"); //$NON-NLS-1$
		}

		BigDecimal bdChargeRate = convertToBigDecimal(serviceChargeRate / 100.0);
		return round(subtotalAmount.multiply(bdChargeRate));
	}

	public double calculateServiceChargeV2(double subtotalAmount) {
		if (!isServiceChargeApplicable()) {
			return 0.0;
		}
		Ticket ticket = getTicket();
		if (ticket == null || !ticket.isServiceChargeApplicable()) {
			return 0.0;
		}

		Double serviceChargePercentage = getServiceChargeRate();
		if (serviceChargePercentage == 0) {
			serviceChargePercentage = ticket.getOutletServiceChargeRate();
		}

		if (serviceChargePercentage == 0.0) {
			return 0; //$NON-NLS-1$
		}

		double serviceCharge = subtotalAmount * (serviceChargePercentage / 100.0);
		if (serviceCharge < 0 && !isServiceChargeRefundable()) {
			serviceCharge = 0;
		}
		return serviceCharge;
	}

	public boolean isMergable(TicketItem otherItem, boolean merge) {
		if (this.isTreatAsSeat() || otherItem.isTreatAsSeat()) {
			return false;
		}
		if (this.isFractionalUnit() || otherItem.isFractionalUnit()) {
			return false;
		}
		if ((this.getQuantity() > 0 && otherItem.getQuantity() < 0) || (this.getQuantity() < 0 && otherItem.getQuantity() > 0)) {
			return false;
		}
		if (StringUtils.isEmpty(this.getMenuItemId()) || StringUtils.isEmpty(otherItem.getMenuItemId())) {
			return false;
		}
		if (this.hasCookingInstructions() || otherItem.hasCookingInstructions()) {
			return false;
		}
		if (this.isReturned() || otherItem.isReturned()) {
			return false;
		}
		if (!this.getMenuItemId().equals(otherItem.getMenuItemId())) {
			return false;
		}
		if (!this.getUnitPrice().equals(otherItem.getUnitPrice())) {
			return false;
		}
		if (!this.getSeatNumber().equals(otherItem.getSeatNumber())) {
			return false;
		}
		if (!this.isHasModifiers() && !otherItem.isHasModifiers()) {
			return true;
		}
		if (!isMergableModifiers(getTicketItemModifiers(), otherItem.getTicketItemModifiers(), merge)) {
			return false;
		}
		return true;
	}

	public boolean isMergableModifiers(List<TicketItemModifier> thisModifiers, List<TicketItemModifier> thatModifiers, boolean merge) {
		if (thatModifiers == null || thisModifiers == null) {
			return true;
		}
		if (thisModifiers.size() != thatModifiers.size()) {
			return false;
		}

		Comparator<TicketItemModifier> comparator = new Comparator<TicketItemModifier>() {
			@Override
			public int compare(TicketItemModifier o1, TicketItemModifier o2) {
				return o1.getItemId().compareTo(o2.getItemId());
			}
		};

		Collections.sort(thisModifiers, comparator);
		Collections.sort(thatModifiers, comparator);

		Iterator<TicketItemModifier> thisIterator = thisModifiers.iterator();
		Iterator<TicketItemModifier> thatIterator = thatModifiers.iterator();

		while (thisIterator.hasNext()) {
			TicketItemModifier next1 = thisIterator.next();
			TicketItemModifier next2 = thatIterator.next();

			if (comparator.compare(next1, next2) != 0) {
				return false;
			}

			if (merge) {
				next1.merge(next2);
			}
		}
		return true;
	}

	public void doCalculateComboItemPrice() {
		if (!(this instanceof ComboTicketItem)) {
			return;
		}

		ComboTicketItem ticketItem = (ComboTicketItem) this;
		List<ComboGroup> comboGroups = ticketItem.getMenuItem().getComboGroups();
		if (comboGroups == null || comboGroups.isEmpty()) {
			return;
		}

		double comboItemsPrice = ticketItem.getComboItemsPrice();
		for (ComboGroup comboGroup : comboGroups) {
			if (comboGroup == null) {
				continue;
			}
			List<MenuItem> menuItems = comboGroup.getItems();
			if (menuItems == null || menuItems.isEmpty()) {
				continue;
			}
			List<TicketItem> addedItems = ticketItem.getComboItems();
			for (MenuItem menuItem : menuItems) {
				if (menuItem == null || StringUtils.isEmpty(menuItem.getId())) {
					continue;
				}
				for (TicketItem addedTicketItem : addedItems) {
					if (addedTicketItem == null || StringUtils.isEmpty(addedTicketItem.getMenuItemId())
							|| !menuItem.getId().equals(addedTicketItem.getMenuItemId()) || addedTicketItem.getGroupId() == null) {
						continue;
					}
					comboItemsPrice += comboGroup.getItemPrice(menuItem.getId());
				}
			}
		}
		ticketItem.setUnitPrice(comboItemsPrice);
	}

	public void merge(TicketItem otherItem) {
		if (!this.isHasModifiers() && !otherItem.isHasModifiers()) {
			this.setQuantity(this.getQuantity() + otherItem.getQuantity());
			return;
		}
		if (isMergable(otherItem, false)) {
			this.setQuantity(this.getQuantity() + otherItem.getQuantity());
		}
	}

	public boolean hasCookingInstructions() {
		return this.getCookingInstructions() != null && this.getCookingInstructions().size() > 0;
	}

	public BigDecimal calculateDiscount(final BigDecimal subtotalAmount) {
		BigDecimal discount = convertToBigDecimal(0);

		List<TicketItemDiscount> discounts = getDiscounts();
		if (discounts != null) {
			for (TicketItemDiscount ticketItemDiscount : discounts) {
				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
					continue;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					discount = discount.add(convertToBigDecimal(ticketItemDiscount.calculateDiscount(subtotalAmount.doubleValue() - discount.doubleValue())));
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
					double discountAmount = subtotalAmount.doubleValue() == 0.0 ? 0.0
							: Math.abs((ticketItemDiscount.getValue() * 100.0) / subtotalAmount.doubleValue());
					TicketItemDiscount percentDiscount = new TicketItemDiscount();
					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
					percentDiscount.setValue(discountAmount);
					percentDiscount.setTicketItem(this);
					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
					BigDecimal convertToBigDecimal = convertToBigDecimal(percentDiscount.calculateDiscount(subtotalAmount.doubleValue()));
					discount = discount.add(convertToBigDecimal);
					ticketItemDiscount.setAmount(convertToBigDecimal.doubleValue());
				}
				//discount += ticketItemDiscount.calculateDiscount(subtotalAmount);
			}
		}
		buildDiscounts();
		if (discount.compareTo(subtotalAmount.abs()) > 0)
			return subtotalAmount;
		return round(discount);
	}

	public double calculateDiscountV2(final double subtotalAmount) {
		double discount = 0.0;

		List<TicketItemDiscount> discounts = getDiscounts();
		if (discounts != null) {
			for (TicketItemDiscount ticketItemDiscount : discounts) {
				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
					continue;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					discount += ticketItemDiscount.calculateDiscount(subtotalAmount - discount);
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
					double discountAmount = subtotalAmount == 0.0 ? 0.0 : Math.abs((ticketItemDiscount.getValue() * 100.0) / subtotalAmount);
					TicketItemDiscount percentDiscount = new TicketItemDiscount();
					percentDiscount.setOriginalType(ticketItemDiscount.getType());
					percentDiscount.setOriginalValue(ticketItemDiscount.getValue());

					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
					percentDiscount.setValue(discountAmount);
					percentDiscount.setTicketItem(this);
					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
					double perdentDiscAmount = percentDiscount.calculateDiscount(subtotalAmount);
					discount += perdentDiscAmount;
					ticketItemDiscount.setAmount(perdentDiscAmount);
				}
			}
		}
		buildDiscounts();
		if (discount > Math.abs(subtotalAmount))
			return subtotalAmount;
		return discount;
	}

	public double getAmountByType(TicketItemDiscount discount) {

		switch (discount.getType()) {
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return discount.getValue();

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return (discount.getValue() * getUnitPrice()) / 100;

			default:
				break;
		}

		return 0;
	}

	@Deprecated
	private BigDecimal calculateTax(BigDecimal subtotal) {
		Ticket ticket = getTicket();

		BigDecimal totalTaxAmount = convertToBigDecimal("0"); //$NON-NLS-1$
		List<TicketItemTax> taxList = getTaxes();
		if (taxList != null) {
			if (isTaxIncluded()) {
				BigDecimal taxRatePercentage = convertToBigDecimal(0);
				for (TicketItemTax ticketItemTax : taxList) {
					taxRatePercentage = taxRatePercentage
							.add(convertToBigDecimal(ticketItemTax.getRate()).divide(convertToBigDecimal("100"), 4, RoundingMode.HALF_EVEN)); //$NON-NLS-1$
				}
				if (taxRatePercentage.compareTo(convertToBigDecimal("0")) == 0) { //$NON-NLS-1$
					return totalTaxAmount;
				}
				else {
					BigDecimal totalTaxRatePercentage = taxRatePercentage.add(convertToBigDecimal(1d));
					BigDecimal actualPrice = subtotal.divide(totalTaxRatePercentage, 4, RoundingMode.HALF_EVEN);

					for (TicketItemTax ticketItemTax : taxList) {
						taxRatePercentage = convertToBigDecimal(ticketItemTax.getRate()).divide(convertToBigDecimal("100"), 4, RoundingMode.HALF_EVEN); //$NON-NLS-1$
						BigDecimal tax = actualPrice.multiply(taxRatePercentage);
						ticketItemTax.setTaxAmount(tax.doubleValue());
						totalTaxAmount = totalTaxAmount.add(tax);
					}
				}
			}
			else {
				for (TicketItemTax ticketItemTax : taxList) {
					BigDecimal taxRatePercentage = convertToBigDecimal(ticketItemTax.getRate()).divide(convertToBigDecimal("100"), 4, RoundingMode.HALF_EVEN); //$NON-NLS-1$
					if (taxRatePercentage.compareTo(convertToBigDecimal("0")) == 0) { //$NON-NLS-1$
						continue;
					}
					else {
						BigDecimal tax = subtotal.multiply(taxRatePercentage);
						ticketItemTax.setTaxAmount(tax.doubleValue());
						totalTaxAmount = totalTaxAmount.add(tax);
					}
				}
			}
		}
		if (totalTaxAmount.doubleValue() < 0) {
			totalTaxAmount = BigDecimal.ZERO;
		}

		buildTaxes();

		if (ticket != null && ticket.isTaxExempt()) {
			setTaxExemptAmount(totalTaxAmount.doubleValue());
			return BigDecimal.ZERO;
		}
		if (ticket != null && Boolean.valueOf(ticket.getProperty(Ticket.CUSTOMER_TAX_EXEMPT))) {
			setTaxExemptAmount(totalTaxAmount.doubleValue());
			return BigDecimal.ZERO;
		}

		return Tax.applyFloridaTaxRule(subtotal, totalTaxAmount);
	}

	private double calculateTaxV2(double subtotalAmount) {
		Ticket ticket = getTicket();

		double totalTaxAmount = 0; //$NON-NLS-1$
		List<TicketItemTax> taxList = getTaxes();
		if (taxList != null) {
			if (isTaxIncluded()) {
				double taxRatePercentage = 0;
				for (TicketItemTax ticketItemTax : taxList) {
					taxRatePercentage = (taxRatePercentage + ticketItemTax.getRate()) / 100.0; //$NON-NLS-1$
				}
				if (NumberUtil.isZero(taxRatePercentage)) { //$NON-NLS-1$
					return totalTaxAmount;
				}
				else {
					double totalTaxRatePercentage = taxRatePercentage + 1.0;
					double actualPrice = subtotalAmount / totalTaxRatePercentage;

					for (TicketItemTax ticketItemTax : taxList) {
						taxRatePercentage = ticketItemTax.getRate() / 100.0; //$NON-NLS-1$
						double taxAmount = actualPrice * taxRatePercentage;
						ticketItemTax.setTaxAmount(taxAmount);
						totalTaxAmount += taxAmount;
					}
				}
			}
			else {
				for (TicketItemTax ticketItemTax : taxList) {
					double taxRatePercentage = ticketItemTax.getRate() / 100.0;
					if (NumberUtil.isZero(taxRatePercentage)) {
						continue;
					}
					else {
						double tax = subtotalAmount * taxRatePercentage;
						ticketItemTax.setTaxAmount(tax);
						totalTaxAmount += tax;
					}
				}
			}
		}
		totalTaxAmount = Tax.applyFloridaTaxRuleV2(getTicket(), subtotalAmount, totalTaxAmount);
		buildTaxes();
		setTaxExemptAmount(0.0);
		if (ticket != null && ticket.isTaxExempt()) {
			setTaxExemptAmount(totalTaxAmount);
			return 0.0;
		}

		return totalTaxAmount;
	}

	public double getTotalTaxRate() {
		List<TicketItemTax> ticketItemTaxes = getTaxes();
		if (ticketItemTaxes == null || ticketItemTaxes.isEmpty())
			return 0;

		double totalTaxRate = 0;
		for (TicketItemTax tax : ticketItemTaxes) {
			totalTaxRate += tax.getRate();
		}
		return totalTaxRate;
	}

	@Override
	public String getNameDisplay() {
		return getNameDisplay(getName());
	}

	public String getNameDisplay(String name) {
		if (isTreatAsSeat()) {
			return name;
		}

		String displayName = ""; //$NON-NLS-1$

		if (getQuantity() < 0) {
			if (isVoided() && !isItemReturned()) {
				displayName += "*Voided* "; //$NON-NLS-1$
				displayName += getItemQuantityDisplay(true); //$NON-NLS-1$
			}
			else {
				displayName += getItemQuantityDisplay(); //$NON-NLS-1$
			}
		}
		else {
			displayName += getItemQuantityDisplay();
		}
		if (isComboItem()) {
			List<TicketItem> comboItems = getComboItems();
			if (comboItems != null && !comboItems.isEmpty()) {
				displayName += name;
				displayName += "\n"; //$NON-NLS-1$
				for (Iterator iterator = comboItems.iterator(); iterator.hasNext();) {
					TicketItem item = (TicketItem) iterator.next();
					displayName += " #" + NumberUtil.trimDecilamIfNotNeeded(item.getQuantity() / getQuantity(), true) + " " + item.getName(); //$NON-NLS-1$ //$NON-NLS-2$
					List<TicketItemModifier> ticketItemModifiers = item.getTicketItemModifiers();
					if (iterator.hasNext() || ticketItemModifiers != null) {
						displayName += "\n"; //$NON-NLS-1$
					}
					if (ticketItemModifiers != null && !ticketItemModifiers.isEmpty()) {
						for (Iterator iterator2 = ticketItemModifiers.iterator(); iterator2.hasNext();) {
							TicketItemModifier modifier = (TicketItemModifier) iterator2.next();
							displayName += modifier.getNameDisplay();
							if (iterator2.hasNext()) {
								displayName += "\n"; //$NON-NLS-1$
							}
							else if (iterator.hasNext()) {
								displayName += "\n"; //$NON-NLS-1$
							}
						}
					}
				}
				return displayName;
			}
		}
		else if (getSizeModifier() != null) {
			displayName += getSizeModifier().getNameDisplay().replaceAll(" -- ", "") + " " + getName(); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return displayName;
		}

		return displayName += name;
	}

	@Override
	public String getUnitPriceDisplay() {
		if (isTreatAsSeat())
			return null;

		return NumberUtil.formatNumberAcceptNegative(getUnitPrice());
	}

	@Override
	public String getItemQuantityDisplay() {
		return getItemQuantityDisplay(false);
	}

	public String getItemQuantityDisplay(boolean itemVoided) {
		if (isTreatAsSeat())
			return ""; //$NON-NLS-1$

		Double quantity = itemVoided ? Math.abs(getQuantity()) : getQuantity();
		if (!isFractionalUnit() && quantity == 1) {
			return ""; //$NON-NLS-1$
		}

		String itemQuantity = NumberUtil.trimDecilamIfNotNeeded(quantity, true);
		String unitName = getUnitName();
		if (unitName.equals("") || "ea".equals(unitName) || "pc".equals(unitName)) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return itemQuantity + "x "; //$NON-NLS-1$
		}
		return itemQuantity + unitName + " "; //$NON-NLS-1$
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		return NumberUtil.formatNumberAcceptNegative(getTaxAmountWithoutModifiers());
	}

	@Override
	public String getTotalAmountWithoutModifiersDisplay() {
		return NumberUtil.formatNumberAcceptNegative(getTotalAmountWithoutModifiers());
	}

	@Override
	public String getSubTotalAmountWithoutModifiersDisplay() {
		if (isTreatAsSeat())
			return null;
		return NumberUtil.formatNumberAcceptNegative(getSubtotalAmountWithoutModifiers());
	}

	@Override
	public String getItemCode() {
		return String.valueOf(getMenuItemId());
	}

	public List<Printer> getPrinters(OrderType orderType) {
		PrinterGroup printerGroup = getPrinterGroup();

		List<Printer> printerAll = new ArrayList<Printer>();
		PosPrinters printers = DataProvider.get().getPrinters();
		List<Printer> kitchenPrinters = printers.getKitchenPrinters();
		printerAll.addAll(printers.getStickerPrinters());

		if (printerGroup == null && (kitchenPrinters == null || kitchenPrinters.isEmpty()) && printers.isPrintToKds()) {
			printerAll.add(VirtualPrinter.getKdsPrinter());
		}

		if (printerGroup == null && kitchenPrinters != null && kitchenPrinters.size() > 0) {
			Printer printer = kitchenPrinters.get(0);
			if (StringUtils.isBlank(printer.getDeviceName())) {
				printerAll.add(VirtualPrinter.getKdsPrinter());
			}
			else {
				printerAll.add(printer);
			}
			return printerAll;
		}

		if (printerGroup != null) {
			List<String> printerNames = printerGroup.getPrinterNames();
			for (Printer printer : kitchenPrinters) {
				VirtualPrinter virtualPrinter = printer.getVirtualPrinter();
				if (StringUtils.isBlank(printer.getDeviceName())) {
					continue;
				}
				if (printerNames.contains(virtualPrinter.getName())) {
					printerAll.add(printer);
				}
			}
		}
		if (printerAll.isEmpty() && printers.isPrintToKds()) {
			printerAll.add(VirtualPrinter.getKdsPrinter());
		}

		return printerAll;
	}

	@Override
	public boolean canAddDiscount() {
		return true;
	}

	@Override
	public boolean canVoid() {
		return true;
	}

	@Override
	public boolean canAddAdOn() {
		return true;
	}

	public MenuItem getMenuItem() {
		if (menuItem == null) {
			String itemId = getMenuItemId();
			if (StringUtils.isEmpty(itemId)) {
				return null;
			}
			menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}

		return menuItem;
	}

	@XmlTransient
	public void setMenuItem(MenuItem menuItem) {
		this.menuItem = menuItem;
		//		setMenuItemId(menuItem.getId());
		String menuItemId = null;
		if (menuItem != null) {
			menuItemId = menuItem.getId();
		}
		super.setMenuItemId(menuItemId);
	}

	public KitchenStatus getKitchenStatusValue() {
		return KitchenStatus.fromString(super.getKitchenStatus());
	}

	public void setKitchenStatusValue(KitchenStatus kitchenStatus) {
		super.setKitchenStatus(kitchenStatus.name());
	}

	/**
	 * Set kitchen status of this item, modifier of this item and cooking instructions of this item.
	 * 
	 * @param kitchenStatus
	 */
	public void setKitchenStatusValueWithChildren(KitchenStatus kitchenStatus) {
		super.setKitchenStatus(kitchenStatus.name());
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				ticketItemModifier.setKitchenStatusValue(kitchenStatus);
			}
		}
		List<TicketItemCookingInstruction> cookingInstructions = getCookingInstructions();
		if (cookingInstructions != null) {
			for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
				ticketItemCookingInstruction.setKitchenStatusValue(kitchenStatus);
			}
			buildCoookingInstructions();
		}
	}

	@Override
	public String getUnitName() {
		if (super.getUnitName() == null) {
			return ""; //$NON-NLS-1$
		}
		return super.getUnitName();
	}

	public Double getQuantityToShip() {
		return quantityToShip == null ? 0.0 : quantityToShip;
	}

	public void setQuantityToShip(Double shipQuantity) {
		this.quantityToShip = shipQuantity;
	}

	@Override
	public String getSubTotalAmountDisplay() {
		if (isTreatAsSeat())
			return null;
		if (isVoided() && !isItemReturned()) {
			return "(" + NumberUtil.format(Math.abs(getSubtotalAmount())) + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return NumberUtil.format(getSubtotalAmount());
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			String menuItemMGId = menuModifier.getMenuItemModifierGroup() == null ? null : menuModifier.getMenuItemModifierGroup().getId();
			String ticketItemMGId = ticketItemModifier.getGroupId();
			String multiplierName = ticketItemModifier.getMultiplierName();
			boolean idIsNull = itemId == null || menuItemMGId == null || ticketItemMGId == null;
			if (idIsNull) {
				return null;
			}
			if (multiplier != null && !multiplier.getId().equals(multiplierName)) {
				return null;
			}
			if (menuItemMGId.equals(ticketItemMGId) && itemId.equals(menuModifier.getId())) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public List<TicketItemModifier> findTicketItemModifiersFor(MenuModifier menuModifier, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		List<TicketItemModifier> ticketItemModifiers = new ArrayList<>();
		if (modifiers == null) {
			return ticketItemModifiers;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (itemId != null && multiplier != null && itemId.equals(menuModifier.getId())
					&& multiplier.getId().equals(ticketItemModifier.getMultiplierName())) {
				ticketItemModifiers.add(ticketItemModifier);
			}
		}
		return ticketItemModifiers;
	}

	public double findTicketItemModifierByMenuModifier(MenuModifier menuModifier) {
		if (menuModifier == null) {
			return 0;
		}
		double totalAmount = 0;
		List<Multiplier> multiplierList = DataProvider.get().getMultiplierList();
		if (multiplierList != null) {
			for (Multiplier multiplier : multiplierList) {
				List<TicketItemModifier> ticketItemModifiers = findTicketItemModifiersFor(menuModifier, multiplier);
				if (ticketItemModifiers != null && !ticketItemModifiers.isEmpty()) {
					for (TicketItemModifier modifier : ticketItemModifiers) {
						totalAmount += modifier.getItemQuantity();
					}
				}
			}
		}
		return totalAmount;
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, String sectionName) {
		return findTicketItemModifierFor(menuModifier, sectionName, null);
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, String sectionName, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (multiplier != null) {
				if ((itemId != null && itemId.equals(menuModifier.getId())) && (sectionName != null && sectionName.equals(ticketItemModifier.getSectionName())
						&& (multiplier != null && multiplier.getId().equals(ticketItemModifier.getMultiplierName())))) {
					return ticketItemModifier;
				}
			}

			if ((itemId != null && itemId.equals(menuModifier.getId())) && (sectionName != null && sectionName.equals(ticketItemModifier.getSectionName()))) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public int countModifierFromGroup(MenuItemModifierSpec menuItemModifierGroup) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return 0;
		}
		int modifierFromGroupCount = 0;
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String groupId = ticketItemModifier.getGroupId();
			if (groupId != null && groupId.equals(menuItemModifierGroup.getId())) {
				modifierFromGroupCount += ticketItemModifier.getItemQuantity();
			}
		}
		return modifierFromGroupCount;
	}

	public boolean requiredModifiersAdded(MenuItemModifierSpec menuItemModifierGroup) {
		int minQuantity = menuItemModifierGroup.getMinQuantity();
		if (minQuantity == 0) {
			return true;
		}
		return countModifierFromGroup(menuItemModifierGroup) >= minQuantity;
	}

	public boolean deleteTicketItemModifier(TicketItemModifier ticketItemModifierToRemove) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null || modifiers.isEmpty()) {
			return false;
		}
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			TicketItemModifier ticketItemModifier = (TicketItemModifier) iterator.next();
			if (ticketItemModifier == ticketItemModifierToRemove) {
				iterator.remove();
				return true;
			}
		}

		return false;
	}

	public boolean deleteTicketItemModifierByName(TicketItemModifier ticketItemModifierToRemove) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return false;
		}
		for (Iterator iterator = modifiers.iterator(); iterator.hasNext();) {
			TicketItemModifier ticketItemModifier = (TicketItemModifier) iterator.next();
			if (ticketItemModifier.getName().equals(ticketItemModifierToRemove.getName())) {
				iterator.remove();
				return true;
			}
		}

		return false;
	}

	public enum PIZZA_SECTION_MODE {
		FULL(1), HALF(2), QUARTER(3);

		private final int value;

		private PIZZA_SECTION_MODE(int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}

		public static PIZZA_SECTION_MODE from(int value) {
			if (value == 2) {
				return HALF;
			}
			if (value == 3) {
				return QUARTER;
			}
			return FULL;
		}

		@Override
		public String toString() {
			return name();
		}
	}

	public void setPizzaSectionMode(PIZZA_SECTION_MODE pizzaSectionMode) {
		setPizzaSectionModeType(pizzaSectionMode.getValue());
	}

	public PIZZA_SECTION_MODE getPizzaSectionMode() {
		return PIZZA_SECTION_MODE.from(getPizzaSectionModeType());
	}

	public TicketItemModifier findTicketItemModifierByPageItem(MenuModifier menuModifier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getPageItemId();
			if (itemId != null && itemId.equals(menuModifier.getPageItemId())) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public TicketItemModifier findTicketItemComboModifierFor(MenuModifier menuModifier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (itemId != null && itemId.equals(menuModifier.getId()) && ticketItemModifier.getModifierType() == TicketItemModifier.EXTRA_MODIFIER) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public TicketItemModifier removeTicketItemModifierByPageItem(TicketItemModifier ticketItemModifier) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers == null)
			return ticketItemModifier;

		for (Iterator iter = ticketItemModifiers.iterator(); iter.hasNext();) {
			TicketItemModifier oldTicketItemModifier = (TicketItemModifier) iter.next();
			String pageItemId = oldTicketItemModifier.getPageItemId();
			if (pageItemId != null && pageItemId.equals(ticketItemModifier.getPageItemId())
					&& oldTicketItemModifier.getModifierType() == ticketItemModifier.getModifierType()) {
				iter.remove();
				return oldTicketItemModifier;
			}
		}
		return ticketItemModifier;
	}

	public void removeTicketItemDiscount(TicketItemDiscount itemDiscount) {
		List<TicketItemDiscount> discounts = getDiscounts();
		if (discounts == null) {
			return;
		}

		for (Iterator iterator = discounts.iterator(); iterator.hasNext();) {
			TicketItemDiscount ticketItemDiscount = (TicketItemDiscount) iterator.next();
			if (ticketItemDiscount.getTableRowNum() == itemDiscount.getTableRowNum()) {
				iterator.remove();
				return;
			}
		}
	}

	public boolean isInventoryAdjusted() {
		if (isTreatAsSeat()) {
			return false;
		}
		return getQuantity().doubleValue() == getInventoryAdjustQty().doubleValue();
	}

	public Boolean isIncludeVoidQuantity() {
		return includeVoidQuantity == null ? false : includeVoidQuantity;
	}

	public void setIncludeVoidQuantity(Boolean includeVoidQuantity) {
		this.includeVoidQuantity = includeVoidQuantity;
	}

	public void setVoidItem(VoidItem voidItem) {
		this.voidItem = voidItem;
		if (voidItem == null) {
			removeProperty(TicketItem.JSON_PROP_VOID_REASON);
			removeProperty(TicketItem.JSON_PROP_VOID_QUANTITY);
		}
	}

	@XmlTransient
	public VoidItem getVoidItem() {
		if (voidItem != null) {
			return voidItem;
		}
		String voidReason = getProperty(JSON_PROP_VOID_REASON);
		if (StringUtils.isBlank(voidReason)) {
			return null;
		}
		double voidQuantity = POSUtil.parseDouble(getProperty(JSON_PROP_VOID_QUANTITY));
		if (voidQuantity == 0) {
			return null;
		}
		//TODO: 
		//voidItem = createVoidItem(voidReason, POSUtil.getBoolean(getProperty(JSON_PROP_WASTED)), voidQuantity);
		return voidItem;
	}

	public void markVoided(String voidReason, boolean itemWasted, double quantity, boolean returned) {
		setCreateDate(StoreDAO.getServerTimestamp());

		setVoidDate(StoreDAO.getServerTimestamp());
		setQuantity(-1 * quantity);
		if (returned) {
			setReturned(true);
		}
		else {
			setVoided(true);
		}

		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				ticketItemModifier.setItemQuantity(-1 * ticketItemModifier.getItemQuantity());
			}
		}
		if (isComboItem()) {
			if (getComboItems() != null) {
				for (TicketItem comboTicketItem : getComboItems()) {
					comboTicketItem.markVoided(voidReason, itemWasted, comboTicketItem.getQuantity(), returned);
				}
			}
		}
		setVoidProperties(voidReason, itemWasted, quantity);
	}

	public void markReturnedItemVoided(String voidReason, boolean itemWasted, double quantity) {
		setVoided(true);
		setVoidDate(StoreDAO.getServerTimestamp());
		setQuantity(quantity);
		setReturned(false);
		setVoidedItemId(getId());
		if (itemWasted) {
			setInventoryAdjustQty(quantity);
		}
		else {
			setInventoryAdjustQty(0.0);
		}
		setPrintedToKitchen(false);
		setVoidProperties(voidReason, itemWasted, quantity);
		addProperty(JSON_PROP_RETURNED_ITEM_VOIDED, String.valueOf(true));
	}

	public void removeVoidProperties() {
		removeProperty(JSON_PROP_VOID_REASON);
		removeProperty(JSON_PROP_WASTED);
		removeProperty(JSON_PROP_VOID_QUANTITY);
		removeProperty(JSON_PROP_VOID_ID);
	}

	public void setVoidProperties(String voidReason, boolean wasted, double quantity) {
		setVoidProperties(voidReason, wasted, quantity, isPrintedToKitchen());
	}

	public void setVoidProperties(String voidReason, boolean wasted, double quantity, boolean printedToKitchen) {
		addProperty(JSON_PROP_VOID_REASON, voidReason);
		addProperty(JSON_PROP_WASTED, String.valueOf(wasted));
		addProperty(JSON_PROP_VOID_QUANTITY, String.valueOf(quantity));
		addProperty(JSON_PROP_PRINTED_TO_KITCHEN, String.valueOf(printedToKitchen));

		setVoidReason(voidReason);
		setItemWasted(wasted);
	}

	public VoidItem createVoidItem(String voidReason, boolean wasted, double quantity) {
		TicketItem clonedTicket = this.cloneAsNew();
		clonedTicket.setQuantity(quantity);
		clonedTicket.calculatePrice();

		VoidItem newVoidItem = new VoidItem();
		if (clonedTicket.getTicket() != null) {
			newVoidItem.setTicketId(clonedTicket.getTicket().getId());
		}
		newVoidItem.setVoidReason(voidReason);
		newVoidItem.setItemId(clonedTicket.getId());
		newVoidItem.setMenuItemId(clonedTicket.getMenuItemId());
		newVoidItem.setMenuItemName(clonedTicket.getName());
		newVoidItem.setUnitPrice(clonedTicket.getUnitPrice());
		newVoidItem.setQuantity(quantity);
		newVoidItem.setItemWasted(wasted);
		newVoidItem.setVoidDate(new Date());
		newVoidItem.setModifier(false);
		newVoidItem.setPrinterGroup(clonedTicket.getPrinterGroup());
		User currentUser = Application.getCurrentUser();
		newVoidItem.setVoidByUser(currentUser);
		newVoidItem.setTerminal(Application.getInstance().getTerminal());
		if (currentUser != null && currentUser.getActiveDrawerPullReport() != null) {
			newVoidItem.setCashDrawerId(currentUser.getActiveDrawerPullReport().getId());
		}

		newVoidItem.setTaxAmount(clonedTicket.getTaxAmountWithoutModifiers());
		newVoidItem.setTotalPrice(clonedTicket.getAdjustedTotalWithoutModifiers());
		//		newVoidItem.setTotalPrice(clonedTicket.getTotalAmount());
		//		double subtotal = clonedTicket.getUnitPrice() * newVoidItem.getQuantity();

		double ticketItemQuantity = clonedTicket.getQuantity();
		double voidItemQuantity = newVoidItem.getQuantity();
		List<VoidItem> modifierVoidItems = new ArrayList<>();
		if (clonedTicket.isHasModifiers()) {
			for (TicketItemModifier ticketItemModifier : clonedTicket.getTicketItemModifiers()) {
				VoidItem modifierVoidItem = new VoidItem();
				double voidModifierquantity = (ticketItemModifier.getItemQuantity() * voidItemQuantity) / ticketItemQuantity;
				if (voidModifierquantity > 0) {
					String tfvoidItemReason = newVoidItem.getVoidReason();
					if (!StringUtils.isEmpty(tfvoidItemReason)) {
						modifierVoidItem.setVoidReason(tfvoidItemReason);
					}

					modifierVoidItem.setModifierId(ticketItemModifier.getItemId());
					modifierVoidItem.setItemId(ticketItemModifier.getId());
					modifierVoidItem.setMenuItemName(ticketItemModifier.getNameDisplay());
					modifierVoidItem.setUnitPrice(ticketItemModifier.getUnitPrice());
					modifierVoidItem.setQuantity(voidModifierquantity);
					modifierVoidItem.setItemWasted(newVoidItem.isItemWasted());
					modifierVoidItem.setVoidDate(new Date());
					modifierVoidItem.setModifier(true);
					modifierVoidItem.setVoidByUser(currentUser);
					modifierVoidItem.setTerminal(Application.getInstance().getTerminal());
					modifierVoidItem.setCashDrawerId(currentUser.getActiveDrawerPullReport().getId());

					//					double modifierSubtotal = voidModifierquantity * ticketItemModifier.getUnitPrice();
					//					double modifierTaxAmount = subtotal * (ticketItemModifier.getTotalTaxRate() / 100);
					//					modifierVoidItem.setTaxAmount(modifierSubtotal);
					//					modifierVoidItem.setTotalPrice(modifierSubtotal + modifierTaxAmount);

					//					modifierVoidItem.setTaxAmount((-1) * ticketItemModifier.getTaxAmount());
					//					modifierVoidItem.setTotalPrice((-1) *(ticketItemModifier.getAdjustedSubtotal()+ticketItemModifier.getTaxAmount()));

					modifierVoidItem.setTaxAmount(ticketItemModifier.getTaxAmount());
					modifierVoidItem.setTotalPrice(ticketItemModifier.getAdjustedSubtotal() + ticketItemModifier.getTaxAmount());

					modifierVoidItem.setTicketId(newVoidItem.getTicketId());
					modifierVoidItems.add(modifierVoidItem);
				}
			}
		}
		newVoidItem.setVoidModifiers(modifierVoidItems);
		return newVoidItem;
	}

	@Override
	public boolean isSaved() {
		return getId() != null;
	}

	public boolean isRefundable() {
		//		List<TicketItemDiscount> discounts = getDiscounts();
		//		if (discounts == null || discounts.isEmpty()) {
		//			return true;
		//		}
		//		for (TicketItemDiscount ticketItemDiscount : discounts) {
		//			Discount discount = DiscountDAO.getInstance().get(ticketItemDiscount.getDiscountId());
		//			if (discount == null)
		//				continue;
		//			if (!discount.isRefundable())
		//				return false;
		//		}
		return true;
	}

	public java.util.List<TicketItemCookingInstruction> getCookingInstructions() {
		if (cookingInstructions == null) {
			cookingInstructions = new ArrayList<>();

			String property = super.getCookingInstructionsProperty();
			if (StringUtils.isNotEmpty(property)) {
				JsonReader jsonParser = Json.createReader(new StringReader(property));
				JsonArray jsonArray = jsonParser.readArray();
				jsonParser.close();
				for (int i = 0; i < jsonArray.size(); i++) {
					JsonObject jsonObject = (JsonObject) jsonArray.get(i);
					TicketItemCookingInstruction tic = new TicketItemCookingInstruction();
					tic.setDescription(JsonUtil.getString(jsonObject, TicketItemCookingInstruction.PROP_DESCRIPTION));
					tic.setPrintedToKitchen(JsonUtil.getBoolean(jsonObject, TicketItemCookingInstruction.PROP_PRINTED_TO_KITCHEN));
					tic.setSaved(JsonUtil.getBoolean(jsonObject, TicketItemCookingInstruction.PROP_SAVED));
					tic.setKitchenStatus(JsonUtil.getString(jsonObject, TicketItemCookingInstruction.PROP_KITCHEN_STATUS));
					cookingInstructions.add(tic);
				}
			}
		}
		return cookingInstructions;
	}

	public PrinterGroup getPrinterGroup() {
		PrinterGroup printerGroup = DataProvider.get().getPrinterGroupById(getPrinterGroupId());

		if (printerGroup == null) {
			DataProvider.get().getDefaultPrinterGroup();
		}

		return printerGroup;
	}

	public void setPrinterGroup(PrinterGroup printerGroup) {
		String printerGroupId = null;
		if (printerGroup != null) {
			printerGroupId = printerGroup.getId();
		}
		super.setPrinterGroupId(printerGroupId);
	}

	public TicketItem getParentTicketItem() {
		return parentTicketItem;
	}

	public void setParentTicketItem(TicketItem parentTicketItem) {
		this.parentTicketItem = parentTicketItem;
	}

	public List<TicketItem> getComboItems() {
		return null;
	}

	public void setComboItems(List<TicketItem> comboItems) {
	}

	public void addTocomboItems(TicketItem item) {

	}

	//	public ImageIcon getCourseIcon() {
	//		if (StringUtils.isEmpty(getCourseId()))
	//			return null;
	//		Course course = DataProvider.get().getCourse(getCourseId());
	//		return course == null ? null : course.getIcon();
	//	}

	public Integer getSortOrder() {
		if (sortOrder != null) {
			return sortOrder;
		}

		if (StringUtils.isEmpty(getCourseId())) {
			return 0;
		}
		Course course = DataProvider.get().getCourse(getCourseId());
		if (course != null) {
			return course.getSortOrder();
		}
		return 0;
	}

	public void setSortOrder(Integer sortOrder) {
		this.sortOrder = sortOrder;
	}

	public boolean isSyncEdited() {
		return syncEdited;
	}

	public void setSyncEdited(boolean syncEdited) {
		this.syncEdited = syncEdited;
	}

	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	public String getProperty(String key) {
		initPropertyContainer();

		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		initPropertyContainer();
		propertiesContainer.remove(propertyName);
		super.setProperties(propertiesContainer.toString());
	}

	public void setUnitSelection(boolean unitSelection) {
		addProperty(AppConstants.ALLOW_UNIT_SELECTION, String.valueOf(unitSelection));
	}

	public boolean isAllowUnitSelection() {
		String isAllowUnitSelection = getProperty(AppConstants.ALLOW_UNIT_SELECTION);
		if (StringUtils.isNotEmpty(isAllowUnitSelection))
			return Boolean.parseBoolean(isAllowUnitSelection);
		return false;
	}

	public void setItemIsGiftCard(String giftCardNo, double balance) {
		setQuantity(1.0);
		setShouldPrintToKitchen(false);
		setInventoryItem(false);
		setName(Messages.getString("GiftCardAddBalanceView.22")); //$NON-NLS-1$
		setCategoryName(Messages.getString("GiftCard")); //$NON-NLS-1$
		setGroupName(Messages.getString("GiftCard")); //$NON-NLS-1$
		setUnitPrice(balance);

		addProperty(AppConstants.IS_GIFT_CARD, String.valueOf(true));
		addProperty(AppConstants.GIFT_CARD_NO, giftCardNo);
	}

	public boolean isGiftCard() {
		return Boolean.valueOf(getProperty(AppConstants.IS_GIFT_CARD));
	}

	public String getGiftCardNo() {
		return getProperty(AppConstants.GIFT_CARD_NO);
	}

	public String getClassType() {
		return classType;
	}

	public void setClassType(String classType) {
		this.classType = classType;
	}

	public void setReturned(Boolean returned) {
		addProperty(JSON_PROP_RETURNED, String.valueOf(returned));
		super.setItemReturned(returned);
	}

	public boolean isReturned() {
		return Boolean.valueOf(getProperty(JSON_PROP_RETURNED));
	}

	public boolean isReturnedItemVoided() {
		return Boolean.valueOf(getProperty(JSON_PROP_RETURNED_ITEM_VOIDED));
	}

	public double getComboItemsPrice() {
		double itemsPrice = 0;
		List<TicketItem> comboItems = getComboItems();
		if (comboItems != null) {
			for (TicketItem comboItem : comboItems) {
				if (comboItem.getGroupId() == null) {
					itemsPrice += comboItem.getUnitPrice();
				}
			}
		}
		return itemsPrice;
	}

	public double getGiftCardPaidAmount() {
		return POSUtil.parseDouble(getProperty(JSON_PROP_GIFT_CARD_PAID_AMOUNT));
	}

	public void setGiftCardPaidAmount(double amount) {
		addProperty(JSON_PROP_GIFT_CARD_PAID_AMOUNT, String.valueOf(amount));
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	public void setServiceChargeWithoutModifiers(double sc) {
		addProperty(JSON_PROP_SC_WITHOUT_MODOFIERS, String.valueOf(sc));
	}

	public double getServiceChargeWithoutModifiers() {
		return getDoubleProperty(JSON_PROP_SC_WITHOUT_MODOFIERS);
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		return propertiesContainer;
	}

	public void setQuantity(double quantity, boolean updateComboItemsQuantity) {
		if (isComboItem() && updateComboItemsQuantity) {
			List<TicketItem> comboticketItems = getComboItems();
			if (comboticketItems != null && comboticketItems.size() > 0) {
				double ticketItemQuantity = getQuantity();
				double adjustQuantity = quantity - ticketItemQuantity;
				for (TicketItem comboTicketItem : comboticketItems) {
					comboTicketItem.setQuantity(comboTicketItem.getQuantity() + (adjustQuantity * comboTicketItem.getQuantity() / ticketItemQuantity));
				}
			}
		}
		setQuantity(quantity);
	}
}