package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.MenuItem;

public class AttributeDAO extends BaseAttributeDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AttributeDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Attribute attribute = (Attribute) obj;
		if (attribute == null) {
			throw new PosException(Messages.getString("AttributeDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(attribute);
		attribute.setDeleted(true);
		attribute.setGroup(null);
		update(attribute, session);
	}

	public void checkForeignRelation(Attribute bean) throws PosException {
		String id = bean.getId();
		StringBuilder ref = new StringBuilder();
		StringBuilder details = new StringBuilder();
		try (Session session = createNewSession()) {
			GenericDAO dao = GenericDAO.getInstance();
			append(dao.getForeignDataListNames(session, MenuItem.class, "attributes", Attribute.PROP_ID, id), Messages.getString("AttributeDAO.1"), ref, //$NON-NLS-1$//$NON-NLS-2$
					details);
		}
		String foreignItemDetails = details.toString();
		if (StringUtils.isNotBlank(foreignItemDetails)) {
			throw new PosException(Messages.getString("AttributeDAO.2") + ref.toString() + Messages.getString("AttributeDAO.3") //$NON-NLS-1$ //$NON-NLS-2$
					+ ref.toString() + ".", foreignItemDetails); //$NON-NLS-1$
		}
	}

	private void append(List<String> foreignDataListNames, String ref, StringBuilder refDetails, StringBuilder details) {
		if (foreignDataListNames == null || foreignDataListNames.isEmpty()) {
			return;
		}
		details.append(Messages.getString("AttributeDAO.4") + ref + ":"); //$NON-NLS-1$//$NON-NLS-2$
		refDetails.append(ref);
		int count = 1;
		for (String itemName : foreignDataListNames) {
			details.append("\n" + count + ". " + itemName); //$NON-NLS-1$ //$NON-NLS-2$
			count++;
		}
		details.append("\n"); //$NON-NLS-1$
	}

	@Override
	public List<Attribute> findAll() {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.createAlias(Attribute.PROP_GROUP, "g"); //$NON-NLS-1$
			criteria.addOrder(Order.asc("g.name")); //$NON-NLS-1$
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public boolean nameExists(Attribute attribute, String code) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Attribute.PROP_NAME, code).ignoreCase());
			if (attribute.getId() != null)
				criteria.add(Restrictions.ne(Attribute.PROP_ID, attribute.getId()));

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();

			if (rowCount == null)
				return false;

			return (rowCount.intValue() > 0) ? true : false;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateAttributeList(List<Attribute> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Attribute> iterator = dataList.iterator(); iterator.hasNext();) {
				Attribute item = (Attribute) iterator.next();
				Attribute existingItem = get(item.getId());
				if (existingItem != null) {
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}