/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Discount;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class DiscountDAO extends BaseDiscountDAO {

	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public DiscountDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private List<MenuItem> getMenuItemByDiscount(Discount discount, Session session) {
		Criteria criteriaMenuItem = session.createCriteria(getReferenceClass());
		criteriaMenuItem.createAlias("menuItems", "menuItem"); //$NON-NLS-1$ //$NON-NLS-2$
		criteriaMenuItem.add(Restrictions.eq(Discount.PROP_ID, discount.getId()));
		criteriaMenuItem.setProjection(Projections.alias(Projections.property("menuItem." + MenuItem.PROP_NAME), MenuItem.PROP_NAME)); //$NON-NLS-1$
		return criteriaMenuItem.setResultTransformer(Transformers.aliasToBean(MenuItem.class)).list();
	}

	private List<MenuGroup> getMenuGroupByDiscount(Discount discount, Session session) {
		Criteria criteriarMenuGroup = session.createCriteria(getReferenceClass());
		criteriarMenuGroup.createAlias("menuGroups", "menuGroup"); //$NON-NLS-1$ //$NON-NLS-2$
		criteriarMenuGroup.add(Restrictions.eq(Discount.PROP_ID, discount.getId()));
		criteriarMenuGroup.setProjection(Projections.alias(Projections.property("menuGroup." + MenuGroup.PROP_NAME), MenuGroup.PROP_NAME)); //$NON-NLS-1$
		return criteriarMenuGroup.setResultTransformer(Transformers.aliasToBean(MenuGroup.class)).list();
	}

	private List<MenuCategory> getMenuCategoryByDiscount(Discount discount, Session session) {
		Criteria criteriarMenuCategory = session.createCriteria(getReferenceClass());
		criteriarMenuCategory.createAlias("menuCategories", "menuCategory"); //$NON-NLS-1$ //$NON-NLS-2$
		criteriarMenuCategory.add(Restrictions.eq(Discount.PROP_ID, discount.getId()));
		criteriarMenuCategory.setProjection(Projections.alias(Projections.property("menuCategory." + MenuCategory.PROP_NAME), MenuCategory.PROP_NAME)); //$NON-NLS-1$
		return criteriarMenuCategory.setResultTransformer(Transformers.aliasToBean(MenuCategory.class)).list();
	}

	@Override
	protected void delete(Object obj, Session session) {
		if (obj instanceof Discount) {
			Discount discount = (Discount) obj;

			StringBuilder details = new StringBuilder();
			List<MenuItem> menuItems = getMenuItemByDiscount(discount, session);
			if (menuItems != null && !menuItems.isEmpty()) {
				details.append(this.constructExceptionDetailsByMenuitem(discount, menuItems));
				details.append(EMPTY_NEWLINE_STRING);
			}

			List<MenuGroup> menuGroups = getMenuGroupByDiscount(discount, session);
			if (menuGroups != null && !menuGroups.isEmpty()) {
				details.append(this.constructExceptionDetailsByMenuGroup(discount, menuGroups));
				details.append(EMPTY_NEWLINE_STRING);
			}

			List<MenuCategory> menuCategories = getMenuCategoryByDiscount(discount, session);
			if (menuCategories != null && !menuCategories.isEmpty()) {
				details.append(this.constructExceptionDetailsByMenuCategory(discount, menuCategories));
				details.append(EMPTY_NEWLINE_STRING);
			}

			String message = Messages.getString("DiscountDAO.0"); //$NON-NLS-1$
			String detailsMessage = details.toString();
			if (StringUtils.isNotBlank(detailsMessage)) {
				throw new PosException(message, detailsMessage);
			}

			discount.setDeleted(Boolean.TRUE);
			super.update(discount, session);
		}
		else {
			super.delete(obj, session);
		}
	}

	private String constructExceptionDetailsByMenuitem(Discount discount, List<MenuItem> menuItems) {
		if (menuItems != null && !menuItems.isEmpty()) {
			StringBuilder builder = new StringBuilder(discount.getName() + " " + Messages.getString("DiscountDAO.3")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < menuItems.size(); i++) {
				String message = (i + 1) + "." + " " + menuItems.get(i).getName(); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByMenuCategory(Discount discount, List<MenuCategory> menuCategories) {
		if (menuCategories != null && !menuCategories.isEmpty()) {
			StringBuilder builder = new StringBuilder(discount.getName() + " " + Messages.getString("DiscountDAO.4")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < menuCategories.size(); i++) {
				String message = (i + 1) + "." + " " + menuCategories.get(i).getName(); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByMenuGroup(Discount discount, List<MenuGroup> menuGroups) {
		if (menuGroups != null && !menuGroups.isEmpty()) {
			StringBuilder builder = new StringBuilder(discount.getName() + " " + Messages.getString("DiscountDAO.5")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < menuGroups.size(); i++) {
				String message = (i + 1) + "." + " " + menuGroups.get(i).getName(); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	public List<Discount> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Discount.PROP_NAME));
			return criteria.list();
		}
	}

	public Discount getInitialized(String discountId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Discount.PROP_ID, discountId));

			Discount discount = (Discount) criteria.uniqueResult();
			Hibernate.initialize(discount.getMenuItems());
			Hibernate.initialize(discount.getMenuGroups());
			Hibernate.initialize(discount.getMenuCategories());

			return discount;
		}
	}

	public List<Discount> findAllValidCoupons() {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			return criteria.list();
		}

	}

	public List<Discount> getValidItemCoupons() {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Discount.PROP_QUALIFICATION_TYPE, Discount.QUALIFICATION_TYPE_ITEM));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			return criteria.list();
		}

	}

	public List<Discount> getValidCoupon(MenuItem menuItem) {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.createAlias("menuItems", "item"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.eq("item.id", menuItem.getId())); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_QUALIFICATION_TYPE, Discount.QUALIFICATION_TYPE_ITEM),
					Restrictions.eq(Discount.PROP_TYPE, Discount.DISCOUNT_TYPE_REPRICE)));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			return criteria.list();
		}
	}

	public List<Discount> getTicketValidCoupon() {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_QUALIFICATION_TYPE, Discount.QUALIFICATION_TYPE_ORDER),
					Restrictions.eq(Discount.PROP_TYPE, Discount.DISCOUNT_TYPE_REPRICE)));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			return criteria.list();
		}
	}

	public boolean isApplicable(String itemId, String discountId) {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			criteria.createAlias("menuItems", "item"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.eq("item.id", itemId)); //$NON-NLS-1$
			criteria.add(Restrictions.eq(Discount.PROP_ID, discountId));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	public Discount getDiscountById(String id, int couponType) {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Discount.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ID, id));
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.or(
					Restrictions.eq(Discount.PROP_QUALIFICATION_TYPE,
							couponType == Discount.QUALIFICATION_TYPE_ITEM ? Discount.QUALIFICATION_TYPE_ITEM : Discount.QUALIFICATION_TYPE_ORDER),
					Restrictions.eq(Discount.PROP_TYPE, Discount.DISCOUNT_TYPE_REPRICE)));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			List<Discount> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (Discount) result.get(0);
		}
	}

	public Discount getDiscountByBarcode(String barcode, int couponType) {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Discount.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.like(Discount.PROP_BARCODE, barcode));
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.or(
					Restrictions.eq(Discount.PROP_QUALIFICATION_TYPE,
							couponType == Discount.QUALIFICATION_TYPE_ITEM ? Discount.QUALIFICATION_TYPE_ITEM : Discount.QUALIFICATION_TYPE_ORDER),
					Restrictions.eq(Discount.PROP_TYPE, Discount.DISCOUNT_TYPE_REPRICE)));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			List<Discount> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (Discount) result.get(0);
		}
	}

	public List<Discount> getDiscountsByMenuCategory(MenuCategory category, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("menuCategories", "menuCategory"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("menuCategory." + MenuCategory.PROP_ID, category.getId())); //$NON-NLS-1$
		return criteria.list();
	}

	public List<Discount> getDiscountsByMenuGroup(MenuGroup menuGroup, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("menuGroups", "menugroup"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("menugroup." + MenuGroup.PROP_ID, menuGroup.getId())); //$NON-NLS-1$
		return criteria.list();
	}

	public List<Discount> getDiscountsByMenuItem(MenuItem menuItem, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("menuItems", "menuItem"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("menuItem." + MenuItem.PROP_ID, menuItem.getId())); //$NON-NLS-1$
		return criteria.list();
	}

	public String getCurrentDayPropertyName(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);

		String propertyName = ""; //$NON-NLS-1$

		int dayOfWeek = c.get(Calendar.DAY_OF_WEEK);
		switch (dayOfWeek) {
			case Calendar.SATURDAY:
				propertyName = Discount.PROP_APPLY_TO_SATURDAY_ONLY;
				break;
			case Calendar.SUNDAY:
				propertyName = Discount.PROP_APPLY_TO_SUNDAY_ONLY;
				break;
			case Calendar.MONDAY:
				propertyName = Discount.PROP_APPLY_TO_MONDAY_ONLY;
				break;
			case Calendar.TUESDAY:
				propertyName = Discount.PROP_APPLY_TO_TUESDAY_ONLY;
				break;
			case Calendar.WEDNESDAY:
				propertyName = Discount.PROP_APPLY_TO_WEDNESDAY_ONLY;
				break;
			case Calendar.THURSDAY:
				propertyName = Discount.PROP_APPLY_TO_THURSDAY_ONLY;
				break;
			case Calendar.FRIDAY:
				propertyName = Discount.PROP_APPLY_TO_FRIDAY_ONLY;
				break;

			default:
				break;
		}
		return propertyName;
	}

	public List<Discount> getValidCoupons() {
		Date currentDate = new Date();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Discount.PROP_ENABLED, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.eq(Discount.PROP_NEVER_EXPIRE, Boolean.TRUE), Restrictions.ge(Discount.PROP_EXPIRY_DATE, currentDate)));
			return criteria.list();
		}
	}

	public void saveOrUpdateDiscountList(List<Discount> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		Map<String, MenuItem> itemMap = new HashMap<>();
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Discount> iterator = dataList.iterator(); iterator.hasNext();) {
				Discount item = (Discount) iterator.next();
				List<MenuItem> menuItems = item.getMenuItems();
				item.setMenuItems(null);
				if (menuItems != null && menuItems.size() > 0) {
					for (MenuItem menuItem : menuItems) {
						MenuItem existingMenuItem = itemMap.get(menuItem.getId());
						if (existingMenuItem == null) {
							existingMenuItem = MenuItemDAO.getInstance().get(menuItem.getId());
							if (existingMenuItem == null) {
								throw new Exception("No item found with id " + menuItem.getId()); //$NON-NLS-1$
							}
							itemMap.put(existingMenuItem.getId(), existingMenuItem);
						}
						item.addTomenuItems(existingMenuItem);
					}
				}
				Discount existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}
}