/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.ImageIcon;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.PrintException;
import com.floreantpos.config.AppConfig;
import com.floreantpos.config.CardConfig;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.extension.PaymentGatewayPlugin;
import com.floreantpos.main.Application;
import com.floreantpos.model.Address;
import com.floreantpos.model.Currency;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.LabelItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosPrinters;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Printer;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.ReceiptParam;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Store;
import com.floreantpos.model.SubOrderType;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalPrinters;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemTax;
import com.floreantpos.model.User;
import com.floreantpos.model.VirtualPrinter;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.GiftCardDAO;
import com.floreantpos.model.dao.PrinterGroupDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TerminalPrintersDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.ext.KitchenStickerPaperSize;
import com.floreantpos.model.ext.ReciptPaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.MqttCommand;
import com.floreantpos.print.EscPosPrintService;
import com.floreantpos.print.JasperPrintService;
import com.floreantpos.print.ReceiptPrintServiceProvider;
import com.floreantpos.report.KitchenStickerModel.KitchenSticker;
import com.floreantpos.ui.dialog.POSMessageDialog;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.PrintServiceUtil;
import com.floreantpos.util.ReceiptUtil;
import com.orocube.common.util.TerminalUtil;
import com.orocube.common.util.TicketStatus;
import com.orocube.rest.service.mqtt.OroMqttClient;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class ReceiptPrintService {
	public static final String SYSTEM_INFORMATION = "System Information"; //$NON-NLS-1$
	public static final String ADDITIONAL_ORDER_INFO = "additionalOrderInfo"; //$NON-NLS-1$
	public static final String SPLIT_TICKET_ID = "splitTicketId"; //$NON-NLS-1$
	public static final String PREVIOUS_DUE = "previousDue"; //$NON-NLS-1$
	public static final String PREVIOUS_DUE_TEXT = "previousDueText"; //$NON-NLS-1$
	public static final String CASH_REFUND = "cashRefund"; //$NON-NLS-1$
	public static final String CASH_REFUND_TEXT = "cashRefundText"; //$NON-NLS-1$
	public static final String TICKET_HEADER = "ticketHeader";//$NON-NLS-1$
	public static final String TAX_BREAKDOWN_TEXT = "taxBreakdownText";//$NON-NLS-1$
	public static final String IS_SHOW_TAX_BREAKDOWN = "isShowTaxBreakdown";//$NON-NLS-1$
	public static final String SHOW_TIPS_BLOCK = "showTipsBlock";//$NON-NLS-1$
	public static final String SHOW_TIPS = "showTips";//$NON-NLS-1$
	public static final String STORE_LOGO_IMAGE = "storeLogoIcon";//$NON-NLS-1$
	public static final String STORE_LOGO_ICON = "storeLogoIcon";//$NON-NLS-1$
	public static final String ADDITIONAL_PROPERTIES = "additionalProperties";//$NON-NLS-1$
	public static final String ADDITIONAL_PAYMENT_PROPERTIES = "additionalPaymentProperties";//$NON-NLS-1$
	public static final String APPROVAL_CODE = "approvalCode";//$NON-NLS-1$
	public static final String CARD_PAYMENT = "cardPayment";//$NON-NLS-1$
	public static final String CHANGED_AMOUNT = "changedAmount";//$NON-NLS-1$
	public static final String COPY_TYPE = "copyType";//$NON-NLS-1$
	public static final String BOTTOM_MESSAGE = "bottomMessage";//$NON-NLS-1$
	public static final String FOOTER_MESSAGE = "footerMessage";//$NON-NLS-1$
	public static final String GRAND_SUBTOTAL = "grandSubtotal";//$NON-NLS-1$
	public static final String DUE_AMOUNT = "dueAmount";//$NON-NLS-1$
	public static final String REFUND_AMOUNT = "refundAmount";//$NON-NLS-1$
	public static final String REFUND_AMOUNT_TEXT = "refundAmountText";//$NON-NLS-1$
	public static final String TENDER_AMOUNT = "tenderAmount";//$NON-NLS-1$
	public static final String TENDER_AMOUNT_TEXT = "tenderAmountText";//$NON-NLS-1$
	public static final String PAID_AMOUNT = "paidAmount";//$NON-NLS-1$
	public static final String NET_AMOUNT = "netAmount";//$NON-NLS-1$
	public static final String FEE_AMOUNT = "feeAmount";//$NON-NLS-1$
	public static final String FEE_AMOUNT_TEXT = "feeAmountText";//$NON-NLS-1$
	public static final String CHANGE_AMOUNT_TEXT = "changeAmountText";//$NON-NLS-1$
	public static final String DUE_AMOUNT_TEXT = "dueAmountText";//$NON-NLS-1$
	public static final String PAID_AMOUNT_TEXT = "paidAmountText";//$NON-NLS-1$
	public static final String NET_AMOUNT_TEXT = "netAmountText";//$NON-NLS-1$
	public static final String TIPS_TEXT = "tipsText";//$NON-NLS-1$
	public static final String DELIVERY_CHARGE_TEXT = "deliveryChargeText";//$NON-NLS-1$
	public static final String SERVICE_CHARGE_TEXT = "serviceChargeText";//$NON-NLS-1$
	public static final String TAX_TEXT = "taxText";//$NON-NLS-1$
	public static final String DISCOUNT_TEXT = "discountText";//$NON-NLS-1$
	public static final String TOTAL_TEXT = "totalText";//$NON-NLS-1$
	public static final String DATA = "data"; //$NON-NLS-1$
	public static final String TITLE = "title"; //$NON-NLS-1$
	public static final String ORDER_ = "ORDER-"; //$NON-NLS-1$
	public static final String PROP_PRINTER_NAME = "printerName"; //$NON-NLS-1$
	public static final String TIP_AMOUNT = "tipAmount"; //$NON-NLS-1$
	public static final String SERVICE_CHARGE = "serviceCharge"; //$NON-NLS-1$
	public static final String DELIVERY_CHARGE = "deliveryCharge"; //$NON-NLS-1$
	public static final String TAX_AMOUNT = "taxAmount"; //$NON-NLS-1$
	public static final String DISCOUNT_AMOUNT = "discountAmount"; //$NON-NLS-1$
	public static final String HEADER_LINE1 = "headerLine1"; //$NON-NLS-1$
	private static final String REPORT_DATE = "reportDate"; //$NON-NLS-1$
	private static final String SHOW_FOOTER = "showFooter"; //$NON-NLS-1$
	public static final String SHOW_HEADER_SEPARATOR = "showHeaderSeparator"; //$NON-NLS-1$
	private static final String SHOW_SUBTOTAL = "showSubtotal"; //$NON-NLS-1$
	private static final String RECEIPT_TYPE = "receiptType"; //$NON-NLS-1$
	public static final String SUB_TOTAL_TEXT = "subTotalText"; //$NON-NLS-1$
	private static final String QUANTITY_TEXT = "quantityText"; //$NON-NLS-1$
	private static final String ITEM_TEXT = "itemText"; //$NON-NLS-1$
	public static final String CUSTOMER_COPY = "Customer Copy"; //$NON-NLS-1$
	public static final String DRIVER_COPY = "Driver Copy"; //$NON-NLS-1$
	public static final String CENTER = "center"; //$NON-NLS-1$
	public static final String LEFT = "left"; //$NON-NLS-1$
	public static final String RIGHT = "right"; //$NON-NLS-1$

	public static final String OROPOS_PDF_PRINTER = Messages.getString("ReceiptPrintService.11"); //$NON-NLS-1$

	private static File pdfPrinterDir;
	private static Log logger = LogFactory.getLog(ReceiptPrintService.class);

	static {
		String homeDir = System.getProperty("user.home"); //$NON-NLS-1$
		pdfPrinterDir = new File(homeDir, "oropos-pdf-print"); //$NON-NLS-1$
		if (!pdfPrinterDir.exists()) {
			pdfPrinterDir.mkdirs();
		}
	}

	public static void printGenericReport(String title, String data) throws Exception {
		HashMap<String, Object> map = new HashMap<String, Object>(2);
		map.put(TITLE, title);
		map.put(DATA, data);
		getJasperPrintService().printGenericReport(map);
	}

	public static void testPrinter(Printer printer, String title, String data) throws Exception {
		HashMap<String, Object> map = new HashMap<String, Object>(2);
		map.put(TITLE, title);
		map.put(DATA, data);
		setPrintProperty(map, printer);
		getReceiptPrintServiceProvider(printer).testPrinter(map, printer.getDeviceName());
	}

	public static String getSystemInfo(String printerName, String deviceName) {
		String data = printerName + "-" + deviceName; //$NON-NLS-1$
		data += "\n Terminal : " + Application.getInstance().getTerminal().getName(); //$NON-NLS-1$
		data += "\n Current User : " + Application.getCurrentUser().getFirstName(); //$NON-NLS-1$
		data += "\n Database Name : " + AppConfig.getDatabaseName() + AppConfig.getDatabaseHost() + AppConfig.getDatabasePort(); //$NON-NLS-1$
		return data;
	}

	private static void setPrintProperty(HashMap map, Printer printer) {
		map.put(AppConstants.RECEIPT_PRINT_SYSTEM, printer.getPrintSystem());
		map.put(TerminalPrinters.PRINTER_TYPE, printer.getPrinterType());
		map.put(TerminalPrinters.IP_ADDRESS, printer.getIpAddress());
		map.put(TerminalPrinters.IP_PORT, printer.getIpPort());
		map.put(TerminalPrinters.SERIAL_PORT, printer.getSerialPort());
		map.put(TerminalPrinters.TEXT_LENGTH, printer.getTextLength());
	}

	public static void printTicket(Ticket ticket) {
		printTicket(ticket, true);
	}

	public static void printTicket(Ticket ticket, boolean eatException) {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true);
		printProperties.setPrintCookingInstructions(false);
		printTicket(ticket, populateTicketProperties(ticket, printProperties, null), eatException);
	}

	public static void printTicket(Ticket ticket, String copyType) {
		printTicket(ticket, copyType, true);
	}

	public static void printTicket(Ticket ticket, String copyType, boolean eatException) {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); // $NON-NLS-1$
																									// //$NON-NLS-2$
		printProperties.setPrintCookingInstructions(false);
		HashMap map = populateTicketProperties(ticket, printProperties, null);
		map.put(COPY_TYPE, copyType); //$NON-NLS-1$
		map.put(CARD_PAYMENT, true); //$NON-NLS-1$

		printTicket(ticket, map, eatException);
	}

	public static void printTicket(Ticket ticket, HashMap map) {
		printTicket(ticket, map, true);
	}

	public static void printTicket(Ticket ticket, HashMap map, boolean eatException) {
		try {
			ticket.setShouldUpdateStock(true);
			PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGateway();
			boolean printUsingPaymentGateway = paymentGateway != null && paymentGateway.printUsingThisTerminal();

			if (printUsingPaymentGateway) {
				updatePrintCount(ticket);
				paymentGateway.printTicket(ticket);
				return;
			}

			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			if (receiptPrinters == null) {
				return;
			}
			updatePrintCount(ticket);
			for (Printer receiptPrinter : receiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					logger.info(
							String.format("Printing ticket %s payment using printer %s. Total amount %s", ticket.getId(), deviceName, ticket.getTotalAmount())); //$NON-NLS-1$
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printTicket(ticket, map, deviceName);
				}
			}
			saveJournal(ticket);
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
			if (!eatException) {
				throw new PrintException(Messages.getString("SettleTicketProcessor.23"), e); //$NON-NLS-1$
			}
		} finally {
			ticket.setShouldUpdateStock(false);
		}
	}

	/*public static void printTicket(Ticket ticket, Terminal terminal) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); // $NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, null);
	
			List<TerminalPrinters> terminalPrinters = TerminalPrintersDAO.getInstance().findTerminalPrinters(terminal);
			List<Printer> activeReceiptPrinters = new ArrayList<Printer>();
			for (TerminalPrinters terminalPrinters2 : terminalPrinters) {
				int printerType = terminalPrinters2.getVirtualPrinter().getType();
				if (printerType == VirtualPrinter.RECEIPT) {
					Printer printer = new Printer(terminalPrinters2.getVirtualPrinter(), terminalPrinters2.getPrinterName());
					activeReceiptPrinters.add(printer);
				}
			}
			for (Printer receiptPrinter : activeReceiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printTicket(ticket, map, deviceName);
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}*/

	public static void printTransaction(PosTransaction transaction, boolean printCustomerCopy) {
		printTransaction(transaction, false, printCustomerCopy);
	}

	public static void printTransaction(PosTransaction transaction, boolean printStoreCopy, boolean printCustomerCopy) {
		try {
			Ticket ticket = transaction.getTicket();
			PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGateway();
			if (paymentGateway != null && paymentGateway.printUsingThisTerminal()) {
				paymentGateway.printTransaction(transaction, printStoreCopy, printCustomerCopy);
				return;
			}
			TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.6"), true, true, true); //$NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, transaction);

			if (transaction != null && transaction.isCard()) {
				map.put(CARD_PAYMENT, true); //$NON-NLS-1$
				map.put(COPY_TYPE, Messages.getString("ReceiptPrintService.7")); //$NON-NLS-1$ //$NON-NLS-2$

				printTransaction(transaction, map);

				if (printCustomerCopy) {
					map.put(COPY_TYPE, Messages.getString("ReceiptPrintService.8")); //$NON-NLS-1$ //$NON-NLS-2$
					printTransaction(transaction, map);
				}
			}
			else {
				printTransaction(transaction, map);
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
			throw new PrintException(Messages.getString("SettleTicketProcessor.23"), e); //$NON-NLS-1$
		}
	}

	public static void printTransaction(PosTransaction transaction, HashMap map) {
		try {
			Ticket ticket = transaction.getTicket();

			PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGateway();
			boolean printUsingPaymentGateway = paymentGateway != null && paymentGateway.printUsingThisTerminal();
			if (printUsingPaymentGateway) {
				paymentGateway.printTransaction(transaction, false, false);
				return;
			}

			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			for (Printer receiptPrinter : receiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					logger.info(
							String.format("Printing ticket %s payment using printer %s. Total amount %s", ticket.getId(), deviceName, ticket.getTotalAmount())); //$NON-NLS-1$
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printTransaction(ticket, map, transaction, deviceName);
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
			throw new PrintException(Messages.getString("SettleTicketProcessor.23"), e); //$NON-NLS-1$
		}
	}

	public static void printTransaction(PosTransaction transaction) {
		Ticket ticket = transaction.getTicket();

		PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGateway();
		boolean paymentTypeCash = transaction.getPaymentType().equals(PaymentType.CASH);
		boolean printUsingPaymentGateway = paymentGateway != null && paymentGateway.printUsingThisTerminal();

		if (printUsingPaymentGateway && paymentTypeCash) {
			return;
		}

		if (printUsingPaymentGateway) {
			paymentGateway.printTransaction(transaction, false, false);
			return;
		}

		String receiptPrinter = Application.getPrinters().getReceiptPrinter();
		if (StringUtils.isEmpty(receiptPrinter)) {
			return;
		}

		TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.3"), true, true, true); //$NON-NLS-1$
		printProperties.setPrintCookingInstructions(false);

		HashMap map = populateTicketProperties(ticket, printProperties, transaction);
		if (transaction != null && transaction.isCard()) {
			map.put(CARD_PAYMENT, true); //$NON-NLS-1$
			map.put(COPY_TYPE, Messages.getString("ReceiptPrintService.4")); //$NON-NLS-1$ //$NON-NLS-2$
			printTransaction(transaction, map);

			map.put(COPY_TYPE, Messages.getString("ReceiptPrintService.5")); //$NON-NLS-1$ //$NON-NLS-2$
			printTransaction(transaction, map);
		}
		else {
			printTransaction(transaction, map);
		}
	}

	public static void printRefundTicket(Ticket ticket, List<PosTransaction> posTransactions) {
		try {
			if (posTransactions == null || posTransactions.isEmpty())
				return;
			double refundAmount = 0;
			if (posTransactions != null) {
				for (PosTransaction t : posTransactions) {
					if (t instanceof RefundTransaction || t.isVoided())
						refundAmount += t.getAmount();
				}
			}
			TicketPrintProperties printProperties = new TicketPrintProperties("*** REFUND RECEIPT ***", true, true, //$NON-NLS-1$
					true);
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, null, false, true);
			map.put(REFUND_AMOUNT_TEXT, Messages.getString("ReceiptPrintService.1")); //$NON-NLS-1$ //$NON-NLS-2$
			map.put(REFUND_AMOUNT, NumberUtil.formatNumber(refundAmount)); //$NON-NLS-1$
			map.put(CASH_REFUND_TEXT, Messages.getString("ReceiptPrintService.2")); //$NON-NLS-1$ //$NON-NLS-2$
			map.put(CASH_REFUND, NumberUtil.formatNumber(refundAmount)); //$NON-NLS-1$
			//String deviceName = DataProvider.get().getPrinters().getReceiptPrinter();
			//TODO: needs to check refund ticket printed only default printer
			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			for (Printer receiptPrinter : receiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					logger.info(String.format("Printing refund ticket %s using printer %s. Refund amount %s", ticket.getId(), deviceName, //$NON-NLS-1$
							refundAmount));
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printRefundTicket(ticket, map, deviceName);
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}

	public static void printRefundTicket(Ticket ticket, RefundTransaction posTransaction) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties("*** REFUND RECEIPT ***", true, true, //$NON-NLS-1$
					true);
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, posTransaction, false, true);
			map.put(REFUND_AMOUNT_TEXT, Messages.getString("ReceiptPrintService.1") + CurrencyUtil.getCurrencySymbol()); //$NON-NLS-1$ //$NON-NLS-2$
			map.put(REFUND_AMOUNT, NumberUtil.formatNumber(posTransaction.getAmount())); //$NON-NLS-1$
			map.put(CASH_REFUND_TEXT, Messages.getString("ReceiptPrintService.2") + CurrencyUtil.getCurrencySymbol()); //$NON-NLS-1$ //$NON-NLS-2$
			map.put(CASH_REFUND, NumberUtil.formatNumber(posTransaction.getAmount())); //$NON-NLS-1$
			//String deviceName = DataProvider.get().getPrinters().getReceiptPrinter();
			//TODO:
			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			for (Printer receiptPrinter : receiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					logger.info(String.format("Printing refund ticket %s using printer %s. Refund amount %s", ticket.getId(), deviceName, //$NON-NLS-1$
							posTransaction.getAmount()));
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printRefundTicket(ticket, map, deviceName);
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}

	public static void printVoidTicket(Ticket ticket) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties("*** VOID RECEIPT ***", true, true, true); //$NON-NLS-1$ //$NON-NLS-2$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, null);

			String refundText = ""; //$NON-NLS-1$
			if (ticket.getTransactions() != null) {
				Set<PosTransaction> posTransactions = ticket.getTransactions();
				double refundAmount = 0;
				if (posTransactions != null) {
					for (PosTransaction t : posTransactions) {
						if (t instanceof RefundTransaction || t.isVoided())
							refundAmount += t.getAmount();
					}
				}
				refundAmount = NumberUtil.roundToTwoDigit(refundAmount);
			}
			map.put(ADDITIONAL_PROPERTIES, "<html><b>" + refundText + "</b></html>");//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			map.put(ADDITIONAL_PAYMENT_PROPERTIES, ""); //$NON-NLS-1$ //$NON-NLS-2$
			//String deviceName = DataProvider.get().getPrinters().getReceiptPrinter();
			//TODO: 
			List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
			for (Printer receiptPrinter : receiptPrinters) {
				String deviceName = receiptPrinter.getDeviceName();
				if (deviceName != null) {
					logger.info(String.format("Printing voided ticket %s using printer %s. Voided amount %s", ticket.getId(), deviceName, //$NON-NLS-1$
							refundText));
					setPrintProperty(map, receiptPrinter);
					setPaginationProperty(map, deviceName);
					getReceiptPrintServiceProvider(receiptPrinter).printVoidTicket(ticket, map, deviceName);
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}

	public static JasperPrint createPurchaseOrderPrint(PurchaseOrder order) throws Exception {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); // $NON-NLS-1$
																									// //$NON-NLS-2$
		printProperties.setPrintCookingInstructions(false);
		HashMap map = populatePurchaseOrderProperties(order, printProperties);
		map.put(COPY_TYPE, ""); //$NON-NLS-1$ //$NON-NLS-2$

		OrderDataSource dataSource = new OrderDataSource(order);
		String receiptName = "order_receipt"; //$NON-NLS-1$
		map.put(PREVIOUS_DUE_TEXT, "Previous Due  "); //$NON-NLS-1$ //$NON-NLS-2$
		map.put(PREVIOUS_DUE, ""); //$NON-NLS-1$ //$NON-NLS-2$

		return getJasperPrintService().createJasperPrint(ReportUtil.getReport(receiptName), map, new JRTableModelDataSource(dataSource)); // $NON-NLS-1$
	}

	public static JasperPrint createPurchaseOrderItemsBarcodePrint(List<LabelItem> orderedItemList) throws Exception {
		HashMap map = new HashMap();
		LabelPrinterTableModel dataSource = new LabelPrinterTableModel(orderedItemList);
		String receiptName = "barcode_report"; //$NON-NLS-1$

		return getJasperPrintService().createJasperPrint(ReportUtil.getReport(receiptName), map, new JRTableModelDataSource(dataSource)); // $NON-NLS-1$
	}

	public static JasperPrint printKitchenStickerItems(List<KitchenSticker> stickerItemList) throws Exception {
		try {
			HashMap map = new HashMap();
			KitchenStickerModel dataSource = new KitchenStickerModel(stickerItemList);
			String receiptName = "kitchenStickerReport"; //$NON-NLS-1$

			KitchenStickerPaperSize kitchenStickerPaperSize = getKitchenStickerPaperSize(AppConstants.KITCHEN_STICKER_PAPER_SIZE_IN_MM);

			return getJasperPrintService().createJasperPrint(ReportUtil.getReport(kitchenStickerPaperSize.getReportNameAccording2Size(receiptName)), map,
					new JRTableModelDataSource(dataSource)); // $NON-NLS-1$
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e.getMessage(), e);
		}
		return null;
	}

	public static byte[] createPDFStream(Ticket ticket) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true);
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, null);
			map.put(JRParameter.IS_IGNORE_PAGINATION, true);

			JasperPrint jasperPrint = getJasperPrintService().createPrint(ticket, map, null);
			jasperPrint.setName("TICKET_RECEIPT" + ticket.getId()); //$NON-NLS-1$
			jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());

			return JasperExportManager.exportReportToPdf(jasperPrint);
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e);
		}
		return null;
	}

	public static byte[] createPDFStream(PosTransaction transaction) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.6"), true, true, true); //$NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			Ticket ticket = transaction.getTicket();
			HashMap map = populateTicketProperties(ticket, printProperties, transaction);
			map.put(JRParameter.IS_IGNORE_PAGINATION, true);

			JasperPrint jasperPrint = getJasperPrintService().createPrint(ticket, map, null);
			jasperPrint.setName("TICKET_RECEIPT" + ticket.getId()); //$NON-NLS-1$
			jasperPrint.setProperty(ReceiptPrintService.PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());

			return JasperExportManager.exportReportToPdf(jasperPrint);
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e);
		}
		return null;
	}

	public static JasperPrintService getJasperPrintService() {
		return new JasperPrintService();
	}

	public static ReceiptPrintServiceProvider getReceiptPrintService() {
		List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
		if (receiptPrinters == null || receiptPrinters.isEmpty()) {
			return getJasperPrintService();
		}
		return getReceiptPrintServiceProvider(receiptPrinters.get(0));
	}

	public static ReceiptPrintServiceProvider getKitchenReceiptPrintPreviewService() {
		String printPreview = (String) DataProvider.get().getCurrentTerminal().getProperty(AppConstants.KITCHEN_RECEIPT_PRINT_PREVIEW_SYSTEM);
		if (printPreview != null && printPreview.equals(AppConstants.RECEIPT_ESC)) {
			return new EscPosPrintService();
		}
		return new JasperPrintService();
	}

	public static ReceiptPrintServiceProvider getMainReceiptPrintPreviewService() {
		String printPreview = (String) DataProvider.get().getCurrentTerminal().getProperty(AppConstants.MAIN_RECEIPT_PRINT_PREVIEW_SYSTEM);
		if (printPreview != null && printPreview.equals(AppConstants.RECEIPT_ESC)) {
			return new EscPosPrintService();
		}
		return new JasperPrintService();
	}

	private static ReceiptPrintServiceProvider getReceiptPrintServiceProvider(Printer receiptPrinter) {
		if (receiptPrinter == null) {
			return new JasperPrintService();
		}
		String printerName = receiptPrinter.getDeviceName();
		if (printerName != null && printerName.equals(ReceiptPrintService.OROPOS_PDF_PRINTER)) {
			return new JasperPrintService();
		}
		else if (receiptPrinter.getPrintSystem().equals(AppConstants.RECEIPT_ESC)) {
			return new EscPosPrintService();
		}
		return new JasperPrintService();
	}

	private static String createMemberAccountInfo(Ticket ticket) {
		Customer customer = CustomerDAO.getInstance().findById(ticket.getCustomerId());
		String memberAccInfo = new String();
		memberAccInfo += lineBreak();
		memberAccInfo += Messages.getString("ReceiptPrintService.25") + lineBreak();//$NON-NLS-1$
		memberAccInfo += Messages.getString("ReceiptPrintService.26") + customer.getId() + lineBreak(); //$NON-NLS-1$
		memberAccInfo += Messages.getString("NAME") + ": " + customer.getName() + lineBreak(); //$NON-NLS-1$ //$NON-NLS-2$ 
		memberAccInfo += Messages.getString("Balance") + ": \\" + CurrencyUtil.getCurrencySymbol() //$NON-NLS-1$ //$NON-NLS-2$
				+ NumberUtil.formatNumberAcceptNegative(customer.getBalance());

		return memberAccInfo;
	}

	private static void beginRow(StringBuilder html) {
		html.append("<div>"); //$NON-NLS-1$
	}

	private static String lineBreak() {
		return "<br>";//$NON-NLS-1$
	}

	private static void endRow(StringBuilder html) {
		html.append("</div>\n"); //$NON-NLS-1$
	}

	private static void addColumn(StringBuilder html, String columnText) {
		html.append("<span>" + columnText + "</span>"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static HashMap populateTicketProperties(Ticket ticket, TicketPrintProperties printProperties, PosTransaction transaction) {
		return populateTicketProperties(ticket, printProperties, transaction, false, false);
	}

	public static HashMap populateTicketProperties(Ticket ticket, TicketPrintProperties printProperties, PosTransaction transaction, boolean kitchenReceipt,
			boolean refundReceipt) {
		Application.getInstance().refreshStore();
		Store store = DataProvider.get().getStore();

		String extraOrderInfo = ""; //$NON-NLS-1$
		OrderType orderType = ticket.getOrderType();
		String orderTypeId = ""; //$NON-NLS-1$
		HashMap parameters = new HashMap();

		if (orderType != null) {
			orderTypeId = orderType.getId();
		}
		String header = ""; //$NON-NLS-1$
		String footer = ""; //$NON-NLS-1$
		String bottom = ""; //$NON-NLS-1$
		String orderInfo = ""; //$NON-NLS-1$

		if (kitchenReceipt) {
			header = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_KITCHEN_TICKET_HEADER, orderTypeId);
			orderInfo = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_KITCHEN_ORDER_INFO, orderTypeId);
			extraOrderInfo = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_KITCHEN_ADDITIONAL_ORDER_INFO_1, orderTypeId);
			footer = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_KITCHEN_TICKET_FOOTER, orderTypeId);
			bottom = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_KITCHEN_TICKET_BOTTOM, orderTypeId);
		}
		else {
			header = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_TICKET_HEADER, orderTypeId);
			orderInfo = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_ORDER_INFO, orderTypeId);
			extraOrderInfo = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_ADDITIONAL_ORDER_INFO_1, orderTypeId);
			footer = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_TICKET_FOOTER, orderTypeId);
			bottom = ReceiptUtil.getReceiptSection(store, AppConstants.PROP_TICKET_BOTTOM, orderTypeId);
			setPaginationProperty(parameters, DataProvider.get().getPrinters().getReceiptPrinter());
		}
		if (getReceiptPrintService() instanceof EscPosPrintService) {
			parameters.put(ReceiptParam.STORE_LOGO.getParamName(), " "); //$NON-NLS-1$
		}
		boolean showHeaderLogo = Boolean.valueOf(ReceiptUtil.getReceiptSection(store,
				kitchenReceipt ? AppConstants.PROP_SHOW_KITCHEN_HEADER_LOGO : AppConstants.PROP_SHOW_HEADER_LOGO, orderTypeId));
		double totalAmount = ticket.getTotalAmount();
		double tipAmount = 0;

		String currencySymbol = CurrencyUtil.getCurrencySymbol();

		parameters.put(ReceiptParam.STORE_NAME.getParamName(), store.getName());
		parameters.put(ReceiptParam.STORE_ADDRESS1.getParamName(), store.getAddressLine1());
		parameters.put(ReceiptParam.STORE_ADDRESS2.getParamName(), store.getAddressLine2());
		parameters.put(ReceiptParam.STORE_ADDRESS3.getParamName(), store.getAddressLine3());
		parameters.put(ReceiptParam.STORE_PHONE_NO.getParamName(), store.getTelephone());
		parameters.put(ReceiptParam.CURRENCY_SYMBOL.getParamName(), currencySymbol);
		parameters.put(ReceiptParam.TICKET_ID.getParamName(), ticket.getId());
		parameters.put(ReceiptParam.TICKET_SHORT_ID.getParamName(), ticket.getShortId());
		parameters.put(ReceiptParam.ORDER_DATE.getParamName(), DateUtil.formatFullDateAndTimeAsString(ticket.getCreateDate()));
		parameters.put(ReceiptParam.TOKEN_NO.getParamName(), "" + ticket.getTokenNo()); //$NON-NLS-1$
		if (StringUtils.isNotBlank(ticket.getNote())) {
			parameters.put(ReceiptParam.GUEST_NOTE.getParamName(), ticket.getNote());
		}
		parameters.put(ReceiptParam.BARTAB_NAME.getParamName(), ticket.getBartabName());

		if (transaction != null && transaction.getPaymentType() == PaymentType.MEMBER_ACCOUNT) {
			parameters.put(ReceiptParam.MEMBER_ACCOUNT_INFO.getParamName(), createMemberAccountInfo(ticket));
		}

		if (orderType != null && (orderType.isDelivery() || orderType.isPickup())) {
			if (ticket.isCustomerWillPickup()) {
				parameters.put(ReceiptParam.ORDER_TYPE.getParamName(), "PICKUP"); //$NON-NLS-1$
			}
			else {
				parameters.put(ReceiptParam.ORDER_TYPE.getParamName(), "DELIVERY"); //$NON-NLS-1$
			}
			String deliveryAddress = ticket.getDeliveryAddress() == null ? "" : ticket.getDeliveryAddress();// $NON-NLS-1$ //$NON-NLS-1$
			String extraDeliveryInfo = ticket.getExtraDeliveryInfo() == null ? "" : ticket.getExtraDeliveryInfo();// $NON-NLS-1$ //$NON-NLS-1$
			if (StringUtils.isNotBlank(deliveryAddress) || StringUtils.isNotBlank(extraDeliveryInfo)) {
				parameters.put(ReceiptParam.DELIVERY_ADDRESS.getParamName(), deliveryAddress + extraDeliveryInfo);
			}
			if (ticket.getDeliveryDate() != null) {
				parameters.put(ReceiptParam.DELIVERY_DATE.getParamName(), DateUtil.formatFullDateAndTimeAsString(ticket.getDeliveryDate())); // $NON-NLS-1$
			}
		}
		else {
			if (orderType != null && orderType.isHasForHereAndToGo()) {
				SubOrderType subOrderType = ticket.getSubOrderType();
				parameters.put(ReceiptParam.ORDER_TYPE.getParamName(), subOrderType != null ? subOrderType.getDisplayString() : ticket.getOrderType()); // $NON-NLS-1$
			}
			else {
				parameters.put(ReceiptParam.ORDER_TYPE.getParamName(), ticket.getOrderType()); // $NON-NLS-1$
			}
		}
		Terminal terminal = ticket.getTerminal();
		if (terminal != null) {
			parameters.put(ReceiptParam.TERMINAL_ID.getParamName(), terminal.getId()); // $NON-NLS-1$
			parameters.put(ReceiptParam.TERMINAL_NAME.getParamName(), terminal.getName()); // $NON-NLS-1$
		}
		User owner = ticket.getOwner();
		if (owner != null) {
			parameters.put(ReceiptParam.SERVER_NAME.getParamName(), owner.getFullName()); // $NON-NLS-1$
			parameters.put(ReceiptParam.SERVER_ID.getParamName(), owner.getId()); // $NON-NLS-1$
		}
		parameters.put(ReceiptParam.GUEST_COUNT.getParamName(), ticket.getNumberOfGuests().toString()); // $NON-NLS-1$

		int printCount = POSUtil.parseInteger(ticket.getProperty(AppConstants.PRINT_COUNT));
		if (printCount > 0) {
			parameters.put(ReceiptParam.PRINT_COUNT.getParamName(), printCount);
		}
		parameters.put(ReceiptParam.TABLE_NO.getParamName(), ticket.getTableNameDisplay());
		if (kitchenReceipt && orderType.isBarTab()) {
			parameters.put(ReceiptParam.TABLE_NO.getParamName(), ticket.getBartabName());
		}
		if (ticket.getCustomer() != null) {
			Customer customer = ticket.getCustomer();
			parameters.put(ReceiptParam.CUSTOMER_NAME.getParamName(), customer.getName()); // $NON-NLS-1$
			parameters.put(ReceiptParam.CUSTOMER_ID.getParamName(), customer.getMemberId() != null ? customer.getMemberId() : ""); // $NON-NLS-1$ //$NON-NLS-1$
			parameters.put(ReceiptParam.CUSTOMER_PHONE.getParamName(), customer.getMobileNo()); // $NON-NLS-1$
			parameters.put(ReceiptParam.CUSTOMER_SIGNATURE.getParamName(), customer.getSignatureImageId());
			if (POSUtil.getBoolean(store.getProperty(AppConstants.LOYALTY_ENABLED))) {
				if (transaction != null) {
					String chargedPoint = transaction.getExtraProperty(AppConstants.LOYALTY_CHARGED_AMOUNT);
					if (StringUtils.isNotBlank(chargedPoint)) {
						parameters.put(ReceiptParam.LOYALTY_REDEEM.getParamName(), chargedPoint);
					}

					String earnedPoint = transaction.getExtraProperty(AppConstants.LOYALTY_POINT_EARNED);
					String loyaltyEarned = earnedPoint == null ? String.valueOf(0) : earnedPoint;
					parameters.put(ReceiptParam.LOYALTY_EARNED.getParamName(), loyaltyEarned);
				}
				parameters.put(ReceiptParam.LOYALTY_TOTAL.getParamName(), customer.getLoyaltyPoint());
			}
		}
		if (ticket.getSalesArea() != null) {
			SalesArea salesArea = ticket.getSalesArea();
			parameters.put(ReceiptParam.SALES_AREA.getParamName(), salesArea.getName()); // $NON-NLS-1$
		}
		if (ticket.getTransactions() != null) {
			String strPaymentType = findLastTxPaymentTypeName(ticket);
			parameters.put(ReceiptParam.PAYMENT_TYPE.getParamName(), strPaymentType); // $NON-NLS-1$
		}
		if (ticket.getAssignedDriver() != null) {
			User driver = ticket.getAssignedDriver();
			parameters.put(ReceiptParam.DRIVER_NAME.getParamName(), driver.getFullName()); // $NON-NLS-1$
			parameters.put(ReceiptParam.DRIVER_ID.getParamName(), driver.getId()); // $NON-NLS-1$
		}
		String barcodeParamName = ReceiptParam.BARCODE.getParamName();
		if (header.contains(barcodeParamName) || orderInfo.contains(barcodeParamName) || extraOrderInfo.contains(barcodeParamName)
				|| footer.contains(barcodeParamName) || bottom.contains(barcodeParamName)) {
			parameters.put(barcodeParamName, String.valueOf(ticket.getId())); // $NON-NLS-1$
		}
		parameters.put(ReceiptParam.RECEIPT_TYPE.getParamName(), printProperties.getReceiptTypeName());

		String splitTicketId = ticket.getProperty("SPLIT_TICKET"); //$NON-NLS-1$
		if (StringUtils.isNotEmpty(splitTicketId)) {
			parameters.put(SPLIT_TICKET_ID, splitTicketId); //$NON-NLS-1$
		}
		parameters.put(ReceiptParam.PRINT_DATE.getParamName(), DateUtil.formatFullDateAndTimeAsString(new Date())); // $NON-NLS-1$
		parameters.put(ITEM_TEXT, POSConstants.RECEIPT_REPORT_ITEM_LABEL);
		parameters.put(QUANTITY_TEXT, POSConstants.RECEIPT_REPORT_QUANTITY_LABEL);
		parameters.put(SUB_TOTAL_TEXT, POSConstants.RECEIPT_REPORT_SUBTOTAL_LABEL);
		parameters.put(SHOW_SUBTOTAL, Boolean.valueOf(printProperties.isShowSubtotal()));
		parameters.put(SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		parameters.put(SHOW_FOOTER, Boolean.valueOf(printProperties.isShowFooter()));

		String ticketHeaderString = getReceiptParamValuesAsString(parameters, orderInfo);
		if (ticket.isVoided()) {
			ticketHeaderString += Messages.getString("ReceiptPrintService.16") + "<br>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		parameters.put(TICKET_HEADER, ticketHeaderString); //$NON-NLS-1$
		parameters.put(ADDITIONAL_ORDER_INFO, getReceiptParamValuesAsString(parameters, extraOrderInfo)); //$NON-NLS-1$
		parameters.put(HEADER_LINE1, getReceiptParamValuesAsString(parameters, header));

		if (printProperties.isShowFooter()) {
			if (ticket.getDiscountAmount() != 0.0) {
				parameters.put(DISCOUNT_AMOUNT, NumberUtil.formatNumber(ticket.getDiscountAmount(), true));
			}
			String TAX_STRING = POSConstants.RECEIPT_REPORT_TAX_LABEL + currencySymbol;
			String taxAmount = NumberUtil.formatNumber(ticket.getTaxAmount(), true);
			if (ticket.isTaxIncluded()) {
				TAX_STRING += Messages.getString("ReceiptPrintService.18"); //$NON-NLS-1$
				if (!store.isShowSubtotalWithoutTax()) {
					taxAmount = " (" + taxAmount + ")"; //$NON-NLS-1$ //$NON-NLS-2$
				}
			}

			if (ticket.getTaxAmount() != 0.0) {
				parameters.put(TAX_AMOUNT, taxAmount);
			}

			if (ticket.getServiceCharge() != 0.0) {
				parameters.put(SERVICE_CHARGE, NumberUtil.formatNumberAcceptNegative(ticket.getServiceCharge()));
			}

			if (ticket.getDeliveryCharge() > 0.0) {
				parameters.put(DELIVERY_CHARGE, NumberUtil.formatNumberAcceptNegative(ticket.getDeliveryCharge()));
			}

			if (ticket.getGratuity() != null) {
				tipAmount = ticket.getGratuity().getAmount();
				parameters.put(TIP_AMOUNT, NumberUtil.formatNumber(tipAmount));
			}

			parameters.put(TOTAL_TEXT, POSConstants.SUBTOTAL + currencySymbol); //$NON-NLS-1$
			parameters.put(DISCOUNT_TEXT, POSConstants.RECEIPT_REPORT_DISCOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(TAX_TEXT, TAX_STRING); //$NON-NLS-1$
			parameters.put(SERVICE_CHARGE_TEXT, POSConstants.RECEIPT_REPORT_SERVICE_CHARGE_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(DELIVERY_CHARGE_TEXT, POSConstants.RECEIPT_REPORT_DELIVERY_CHARGE_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(TIPS_TEXT, POSConstants.RECEIPT_REPORT_TIPS_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(NET_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_TOTAL_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(PAID_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_PAIDAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(DUE_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_DUEAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(CHANGE_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_CHANGEAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			parameters.put(FEE_AMOUNT_TEXT, POSConstants.FEE_AMOUNT + currencySymbol); //$NON-NLS-1$

			if (ticket.getFeeAmount() > 0) {
				parameters.put(FEE_AMOUNT, NumberUtil.formatNumber(ticket.getFeeAmount())); //$NON-NLS-1$
			}

			parameters.put(NET_AMOUNT, NumberUtil.formatNumber(totalAmount, true)); //$NON-NLS-1$
			parameters.put(PAID_AMOUNT, NumberUtil.formatNumber(ticket.getPaidAmount())); //$NON-NLS-1$
			Double refundAmount = ticket.getRefundAmount();
			if (transaction != null) {
				Double tenderAmount = transaction.getTenderAmount();
				parameters.put(TENDER_AMOUNT_TEXT, POSConstants.RECEIPT_TENDERED_AMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
				parameters.put(TENDER_AMOUNT, NumberUtil.formatNumber(tenderAmount, true)); //$NON-NLS-1$
			}
			if (refundAmount > 0) {
				parameters.put(REFUND_AMOUNT_TEXT, Messages.getString("ReceiptPrintService.40") + currencySymbol); //$NON-NLS-1$ //$NON-NLS-2$
				parameters.put(REFUND_AMOUNT, NumberUtil.formatNumber(refundAmount)); //$NON-NLS-1$
			}
			Double subtotalAmount = ticket.getSubtotalAmount();
			if (store.isShowSubtotalWithoutTax()) {
				subtotalAmount = ticket.getSubtotalWithoutIncludedTax();
			}
			parameters.put(DUE_AMOUNT, NumberUtil.formatNumber(ticket.getDueAmount(), true)); //$NON-NLS-1$
			parameters.put(GRAND_SUBTOTAL, NumberUtil.formatNumber(subtotalAmount, true)); //$NON-NLS-1$
			parameters.put(FOOTER_MESSAGE, getReceiptParamValuesAsString(parameters, footer)); //$NON-NLS-1$
			parameters.put(BOTTOM_MESSAGE, getReceiptParamValuesAsString(parameters, bottom)); //$NON-NLS-1$
			parameters.put(COPY_TYPE, printProperties.getReceiptCopyType()); //$NON-NLS-1$
			if (refundReceipt) {
				populateRefundProperties(transaction, ticket.getTransactions(), parameters);
			}
			if (transaction != null) {
				double changedAmount = transaction.getTenderAmount() - transaction.getAmount();
				if (changedAmount < 0) {
					changedAmount = 0;
				}
				parameters.put(CHANGED_AMOUNT, NumberUtil.formatNumber(changedAmount)); //$NON-NLS-1$
				if (transaction.isCard()) {
					if (!orderType.isRetailOrder()) {
						parameters.put(CARD_PAYMENT, true); //$NON-NLS-1$
					}
					String cardInformationForReceipt = CardConfig.getPaymentGateway().getProcessor().getCardInformationForReceipt(transaction);
					if (StringUtils.isEmpty(cardInformationForReceipt)) {
						cardInformationForReceipt = getCardInformation(transaction);
					}
					parameters.put(APPROVAL_CODE, cardInformationForReceipt); //$NON-NLS-1$
				}

				if (transaction instanceof CustomerAccountTransaction) {
					if (!orderType.isRetailOrder()) {
						parameters.put(CARD_PAYMENT, true); //$NON-NLS-1$
					}
				}

				if (transaction instanceof GiftCertificateTransaction) {
					parameters.put(CARD_PAYMENT, true); //$NON-NLS-1$
					parameters.put(APPROVAL_CODE, createGiftCertificateInfo(transaction));
				}
			}
			StringBuilder paymentSummary = null;
			if (StringUtils.isEmpty(splitTicketId)) {
				paymentSummary = buildPayments(ticket);
			}
			if (paymentSummary != null) {
				parameters.put(ADDITIONAL_PAYMENT_PROPERTIES, paymentSummary.toString()); //$NON-NLS-1$
			}

			if (!refundReceipt && Application.getInstance().getTerminal().isEnableMultiCurrency()) {
				if (transaction != null) {
					Set<PosTransaction> transactions = new HashSet<>();
					transactions.add(transaction);
					StringBuilder multiCurrencyBreakdownCashBack = buildAllMultiCurrency(ticket, transactions, false);
					if (multiCurrencyBreakdownCashBack != null) {
						parameters.put(ADDITIONAL_PROPERTIES, multiCurrencyBreakdownCashBack.toString()); //$NON-NLS-1$
					}
				}
				else {
					StringBuilder multiCurrencyTotalAmount = buildAllMultiCurrency(ticket, ticket.getTransactions(), false);
					if (multiCurrencyTotalAmount != null && splitTicketId == null)
						parameters.put(ADDITIONAL_PROPERTIES, multiCurrencyTotalAmount.toString()); //$NON-NLS-1$

				}
			}
		}
		if (showHeaderLogo) {
			ImageIcon storeLogo = store.getStoreLogo();
			if (storeLogo != null) {
				parameters.put(STORE_LOGO_ICON, storeLogo.getImage()); //$NON-NLS-1$
			}
		}
		boolean isShowTipsSuggestion = ReceiptUtil.getReceiptBooleanProp(store, AppConstants.PROP_SHOW_TIPS_SUGGESTION, orderTypeId);
		if (isShowTipsSuggestion) {
			parameters.put(SHOW_TIPS, tipsCalculation(ticket)); //$NON-NLS-1$
		}

		boolean isShowTipsBlock = ReceiptUtil.getReceiptBooleanProp(store, AppConstants.PROP_SHOW_TIPS_BLOCK, orderTypeId);
		parameters.put(SHOW_TIPS_BLOCK, isShowTipsBlock); //$NON-NLS-1$

		boolean isShowTaxBreakdown = ReceiptUtil.getReceiptBooleanProp(store, AppConstants.PROP_SHOW_TAX_BREAKDOWN, orderTypeId);
		parameters.put(IS_SHOW_TAX_BREAKDOWN, isShowTaxBreakdown); //$NON-NLS-1$
		if (isShowTaxBreakdown) {
			StringBuilder taxBreakdown = getTaxBreakdown(ticket, store);
			if (taxBreakdown != null) {
				parameters.put(TAX_BREAKDOWN_TEXT, taxBreakdown.toString()); //$NON-NLS-1$
			}
		}
		return parameters;
	}

	private static String createGiftCertificateInfo(PosTransaction transaction) {
		GiftCard giftCard = GiftCardDAO.getInstance().get(transaction.getGiftCertNumber());
		if (giftCard == null) {
			return ""; //$NON-NLS-1$
		}

		String cardNo = giftCard.getCardNumber();
		String lastFourDigitsOfGiftCardNo;
		if (cardNo.length() > 4) {
			lastFourDigitsOfGiftCardNo = "***" + cardNo.substring(cardNo.length() - 4); //$NON-NLS-1$
		}
		else {
			lastFourDigitsOfGiftCardNo = cardNo;
		}

		String giftCertificateInfo = new String();
		giftCertificateInfo += lineBreak();
		giftCertificateInfo += Messages.getString("ReceiptPrintService.30") + "------------" + lineBreak(); //$NON-NLS-1$ //$NON-NLS-2$
		giftCertificateInfo += Messages.getString("ReceiptPrintService.32") + ": " + lastFourDigitsOfGiftCardNo + lineBreak(); //$NON-NLS-1$//$NON-NLS-2$
		giftCertificateInfo += Messages.getString("ReceiptPrintService.33") + ": " + CurrencyUtil.getCurrencySymbol() + giftCard.getBalance(); //$NON-NLS-1$//$NON-NLS-2$
		return giftCertificateInfo;
	}

	private static void setPaginationProperty(HashMap parameters, String printerName) {
		if (printerName != null && (printerName.equalsIgnoreCase(OROPOS_PDF_PRINTER))) {
			parameters.put(JRParameter.IS_IGNORE_PAGINATION, Boolean.TRUE);
		}
		else {
			parameters.put(JRParameter.IS_IGNORE_PAGINATION, false);
		}
	}

	public static void setPaginationProperty(HashMap parameters, String printerName, boolean ignorePagination) {
		if (printerName != null && (printerName.equalsIgnoreCase(OROPOS_PDF_PRINTER))) {
			parameters.put(JRParameter.IS_IGNORE_PAGINATION, Boolean.TRUE);
		}
		else {
			parameters.put(JRParameter.IS_IGNORE_PAGINATION, ignorePagination);
		}
	}

	private static String getReceiptParamValuesAsString(Map map, String info) {
		if (info == null)
			return ""; //$NON-NLS-1$
		ReceiptParam[] receiptParams = ReceiptParam.values();
		for (ReceiptParam receiptParam : receiptParams) {
			String paramName = receiptParam.getParamName();
			Object object = map.get(paramName);
			if (object == null || StringUtils.isEmpty(object.toString()) || object.toString().equals("[]")) { //$NON-NLS-1$
				String text = "<" + receiptParam.getParamName() + ">.*</" + receiptParam.getParamName() + ">"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				info = info.replaceAll("<br>" + text, ""); //$NON-NLS-1$ //$NON-NLS-2$
				info = info.replaceAll(text, ""); //$NON-NLS-1$
				continue;
			}
			String val = object.toString();
			try {
				info = info.replaceAll("\\$" + paramName, val); //$NON-NLS-1$
			} catch (Exception e) {
				info = info.replaceAll("\\$" + paramName, "\\" + val); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		String addSpaces = replacePatternBySpace(info);
		return addSpaces;
	}

	private static String replacePatternBySpace(String info) {
		Pattern pattern = Pattern.compile("\\{\\{(\\d+)\\}\\}"); //NON-NLS-1$ //$NON-NLS-1$
		Matcher matcher = pattern.matcher(info);
		StringBuffer buffer = new StringBuffer();
		while (matcher.find()) {
			String group = matcher.group(1);
			try {
				int count = Integer.parseInt(group);
				String space = ""; //NON-NLS-1$ //$NON-NLS-1$
				for (int i = 0; i < count; i++) {
					space += " "; //NON-NLS-1$ //$NON-NLS-1$
				}
				matcher.appendReplacement(buffer, space);
			} catch (Exception e) {
			}
		}
		matcher.appendTail(buffer);
		return buffer.toString();
	}

	private static void populateRefundProperties(PosTransaction t, Set<PosTransaction> transactions, HashMap map) {
		if (transactions == null)
			return;
		TicketPrintProperties printProperties = new TicketPrintProperties(Messages.getString("ReceiptPrintService.36"), true, true, true); //$NON-NLS-1$
		printProperties.setPrintCookingInstructions(false);
		String refundText = ""; //$NON-NLS-1$
		if (t != null) {
			refundText = "<br>" + t.getPaymentType() + Messages.getString("ReceiptPrintService.66") + CurrencyUtil.getCurrencySymbol() + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ t.getAmount() + "<br>"; //$NON-NLS-1$
		}
		else {
			double refundAmount = 0;
			for (PosTransaction transaction : transactions) {
				if (transaction instanceof RefundTransaction || transaction.isVoided())
					refundAmount += transaction.getAmount();
			}
			refundText = "<br>" + Messages.getString("ReceiptPrintService.68") + CurrencyUtil.getCurrencySymbol() + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ NumberUtil.formatNumber(refundAmount) + "<br>"; //$NON-NLS-1$
		}
		map.put(ADDITIONAL_PROPERTIES, "<html><b>" + refundText + "</b></html>");//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

	public static HashMap populatePurchaseOrderProperties(PurchaseOrder order, TicketPrintProperties printProperties) {
		Store store = StoreDAO.getRestaurant();
		double totalAmount = order.getTotalAmount();

		HashMap map = new HashMap();
		setPaginationProperty(map, DataProvider.get().getPrinters().getReceiptPrinter());

		String currencySymbol = CurrencyUtil.getCurrencySymbol();
		map.put("currencySymbol", currencySymbol); //$NON-NLS-1$
		map.put("nameText", Messages.getString("InventoryStockInForm.7") + ": "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("addressText", Messages.getString("DepartmentExplorer.6") + ": "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		map.put("slNoText", Messages.getString("ReceiptPrintService.41")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("itemText", POSConstants.DESCRIPTION); //$NON-NLS-1$
		map.put(QUANTITY_TEXT, POSConstants.RECEIPT_REPORT_QUANTITY_LABEL); //$NON-NLS-1$
		map.put("priceText", Messages.getString("InventoryItemEntryDialog.15") + CurrencyUtil.getCurrencySymbolWithBracket()); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("unitText", POSConstants.UNIT); //$NON-NLS-1$ //$NON-NLS-2$
		map.put(SUB_TOTAL_TEXT, POSConstants.AMOUNT + CurrencyUtil.getCurrencySymbolWithBracket()); //$NON-NLS-1$
		map.put("note", Messages.getString("ReceiptPrintService.75")); //$NON-NLS-1$ //$NON-NLS-2$ 
		map.put("shipDateTxt", Messages.getString("ReceiptPrintService.76")); //$NON-NLS-1$ //$NON-NLS-2$ 

		String shipmentDate = order.getShipDate() != null ? DateUtil.formatFullDateAndTimeAsString(order.getShipDate()) : null;
		map.put("shipDate", shipmentDate); //$NON-NLS-1$ //$NON-NLS-2$

		map.put("signature1", Messages.getString("ReceiptPrintService.42")); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("signature2", Messages.getString("ReceiptPrintService.43")); //$NON-NLS-1$ //$NON-NLS-2$
		InventoryVendor vendor = order.getVendor();
		if (vendor != null) {
			String vendorName = vendor.getName();
			String vendorAddress = vendor.getAddress();
			String vendorMobile = vendor.getPhone();
			map.put("vendorName", vendorName); //$NON-NLS-1$
			String address = ""; //$NON-NLS-1$
			if (vendorAddress != null && !vendorAddress.isEmpty()) {
				address += vendorAddress;
			}
			if (vendorMobile != null && !vendorMobile.isEmpty()) {
				address += ", " + Messages.getString("ReceiptPrintService.44") + ":" + vendorMobile; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
			map.put("vendorAddress", StringUtils.isNotEmpty(address) ? address : null); //$NON-NLS-1$
		}
		map.put(RECEIPT_TYPE, printProperties.getReceiptTypeName());
		map.put(SHOW_SUBTOTAL, Boolean.valueOf(printProperties.isShowSubtotal()));
		map.put(SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		map.put(SHOW_FOOTER, Boolean.valueOf(printProperties.isShowFooter()));

		map.put(REPORT_DATE, POSConstants.RECEIPT_REPORT_DATE_LABEL + DateUtil.formatFullDateAndTimeAsString(new Date()));

		StringBuilder ticketHeaderBuilder2 = buildOrderInfo(order, printProperties);

		map.put("ticketHeader2", ticketHeaderBuilder2.toString()); //$NON-NLS-1$
		map.put("ticketHeader3", Messages.getString("InventoryLocationDAO.4")); //$NON-NLS-1$ //$NON-NLS-2$
		String invoiceNumber = order.getInvoiceNumber() != null ? Messages.getString("ReceiptPrintService.77") + order.getInvoiceNumber() : null; //$NON-NLS-1$
		map.put("invoiceNumber", invoiceNumber); //$NON-NLS-1$ //$NON-NLS-2$

		map.put(HEADER_LINE1, store.getName());
		map.put("headerLine9", store.getTelephone()); //$NON-NLS-1$

		Outlet outlet = DataProvider.get().getOutlet();
		if (outlet != null) {
			map.put("headerLine2", outlet.getAddressLine1()); //$NON-NLS-1$
			map.put("headerLine3", outlet.getAddressLine2()); //$NON-NLS-1$
			map.put("headerLine4", outlet.getAddressLine3()); //$NON-NLS-1$
			map.put("headerLine5", outlet.getCity()); //$NON-NLS-1$
			map.put("headerLine6", outlet.getState()); //$NON-NLS-1$
			map.put("headerLine7", outlet.getZipCode()); //$NON-NLS-1$
			map.put("headerLine8", outlet.getCountry()); //$NON-NLS-1$
		}
		else {
			map.put("headerLine2", store.getAddressLine1()); //$NON-NLS-1$
			map.put("headerLine3", store.getAddressLine2()); //$NON-NLS-1$
			map.put("headerLine4", store.getAddressLine3()); //$NON-NLS-1$
			map.put("headerLine6", store.getZipCode()); //$NON-NLS-1$
		}
		if (printProperties.isShowFooter()) {
			if (order.getDiscountAmount() > 0.0) {
				map.put(DISCOUNT_AMOUNT, NumberUtil.formatNumber(order.getDiscountAmount()));
			}
			if (order.getTaxAmount() > 0.0) {
				map.put(TAX_AMOUNT, NumberUtil.formatNumber(order.getTaxAmount()));
			}
			map.put(TOTAL_TEXT, POSConstants.RECEIPT_REPORT_TOTAL_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(DISCOUNT_TEXT, POSConstants.RECEIPT_REPORT_DISCOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(TAX_TEXT, POSConstants.RECEIPT_REPORT_TAX_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(SERVICE_CHARGE_TEXT, POSConstants.RECEIPT_REPORT_SERVICE_CHARGE_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(TIPS_TEXT, POSConstants.RECEIPT_REPORT_TIPS_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(NET_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_NETAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(PAID_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_PAIDAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(DUE_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_DUEAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(CHANGE_AMOUNT_TEXT, POSConstants.RECEIPT_REPORT_CHANGEAMOUNT_LABEL + currencySymbol); //$NON-NLS-1$
			map.put(NET_AMOUNT, NumberUtil.getCurrencyFormatWithoutCurrencySymbol(totalAmount)); //$NON-NLS-1$
			map.put(PAID_AMOUNT, NumberUtil.getCurrencyFormatWithoutCurrencySymbol(order.getPaidAmount())); //$NON-NLS-1$
			map.put(DUE_AMOUNT, NumberUtil.getCurrencyFormatWithoutCurrencySymbol(order.getDueAmount())); //$NON-NLS-1$
			map.put(GRAND_SUBTOTAL, NumberUtil.getCurrencyFormatWithoutCurrencySymbol(order.getSubtotalAmount())); //$NON-NLS-1$
			map.put(FOOTER_MESSAGE, store.getTicketFooterMessage()); //$NON-NLS-1$
			map.put(COPY_TYPE, printProperties.getReceiptCopyType()); //$NON-NLS-1$
		}
		return map;
	}

	private static String tipsCalculation(Ticket ticket) {
		StringBuilder buildTips = new StringBuilder();
		buildTips.append("<html>"); //$NON-NLS-1$
		double totalAmount = ticket.getTotalAmountWithTips();

		String tipsSection = ReceiptUtil.getReceiptSection(DataProvider.get().getStore(), AppConstants.TIPS_SUGGESTIONS, ticket.getOrderType().getId());
		if (tipsSection == null) {
			tipsSection = "[{\"percentage\":25,\"sortOrder\":1},{\"percentage\":20,\"sortOrder\":2},{\"percentage\":18,\"sortOrder\":3},{\"percentage\":15,\"sortOrder\":4}]"; //$NON-NLS-1$
		}
		JSONArray arrays = new JSONArray(tipsSection);
		beginRow(buildTips);
		for (int i = 0; i < arrays.length(); i++) {
			Object ts = arrays.get(i);
			if (ts == null) {
				continue;
			}
			JSONObject object = new JSONObject(ts.toString());
			double percentage = object.getDouble(AppConstants.PERCENTAGE);//$NON-NLS-1$
			addColumn(buildTips, "<br>" + NumberUtil.trimDecilamIfNotNeeded(percentage) + "% " + Messages.getString("ReceiptPrintService.99") //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
					+ NumberUtil.formatNumber(totalAmount * percentage / 100));
		}
		//		for (Object ts : array) {
		//			JSONObject object = new JSONObject(ts.toString());
		//			double percentage = object.getDouble(AppConstants.PERCENTAGE);//$NON-NLS-1$
		//			addColumn(buildTips, "<br>" + NumberUtil.trimDecilamIfNotNeeded(percentage) + "% " + Messages.getString("ReceiptPrintService.90") //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		//					+ NumberUtil.formatNumber(totalAmount * percentage / 100));
		//		}
		endRow(buildTips);
		buildTips.append("</html>"); //$NON-NLS-1$
		return buildTips.toString();
	}

	private static StringBuilder buildOrderInfo(PurchaseOrder order, TicketPrintProperties printProperties) {
		String orderNo = " "; //$NON-NLS-1$
		if (order.getOrderId() != null) {
			orderNo = order.getOrderId();
		}
		StringBuilder ticketHeaderBuilder = new StringBuilder();
		ticketHeaderBuilder.append("<html>"); //$NON-NLS-1$

		InventoryLocation location = order.getInventoryLocation();

		String inventoryLocation = ""; //$NON-NLS-1$
		String locationAddress = ""; //$NON-NLS-1$

		if (location != null) {
			inventoryLocation = location.getName();
			locationAddress = location.getAddress();
		}

		if (order.getVarificationDate() != null) {
			beginRow(ticketHeaderBuilder);
			addColumn(ticketHeaderBuilder, "Date: " + DateUtil.formatFullDateAndTimeAsString(order.getVarificationDate())); //$NON-NLS-1$
			endRow(ticketHeaderBuilder);
		}
		beginRow(ticketHeaderBuilder);
		addColumn(ticketHeaderBuilder, "Purchase Order #" + orderNo); //$NON-NLS-1$
		endRow(ticketHeaderBuilder);

		if (StringUtils.isNotEmpty(inventoryLocation)) {
			beginRow(ticketHeaderBuilder);
			addColumn(ticketHeaderBuilder, "Inventory Location: " + inventoryLocation); //$NON-NLS-1$
			endRow(ticketHeaderBuilder);
		}

		if (StringUtils.isNotEmpty(locationAddress)) {
			beginRow(ticketHeaderBuilder);
			addColumn(ticketHeaderBuilder, "Inventory Location Address: " + locationAddress); //$NON-NLS-1$
			endRow(ticketHeaderBuilder);
		}

		ticketHeaderBuilder.append("</html>"); //$NON-NLS-1$
		return ticketHeaderBuilder;
	}

	public static StringBuilder getTaxBreakdown(Ticket ticket, Store store) {
		return getTaxBreakdown(ticket, 20, 27, false, store);
	}

	public static StringBuilder getTaxBreakdown(Ticket ticket, int firstColLength, int secondColLegth, boolean ecsPrint, Store store) {
		StringBuilder taxBreakdownBuilder = new StringBuilder();
		Map<String, Double> taxProperties = populateTaxBreakdownMap(ticket);
		if (taxProperties.isEmpty()) {
			return null;
		}
		String currencySymbol = CurrencyUtil.getCurrencySymbol();
		taxBreakdownBuilder.append("<html><table>"); //$NON-NLS-1$
		Boolean includesTax = ticket.isTaxIncluded();
		String taxText = "(" + POSConstants.RECEIPT_REPORT_TAX_LABEL + ") ";//$NON-NLS-1$//$NON-NLS-2$
		for (Iterator iterator = taxProperties.entrySet().iterator(); iterator.hasNext();) {
			Map.Entry<String, Double> entry = (Entry<String, Double>) iterator.next();
			String taxValue = NumberUtil.formatNumber(entry.getValue());
			String taxName = entry.getKey();
			if (!ecsPrint && taxName.length() > 15) {
				taxName = taxName.substring(0, 13) + "..";//$NON-NLS-1$
			}
			taxName += currencySymbol;
			if (includesTax) {
				taxName += Messages.getString("ReceiptPrintService.18"); //$NON-NLS-1$
				taxValue = " (" + taxValue + ")"; //$NON-NLS-1$ //$NON-NLS-2$
			}
			beginRow(taxBreakdownBuilder);
			addColumn(taxBreakdownBuilder, getHtmlText(taxText + taxName, taxName.length(), RIGHT));// $NON-NLS-1$
			addColumn(taxBreakdownBuilder, getHtmlText(taxValue, secondColLegth - (ecsPrint ? 0 : taxValue.length()), RIGHT));// $NON-NLS-1$
			if (iterator.hasNext()) {
				endRow(taxBreakdownBuilder);
			}
		}
		taxBreakdownBuilder.append("</div></table></html>"); //$NON-NLS-1$
		return taxBreakdownBuilder;
	}

	public static Map<String, Double> populateTaxBreakdownMap(Ticket ticket) {
		Map<String, Double> taxProperties = new HashMap<String, Double>();
		List<TicketItem> ticketItems = ticket.getTicketItems();
		List<TicketItemTax> taxes = new ArrayList<TicketItemTax>();
		for (TicketItem ticketItem : ticketItems) {
			for (TicketItemTax ticketItemTax : ticketItem.getTaxes()) {
				String key = ticketItemTax.getName();
				Double taxAmount = taxProperties.get(key);
				if (taxAmount == null) {
					taxAmount = 0.0;
				}
				taxAmount += ticketItemTax.getTaxAmount();
				taxProperties.put(key, taxAmount);
				taxes.add(ticketItemTax);
			}
		}
		return taxProperties;
	}

	@Deprecated
	public static StringBuilder buildMulticurrencyBreakdown(Ticket ticket, TicketPrintProperties printProperties, boolean escPrint) {
		DecimalFormat decimalFormat = new DecimalFormat("0.00"); //$NON-NLS-1$

		StringBuilder currencyAmountBuilder = new StringBuilder();
		currencyAmountBuilder.append("<html><table>"); //$NON-NLS-1$

		String sep = "------------------------------------";//$NON-NLS-1$

		beginRow(currencyAmountBuilder);
		String space = /* "&nbsp;"*/" "; //$NON-NLS-1$
		addColumn(currencyAmountBuilder, space);//$NON-NLS-1$
		addColumn(currencyAmountBuilder, space);//$NON-NLS-1$
		addColumn(currencyAmountBuilder, space);//$NON-NLS-1$
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, "<b>Currency breakdown</b>"); //$NON-NLS-1$
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, sep);
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, getHtmlText("", 13, RIGHT));//$NON-NLS-1$
		addColumn(currencyAmountBuilder, getHtmlText("Net Amount", escPrint ? 13 : 11, RIGHT));//$NON-NLS-1$
		addColumn(currencyAmountBuilder, getHtmlText("Due", escPrint ? 13 : 17, RIGHT));//$NON-NLS-1$
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, sep);
		endRow(currencyAmountBuilder);

		int rowCount = 0;
		List<Currency> allCurrency = CurrencyUtil.getAllCurrency(ticket.getOutletId());
		if (allCurrency != null) {
			for (Currency currency : allCurrency) {
				if (currency == null) {
					continue;
				}
				String key = currency.getName();
				double rate = currency.getExchangeRate();
				beginRow(currencyAmountBuilder);
				addColumn(currencyAmountBuilder, getHtmlText(key, key.length(), RIGHT));

				String totalAmountText = decimalFormat.format(ticket.getTotalAmountWithTips() * rate);
				String dueAmountText = decimalFormat.format(ticket.getDueAmount() * rate);

				addColumn(currencyAmountBuilder, getHtmlText(totalAmountText, escPrint ? 13 : 20 - totalAmountText.length(), RIGHT));
				addColumn(currencyAmountBuilder, getHtmlText(dueAmountText, escPrint ? 13 : 20 - dueAmountText.length(), RIGHT));
				endRow(currencyAmountBuilder);
				rowCount++;
			}
		}
		if (rowCount == 0) {
			return null;
		}
		currencyAmountBuilder.append("</table></html>"); //$NON-NLS-1$
		return currencyAmountBuilder;
	}

	//	@Deprecated
	//	private static StringBuilder buildMultiCurrency(Ticket ticket, PosTransaction transaction, TicketPrintProperties printProperties) {
	//		return buildMultiCurrency(ticket, transaction, printProperties, false);
	//	}

	public static StringBuilder buildAllMultiCurrency(Ticket ticket, Set<PosTransaction> posTransactions, boolean escPrint) {
		DecimalFormat decimalFormat = new DecimalFormat("0.000"); //$NON-NLS-1$

		StringBuilder currencyAmountBuilder = new StringBuilder();
		currencyAmountBuilder.append("<html>"); //$NON-NLS-1$

		String sep = "------------------------------------";//$NON-NLS-1$

		String groupSettleTickets = ticket.getProperty("GROUP_SETTLE_TICKETS"); //$NON-NLS-1$
		if (groupSettleTickets == null) {
			groupSettleTickets = ""; //$NON-NLS-1$
		}

		buildDueAmountMultiCurrency(escPrint, decimalFormat, currencyAmountBuilder, sep, groupSettleTickets, ticket);
		String paidAmountBuilder = buildPaidAmountMultiCurrency(posTransactions, escPrint, decimalFormat, sep, groupSettleTickets);
		if (paidAmountBuilder != null) {
			currencyAmountBuilder.append(paidAmountBuilder);
		}

		if (!escPrint) {
			beginRow(currencyAmountBuilder);
			lineBreak();
			endRow(currencyAmountBuilder);
		}
		currencyAmountBuilder.append("</html>"); //$NON-NLS-1$
		return currencyAmountBuilder;
	}

	private static String buildPaidAmountMultiCurrency(Set<PosTransaction> posTransactions, boolean escPrint, DecimalFormat decimalFormat, String sep,
			String groupSettleTickets) {
		if (posTransactions == null || posTransactions.isEmpty()) {
			return null;
		}
		StringBuilder paidAmountBuilder = new StringBuilder();
		beginRow(paidAmountBuilder);
		lineBreak();
		endRow(paidAmountBuilder);

		beginRow(paidAmountBuilder);
		addColumn(paidAmountBuilder, groupSettleTickets + "<b>" + Messages.getString("ReceiptPrintService.34") + "</b>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		endRow(paidAmountBuilder);

		beginRow(paidAmountBuilder);
		addColumn(paidAmountBuilder, sep);
		endRow(paidAmountBuilder);

		beginRow(paidAmountBuilder);
		addColumn(paidAmountBuilder, getHtmlText("", 10, RIGHT));//$NON-NLS-1$
		addColumn(paidAmountBuilder, getHtmlText(Messages.getString("ReceiptPrintService.31"), escPrint ? 13 : 16, RIGHT)); //$NON-NLS-1$
		addColumn(paidAmountBuilder, getHtmlText(Messages.getString("ReceiptPrintService.35"), escPrint ? 13 : 12, RIGHT)); //$NON-NLS-1$
		endRow(paidAmountBuilder);

		beginRow(paidAmountBuilder);
		addColumn(paidAmountBuilder, sep);
		endRow(paidAmountBuilder);

		int rowCount = 0;
		for (Currency currency : CurrencyUtil.getAllCurrency(new ArrayList<>(posTransactions).get(0).getOutletId())) {
			if (currency == null) {
				continue;
			}
			String currencyName = currency.getName();
			String tenderedKey = currency.getId() + PosTransaction.PROP_TENDERED_POSTFIX;
			String cashBackKey = currency.getId() + PosTransaction.PROP_CASH_BACK_POSTFIX;

			double paidAmount = 0.0;
			double cashBackAmount = 0.0;
			for (PosTransaction posTransaction : posTransactions) {
				String tenderAmount = posTransaction.getProperty(tenderedKey);
				String cashbackAmount = posTransaction.getProperty(cashBackKey);
				paidAmount += tenderAmount == null ? 0 : NumberUtil.parseDouble(tenderAmount);
				cashBackAmount += cashbackAmount == null ? 0 : NumberUtil.parseDouble(cashbackAmount);
			}

			if (paidAmount == 0.0 && cashBackAmount == 0.0) {
				continue;
			}
			String paidAmountText = decimalFormat.format(paidAmount);
			String changeAmountText = decimalFormat.format(cashBackAmount);

			beginRow(paidAmountBuilder);
			addColumn(paidAmountBuilder, getHtmlText(currencyName, currencyName.length(), RIGHT));
			addColumn(paidAmountBuilder, getHtmlText(paidAmountText, escPrint ? 13 : 20 - paidAmountText.length(), RIGHT));
			addColumn(paidAmountBuilder, getHtmlText(changeAmountText, escPrint ? 13 : 20 - changeAmountText.length(), RIGHT));
			endRow(paidAmountBuilder);
			rowCount++;
		}

		if (rowCount == 0) {
			return null;
		}
		return paidAmountBuilder.toString();
	}

	private static void buildDueAmountMultiCurrency(boolean escPrint, DecimalFormat decimalFormat, StringBuilder currencyAmountBuilder, String sep,
			String groupSettleTickets, Ticket ticket) {

		List<Currency> allCurrency = CurrencyUtil.getAllCurrency(ticket.getOutletId());
		if (ticket.getDueAmount() > 0 && allCurrency != null) {
			beginRow(currencyAmountBuilder);
			lineBreak();
			endRow(currencyAmountBuilder);

			beginRow(currencyAmountBuilder);
			addColumn(currencyAmountBuilder, groupSettleTickets + "<b>" + Messages.getString("ReceiptPrintService.37") + "</b>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			endRow(currencyAmountBuilder);

			beginRow(currencyAmountBuilder);
			addColumn(currencyAmountBuilder, sep);
			endRow(currencyAmountBuilder);

			beginRow(currencyAmountBuilder);
			addColumn(currencyAmountBuilder, getHtmlText("", 10, RIGHT));//$NON-NLS-1$
			addColumn(currencyAmountBuilder, getHtmlText(Messages.getString("ReceiptPrintService.39"), escPrint ? 13 : 16, RIGHT)); //$NON-NLS-1$
			endRow(currencyAmountBuilder);

			beginRow(currencyAmountBuilder);
			addColumn(currencyAmountBuilder, sep);
			endRow(currencyAmountBuilder);
			for (Currency currency : allCurrency) {
				if (currency == null) {
					continue;
				}
				String key = currency.getName();
				double rate = currency.getExchangeRate();
				beginRow(currencyAmountBuilder);
				addColumn(currencyAmountBuilder, getHtmlText(key, key.length(), RIGHT));

				String totalAmountText = decimalFormat.format(ticket.getDueAmount() * rate);

				addColumn(currencyAmountBuilder, getHtmlText(totalAmountText, escPrint ? 13 : 20 - totalAmountText.length(), RIGHT));
				endRow(currencyAmountBuilder);
			}
		}
	}

	@Deprecated
	public static StringBuilder buildMultiCurrency(Ticket ticket, PosTransaction transaction, TicketPrintProperties printProperties, boolean escPrint) {
		if (transaction == null) {
			return null;
		}
		DecimalFormat decimalFormat = new DecimalFormat("0.000"); //$NON-NLS-1$

		StringBuilder currencyAmountBuilder = new StringBuilder();
		currencyAmountBuilder.append("<html><table>"); //$NON-NLS-1$

		String sep = "------------------------------------";//$NON-NLS-1$

		beginRow(currencyAmountBuilder);
		String space = " "; //$NON-NLS-1$
		addColumn(currencyAmountBuilder, space);
		//addColumn(currencyAmountBuilder, space);//$NON-NLS-1$
		//addColumn(currencyAmountBuilder, space);//$NON-NLS-1$
		endRow(currencyAmountBuilder);

		String groupSettleTickets = ticket.getProperty("GROUP_SETTLE_TICKETS"); //$NON-NLS-1$
		if (groupSettleTickets == null) {
			groupSettleTickets = ""; //$NON-NLS-1$
		}
		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, groupSettleTickets + "<b>\nCurrency breakdown</b>"); //$NON-NLS-1$
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, sep);
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, getHtmlText("", 10, RIGHT));//$NON-NLS-1$
		addColumn(currencyAmountBuilder, getHtmlText("Paid", escPrint ? 13 : 16, RIGHT));//$NON-NLS-1$
		addColumn(currencyAmountBuilder, getHtmlText("Cashback", escPrint ? 13 : 12, RIGHT));//$NON-NLS-1$
		endRow(currencyAmountBuilder);

		beginRow(currencyAmountBuilder);
		addColumn(currencyAmountBuilder, sep);
		endRow(currencyAmountBuilder);

		int rowCount = 0;
		for (Currency currency : CurrencyUtil.getAllCurrency(ticket.getOutletId())) {
			if (currency == null) {
				continue;
			}
			String currencyName = currency.getName();
			String tenderedKey = currency.getId() + PosTransaction.PROP_TENDERED_POSTFIX;
			String cashBackKey = currency.getId() + PosTransaction.PROP_CASH_BACK_POSTFIX;
			String paidAmount = transaction.getProperty(tenderedKey);
			String cashBackAmount = transaction.getProperty(cashBackKey);

			if (paidAmount == null) {
				paidAmount = "0";//$NON-NLS-1$
			}
			if (cashBackAmount == null) {
				cashBackAmount = "0";//$NON-NLS-1$
			}
			Double paid = Double.valueOf(paidAmount);
			Double changeDue = Double.valueOf(cashBackAmount);
			if (paid == 0 && changeDue == 0) {
				continue;
			}
			String paidAmountText = decimalFormat.format(paid);
			String changeAmountText = decimalFormat.format(changeDue);

			beginRow(currencyAmountBuilder);
			addColumn(currencyAmountBuilder, getHtmlText(currencyName, currencyName.length(), RIGHT));
			addColumn(currencyAmountBuilder, getHtmlText(paidAmountText, escPrint ? 13 : 20 - paidAmountText.length(), RIGHT));
			addColumn(currencyAmountBuilder, getHtmlText(changeAmountText, escPrint ? 13 : 20 - changeAmountText.length(), RIGHT));
			endRow(currencyAmountBuilder);
			rowCount++;
		}

		if (rowCount == 0) {
			return null;
		}
		currencyAmountBuilder.append("</table></html>"); //$NON-NLS-1$
		return currencyAmountBuilder;
	}

	private static StringBuilder buildPayments(Ticket ticket) {
		return buildPayments(ticket, false);
	}

	public static StringBuilder buildPayments(Ticket ticket, boolean escPrint) {
		Set<PosTransaction> transactionList = ticket.getTransactions();
		if (transactionList == null || transactionList.size() <= 1)
			return null;

		ArrayList<PosTransaction> transactions = new ArrayList<>(transactionList);
		Collections.sort(transactions, new Comparator<PosTransaction>() {

			@Override
			public int compare(PosTransaction o1, PosTransaction o2) {
				if (o1.getTransactionTime() == null) {
					return -1;
				}
				else if (o2.getTransactionTime() == null) {
					return 1;
				}
				return o1.getTransactionTime().compareTo(o2.getTransactionTime());
			}
		});

		StringBuilder paymentsBuilder = new StringBuilder();
		paymentsBuilder.append("<html><table>"); //$NON-NLS-1$

		beginRow(paymentsBuilder);
		String sp = /*"&nbsp;"*/" "; //$NON-NLS-1$
		addColumn(paymentsBuilder, sp);//$NON-NLS-1$
		addColumn(paymentsBuilder, sp);//$NON-NLS-1$
		addColumn(paymentsBuilder, sp);//$NON-NLS-1$
		endRow(paymentsBuilder);

		beginRow(paymentsBuilder);
		addColumn(paymentsBuilder, "Settlements:"); //$NON-NLS-1$
		endRow(paymentsBuilder);

		String tagStart = ""; //$NON-NLS-1$
		String tagEng = ""; //$NON-NLS-1$
		int amtLength = 13;
		int tipsLength = 0;
		boolean tipsExists = false;
		String currencySymbol = CurrencyUtil.getCurrencySymbol();
		for (PosTransaction transaction : transactions) {
			if (transaction.getTipsAmount() > 0) {
				tipsExists = true;
				tipsLength = 13;
				break;
			}
		}
		for (PosTransaction transaction : transactions) {
			beginRow(paymentsBuilder);
			tagStart = transaction.isVoided() ? "<strike>" : ""; //$NON-NLS-1$ //$NON-NLS-2$
			tagEng = transaction.isVoided() ? "</strike>" : ""; //$NON-NLS-1$ //$NON-NLS-2$
			Double amount = transaction.getAmount();
			String amountText = NumberUtil.formatNumberAcceptNegative((transaction instanceof RefundTransaction) ? -amount : amount);
			String tipsAmountText = transaction.getTipsAmount() > 0 ? NumberUtil.formatNumber(transaction.getTipsAmount()) : ""; //$NON-NLS-1$
			if (!escPrint) {
				amtLength = amountText.length() + ((14 - amountText.length()) * 2);
				tipsLength = tipsAmountText.length() + ((7 - tipsAmountText.length()) * 2);
			}
			String paymentTypeText = transaction.getPaymentType().getDisplayString() + currencySymbol;
			addColumn(paymentsBuilder, tagStart + getHtmlText(paymentTypeText, paymentTypeText.length(), RIGHT) + tagEng); //$NON-NLS-1$
			if (tipsExists) {
				addColumn(paymentsBuilder, tagStart + getHtmlText(tipsAmountText, tipsLength, RIGHT) + tagEng); //$NON-NLS-1$
			}
			addColumn(paymentsBuilder, getHtmlText(tagStart + amountText, amtLength, RIGHT) + tagEng); //$NON-NLS-1$
			endRow(paymentsBuilder);
		}
		paymentsBuilder.append("</table></html>"); //$NON-NLS-1$
		return paymentsBuilder;
	}

	public static String getHtmlText(String txt, int length, String align) {
		String sp = "&nbsp;";//$NON-NLS-1$
		/*if (txt.length() > 30) {
			txt = txt.substring(0, 30);
		}*/
		if (align.equals(CENTER)) {
			int space = (length - txt.length()) / 2;
			for (int i = 1; i <= space; i++) {
				txt = sp + txt + sp;//$NON-NLS-1$//$NON-NLS-2$
			}
		}
		else if (align.equals(RIGHT)) {
			int space = (length - txt.length());
			for (int i = 1; i <= space; i++) {
				txt = sp + txt;//$NON-NLS-1$
			}
		}
		else if (align.equals(LEFT)) {
			int space = (length - txt.length());
			for (int i = 1; i <= space; i++) {
				txt = txt + sp;//$NON-NLS-1$
			}
		}
		return txt;
	}

	public static void printVoidKitchenTicket(String virtualPrinterName, KitchenTicket kitchenTicket, String deviceName) throws Exception {
		HashMap map = new HashMap();

		map.put(HEADER_LINE1, DataProvider.get().getStore().getName());
		map.put(CARD_PAYMENT, true); //$NON-NLS-1$
		map.put(SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		map.put(SHOW_HEADER_SEPARATOR, Boolean.TRUE);
		map.put(ReceiptParam.TICKET_ID.getParamName(),
				POSConstants.RECEIPT_REPORT_TICKET_NO_LABEL + kitchenTicket.getTicketId() /* + "-" + ticket.getSequenceNumber() */); // $NON-NLS-1$

		if (kitchenTicket.getTableNumbers() != null && kitchenTicket.getTableNumbers().size() > 0) {
			map.put(ReceiptParam.TABLE_NO.getParamName(), POSConstants.RECEIPT_REPORT_TABLE_NO_LABEL + kitchenTicket.getTableNumbers());
		}
		if (StringUtils.isNotEmpty(kitchenTicket.getCustomerName())) {
			map.put("customer", Messages.getString("ReceiptPrintService.0") + kitchenTicket.getCustomerName()); //$NON-NLS-1$ //$NON-NLS-2$
		}
		map.put(ReceiptParam.SERVER_NAME.getParamName(), POSConstants.RECEIPT_REPORT_SERVER_LABEL + kitchenTicket.getServerName());
		map.put(REPORT_DATE, Messages.getString("ReceiptPrintService.119") + DateUtil.getReportDate()); //$NON-NLS-1$
		map.put(TICKET_HEADER, Messages.getString("ReceiptPrintService.12")); //$NON-NLS-1$ //$NON-NLS-2$ 

		String ticketType = kitchenTicket.getOrderType().toString();
		if (StringUtils.isNotEmpty(ticketType)) {
			ticketType = ticketType.replaceAll("_", " "); //$NON-NLS-1$ //$NON-NLS-2$
		}
		map.put(ReceiptParam.ORDER_TYPE.getParamName(), "** " + "VOID" + " **"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		map.put(PROP_PRINTER_NAME, Messages.getString("ReceiptPrintService.14") + virtualPrinterName); //$NON-NLS-1$ //$NON-NLS-2$ 
		getJasperPrintService().printVoidKitchenTicket(kitchenTicket, map, virtualPrinterName, deviceName);
	}

	public static void printToKitchen(Ticket ticket) {
		printToKitchen(ticket, true);
	}

	public static void printToKitchen(Ticket ticket, boolean isFilterKitchenPrintedItems) {
		printToKitchen(ticket, isFilterKitchenPrintedItems, true);
	}

	public static void printToKitchen(Ticket ticket, boolean isFilterKitchenPrintedItems, boolean saveAndUpdateStatus) {
		try {
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, isFilterKitchenPrintedItems);
			if (kitchenTickets == null || kitchenTickets.isEmpty()) {
				return;
			}
			doPrintToKitchenAndSaveStatus(ticket, kitchenTickets, saveAndUpdateStatus);

			//			if (PosWebService.get().isCloudConfigured()) {
			//				String jsonString = ServiceUtils.convertKitTicketsToJsonString(kitchenTickets);
			//				PosLog.debug(ReceiptPrintService.class, jsonString);
			//				OroMqttClient.getInstance().publishData(MqttTopics.PUBLIC, jsonString, true);
			//			}
			sendUpdateNotification();
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e);
			throw new PrintException(Messages.getString("ReceiptPrintService.28"), e); //$NON-NLS-1$
		}
	}

	private static void sendUpdateNotification() {
		try {
			JSONObject jsonObject = new JSONObject();
			jsonObject.put("terminalKey", TerminalUtil.getSystemUID()); //$NON-NLS-1$
			jsonObject.put("command", MqttCommand.CMD_REFRESH_KITCHEN_ORDER); //$NON-NLS-1$
			jsonObject.put("isKds", Boolean.FALSE); //$NON-NLS-1$

			OroMqttClient.getInstance().publishData(MqttCommand.TOPIC_KIT_DIS_UPDATE, jsonObject.toString());
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e);
		}
	}

	public static void printCourseItemsToKitchen(Ticket ticket, List<TicketItem> ticketItems, boolean shouldKdsPrint) {
		try {
			if (ticket == null && ticketItems == null) {
				return;
			}
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, shouldKdsPrint, ticketItems);
			doPrintToKitchenAndSaveStatus(ticket, kitchenTickets, shouldKdsPrint);
			POSMessageDialog.showMessage(POSUtil.getFocusedWindow(),
					Messages.getString("CourseOrganizeTableView.0") + " " + POSConstants.COURSE + " " + ticketItems.get(0).getCourseName());//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		} catch (Exception e) {
			POSMessageDialog.showError(null, e.getMessage(), e);
		}
	}

	public static void printItemsToKitchen(Ticket ticket, List<TicketItem> ticketItems) {
		try {
			if (ticket == null && ticketItems == null) {
				return;
			}
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, true, ticketItems);
			doPrintToKitchenAndSaveStatus(ticket, kitchenTickets, true);
		} catch (Exception e) {
			POSMessageDialog.showError(null, e.getMessage(), e);
		}
	}

	public static void doPrintToKitchenAndSaveStatus(Ticket ticket, List<KitchenTicket> kitchenTickets, boolean saveAndUpdateStatus) throws Exception {
		if (saveAndUpdateStatus) {
			ticket.setShouldUpdateStock(true);
		}
		for (KitchenTicket kitchenTicket : kitchenTickets) {
			kitchenTicket.setParentTicket(ticket);
			Printer printer = kitchenTicket.getPrinter();
			String deviceName = printer.getDeviceName();
			if (printer.getVirtualPrinter().getType() == VirtualPrinter.KITCHEN && deviceName != null) {
				logger.info("Printing to kitchen using printer: " + deviceName); //$NON-NLS-1$
				HashMap map = populateKitchenTicketProperties(kitchenTicket, deviceName, printer.getVirtualPrinter().getName(), saveAndUpdateStatus);
				setPrintProperty(map, printer);
				getReceiptPrintServiceProvider(printer).printKitchenTicket(kitchenTicket, map, ticket, printer.getVirtualPrinter().getName(), deviceName);

				if (saveAndUpdateStatus && DataProvider.get().getStore().isKDSenabled()) {
					TicketDAO.getInstance().saveKitchenPrintStatus(ticket, kitchenTicket);
				}
			}
			else if (printer.getVirtualPrinter().getType() == VirtualPrinter.STICKER && deviceName != null) {
				logger.info("Printing to sticker using printer: " + deviceName); //$NON-NLS-1$
				doPrintKitchenStickers(kitchenTicket, deviceName);

				//if (DataProvider.get().getStore().isKDSenabled()) {
				//	TicketDAO.getInstance().saveKitchenPrintStatus(ticket, kitchenTicket);
				//}
			}
			else if (saveAndUpdateStatus && printer.getVirtualPrinter().getType() == VirtualPrinter.KITCHEN_DISPLAY) {
				TicketDAO.getInstance().saveKitchenPrintStatus(ticket, kitchenTicket);
			}
		}
		if (saveAndUpdateStatus) {
			ticket.clearDeletedItems();
			ticket.setTicketStatus(TicketStatus.Preparing);
			TicketDAO.getInstance().saveOrUpdate(ticket);
		}
	}

	/**
	 * This method is used to printing all ticket item from ticket using all packing Printers
	 * 
	 * @param ticket Ticket.
	 * 
	  */
	public static void doPrintKitchenTicketAfterBump(Ticket ticket) throws Exception {
		if (ticket == null) {
			return;
		}

		List<Printer> packingPrinters = DataProvider.get().getPrinters().getPackingPrinters();
		if (packingPrinters == null || packingPrinters.size() == 0) {
			POSMessageDialog.showError(POSUtil.getFocusedWindow(), Messages.getString("NO_PACKING_PRINTER_CONFIGURED")); //$NON-NLS-1$
			return;
		}

		for (Printer printer : packingPrinters) {
			String deviceName = printer.getDeviceName();

			if (printer.getVirtualPrinter().getType() == VirtualPrinter.PACKING && deviceName != null) {
				logger.info("Printing to kitchen using packing printer: " + deviceName); //$NON-NLS-1$
				TicketPrintProperties printProperties = new TicketPrintProperties(null, true, true, true);
				HashMap map = ReceiptPrintService.populateTicketProperties(ticket, printProperties, null);
				setPrintProperty(map, printer);
				getReceiptPrintServiceProvider(printer).printTicket(ticket, map, deviceName);
			}
		}
	}

	public static void doPrintKitchenStickers(Ticket ticket) {
		try {
			List<Printer> stickerPrinters = DataProvider.get().getPrinters().getStickerPrinters();
			if (stickerPrinters.isEmpty()) {
				return;
			}
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, true);

			for (KitchenTicket kitchenTicket : kitchenTickets) {
				Printer printer = kitchenTicket.getPrinter();
				String deviceName = printer.getDeviceName();
				kitchenTicket.setParentTicket(ticket);

				if (printer.getVirtualPrinter().getType() == VirtualPrinter.STICKER && deviceName != null) {
					doPrintKitchenStickers(kitchenTicket, deviceName);
				}
			}
		} catch (Exception e) {
			POSMessageDialog.showError(null, e.getMessage(), e);
		}
	}

	private static void doPrintKitchenStickers(KitchenTicket kitchenTicket, String deviceName) {
		try {
			List<KitchenTicketItem> ticketItems = kitchenTicket.getTicketItems();
			List<KitchenSticker> kitchenStickers = new ArrayList<KitchenSticker>();

			int count = 0;
			int quantity = getTotalQty(ticketItems);
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) iterator.next();
				if (!kitchenTicketItem.isPrintKitchenSticker()) {
					continue;
				}
				if (kitchenTicketItem.isModifierItem() || kitchenTicketItem.isCookingInstruction()) {
					continue;
				}
				for (int i = 0; i < kitchenTicketItem.getQuantity(); i++) {
					KitchenSticker kitchenSticker = new KitchenSticker();
					kitchenSticker.setToken(kitchenTicket.getTokenNo());
					kitchenSticker.setCustomerName(addCustomerName(kitchenTicket));
					String menuItemName = kitchenTicketItem.getMenuItemName();
					if (StringUtils.isNotEmpty(menuItemName)) {
						menuItemName = (kitchenTicketItem.getQuantity() > 1) ? menuItemName.substring(menuItemName.indexOf(' ') + 1) : menuItemName;
					}
					kitchenSticker.setItemName(menuItemName);
					kitchenSticker.setModifiers(kitchenTicket.getModifiersForTicketItem(kitchenTicketItem.getTicketItemId()));
					kitchenSticker.setCookingInstructions(kitchenTicket.getCookingInstructionForTicketItem(kitchenTicketItem.getTicketItemId()));
					kitchenSticker.setTime(Messages.getString("ReceiptPrintService.102") + DateUtil.formatDateWithTime(new Date())); //$NON-NLS-1$
					if (kitchenTicket.getOrderType() != null) {
						kitchenSticker.setOrderType(kitchenTicket.getOrderType().getName());
					}

					kitchenSticker.setItemCount(
							Messages.getString("ReceiptPrintService.103") + (count + 1) + Messages.getString("ReceiptPrintService.104") + quantity); //$NON-NLS-1$ //$NON-NLS-2$
					kitchenStickers.add(kitchenSticker);
					count++;
				}
			}

			JasperPrint jasperPrint = printKitchenStickerItems(kitchenStickers);

			jasperPrint.setName("Kitchen_Sticker_" + kitchenTicket.getTokenNo()); //$NON-NLS-1$
			jasperPrint.setProperty(PROP_PRINTER_NAME, deviceName);
			JasperPrintService.printQuitely(jasperPrint);
		} catch (Exception e) {
			PosLog.error(ReceiptPrintService.class, e.getMessage(), e);
		}
	}

	private static String addCustomerName(KitchenTicket kitchenTicket) {
		Ticket parentTicket = kitchenTicket.getParentTicket();
		if (parentTicket == null) {
			return null;
		}
		Customer customer = parentTicket.getCustomer();
		String note = parentTicket.getNote();
		String customerNameAndNote = "";//$NON-NLS-1$
		if (customer != null) {
			customerNameAndNote = customer.getName();
		}
		if (StringUtils.isNotBlank(note)) {
			if (StringUtils.isNotBlank(customerNameAndNote)) {
				customerNameAndNote += "/" + note;//$NON-NLS-1$
			}
			else {
				customerNameAndNote = note;
			}
		}
		return customerNameAndNote;
	}

	private static int getTotalQty(List<KitchenTicketItem> ticketItems) {
		int quantity = 0;
		for (KitchenTicketItem kitchenTicketItem : ticketItems) {
			if (!kitchenTicketItem.isPrintKitchenSticker()) {
				continue;
			}
			if (kitchenTicketItem.isModifierItem() || kitchenTicketItem.isCookingInstruction()) {
				continue;
			}
			quantity += kitchenTicketItem.getQuantity();
		}
		return quantity;
	}

	@Deprecated
	public static JasperPrint getKitchenJasperPrint(Ticket ticket, boolean isFilterKitchenPrintedItems) {
		return getKitchenJasperPrint(ticket, isFilterKitchenPrintedItems, false);
	}

	@Deprecated
	public static JasperPrint getKitchenJasperPrint(Ticket ticket, boolean isFilterKitchenPrintedItems, boolean ignorePagination) {
		try {
			List<KitchenTicket> kitchenTickets = KitchenTicket.fromTicket(ticket, isFilterKitchenPrintedItems);

			for (KitchenTicket kitchenTicket : kitchenTickets) {
				kitchenTicket.setParentTicket(ticket);
				Printer printer = kitchenTicket.getPrinter();
				String deviceName = printer.getDeviceName();
				if (deviceName != null) {
					HashMap map = populateKitchenTicketProperties(kitchenTicket, deviceName, printer.getVirtualPrinter().getName(), ignorePagination);
					JasperPrint jasperPrint = getJasperPrintService().createKitchenPrint(map, printer.getVirtualPrinter().getName(), kitchenTicket, deviceName,
							ignorePagination);

					jasperPrint.setName("FP_KitchenReceipt_" + ticket.getId() + "_" + kitchenTicket.getSequenceNumber()); //$NON-NLS-1$ //$NON-NLS-2$
					jasperPrint.setProperty(PROP_PRINTER_NAME, deviceName);
					return jasperPrint;
				}
			}
		} catch (Exception e) {
			POSMessageDialog.showError(null, e.getMessage(), e);
		}
		return null;
	}

	public static void printVoidItemsToKitchen(Ticket ticket) {
		try {
			if (!ticket.getOrderType().isShouldPrintToKitchen())
				return;
			List<VoidItem> voidItems = new ArrayList<>();
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				VoidItem voidItem = ticketItem.getVoidItem();
				if (voidItem != null) {
					if (voidItem.getVoidedModifiers() != null) {
						voidItems.addAll(voidItem.getVoidedModifiers());
					}
					voidItems.add(voidItem);
				}
			}
			Map<Printer, KitchenTicket> itemMap = new HashMap<Printer, KitchenTicket>();
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			for (VoidItem voidItem : voidItems) {
				if (voidItem.isCooked())
					continue;

				PrinterGroup printerGroup = voidItem.getPrinterGroup();
				List<Printer> printers = getPrinters(printerGroup);
				if (printers == null) {
					continue;
				}

				for (Printer printer : printers) {
					KitchenTicket kitchenTicket = itemMap.get(printer);
					if (kitchenTicket == null) {
						kitchenTicket = new KitchenTicket();
						kitchenTicket.setPrinterGroup(printerGroup);
						kitchenTicket.setTicketId(ticket.getId());
						kitchenTicket.setTokenNo(ticket.getTokenNo());
						kitchenTicket.setCreateDate(serverTimestamp);
						kitchenTicket.setOrderType(ticket.getOrderType());

						if (ticket.getTableNumbers() != null) {
							kitchenTicket.setTableNumbers(new ArrayList<Integer>(ticket.getTableNumbers()));
						}

						kitchenTicket.setServerName(ticket.getOwner().getFirstName());
						kitchenTicket.setStatus(KitchenStatus.WAITING.name());

						if (StringUtils.isNotEmpty(ticket.getProperty(Ticket.CUSTOMER_NAME))) {
							kitchenTicket.setCustomerName(ticket.getProperty(Ticket.CUSTOMER_NAME));
						}
						kitchenTicket.setPrinter(printer);
						itemMap.put(printer, kitchenTicket);
					}
					KitchenTicketItem item = new KitchenTicketItem();
					item.setTicketItemId(voidItem.getId());
					item.setMenuItemCode(String.valueOf(voidItem.getMenuItemId()));
					item.setMenuItemName(voidItem.getMenuItemName());
					item.setQuantity(voidItem.getQuantity());
					item.setUnitName(""); //$NON-NLS-1$
					item.setMenuItemGroupName("VOID"); //$NON-NLS-1$
					item.setSortOrder(10001);
					item.setStatus(KitchenStatus.VOID.name());
					item.setKitchenTicket(kitchenTicket);
					kitchenTicket.addToticketItems(item);
				}
			}
			for (KitchenTicket kitchenTicket : itemMap.values()) {
				Printer printer = kitchenTicket.getPrinter();
				String deviceName = printer.getDeviceName();
				printVoidKitchenTicket(printer.getVirtualPrinter().getName(), kitchenTicket, deviceName);
			}
			// TicketDAO.getInstance().saveKitchenPrintStatua(ticket,
			// kitchenTickets);
		} catch (Exception e) {
			POSMessageDialog.showError(null, e.getMessage(), e);
		}
	}

	public static List<Printer> getPrinters(PrinterGroup printerGroup) {
		PosPrinters printers = DataProvider.get().getPrinters();

		if (printerGroup == null) {
			printerGroup = PrinterGroupDAO.getInstance().getDefaultPrinterGroup();
		}
		List<Printer> printerAll = new ArrayList<Printer>();
		if (printerGroup == null) {
			printerAll.addAll(printers.getKitchenPrinters());
			return printerAll;
		}
		List<String> printerNames = printerGroup.getPrinterNames();
		List<Printer> kitchenPrinters = printers.getKitchenPrinters();
		for (Printer printer : kitchenPrinters) {
			if (printerNames.contains(printer.getVirtualPrinter().getName())) {
				printerAll.add(printer);
			}
		}
		if (printerAll.isEmpty() && PrintServiceUtil.getFallBackPrinter() != null) {
			printerAll.add(PrintServiceUtil.getFallBackPrinter());
		}
		return printerAll;
	}

	public static Log getLogger() {
		return logger;
	}

	public static File getPdfPrinterDir() {
		return pdfPrinterDir;
	}

	/*private static String getCardNumber(BankCardMagneticTrack track) {
		String no = ""; //$NON-NLS-1$
	
		try {
			if (track.getTrack1().hasPrimaryAccountNumber()) {
				no = track.getTrack1().getPrimaryAccountNumber().getAccountNumber();
				no = "************" + no.substring(12); //$NON-NLS-1$
			}
			else if (track.getTrack2().hasPrimaryAccountNumber()) {
				no = track.getTrack2().getPrimaryAccountNumber().getAccountNumber();
				no = "************" + no.substring(12); //$NON-NLS-1$
			}
		} catch (Exception e) {
			logger.error(e);
		}
	
		return no;
	}*/

	private static String getCardInformation(PosTransaction transaction) {
		String br = "<br>"; //$NON-NLS-1$
		String string = br + Messages.getString("ReceiptPrintService.15") + " ------------------------"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		try {
			if (transaction.getCardReader() != null) {
				string += br + Messages.getString("ReceiptPrintService.17") + transaction.getCardReader(); //$NON-NLS-1$ 
			}
			if (transaction.getCardType() != null) {
				string += br + Messages.getString("ReceiptPrintService.20") + transaction.getCardType(); //$NON-NLS-1$ 
			}
			String cardNumber = transaction.getCardNumber();
			if (cardNumber != null) {
				string += br + Messages.getString("ReceiptPrintService.21") + " **** **** **** " //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
						+ cardNumber.substring(cardNumber.length() - 4, cardNumber.length());
			}
			if (transaction.getCardHolderName() != null) {
				string += br + Messages.getString("ReceiptPrintService.23") + transaction.getCardHolderName(); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (transaction.getCardTransactionId() != null) {
				string += br + Messages.getString("ReceiptPrintService.24") + transaction.getCardTransactionId(); //$NON-NLS-1$ //$NON-NLS-2$
			}
			string += br + Messages.getString("ReceiptPrintService.29") + transaction.getCardAuthCode(); //$NON-NLS-1$ //$NON-NLS-2$
		} catch (Exception e) {
			logger.error(e);
		}
		return string;
	}

	//	private static String getCustBlnceInformation(PosTransaction transaction) {
	//		CustomerAccountTransaction custBlnceTrans = (CustomerAccountTransaction) transaction;
	//		String string = "";
	//		try {
	//			Customer customer = CustomerDAO.getInstance().findById(custBlnceTrans.getCustomerId());
	//			if (customer == null) {
	//				return string;
	//			}
	//			string += "<br/>PAY FROM CUSTOMER BALANCE: ------------------------";
	//			string += "<br/>CUSTOMER ID: " + customer.getId();
	//			string += "<br/>CUSTOEMER NAME: " + customer.getName();
	//			string += "<br/>AMOUNT: " + CurrencyUtil.getCurrencySymbol() + transaction.getAmount();
	//			string += "<br/>BALANCE: " + CurrencyUtil.getCurrencySymbol() + customer.getBalance();
	//		} catch (Exception e) {
	//			logger.equals(e);
	//		}
	//		return string;
	//	}

	/*
	 * this method is used to print cloud ticket only
	 */
	public static void printCloudTicket(Ticket ticket, Terminal terminal, PosTransaction transaction) {
		try {
			TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true); // $NON-NLS-1$
			printProperties.setPrintCookingInstructions(false);
			HashMap map = populateTicketProperties(ticket, printProperties, transaction);

			List<TerminalPrinters> terminalPrinters = TerminalPrintersDAO.getInstance().findTerminalPrinters(terminal);
			List<Printer> activeReceiptPrinters = new ArrayList<Printer>();
			for (TerminalPrinters terminalPrinters2 : terminalPrinters) {
				int printerType = terminalPrinters2.getVirtualPrinter().getType();
				if (printerType == VirtualPrinter.RECEIPT) {
					Printer printer = new Printer(terminalPrinters2.getVirtualPrinter(), terminalPrinters2.getPrinterName());
					activeReceiptPrinters.add(printer);
				}
			}
			if (activeReceiptPrinters == null || activeReceiptPrinters.isEmpty()) {
				JasperPrint jasperPrint = getJasperPrintService().createPrint(ticket, map, null);
				jasperPrint.setName(ORDER_ + ticket.getId());
				jasperPrint.setProperty(PROP_PRINTER_NAME, DataProvider.get().getPrinters().getReceiptPrinter());
				JasperPrintService.printQuitely(jasperPrint);
			}
			else {
				for (Printer activeReceiptPrinter : activeReceiptPrinters) {
					String deviceName = activeReceiptPrinter.getDeviceName();
					if (deviceName != null) {
						JasperPrint jasperPrint = getJasperPrintService().createPrint(ticket, map, null);
						jasperPrint.setName(ORDER_ + ticket.getId() + deviceName);
						jasperPrint.setProperty(PROP_PRINTER_NAME, deviceName);
						JasperPrintService.printQuitely(jasperPrint);
					}
				}
			}
		} catch (Exception e) {
			logger.error(com.floreantpos.POSConstants.PRINT_ERROR, e);
		}
	}

	public static ReciptPaperSize getReceiptPaperSize(String receiptPropertyName) {
		Terminal currentTerminal = DataProvider.get().getCurrentTerminal();
		String receiptProperty = currentTerminal.getProperty(receiptPropertyName);
		ReciptPaperSize paperSize = ReciptPaperSize.Eighty;
		try {
			if (StringUtils.isNotEmpty(receiptProperty)) {
				paperSize = ReciptPaperSize.valueOf(receiptProperty);
			}
		} catch (Exception e) {
			paperSize = ReciptPaperSize.Eighty;
		}
		return paperSize;
	}

	private static KitchenStickerPaperSize getKitchenStickerPaperSize(String receiptPropertyName) {
		Terminal currentTerminal = DataProvider.get().getCurrentTerminal();
		String receiptProperty = currentTerminal.getProperty(receiptPropertyName);
		KitchenStickerPaperSize paperSize = KitchenStickerPaperSize.FiftyOneMM;
		try {
			if (StringUtils.isNotEmpty(receiptProperty)) {
				paperSize = KitchenStickerPaperSize.fromName(receiptProperty);
			}
		} catch (Exception e) {
			paperSize = KitchenStickerPaperSize.FiftyOneMM;
		}
		return paperSize;
	}

	public static boolean hasNoReceiptPrinters() {
		PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGateway();
		boolean printUsingPaymentGateway = paymentGateway != null && paymentGateway.printUsingThisTerminal();
		List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
		boolean isReceiptPrintersNotExist = receiptPrinters == null || receiptPrinters.isEmpty() || receiptPrinters.size() == 0;
		if (isReceiptPrintersNotExist && (!printUsingPaymentGateway)) {
			return true;
		}
		return false;
	}

	public static boolean hasNoKitchenReceiptPrinters() {
		PosPrinters printers = DataProvider.get().getPrinters();
		if (printers == null || printers.getKitchenPrinters() == null
				|| printers.getKitchenPrinters().size() == 0 && printers.getStickerPrinters().size() == 0 && !printers.isPrintToKds()) {
			return true;
		}
		return false;
	}

	private static void updatePrintCount(Ticket ticket) {
		int printCountInt = POSUtil.getIntegerOrZero(ticket.getProperty(AppConstants.PRINT_COUNT));
		ticket.addProperty(AppConstants.PRINT_COUNT, String.valueOf(++printCountInt));
		TicketDAO.getInstance().saveOrUpdate(ticket);
	}

	public static HashMap populateKitchenTicketProperties(KitchenTicket ticket, String deviceName, String virtualPrinterName, boolean ignorePagination) {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true);
		HashMap map = ReceiptPrintService.populateTicketProperties(ticket.getParentTicket(), printProperties, null, true, false);
		ReceiptPrintService.setPaginationProperty(map, deviceName, ignorePagination);
		if (containsVoidItemOnly(ticket)) {
			map.put(ReceiptPrintService.HEADER_LINE1, "*" + "" + "*");//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		String ticketType = ticket.getOrderType().toString();
		if (StringUtils.isNotEmpty(ticketType)) {
			ticketType = ticketType.replaceAll("_", " "); //$NON-NLS-1$ //$NON-NLS-2$
		}
		OrderType orderType = ticket.getOrderType();
		if (orderType.isDelivery() || orderType.isPickup()) {
			if (ticket.getParentTicket().isCustomerWillPickup()) {
				map.put(ReceiptParam.ORDER_TYPE.getParamName(), "*" + Messages.getString("PICKUP") + "*"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			else {
				map.put(ReceiptParam.ORDER_TYPE.getParamName(), "*" + Messages.getString("DELIVERY") + "*"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
		}
		else {
			map.put(ReceiptParam.ORDER_TYPE.getParamName(), "* " + ticketType + " *"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		map.put(PROP_PRINTER_NAME, "Printer Name : " + virtualPrinterName); //$NON-NLS-1$ //$NON-NLS-2$
		return map;
	}

	private static boolean containsVoidItemOnly(KitchenTicket ticket) {
		for (KitchenTicketItem item : ticket.getTicketItems()) {
			if (!item.isVoided()) {
				return false;
			}
		}
		return true;
	}

	public static void printQuitely(JasperPrint jasperPrint) throws JRException {
		JasperPrintService.printQuitely(jasperPrint);
	}

	private static void saveJournal(Ticket ticket) {
		String description = "Ticket is printed at %s. Total: %s"; //$NON-NLS-1$
		description = String.format(description, new Date(), ticket.getTotalAmount());
	}

	public static void printTestTicket(Ticket ticket) throws Exception {
		TicketPrintProperties printProperties = new TicketPrintProperties(null, false, true, true);
		printProperties.setPrintCookingInstructions(false);
		printTestTicket(ticket, populateTicketProperties(ticket, printProperties, null));
	}

	private static void printTestTicket(Ticket ticket, HashMap map) throws Exception {
		List<Printer> receiptPrinters = DataProvider.get().getPrinters().getReceiptPrinters();
		if (receiptPrinters == null) {
			return;
		}
		for (Printer receiptPrinter : receiptPrinters) {
			String deviceName = receiptPrinter.getDeviceName();
			if (deviceName != null) {
				logger.info(String.format("Printing ticket %s payment using printer %s. Total amount %s", ticket.getId(), deviceName, ticket.getTotalAmount())); //$NON-NLS-1$
				setPrintProperty(map, receiptPrinter);
				setPaginationProperty(map, deviceName);
				getReceiptPrintServiceProvider(receiptPrinter).printTicket(ticket, map, deviceName);
			}
		}
	}

	private static String findLastTxPaymentTypeName(Ticket ticket) {
		Set<PosTransaction> transactions = ticket.getTransactions();
		if (transactions == null || transactions.isEmpty()) {
			return null;
		}

		PosTransaction currentTrans = null;
		for (PosTransaction posTransaction : transactions) {
			if (currentTrans == null) {
				currentTrans = posTransaction;
				continue;
			}

			if (posTransaction.getTransactionTime().compareTo(currentTrans.getTransactionTime()) > 0) {
				currentTrans = posTransaction;
			}
		}

		if (currentTrans instanceof CustomPaymentTransaction) {
			return currentTrans.getCustomPaymentName();
		}

		return currentTrans.getPaymentTypeString();
	}
}