/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseDiscount;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "originalValue" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class Discount extends BaseDiscount implements TimedModel, PropertyContainer {
	private static final long serialVersionUID = 1L;

	private static final String LOYALTY_DISCOUNT_ID = "loyalty_discount"; //$NON-NLS-1$
	public static final String LOYALTY_DISCOUNT_NAME = "Loyalty"; //$NON-NLS-1$
	public static final String LOYALTY_DISCOUNT_BARCODE = "loyalty"; //$NON-NLS-1$

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Discount() {
	}

	/**
	 * Constructor for primary key
	 */
	public Discount(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public final static int FREE_AMOUNT = 0;
	public final static int FIXED_PER_CATEGORY = 1;
	public final static int FIXED_PER_ITEM = 2;
	public final static int FIXED_PER_ORDER = 3;
	public final static int PERCENTAGE_PER_CATEGORY = 4;
	public final static int PERCENTAGE_PER_ITEM = 5;
	public final static int PERCENTAGE_PER_ORDER = 6;

	public final static int DISCOUNT_TYPE_AMOUNT = 0;
	public final static int DISCOUNT_TYPE_PERCENTAGE = 1;
	public final static int DISCOUNT_TYPE_REPRICE = 2;
	public final static int DISCOUNT_TYPE_LOYALTY = 3;
	public final static int DISCOUNT_TYPE_ALT_PRICE = 4;

	public final static int QUALIFICATION_TYPE_ITEM = 0;
	public final static int QUALIFICATION_TYPE_ORDER = 1;

	public final static String[] COUPON_TYPE_NAMES = { "AMOUNT", "PERCENTAGE", "REPRICE", "LOYALTY" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

	public final static String[] COUPON_QUALIFICATION_NAMES = { "ITEM", "ORDER" }; //$NON-NLS-1$ //$NON-NLS-2$
	private List<String> menuItemIds;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private double originalValue;
	private transient JsonObject propertiesContainer;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String toString() {
		return super.getName();
	}

	public double calculateDiscount(ITicketItem ticketItem) {

		Double unitPrice = ticketItem.getUnitPrice();
		switch (getType()) {
			case DISCOUNT_TYPE_AMOUNT:
				return unitPrice;

			case DISCOUNT_TYPE_PERCENTAGE:
				if (unitPrice != null) {
					return (getValue() * unitPrice) / 100;
				}
				return 0;

			default:
				break;
		}

		return 0;
	}

	public double getAmountByType(double price) {
		switch (getType()) {
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return price - getValue();

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return price - (getValue() / 100);

			default:
				break;
		}
		return 0;
	}

	public List<String> getMenuItemIds() {
		return menuItemIds;
	}

	public void setMenuItemIds(List<String> menuItemIds) {
		this.menuItemIds = menuItemIds;
	}

	public static String getLoyaltyDiscountId() {
		return LOYALTY_DISCOUNT_ID;
	}

	public String getTypeName() {
		return Discount.COUPON_TYPE_NAMES[getType()];
	}

	public void setTypeName(String typeName) {
	}

	public String getQualificationName() {
		return Discount.COUPON_QUALIFICATION_NAMES[getQualificationType()];
	}

	public void setQualificationName(String qualificationName) {
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	@XmlTransient
	public double getOriginalValue() {
		return originalValue;
	}

	public void setOriginalValue(double originalValue) {
		this.originalValue = originalValue;
	}
}