package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ModifierGroupDAO extends BaseModifierGroupDAO {
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ModifierGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public List<ModifierGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	private String constructExceptionDetailsByMenuitem(ModifierGroup modifierGroup, List<String> foreignItemNames) {
		if (foreignItemNames != null && !foreignItemNames.isEmpty()) {
			StringBuilder builder = new StringBuilder(modifierGroup.getName() + " " + Messages.getString("DiscountDAO.3")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < foreignItemNames.size(); i++) {
				String message = (i + 1) + "." + " " + foreignItemNames.get(i); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	protected void delete(Object obj, Session session) {
		ModifierGroup modifierGroup = (ModifierGroup) obj;
		if (modifierGroup == null) {
			throw new PosException(Messages.getString("ModifierGroupDAO.0")); //$NON-NLS-1$
		}

		StringBuilder details = new StringBuilder();
		List<String> foreignItemNames = getMenuItemByModifierGroup(modifierGroup, session);
		if (foreignItemNames != null && !foreignItemNames.isEmpty()) {
			details.append(this.constructExceptionDetailsByMenuitem(modifierGroup, foreignItemNames));
			details.append(EMPTY_NEWLINE_STRING);
		}

		String message = Messages.getString("ModifierGroupDAO.1"); //$NON-NLS-1$
		String detailsMessage = details.toString();
		if (StringUtils.isNotBlank(detailsMessage)) {
			throw new PosException(message, detailsMessage);
		}

		modifierGroup.setDeleted(true);
		update(modifierGroup, session);
		//deleteModifierGroupFromModifierSpec(modifierGroup, session);

	}

	private List<String> getMenuItemByModifierGroup(ModifierGroup modifierGroup, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.setProjection(Projections.distinct(Projections.property(MenuItem.PROP_NAME)));
		criteria.createAlias("menuItemModiferSpecs", "specs"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("specs." + MenuItemModifierSpec.PROP_MODIFIER_GROUP_ID, modifierGroup.getId())); //$NON-NLS-1$
		addDeletedFilter(criteria, MenuItem.class);
		return criteria.list();
	}

	private void deleteModifierGroupFromModifierSpec(ModifierGroup modifierGroup, Session session) {

		Criteria criteria = session.createCriteria(MenuItemModifierSpec.class);
		criteria.add(Restrictions.eq(MenuItemModifierSpec.PROP_MODIFIER_GROUP_ID, modifierGroup.getId()));
		List<MenuItemModifierSpec> itemModifierSpecs = criteria.list();
		if (itemModifierSpecs == null || itemModifierSpecs.isEmpty()) {
			return;
		}
		for (MenuItemModifierSpec itemModifierSpec : itemModifierSpecs) {
			removeMenuItemModifierSpecFromMenuItem(itemModifierSpec, session);
			MenuItemModifierSpecDAO.getInstance().delete(itemModifierSpec, session);
		}
	}

	private void removeMenuItemModifierSpecFromMenuItem(MenuItemModifierSpec modifierSpec, Session session) {

		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.createAlias("menuItemModiferSpecs", "modifierSpecs"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("modifierSpecs." + MenuItem.PROP_ID, modifierSpec.getId())); //$NON-NLS-1$
		List<MenuItem> menuItems = criteria.list();
		if (menuItems == null || menuItems.isEmpty()) {
			return;
		}
		for (MenuItem menuItem : menuItems) {
			menuItem.getMenuItemModiferSpecs().remove(modifierSpec);
			MenuItemDAO.getInstance().update(menuItem, session);
		}

	}

	public List<ModifierGroup> findPizzaModifierGroups() {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(ModifierGroup.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ModifierGroup.PROP_PIZZA_MODIFIER_GROUP, Boolean.TRUE)); //$NON-NLS-1$

			return criteria.list();
		} finally {
			session.close();
		}
	}

	public List<ModifierGroup> findNormalModifierGroups() {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(ModifierGroup.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.isNull(ModifierGroup.PROP_PIZZA_MODIFIER_GROUP),
					Restrictions.eq(ModifierGroup.PROP_PIZZA_MODIFIER_GROUP, Boolean.FALSE))); //$NON-NLS-1$

			return criteria.list();
		} finally {
			session.close();
		}
	}

	/*
	 * 
	 * Note: This method is used to check ModifierGroup.PROP_PIZZA_MODIFIER_GROUP 
	 * property contains null value.
	 * 
	 */

	public boolean hasPizzaModifierGroup() {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(ModifierGroup.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(ModifierGroup.PROP_PIZZA_MODIFIER_GROUP, Boolean.TRUE));
			Number rowCount = (Number) criteria.uniqueResult();
			return (rowCount != null && rowCount.intValue() > 0);
		} finally {
			session.close();
		}
	}

	/*
	 * 
	 * Note: This method is used to save ModifierGroup.PROP_PIZZA_MODIFIER_GROUP 
	 * property true for pizza modifier group.
	 * 
	 */

	public void updateModifierGroupBooleanPropertyValue(boolean pizzaGroup) {
		if (hasPizzaModifierGroup()) {
			return;
		}
		Session session = null;
		Criteria criteria = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			criteria = session.createCriteria(ModifierGroup.class);
			criteria.createAlias("modifiers", "m");//$NON-NLS-1$ //$NON-NLS-2$

			if (pizzaGroup)
				criteria.add(Restrictions.eq("m.pizzaModifier", Boolean.TRUE)); //$NON-NLS-1$
			else
				criteria.add(Restrictions.ne("m.pizzaModifier", Boolean.TRUE)); //$NON-NLS-1$

			List list = criteria.list();
			if (list != null && list.size() > 0) {
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					ModifierGroup group = (ModifierGroup) iterator.next();
					group.setPizzaModifierGroup(pizzaGroup);
					saveOrUpdate(group, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
		} finally {
			session.close();
		}
	}

	public ModifierGroup findModifierGroupByName(String name) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ModifierGroup.PROP_NAME, name));
			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (ModifierGroup) list.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}
	}

	public List<ModifierGroup> findAllUnSyncModifierGroup() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(ModifierGroup.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(ModifierGroup.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void updateModifierGroupSync(List<String> ids, Boolean synced) {

		if (ids == null || ids.isEmpty())
			return;

		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		//@formatter:off
		Transaction tx=null;
		Session session = null;
		try {
			session = getSession();
			tx=session.beginTransaction();
		String hqlString = "update ModifierGroup set %s=:synced where %s in"+whereCondition; //$NON-NLS-1$
		hqlString = String.format(hqlString, ModifierGroup.PROP_CLOUD_SYNCED, ModifierGroup.PROP_ID);
		//@formatter:on
			Query query = session.createQuery(hqlString);
			query.setParameter("synced", synced); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	/**
	 * Search Modifier Group By Name (MatchMode.ANYWHERE)
	 */
	public void findModifierGroupByName(PaginationSupport paginationSupport, String itemName, boolean pizzaType) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(ModifierGroup.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			criteria.add(Restrictions.eq(ModifierGroup.PROP_PIZZA_MODIFIER_GROUP, pizzaType)); //$NON-NLS-1$

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(ModifierGroup.PROP_SORT_ORDER));

			paginationSupport.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateModifierGroups(List<ModifierGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				ModifierGroup item = (ModifierGroup) iterator.next();
				ModifierGroup existingItem = get(item.getId());

				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void initialize(ModifierGroup modifierGroup) {
		if (modifierGroup == null || modifierGroup.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(modifierGroup.getModifiers())) {
			return;
		}
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(modifierGroup);
			Hibernate.initialize(modifierGroup.getModifiers());
		} finally {
			closeSession(session);
		}
	}

	public void findModifierGroups(PaginationSupport model, String itemName) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(itemName))
				criteria.add(Restrictions.ilike(ModifierGroup.PROP_NAME, itemName, MatchMode.ANYWHERE));

			model.setNumRows(rowCount(criteria));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.asc(ModifierGroup.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(ModifierGroup.PROP_NAME).ignoreCase());
			model.setRows(criteria.list());
		}
	}
}
