/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.config.CardConfig;
import com.floreantpos.config.TerminalConfig;
import com.floreantpos.main.Application;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.PaymentStatusFilter;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.Shift;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.UserPermission;
import com.floreantpos.model.UserType;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.ext.InvMapKey;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.payment.PaymentPlugin;
import com.floreantpos.report.DeliverySummaryReportData;
import com.floreantpos.report.EndOfDayReportData;
import com.floreantpos.swing.PaginatedTableModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.NumericGlobalIdGenerator;
import com.orocube.common.util.TicketStatus;

public class TicketDAO extends BaseTicketDAO {
	private final static TicketDAO instance = new TicketDAO();

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public TicketDAO() {
	}

	@Override
	public Order getDefaultOrder() {
		return Order.desc(Ticket.PROP_CREATE_DATE);
	}

	@Override
	protected Serializable save(Object obj, Session session) {
		Ticket ticket = (Ticket) obj;

		performPreSaveOperations(ticket);
		Serializable save = super.save(obj, session);
		performPostSaveOperations(session, ticket);

		return save;
	}

	@Override
	protected void update(Object obj, Session session) {
		Ticket ticket = (Ticket) obj;

		performPreSaveOperations(ticket);
		super.update(obj, session);
		performPostSaveOperations(session, ticket);
	}

	@Override
	public synchronized void saveOrUpdate(Ticket ticket) {
		performPreSaveOperations(ticket);

		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(ticket, session);

			tx.commit();

		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	@Override
	public void saveOrUpdate(final Ticket ticket, Session session) {
		updateTime(ticket);
		saveOrUpdate(ticket, session, true);
	}

	public void saveOrUpdate(final Ticket ticket, Session session, boolean updateTableStatus) {
		boolean newTicket = StringUtils.isEmpty(ticket.getId()) ? true : isNewTicket(ticket.getId());
		if (newTicket) {
			if (StringUtils.isEmpty(ticket.getId())) {
				ticket.setId(NumericGlobalIdGenerator.generateGlobalId());
			}
			if (StringUtils.isEmpty(ticket.getShortId())) {
				ticket.setShortId(RandomStringUtils.randomNumeric(7));
			}
			if (ticket.getTokenNo() == 0) {
				ticket.setTokenNo(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_TOKEN, session));
			}
		}

		ticket.setActiveDate(StoreDAO.getServerTimestamp());
		ticket.updateGratuityInfo();

		if (ticket.isPaid() && ticket.getDueAmount() > 0) {
			ticket.setPaid(false);
		}
		if (newTicket) {
			save(ticket, session);
		}
		else {
			update(ticket, session);
		}

		if (updateTableStatus) {
			updateShopTableStatus(ticket, session, newTicket);
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		super.delete(obj, s);
	}

	public void saveOrUpdateTemporaryTicket(Ticket ticket) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			ticket.setActiveDate(StoreDAO.getServerTimestamp());
			boolean newTicket = StringUtils.isEmpty(ticket.getId()) ? true : isNewTicket(ticket.getId());
			if (newTicket) {
				if (StringUtils.isEmpty(ticket.getId())) {
					ticket.setId(NumericGlobalIdGenerator.generateGlobalId());
				}
				if (StringUtils.isEmpty(ticket.getShortId())) {
					ticket.setShortId(RandomStringUtils.randomNumeric(7));
				}
				if (ticket.getTokenNo() == 0) {
					ticket.setTokenNo(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_TOKEN, session));
				}
				updateTime(ticket);
				session.save(ticket);
			}
			else {
				updateTime(ticket);
				session.update(ticket);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	private void performPreSaveOperations(Ticket ticket) {
		//ticket create date
		if (ticket.getCreateDate() == null) {
			Calendar currentTime = DateUtil.getServerTimeCalendar();
			ticket.setCreateDate(currentTime.getTime());
			ticket.setCreationHour(currentTime.get(Calendar.HOUR_OF_DAY));
		}

		if (StringUtils.isEmpty(ticket.getStoreSessionId())) {
			StoreSession storeSession = DataProvider.get().getStoreSession();
			if (storeSession != null) {
				ticket.setStoreSessionId(storeSession.getId());
			}
		}
	}

	private void performPostSaveOperations(Session session, Ticket ticket) {
		updateStock(ticket, session);
		ticket.setShouldUpdateStock(false);
		ticket.clearDeletedItems();
	}

	public synchronized void saveOrUpdateSplitTickets(List<Ticket> tickets, List<Integer> tableNumbers) {
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Ticket ticket : tickets) {
				saveOrUpdate(ticket, session, false);
			}
			ShopTableStatusDAO.getInstance().addTicketsToShopTableStatus(tableNumbers, tickets, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public boolean isNewTicket(String id) {
		Session session = null;
		try {
			session = createNewSession();
			return isNewTicket(id, session);
		} finally {
			closeSession(session);
		}
	}

	private boolean isNewTicket(String id, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.setProjection(Projections.rowCount());
		criteria.add(Restrictions.eq(Ticket.PROP_ID, id));
		Number number = (Number) criteria.uniqueResult();
		return number == null || number.intValue() == 0;
	}

	private void clearVoidedItems(Ticket ticket) {
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			ticketItem.setVoidItem(null);
		}
	}

	public synchronized void saveKitchenPrintStatus(Ticket ticket, KitchenTicket kitchenTicket) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			List<KitchenTicketItem> ticketItems = kitchenTicket.getTicketItems();

			KitchenTicketItem parentItem = null;
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) iterator.next();

				if (kitchenTicketItem.isModifierItem() || kitchenTicketItem.isCookingInstruction()) {
					if (parentItem != null) {
						String nameDisplay = parentItem.getMenuItemName().concat("\n" + kitchenTicketItem.getMenuItemName());
						parentItem.setMenuItemName(nameDisplay);
						iterator.remove();
					}
				}
				else {
					parentItem = kitchenTicketItem;
				}
			}

			//Check voided items
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) iterator.next();
				if (kitchenTicketItem.isVoided()) {
					String ticketItemId = kitchenTicketItem.getVoidedItemId();
					KitchenTicketItemDAO.getInstance().markVoided(ticketItemId, kitchenTicketItem.getQuantity(), session);
					iterator.remove();
				}
			}

			//if kitchen ticket contained voided item only, then do not save the ticket
			if (ticketItems.size() > 0) {
				KitchenTicketDAO.getInstance().saveOrUpdate(kitchenTicket, session);
			}

			ticket.clearDeletedItems();
			if (ticket.getTokenNo() == 0) {
				ticket.setTokenNo(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_TOKEN, session));
			}
			saveOrUpdate(ticket, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void updateShopTableStatus(Ticket ticket, Session session, boolean isNewTicket) {
		List<Integer> tableNumbers = ticket.getTableNumbers();
		if (tableNumbers == null || tableNumbers.isEmpty()) {
			return;
		}
		if (ticket.isClosed()) {
			ShopTableStatusDAO.getInstance().removeTicketFromShopTableStatus(ticket, session);
		}
		else if (isNewTicket || ticket.isShouldUpdateTableStatus()) {
			ShopTableDAO.getInstance().occupyTables(ticket, session);
			ticket.setShouldUpdateTableStatus(false);
		}
	}

	public void voidTicket(Ticket ticket) throws Exception {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			ticket.setShouldUpdateStock(true);
			ticket.setDiscounts(null);
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				if (ticketItem == null || ticketItem.isTreatAsSeat()) {
					continue;
				}
				boolean inventoryAdjusted = ticketItem.getInventoryAdjustQty().doubleValue() >= ticketItem.getQuantity().doubleValue();
				if (!ticketItem.isVoided()) {
					ticket.voidItem(ticketItem, ticket.getVoidReason(), ticket.isWasted(), Math.abs(ticketItem.getQuantity()));
					if (!inventoryAdjusted) {
						ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
					}
				}
				else if (ticketItem.isReturned() && !ticketItem.isReturnedItemVoided()) {
					ticket.voidReturnedItem(ticketItem, ticket.getVoidReason(), ticket.isWasted());
					if (!inventoryAdjusted) {
						ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
					}
				}
			}
			ticket.calculatePrice();

			tx = session.beginTransaction();

			List<KitchenTicket> kitchenTickets = KitchenTicketDAO.getInstance().findByParentId(ticket.getId());
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			if (kitchenTickets != null) {
				for (KitchenTicket kitchenTicket : kitchenTickets) {
					kitchenTicket.setCreateDate(serverTimestamp);
					kitchenTicket.setVoided(true);
					session.saveOrUpdate(kitchenTicket);
				}
			}
			ticket.setVoided(true);
			ticket.setClosed(true);
			ticket.setClosingDate(StoreDAO.getServerTimestamp());
			//			ticket.setLastUpdateTime(new Date());
			TicketDAO.getInstance().saveOrUpdate(ticket, session);

			tx.commit();

		} catch (Exception x) {
			try {
				tx.rollback();
			} catch (Exception e) {
			}
			throw x;
		} finally {
			ticket.setShouldUpdateStock(false);
		}
	}

	@Deprecated
	private void populateVoidItems(Ticket ticket) {
		Map<String, Double> voidedItemQuantityMap = new HashMap();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if (!ticketItem.isVoided())
				continue;
			Double previousValue = voidedItemQuantityMap.get(ticketItem.getMenuItemId());
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double voidedQuantity = 0;
			voidedQuantity = Math.abs(ticketItem.getQuantity());
			voidedQuantity += previousValue;
			if (voidedQuantity == 0) {
				continue;
			}
			voidedItemQuantityMap.put(ticketItem.getMenuItemId(), voidedQuantity);
		}

		Map<TicketItem, VoidItem> toBeVoidedItemsMap = new HashMap<>();

		for (Iterator iterator = ticket.getTicketItems().iterator(); iterator.hasNext();) {
			TicketItem ticketItem = (TicketItem) iterator.next();
			if (ticketItem.getId() == null || ticketItem.isVoided() || ticketItem.getVoidItem() != null || ticketItem.isTreatAsSeat()) {
				continue;
			}

			Double voidedQuantity = voidedItemQuantityMap.get(ticketItem.getMenuItemId());
			double toBeVoidQuantity = ticketItem.getQuantity();
			if (voidedQuantity != null && voidedQuantity > 0) {
				if (voidedQuantity >= toBeVoidQuantity) {
					voidedItemQuantityMap.put(ticketItem.getMenuItemId(), voidedQuantity - toBeVoidQuantity);
					continue;
				}
				else {
					toBeVoidQuantity -= voidedQuantity;
				}
			}
			toBeVoidedItemsMap.put(ticketItem, new VoidItem(ticket.getVoidReason(), ticket.isWasted(), toBeVoidQuantity));
		}
		Set<TicketItem> keys = toBeVoidedItemsMap.keySet();
		for (TicketItem ticketItem : keys) {
			if (ticketItem.isTreatAsSeat()) {
				continue;
			}
			VoidItem voidItem = toBeVoidedItemsMap.get(ticketItem);
			ticket.voidItem(ticketItem, voidItem.getVoidReason(), voidItem.isItemWasted(), voidItem.getQuantity());
		}
	}

	public void loadFullTicket(Ticket ticket) {
		if (ticket.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(ticket.getTicketItems()) && Hibernate.isInitialized(ticket.getTransactions())) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(ticket);

			initialize(ticket, session);
		} finally {
			closeSession(session);
		}
	}

	public void initialize(Ticket ticket, Session session) {
		Hibernate.initialize(ticket.getTicketItems());
		Hibernate.initialize(ticket.getTransactions());
	}

	public Ticket loadFullTicket(String id, String outletId) {
		try (Session session = createNewSession()) {
			Ticket ticket = (Ticket) get(getReferenceClass(), new Ticket(id, outletId), session);
			if (ticket == null)
				return null;

			Hibernate.initialize(ticket.getTicketItems());
			Hibernate.initialize(ticket.getTransactions());

			return ticket;
		}
	}

	public Ticket loadCouponsAndTransactions(String ticketId, String outletId) {
		try (Session session = createNewSession()) {

			Ticket ticket = (Ticket) TicketDAO.getInstance().get(ticketId, outletId, session);
			if (ticket == null) {
				return null;
			}
			Hibernate.initialize(ticket.getTransactions());

			return ticket;
		}
	}

	public List<Ticket> findOpenTickets() {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForGroupSettle(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_DUE_AMOUNT, Double.valueOf(0)));
			criteria.add(Restrictions.isEmpty(Ticket.PROP_TRANSACTIONS));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				Ticket ticket = (Ticket) iterator.next();
				OrderType orderType = ticket.getOrderType();
				if (orderType != null && orderType.isBarTab()) {
					iterator.remove();
				}
			}
			return list;
		}
	}

	public List<Ticket> findMergableTickets(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_DUE_AMOUNT, Double.valueOf(0)));
			criteria.add(Restrictions.isEmpty(Ticket.PROP_TRANSACTIONS));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		}
	}

	public List<Ticket> findOpenTickets(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		}
	}

	public List<Ticket> findOpenTickets(Integer customerId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTickets(Terminal terminal, UserType userType) {
		return findOpenTickets(terminal, userType, null, null, null);
	}

	public List<Ticket> findOpenTickets(Terminal terminal, UserType userType, String orderTypeId, Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, startDate, endDate));
			}
			if (userType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_TYPE_ID, userType.getId()));
			}
			if (terminal != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (StringUtils.isNotBlank(orderTypeId) && orderTypeId != POSConstants.ALL) {
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderTypeId));
			}
			criteria.addOrder(getDefaultOrder());
			return criteria.list();
		}
	}

	@Deprecated
	public int getNumTickets() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			updateCriteriaFilters(criteria);

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadTickets(PaginatedTableModel tableModel) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			updateCriteriaFilters(criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.addOrder(getDefaultOrder());
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			tableModel.setRows(criteria.list());
		}
	}

	public void loadDeliveryTickets(PaginatedTableModel tableModel, String customerId, Date startDate, Date endDate, boolean onlineOrderOnly,
			boolean isPickupOnly, boolean isDeliveryOnly, boolean filterUnassigned) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (StringUtils.isNotEmpty(customerId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			}
			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.between(Ticket.PROP_DELIVERY_DATE, startDate, endDate));
			}
			if (isPickupOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.TRUE));
			}
			if (isDeliveryOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.FALSE));
			}
			if (filterUnassigned) {
				criteria.add(Restrictions.isNull(Ticket.PROP_ASSIGNED_DRIVER_ID));
			}
			List<String> orderTypeIds = OrderTypeDAO.getInstance().getHomeDeliveryOrderTypeIds(session);
			if (orderTypeIds != null && !orderTypeIds.isEmpty()) {
				criteria.add(Restrictions.in(Ticket.PROP_ORDER_TYPE_ID, orderTypeIds));
				tableModel.setNumRows(this.rowCount(criteria));
				criteria.setFirstResult(tableModel.getCurrentRowIndex());
				criteria.setMaxResults(tableModel.getPageSize());
				criteria.addOrder(this.getDefaultOrder());
				List<Ticket> tickets = criteria.list();
				if (onlineOrderOnly) {
					for (Iterator iterator = tickets.iterator(); iterator.hasNext();) {
						Ticket ticket = (Ticket) iterator.next();
						if (!ticket.isSourceOnline()) {
							iterator.remove();
						}
					}
				}
				tableModel.setRows(tickets);
			}
			else {
				if (tableModel.getRows() != null) {
					tableModel.getRows().clear();
				}
				tableModel.setNumRows(0);
				return;
			}
		}
	}

	@SuppressWarnings("unchecked")
	public List<Ticket> findCustomerTickets(String customerId, PaginationSupport tableModel) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(getDefaultOrder());

			return criteria.list();
		}
	}

	public List<Ticket> findNextCustomerTickets(Integer customerId, PaginatedTableModel tableModel, String filter) {
		Session session = null;
		Criteria criteria = null;
		PaymentStatusFilter statusFilter = PaymentStatusFilter.fromString(filter);
		try {
			int nextIndex = tableModel.getNextRowIndex();

			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			if (statusFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}

			criteria.setFirstResult(nextIndex);
			criteria.setMaxResults(tableModel.getPageSize());

			List ticketList = criteria.list();

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				tableModel.setNumRows(rowCount.intValue());

			}

			tableModel.setCurrentRowIndex(nextIndex);

			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findPreviousCustomerTickets(Integer customerId, PaginatedTableModel tableModel, String filter) {
		Session session = null;
		Criteria criteria = null;
		PaymentStatusFilter statusFilter = PaymentStatusFilter.fromString(filter);
		try {

			int previousIndex = tableModel.getPreviousRowIndex();

			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			if (statusFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}

			criteria.setFirstResult(previousIndex);
			criteria.setMaxResults(tableModel.getPageSize());

			List ticketList = criteria.list();

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				tableModel.setNumRows(rowCount.intValue());

			}

			tableModel.setCurrentRowIndex(previousIndex);

			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketByCustomer(Integer customerId) {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			List ticketList = criteria.list();
			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTickets(PaymentStatusFilter psFilter, OrderType orderType) {
		return findTicketsForUser(psFilter, orderType, null);
	}

	public List<Ticket> findTicketsForUser(PaymentStatusFilter psFilter, OrderType orderType, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			if (psFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			else if (psFilter == PaymentStatusFilter.PAID) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.TRUE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			else if (psFilter == PaymentStatusFilter.CLOSED) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));

				Calendar currentTime = Calendar.getInstance();
				currentTime.add(Calendar.HOUR_OF_DAY, -24);

				criteria.add(Restrictions.ge(Ticket.PROP_CLOSING_DATE, currentTime.getTime()));
			}

			if (orderType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
			}

			if (user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public boolean hasOpenTickets(User user) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (user != null)
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			Number result = (Number) criteria.uniqueResult();
			if (result != null) {
				return result.intValue() > 0;
			}
			return false;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTicketsForUser(User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			//criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			//criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user == null ? null : user.getId()));
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsByOutletId(Date lastUpdateTime, String outletId, boolean includeClosedTickets) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			if (!includeClosedTickets) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			criteria.add(Restrictions.ne(Ticket.PROP_STATUS, TicketStatus.Pending.name()));
			if (lastUpdateTime != null)
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findClosedTicketsByOpenTicketIds(Date lastUpdateTime, List<String> openTicketIds) {
		if (openTicketIds == null || openTicketIds.isEmpty()) {
			return null;
		}
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.in(Ticket.PROP_ID, openTicketIds));
			if (lastUpdateTime != null)
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTickets(Date startDate, Date endDate) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findClosedTickets(Date startDate, Date endDate, String outletId) {
		return findClosedTickets(startDate, endDate, "", "");
	}

	public List<Ticket> findClosedTickets(Date startDate, Date endDate, String outletId, String ticketId) {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			if (StringUtils.isNotBlank(ticketId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_ID, ticketId));
				//criteria.add(Restrictions.ilike(Ticket.PROP_ID, ticketId, MatchMode.ANYWHERE));
			}
			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			}
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public void closeOrder(Ticket ticket) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(ticket);

			User driver = ticket.getAssignedDriver();
			if (driver != null) {
				driver.setAvailableForDelivery(true);
				UserDAO.getInstance().saveOrUpdate(driver);
			}

			ShopTableDAO.getInstance().releaseTables(ticket);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(TicketDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTickets(Date startDate, Date endDate, boolean closed, Terminal terminal) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

			if (terminal != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForLaborHour(Date startDate, Date endDate, int hour, Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_ACTIVE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_ACTIVE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_CREATION_HOUR, Integer.valueOf(hour)));
			//criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			//criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForShift(Date startDate, Date endDate, Shift shit, Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_SHIFT_ID, shit == null ? null : shit.getId()));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public static TicketDAO getInstance() {
		return instance;
	}

	private void updateCriteriaFilters(Criteria criteria) {
		User user = Application.getCurrentUser();
		PaymentStatusFilter paymentStatusFilter = TerminalConfig.getPaymentStatusFilter();

		//orderTypeFilter is either order type id, or 'ALL'
		String orderTypeFilter = TerminalConfig.getOrderTypeFilter();
		boolean filterByMyTicket = TerminalConfig.isFilterByOwner();
		OrderType orderType = null;
		//:FIXME
		//if (!POSConstants.ALL.equals(orderTypeFilter)) {
		//	orderType = DataProvider.get().getOrderType(orderTypeFilter);
		//}
		if (paymentStatusFilter == PaymentStatusFilter.OPEN) {
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (!user.canViewAllOpenTickets() || filterByMyTicket) {
				criteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, user.getRoleIds()));
			}
		}
		else if (paymentStatusFilter == PaymentStatusFilter.CLOSED) {
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			//			StoreSessionControl currentStoreOperation = StoreUtil.getCurrentStoreOperation();
			//			if (currentStoreOperation != null && currentStoreOperation.getCurrentData() != null) {
			//				criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, currentStoreOperation.getCurrentData().getId()));
			//			}
			if (!user.canViewAllCloseTickets() || filterByMyTicket) {
				criteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, user.getRoleIds()));
			}
		}

		if (!orderTypeFilter.equals(POSConstants.ALL)) {
			if (orderType != null)
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
		}

	}

	public void deleteTickets(List<Ticket> tickets) {
		deleteTickets(tickets, false);
	}

	public void deleteTickets(List<Ticket> tickets, boolean releaseTables) {
		deleteTickets(tickets, releaseTables, false);
	}

	public void deleteTickets(List<Ticket> tickets, boolean releaseTables, boolean updateStock) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			deleteTickets(session, tickets, releaseTables, updateStock);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(TicketDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void deleteTickets(Session session, List<Ticket> tickets, boolean releaseTables) throws Exception {
		deleteTickets(session, tickets, releaseTables, false);
	}

	public void deleteTickets(Session session, List<Ticket> tickets, boolean releaseTables, boolean updateStock) throws Exception {
		for (Ticket ticket : tickets) {
			ticket = (Ticket) session.merge(ticket);

			if (!ticket.isClosed()) {
				List<TicketItem> removedTicketItems = new ArrayList<>();
				removedTicketItems.addAll(ticket.getTicketItems());
				for (Iterator iterator = removedTicketItems.iterator(); iterator.hasNext();) {
					TicketItem ticketItem = (TicketItem) iterator.next();
					ticket.addDeletedItems(ticketItem);

				}
				//TODO: do we really need to update stock here?

				//this method will be updated stock for normally void ticket.
				if (updateStock && ticket.getPaidAmount() > 0)
					TicketDAO.getInstance().updateStock(ticket, session);
			}
			super.delete(ticket, session);
			if (releaseTables) {
				ShopTableDAO.getInstance().freeTicketTables(ticket, session);
			}
			//KitchenTicketDAO.getInstance().deleteKitchenTicket(ticket.getId());
		}
	}

	//	private void adjustInventory(Ticket ticket, Session session) {
	//		List<TicketItem> items = ticket.getTicketItems();
	//		Map<String, Double> itemStockCountMap = getStockCountAdjustedMap(items);
	//
	//		if (ticket.getId() != null) {
	//			List<Object> deletedItems = ticket.getDeletedItems();
	//			if (deletedItems != null) {
	//				for (Object item : deletedItems) {
	//					if (item instanceof TicketItem) {
	//						TicketItem ticketItem = (TicketItem) item;
	//						if (ticketItem.isPrintedToKitchen())
	//							continue;
	//						MenuItemDAO.getInstance().updateStockQuantity(ticketItem.getMenuItemId(), ticketItem.getQuantity(), true, false, session);
	//					}
	//				}
	//			}
	//		}
	//		if (itemStockCountMap.isEmpty()) {
	//			return;
	//		}
	//
	//		for (String menuItemId : itemStockCountMap.keySet()) {
	//			Double usedAmount = itemStockCountMap.get(menuItemId);
	//			Query query = null;
	//			query = session.createQuery(String.format("update MenuItem set availableUnit = (availableUnit - %s) where id = '%s'", usedAmount, menuItemId));
	//			if (query == null) {
	//				return;
	//			}
	//			query.executeUpdate();
	//		}
	//	}

	//	private Map<String, Double> getStockCountAdjustedMap(List<TicketItem> items) {
	//		HashMap<String, Double> itemStockCountMap = new LinkedHashMap<String, Double>();
	//		for (TicketItem ticketItem : items) {
	//			MenuItem menuItem = ticketItem.getMenuItem();
	//			if (menuItem == null) {
	//				continue;
	//			}
	//			if (!ticketItem.isInventoryItem()) {
	//				continue;
	//			}
	//			if (ticketItem.isInventoryAdjusted()) {
	//				continue;
	//			}
	//
	//			Double previousValue = itemStockCountMap.get(ticketItem.getMenuItemId());
	//
	//			if (previousValue == null) {
	//				previousValue = 0.0;
	//			}
	//			itemStockCountMap.put(ticketItem.getMenuItemId(), ticketItem.getQuantity() + previousValue);
	//
	//			for (RecepieItem recepieItem : menuItem.getRecepieItems()) {
	//				MenuItem inventoryItem = recepieItem.getInventoryItem();
	//				Double itemValue = itemStockCountMap.get(inventoryItem.getId());
	//				Double percentage = recepieItem.getPercentage() / 100.0;
	//				if (itemValue == null)
	//					itemValue = 0.0;
	//
	//				itemValue += (ticketItem.getQuantity() * percentage);
	//
	//				itemStockCountMap.put(inventoryItem.getId(), itemValue);
	//			}
	//			ticketItem.setInventoryAdjusted(true);
	//		}
	//		return itemStockCountMap;
	//	}

	protected void updateStock(Ticket ticket, Session session) {
		try {
			if (shouldUpdateStock(ticket) == false) {
				return;
			}

			InventoryLocation defaultOutInventoryLocation = null;
			InventoryLocation defaultInInventoryLocation = null;
			defaultOutInventoryLocation = DataProvider.get().getDefaultOutLocation();
			HashMap<InvMapKey, Double> itemMap = buildItemMapForInventoryAdjustment(ticket);
			HashMap<InvMapKey, Double> voidedItemsMap = buildItemMapForVoidItems(ticket, session);
			HashMap<InvMapKey, Double> returnedVoidedItemsMap = buildItemMapForReturnedVoidItems(ticket, session);
			adjustInventory(ticket, itemMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_TICKET_SALES, defaultOutInventoryLocation, session);
			if (voidedItemsMap != null && voidedItemsMap.size() > 0) {
				defaultInInventoryLocation = DataProvider.get().getDefaultInLocation();
				adjustInventory(ticket, voidedItemsMap, InventoryTransactionType.IN, InventoryTransaction.REASON_VOID, defaultInInventoryLocation, session);
			}
			if (returnedVoidedItemsMap != null && returnedVoidedItemsMap.size() > 0) {
				adjustInventory(ticket, returnedVoidedItemsMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_VOID, defaultOutInventoryLocation,
						session);
			}
			clearVoidedItems(ticket);
		} catch (Exception e) {
			PosLog.error(getClass(), "Failed to update stock balance for ticket: " + ticket.getId(), e);
		}
	}

	private void adjustInventory(Ticket ticket, HashMap<InvMapKey, Double> itemMap, InventoryTransactionType transactionType, String reason,
			InventoryLocation location, Session session) throws Exception {
		Store store = DataProvider.get().getStore();
		boolean isUpdateOnHandBlncForSale = store.isUpdateOnHandBlncForSale();
		boolean isUpdateAvailBlncForSale = store.isUpdateAvlBlncForSale();
		for (InvMapKey mapKey : itemMap.keySet()) {
			Double unitQuantity = itemMap.get(mapKey);
			MenuItem menuItem = MenuItemDAO.getInstance().getMenuItemWithFields(session, mapKey.getMenuItemId(), MenuItem.PROP_NAME, MenuItem.PROP_PRICE,
					MenuItem.PROP_SKU, MenuItem.PROP_BARCODE, MenuItem.PROP_UNIT_ID, MenuItem.PROP_COST, MenuItem.PROP_AVERAGE_UNIT_PURCHASE_PRICE,
					MenuItem.PROP_AVG_COST, MenuItem.PROP_DEFAULT_RECIPE_ID, MenuItem.PROP_INVENTORY_ITEM);
			if (menuItem == null) {
				continue;
			}
			if (menuItem.isInventoryItem()) {
				InventoryTransaction outTrans = new InventoryTransaction();
				if (mapKey.isItemReturned() && InventoryTransaction.REASON_VOID.equals(reason)) {
					reason = InventoryTransaction.REASON_RETURN;
				}
				outTrans.setReason(reason);
				outTrans.setTransactionDate(new Date());
				outTrans.setMenuItem(menuItem);
				outTrans.setType(transactionType.getType());
				outTrans.setTicketId(ticket.getId());
				outTrans.setUser(ticket.getOwner());
				IUnit transactionUnit = (IUnit) DataProvider.get().getUnitByCode(mapKey.getUnitCode());
				if (transactionUnit instanceof PackagingUnit) {
					session.refresh(menuItem);
					if (!Hibernate.isInitialized(menuItem.getStockUnits())) {
						Hibernate.initialize(menuItem.getStockUnits());
					}
				}
				double baseUnitQuantity = menuItem.getBaseUnitQuantity(mapKey.getUnitCode());
				outTrans.setUnitPrice(menuItem.getPrice() * baseUnitQuantity);
				outTrans.setQuantity(unitQuantity);
				outTrans.setUnit(mapKey.getUnitCode());
				outTrans.setUnitCost(menuItem.getCost());
				if (transactionType == InventoryTransactionType.IN) {
					outTrans.setToInventoryLocation(location);
				}
				else {
					outTrans.setFromInventoryLocation(location);
				}
				outTrans.calculateTotal();
				InventoryTransactionDAO.getInstance().adjustInventoryStock(outTrans, session, isUpdateAvailBlncForSale, isUpdateOnHandBlncForSale);
			}
			//adjust inventory of recipe items
			if (StringUtils.isNotEmpty(menuItem.getDefaultRecipeId())) {
				Recepie recepie = RecepieDAO.getInstance().get(menuItem.getDefaultRecipeId(), session);
				RecepieDAO.getInstance().adjustRecipeItemsFromInventory(unitQuantity, Arrays.asList(recepie), session);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForInventoryAdjustment(Ticket ticket) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		this.populateItemToMap(ticketItems, itemMap);
		return itemMap;
	}

	private void populateItemToMap(List<TicketItem> ticketItems, HashMap<InvMapKey, Double> itemMap) {
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem == null) {
				continue;
			}
			if (ticketItem.getMenuItemId() == null || ticketItem.isVoided() || ticketItem.isItemReturned() || ticketItem.isInventoryAdjusted()) {
				continue;
			}
			InvMapKey key = new InvMapKey(ticketItem.getMenuItemId(), ticketItem.getUnitName());

			Double previousValue = itemMap.get(key);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = ticketItem.getQuantity();
			toBeAdjustQty -= ticketItem.getInventoryAdjustQty();
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty <= 0) {
				continue;
			}
			itemMap.put(key, toBeAdjustQty);
			ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
			if (ticketItem.isComboItem() && ticketItem.getComboItems() != null) {
				this.populateItemToMap(ticketItem.getComboItems(), itemMap);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForVoidItems(Ticket ticket, Session session) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> voidItemMap = new HashMap<>();
		this.populateVoidItemToMap(ticket, session, ticketItems, voidItemMap);
		return voidItemMap;
	}

	private void populateVoidItemToMap(Ticket ticket, Session session, List<TicketItem> ticketItems, HashMap<InvMapKey, Double> voidItemMap) {
		for (TicketItem voidTicketItem : ticketItems) {
			VoidItem voidItem = voidTicketItem.getVoidItem();
			String menuItemId = voidTicketItem.getMenuItemId();

			if (voidItem == null || (!voidTicketItem.isVoided() && !voidTicketItem.isItemReturned()) || voidTicketItem.isReturnedItemVoided()) {
				continue;
			}

			InvMapKey mapKey = new InvMapKey(menuItemId, voidTicketItem.getUnitName());
			mapKey.setItemReturned(voidTicketItem.isItemReturned());

			Double previousValue = voidItemMap.get(mapKey);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double toBeAdjustQty = 0;
			toBeAdjustQty = voidItem.getQuantity();
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty == 0) {
				continue;
			}

			if (!voidItem.isItemWasted() && !voidTicketItem.isInventoryAdjusted()) {
				voidItemMap.put(mapKey, toBeAdjustQty);
				voidTicketItem.setInventoryAdjustQty(voidTicketItem.getQuantity());
			}
			Boolean isPrintedToKitchen = voidTicketItem.isPrintedToKitchen();
			VoidItemDAO.getInstance().saveAndSentToKitchenIfNeeded(voidItem, isPrintedToKitchen, ticket, session);
			voidTicketItem.addProperty(TicketItem.JSON_PROP_VOID_ID, voidItem.getId());

			List<VoidItem> voidedModifiers = voidItem.getVoidedModifiers();
			String propModifierVoidIds = ""; //$NON-NLS-1$
			if (voidedModifiers != null && voidedModifiers.size() > 0) {
				for (Iterator<VoidItem> iterator = voidedModifiers.iterator(); iterator.hasNext();) {
					VoidItem voidedModifier = (VoidItem) iterator.next();
					VoidItemDAO.getInstance().saveAndSentToKitchenIfNeeded(voidedModifier, isPrintedToKitchen, ticket, session);
					propModifierVoidIds += voidedModifier.getId();
					if (iterator.hasNext()) {
						propModifierVoidIds += ","; //$NON-NLS-1$
					}
				}
			}
			if (voidTicketItem.isReturned() && StringUtils.isNotBlank(propModifierVoidIds)) {
				voidTicketItem.addProperty(TicketItem.JSON_PROP_MODIIFIER_VOID_IDS, propModifierVoidIds);
			}
			if (voidTicketItem.isComboItem()) {
				this.populateVoidItemToMap(ticket, session, voidTicketItem.getComboItems(), voidItemMap);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForReturnedVoidItems(Ticket ticket, Session session) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> voidItemMap = new HashMap<>();
		this.populateReturnedVoidItemToMap(ticket, session, ticketItems, voidItemMap);
		return voidItemMap;
	}

	private void populateReturnedVoidItemToMap(Ticket ticket, Session session, List<TicketItem> ticketItems, HashMap<InvMapKey, Double> voidItemMap) {
		for (TicketItem voidTicketItem : ticketItems) {
			if (!voidTicketItem.isReturnedItemVoided()) {
				continue;
			}
			String voidId = voidTicketItem.getProperty(TicketItem.JSON_PROP_VOID_ID);
			if (StringUtils.isBlank(voidId)) {
				continue;
			}
			VoidItem voidItem = VoidItemDAO.getInstance().get(voidId, ticket.getOutletId());
			if (voidItem == null) {
				continue;
			}
			String menuItemId = voidTicketItem.getMenuItemId();
			InvMapKey mapKey = new InvMapKey(menuItemId, voidTicketItem.getUnitName());
			mapKey.setItemReturned(voidTicketItem.isItemReturned());

			Double previousValue = voidItemMap.get(mapKey);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double toBeAdjustQty = 0;
			toBeAdjustQty = Math.abs(voidItem.getQuantity());
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty == 0) {
				continue;
			}
			if (!voidItem.isItemWasted() && !voidTicketItem.isInventoryAdjusted()) {
				voidItemMap.put(mapKey, toBeAdjustQty);
				voidTicketItem.setInventoryAdjustQty(voidTicketItem.getQuantity());
			}
			Boolean isPrintedToKitchen = voidTicketItem.isPrintedToKitchen();
			VoidItemDAO.getInstance().deleteAndSentToKitchenIfNeeded(voidItem, isPrintedToKitchen, ticket, session);
			if (voidTicketItem.isHasModifiers()) {
				String propModifierVoidIds = voidTicketItem.getProperty(TicketItem.JSON_PROP_MODIIFIER_VOID_IDS);
				if (StringUtils.isNotBlank(propModifierVoidIds)) {
					String[] modifierVoidIds = propModifierVoidIds.split(","); //$NON-NLS-1$
					if (modifierVoidIds.length > 0) {
						for (String modifierVoidId : modifierVoidIds) {
							if (StringUtils.isBlank(modifierVoidId)) {
								continue;
							}
							VoidItem existingVoid = VoidItemDAO.getInstance().get(modifierVoidId, ticket.getOutletId());
							if (existingVoid == null) {
								continue;
							}
							VoidItemDAO.getInstance().delete(existingVoid, session);
						}
					}
				}
			}
			if (voidTicketItem.isComboItem()) {
				this.populateReturnedVoidItemToMap(ticket, session, voidTicketItem.getComboItems(), voidItemMap);
			}
		}
	}

	public int getNumTickets(Date start, Date end) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			//updateCriteriaFilters(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (start != null)
				criteria.add(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, start));

			if (end != null)
				criteria.add(Restrictions.le(Ticket.PROP_DELIVERY_DATE, end));

			criteria.add(Restrictions.isNotNull(Ticket.PROP_DELIVERY_DATE));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadTickets(PaginatedTableModel tableModel, Date beginingDeliveryDate, Date endingDeliveryDate) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(getDefaultOrder());
			//updateCriteriaFilters(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (beginingDeliveryDate != null)
				criteria.add(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, beginingDeliveryDate));

			if (endingDeliveryDate != null)
				criteria.add(Restrictions.le(Ticket.PROP_DELIVERY_DATE, endingDeliveryDate));

			//			criteria.add(Restrictions.isNotNull(Ticket.PROP_DELIVERY_DATE));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
			return;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> getTicketsWithSpecificFields(String... fields) {
		return getTicketsWithSpecificFields(true, fields);
	}

	public List<Ticket> getTicketsWithSpecificFields(boolean isOnlyOpenTickets, String... fields) {
		Session session = null;
		Criteria criteria = null;
		User currentUser = Application.getCurrentUser();
		boolean filterUser = !currentUser.isAdministrator() || !currentUser.isManager();
		try {
			session = createNewSession();
			criteria = session.createCriteria(Ticket.class);
			if (isOnlyOpenTickets) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			if (filterUser) {
				//criteria.createAlias(Ticket.PROP_OWNER, "u");
				//criteria.add(Restrictions.eq("u.id", currentUser.getId()));
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, currentUser.getId()));
			}

			ProjectionList projectionList = Projections.projectionList();
			for (String field : fields) {
				projectionList.add(Projections.property(field), field);
			}

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(Ticket.class));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public boolean hasTicketByOnlineOrderId(String onlineOrderId) {
		Session session = null;
		try {
			session = createNewSession();
			String sql = "select * from TICKET_PROPERTIES where PROPERTY_VALUE='%s' and PROPERTY_NAME='%s'"; //$NON-NLS-1$
			sql = String.format(sql, onlineOrderId, Ticket.PROPERTY_ONLINE_ID);
			SQLQuery sqlQuery = session.createSQLQuery(sql);
			List list = sqlQuery.list();
			return list.size() > 0;
		} finally {
			closeSession(session);
		}
	}

	public boolean hasTicketByReservationId(String reservationId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_ID, reservationId));
			criteria.add(Restrictions.eq(Ticket.PROP_TYPE, TicketType.RESERVATION.getTypeNo()));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue() > 0;

			}
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), e);
		} finally {
			closeSession(session);
		}
		return false;
	}

	//	public List<Ticket> findOnlineTickets(PaginatedTableModel tableModel) {
	//		Session session = null;
	//
	//		try {
	//			session = createNewSession();
	//			String sql = "select TICKET_ID from TICKET_PROPERTIES where PROPERTY_VALUE='online' and PROPERTY_NAME='source'"; //$NON-NLS-1$
	//			SQLQuery sqlQuery = session.createSQLQuery(sql);
	//			List<String> list = sqlQuery.list();
	//			List<Ticket> ticketList = new ArrayList<Ticket>();
	//			for (String ticketId : list) {
	//				if (ticketId == null) {
	//					continue;
	//				}
	//				Ticket ticket = get(ticketId);
	//				ticketList.add(ticket);
	//			}
	//
	//			return ticketList;
	//
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public void findTicketForDeliveryDispath(PaginatedTableModel model, OrderType orderType, String customerId, Date startDate, Date endDate,
			boolean onlineOrderOnly, boolean isPickupOnly, boolean isDeliveryOnly, boolean filterUnassigned) {
		User user = Application.getCurrentUser();
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(getDefaultOrder());
			criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));

			if (StringUtils.isNotEmpty(customerId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			}

			if (!user.canViewAllOpenTickets()) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.and(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, startDate), Restrictions.le(Ticket.PROP_DELIVERY_DATE, endDate)));
			}
			if (isPickupOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.TRUE));
			}
			if (isDeliveryOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.FALSE));
			}
			if (filterUnassigned) {
				criteria.add(Restrictions.isNull(Ticket.PROP_ASSIGNED_DRIVER_ID));
			}
			int currentRowIndex = model.getCurrentRowIndex();
			criteria.setFirstResult(currentRowIndex);
			criteria.setMaxResults(model.getPageSize());

			List<Ticket> list = criteria.list();
			if (onlineOrderOnly) {
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Ticket ticket = (Ticket) iterator.next();
					if (!ticket.isSourceOnline()) {
						iterator.remove();
					}
				}
			}

			model.setRows(list);
		} finally {
			closeSession(session);
		}
	}

	public synchronized static void closeOrders(Ticket... tickets) {
		Session session = getInstance().createNewSession();
		Transaction transaction = null;
		if (tickets == null) {
			return;
		}
		try {
			transaction = session.beginTransaction();
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			for (Ticket ticket : tickets) {
				ticket = getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId());
				ticket.setClosed(true);
				ticket.setClosingDate(serverTimestamp);
				//				ticket.setLastUpdateTime(new Date());

				getInstance().saveOrUpdate(ticket, session);
			}
			transaction.commit();
		} catch (Exception e) {
			PosLog.error(TicketDAO.class, e);

			if (transaction != null)
				transaction.rollback();
		} finally {
			session.close();
		}
	}

	public List<Ticket> findBarTabOpenTickets(OrderType orderType) {
		Session session = null;

		try {
			List<String> barTabOrderTypes = new ArrayList<>();
			List<OrderType> orderTypes = DataProvider.get().getOrderTypes();
			if (orderTypes != null) {
				for (OrderType ot : orderTypes) {
					if (ot.isBarTab())
						barTabOrderTypes.add(ot.getId());
				}
			}
			if (barTabOrderTypes == null || barTabOrderTypes.isEmpty())
				return null;

			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			User user = Application.getCurrentUser();
			if (user != null && !user.canViewAllOpenTickets()) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType == null ? null : orderType.getId()));
			//criteria.createAlias(Ticket.PROP_ORDER_TYPE, "orderType");
			//criteria.add(Restrictions.eq("orderType.barTab", true));
			criteria.add(Restrictions.in(Ticket.PROP_ORDER_TYPE_ID, barTabOrderTypes));
			//criteria.add(Restrictions.or(Restrictions.isEmpty("tableNumbers"), Restrictions.isNull("tableNumbers")));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findVoidTicketByDate(Date startDate, Date endDate, Outlet outlet) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, true));
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CLOSING_DATE, endDate));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public User findOwner(String ticketId) {
		if (ticketId == null) {
			return null;
		}
		Session session = getSession();
		Criteria criteria = session.createCriteria(Ticket.class);

		criteria.add(Restrictions.eq(Ticket.PROP_ID, ticketId));

		ProjectionList projectionList = Projections.projectionList();
		//projectionList.add(Projections.property(Ticket.PROP_OWNER));
		projectionList.add(Projections.property(Ticket.PROP_OWNER_ID));
		criteria.setProjection(projectionList);
		if (criteria.list().isEmpty()) {
			return null;
		}
		//TODO:
		User user = UserDAO.getInstance().get((String) criteria.list().get(0), null);
		return user;
	}

	public Ticket findByCustomerAndDeliveryDate(String customerId, Date deliveryDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Ticket.class);

			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.between(Ticket.PROP_DELIVERY_DATE, DateUtil.startOfDay(deliveryDate), DateUtil.endOfDay(deliveryDate)));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, false));
			List list = criteria.list();
			if (!list.isEmpty()) {
				return (Ticket) list.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}

	}

	public void saveMergedTickets(List<Ticket> tickets, Ticket rootTicket) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			rootTicket.setShouldUpdateTableStatus(true);

			session = createNewSession();
			tx = session.beginTransaction();

			//for (Ticket ticket : tickets) {
			//update(ticket, session);
			//}

			deleteTickets(session, tickets, true);
			saveOrUpdate(rootTicket, session, true);

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateTransferedTicketsList(List<Ticket> ticketsToDelete, List<Ticket> ticketsToUpdate) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			if (ticketsToDelete != null || ticketsToDelete.size() > 0) {
				deleteTickets(session, ticketsToDelete, true);
			}

			for (Ticket ticket : ticketsToUpdate) {
				saveOrUpdate(ticket, session);
			}

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void reversePayment(Ticket ticket, PosTransaction transaction, boolean reverseGiftCardBalance) throws Exception {
		if (transaction instanceof CreditCardTransaction || transaction instanceof DebitCardTransaction) {
			CardReader transactionCardReader = CardReader.fromString(transaction.getCardReader());
			if (StringUtils.isBlank(transaction.getCardMerchantGateway()) || transactionCardReader == CardReader.EXTERNAL_TERMINAL) {
				transaction.setVoided(true);
			}
			else {
				CardProcessor cardProcessor = CardConfig.getPaymentGatewayByName(transaction.getCardMerchantGateway()).getProcessor();
				cardProcessor.voidTransaction(transaction);
			}
			transaction.setVoided(true);
			ticket.setPaidAmount(ticket.getPaidAmount() - transaction.getAmount());
			ticket.setDueAmount(ticket.getDueAmount() + transaction.getAmount());
			ticket.setClosed(false);
			ticket.setPaid(false);

			Session session = null;
			Transaction tx = null;
			try {
				session = createNewSession();
				tx = session.beginTransaction();

				if (reverseGiftCardBalance) {
					doReverseGiftCardBalance(ticket, transaction, session);
				}

				update(ticket, session);

				tx.commit();
			} catch (Exception e) {
				if (tx != null) {
					tx.rollback();
				}
				throw e;
			} finally {
				closeSession(session);
			}
		}
		else {
			Transaction tx = null;

			try (Session session = createNewSession()) {
				tx = session.beginTransaction();
				PaymentType paymentType = transaction.getPaymentType();
				PaymentPlugin paymentPlugin = paymentType.getPaymentPlugin();
				if (paymentPlugin != null) {
					paymentPlugin.voidPayment(transaction, session);
				}
				transaction.setVoided(true);
				//				if (paymentType == PaymentType.CASH) {
				//					PosTransactionService.getInstance().adjustMulticurrencyBalance(session, DataProvider.get().getCurrentTerminal(),
				//							Application.getCurrentUser().getActiveDrawerPullReport(), null, transaction);
				//				}
				ticket.setPaidAmount(ticket.getPaidAmount() - transaction.getAmount());
				ticket.setDueAmount(ticket.getDueAmount() + transaction.getAmount());
				ticket.setClosed(false);
				ticket.setPaid(false);

				if (reverseGiftCardBalance) {
					doReverseGiftCardBalance(ticket, transaction, session);
				}

				update(ticket, session);
				tx.commit();
			} catch (Exception e) {
				if (tx != null) {
					tx.rollback();
				}
				throw e;
			}
		}

	}

	public void doReverseGiftCardBalance(Ticket ticket, PosTransaction transaction, Session session) {
		String giftCardInfo = transaction.getGiftCardBalanceAddInfo();
		if (StringUtils.isEmpty(giftCardInfo)) {
			return;
		}

		JSONArray jsonArray = new JSONArray(giftCardInfo);
		for (Object object : jsonArray) {
			JSONObject jsonObject = (JSONObject) object;
			String ticketItemId = jsonObject.getString(PosTransaction.JSON_PROP_GIFT_CARD_TICKET_ITEM_ID);
			String giftCardNumber = jsonObject.getString(PosTransaction.JSON_PROP_GIFT_CARD_GIFT_CARD_NO);
			double addedAmount = jsonObject.getDouble(PosTransaction.JSON_PROP_GIFT_CARD_ADDED_AMOUNT);

			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, giftCardNumber);
			if (giftCard != null) {
				giftCard.setBalance(giftCard.getBalance() - addedAmount);
				GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			}

			List<TicketItem> ticketItems = ticket.getTicketItems();
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.getId().equals(ticketItemId)) {
					ticketItem.setGiftCardPaidAmount(0.0);
				}
			}
		}
	}

	public void loadTicketsForUser(User user, Date from, Date to, PaginatedTableModel listModel) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (user != null)
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			if (from != null)
				criteria.add(Restrictions.or(Restrictions.ge(Ticket.PROP_CREATE_DATE, from), Restrictions.eq(Ticket.PROP_CLOSED, false)));
			if (to != null)
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, to));

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				listModel.setNumRows(rowCount.intValue());
			}
			else {
				listModel.setNumRows(0);
			}
			criteria.setProjection(null);
			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			listModel.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	private boolean shouldUpdateStock(Ticket ticket) {
		//		if (ticket.isSourceOnline()) {
		//			return false;
		//		}
		if (ticket.isReservation()) {
			return true;
		}
		return ticket.isShouldUpdateStock();
	}

	//	/**
	//	 * This method returns all those ticket which have Transaction in this session.
	//	 */
	//	public List<Ticket> getTicketsOfCurrentSession(StoreSession storeSession) {
	//		Session session = null;
	//		Criteria criteria = null;
	//
	//		try {
	//			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeSession);
	//			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
	//				return null;
	//
	//			session = createNewSession();
	//			criteria = session.createCriteria(getReferenceClass());
	//			criteria.add(Restrictions.eq(Ticket.PROP_CLOUD_SYNCED, Boolean.FALSE));
	//
	//			criteria.createAlias("transactions", "tx");
	//			criteria.add(Restrictions.in("tx." + PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
	//			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
	//			return criteria.list();
	//		} finally {
	//			closeSession(session);
	//		}
	//	}
	/**
	 * This method returns all those Unsynced ticket which have Transaction in this session.
	 */
	public List findAllUnSyncTicket(StoreSession storeSession) {

		Session session = null;
		try {

			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeSession);
			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
				return null;

			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			Criterion nullUpdateTime = Restrictions.isNull(Ticket.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(Ticket.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(Ticket.PROP_LAST_UPDATE_TIME, Ticket.PROP_LAST_SYNC_TIME);
			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));

			//			criteria.createAlias("transactions", "tx");
			//			criteria.add(Restrictions.in("tx." + PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
			//			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Updating Tickets and TicketItems Cloud Sync ...
	 */
	public void updateTicketAndTicketItemSync(List<Ticket> ticketOrgin, boolean synced) {
		if (ticketOrgin != null) {
			for (Ticket ticket : ticketOrgin) {
				loadFullTicket(ticket);
				if (!ticket.isCloudSynced()) {
					if (ticket.isClosed()) {
						ticket.setCloudSynced(synced);
						TicketDAO.getInstance().update(ticket);
					}
				}
				List<TicketItem> ticketItems = ticket.getTicketItems();
				if (ticketItems != null) {
					for (TicketItem ticketItem : ticketItems) {
						if (!ticketItem.isCloudSynced()) {
							//if (ticket.isClosed()) {
							ticketItem.setCloudSynced(synced);
							TicketItemDAO.getInstance().update(ticketItem);
							//}
						}
					}
				}
			}
		}
	}

	public void loadClosedTicket(PaginationSupport model, String txtNameOrId, Date fromDate, Date toDate) {
		loadClosedTicket(model, txtNameOrId, fromDate, toDate, null);
	}

	public void loadClosedTicket(PaginationSupport model, String txtNameOrId, Date fromDate, Date toDate, User user) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Customer.class);

			List<String> customerIds = null;
			if (StringUtils.isNotEmpty(txtNameOrId)) {
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, txtNameOrId, MatchMode.ANYWHERE));
				customerIds = criteria.list();
			}

			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());

			Disjunction ticketIdOrCustomerNameCriteria = Restrictions.disjunction();
			if (StringUtils.isNotEmpty(txtNameOrId)) {
				ticketIdOrCustomerNameCriteria.add(Restrictions.ilike(Ticket.PROP_ID, txtNameOrId, MatchMode.ANYWHERE));
			}
			if (customerIds != null && customerIds.size() > 0) {
				ticketIdOrCustomerNameCriteria.add(Restrictions.in(Ticket.PROP_CUSTOMER_ID, customerIds));
			}
			criteria.add(ticketIdOrCustomerNameCriteria);

			if (user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			}

			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<Ticket> result = criteria.list();
			model.setRows(result);
		} finally {
			closeSession(session);
		}
	}

	public List<Date> findTicketWithinDate(String outletId, Date beginDate, Date currentDate, Session session) {
		try {
			Criteria criteria = session.createCriteria(Ticket.class);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
			criteria.setProjection(Projections.property(Ticket.PROP_CREATE_DATE));
			List list = criteria.list();
			if (list != null && list.size() > 0)
				return list;
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return null;
	}

	public List<DeliverySummaryReportData> findTicketsForDeliverySummaryReport(Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			//@formatter:off
			String as = " as "; //$NON-NLS-1$
			String hql = "select " //$NON-NLS-1$
			        		+ "t."+ Ticket.PROP_OWNER_ID + as + DeliverySummaryReportData.PROP_EMPLYOYEE_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CUSTOMER_ID+ as + DeliverySummaryReportData.PROP_CUSTOMER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CUSTOMER_NAME + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_ID+ as + DeliverySummaryReportData.PROP_TICKET_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SUBTOTAL_AMOUNT+ as + DeliverySummaryReportData.PROP_NET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TAX_AMOUNT+ as + DeliverySummaryReportData.PROP_TAX_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TOTAL_AMOUNT+ as + DeliverySummaryReportData.PROP_TOTAL_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SERVICE_CHARGE+ as + DeliverySummaryReportData.PROP_SERVICE_CHARGE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DISCOUNT_AMOUNT+ as + DeliverySummaryReportData.PROP_DISCOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CREATE_DATE+ as + DeliverySummaryReportData.PROP_CREATE_DATE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DELIVERY_DATE+ as + DeliverySummaryReportData.PROP_DELIVERY_DATE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "g."+ Gratuity.PROP_AMOUNT+ as + DeliverySummaryReportData.PROP_GRATUITY_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CASH_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CREDIT_CARD_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_CHARGE_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_OTHERS_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CASH_PAYMENT_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CREDIT_CARD_PAYMENT_DEBIT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_CHARGE_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_OTHERS_PAYMENT_DEBIT  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ " from Ticket as t left outer join t.gratuity as g"; //$NON-NLS-1$

			 String transSql = String.format("select sum(tr.%s) from PosTransaction as tr" //$NON-NLS-1$
			 		+ " where t=tr.%s and tr.%s='%s' and tr.%s=%s",  //$NON-NLS-1$
			 		PosTransaction.PROP_AMOUNT, 
			 		PosTransaction.PROP_TICKET,
			 		PosTransaction.PROP_TRANSACTION_TYPE,
			 		"%s", //$NON-NLS-1$
					PosTransaction.PROP_VOIDED, false);
			 
			 
			String commonPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + "='%s')"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String othersPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + " not in('%s','%s','%s'))"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String customerQuery = "(select c.%s from Customer as c where t.%s=c.%s)"; //$NON-NLS-1$
			
			hql = String.format(hql, 
					String.format(customerQuery, Customer.PROP_NAME, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID), //$NON-NLS-1$
					String.format(customerQuery, Customer.PROP_MEMBER_ID, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID), //$NON-NLS-1$
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()),
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()));

			hql += " where t." + Ticket.PROP_DELIVERY_DATE + " between :fromDate and :toDate"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " and t." + Ticket.PROP_VOIDED + " =:voided"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " order by t." + Ticket.PROP_OWNER_ID + " asc"; //$NON-NLS-1$ //$NON-NLS-2$

			//@formatter:on
			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", fromDate); //$NON-NLS-1$
			query.setTimestamp("toDate", toDate); //$NON-NLS-1$
			query.setBoolean("voided", Boolean.FALSE); //$NON-NLS-1$
			query.setResultTransformer(Transformers.aliasToBean(DeliverySummaryReportData.class));
			return query.list();
		}
	}

	@Deprecated
	public List<EndOfDayReportData> findTicketsGroupedByEmployee(Date fromDate, Date toDate, List<OrderType> orderTypes) {
		try (Session session = createNewSession()) {
			//@formatter:off
			String as = " as "; //$NON-NLS-1$
			String hql = "select " //$NON-NLS-1$
			        		+ "t."+ Ticket.PROP_OWNER_ID + as + EndOfDayReportData.PROP_EMPLYOYEE_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CUSTOMER_ID+ as + EndOfDayReportData.PROP_CUSTOMER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CUSTOMER_NAME + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_ID+ as + EndOfDayReportData.PROP_TICKET_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SUBTOTAL_AMOUNT+ as + EndOfDayReportData.PROP_NET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TAX_AMOUNT+ as + EndOfDayReportData.PROP_TAX_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TOTAL_AMOUNT+ as + EndOfDayReportData.PROP_TOTAL_TICKET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SERVICE_CHARGE+ as + EndOfDayReportData.PROP_SERVICE_CHARGE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DISCOUNT_AMOUNT+ as + EndOfDayReportData.PROP_DISCOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "g."+ Gratuity.PROP_AMOUNT+ as + EndOfDayReportData.PROP_GRATUITY_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CASH_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CREDIT_CARD_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_CHARGE_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_OTHERS_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CASH_PAYMENT_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CREDIT_CARD_PAYMENT_DEBIT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_CHARGE_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_OTHERS_PAYMENT_DEBIT  //$NON-NLS-1$
			        		+ " from Ticket as t left outer join t.gratuity as g"; //$NON-NLS-1$

			 String transSql = String.format("select sum(tr.%s) from PosTransaction as tr" //$NON-NLS-1$
			 		+ " where t=tr.%s and tr.%s='%s' and tr.%s=%s",  //$NON-NLS-1$
			 		PosTransaction.PROP_AMOUNT, 
			 		PosTransaction.PROP_TICKET,
			 		PosTransaction.PROP_TRANSACTION_TYPE,
			 		"%s", //$NON-NLS-1$
					PosTransaction.PROP_VOIDED, false);
			 
			 
			String commonPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + "='%s')"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String othersPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + " not in('%s','%s','%s'))"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String customerQuery = "(select c.%s from Customer as c where t.%s=c.%s)"; //$NON-NLS-1$
			
			hql = String.format(hql, 
					String.format(customerQuery, Customer.PROP_NAME, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(customerQuery, Customer.PROP_MEMBER_ID, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()),
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()));

			hql += " where t." + Ticket.PROP_CREATE_DATE + " >= :fromDate and t." + Ticket.PROP_CREATE_DATE + " < :toDate"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			if (orderTypes != null) {
				String whereCondition = "("; //$NON-NLS-1$
				for (Iterator<OrderType> iterator = orderTypes.iterator(); iterator.hasNext();) {
					OrderType orderType = (OrderType) iterator.next();
					whereCondition += "'" + orderType.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
					if (iterator.hasNext())
						whereCondition += ","; //$NON-NLS-1$
				}
				whereCondition += ")"; //$NON-NLS-1$
				hql += " and t." + Ticket.PROP_ORDER_TYPE_ID + " in " + whereCondition ; //$NON-NLS-1$ //$NON-NLS-2$
			}
			hql += " and t." + Ticket.PROP_VOIDED + " =:voided"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " order by t." + Ticket.PROP_OWNER_ID + " asc"; //$NON-NLS-1$ //$NON-NLS-2$

			//@formatter:on
			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", DateUtil.toUTC(fromDate)); //$NON-NLS-1$
			query.setTimestamp("toDate", DateUtil.toUTC(toDate)); //$NON-NLS-1$
			query.setBoolean("voided", Boolean.FALSE); //$NON-NLS-1$
			query.setResultTransformer(Transformers.aliasToBean(EndOfDayReportData.class));
			return query.list();
		}
	}

	/**
	 * find online Tickets By Customer.
	 */
	public List<Ticket> findTicketsByCustomer(String customerId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.eq(Ticket.PROP_TYPE, TicketType.ONLINE.getTypeNo()));
			return criteria.list();
		}
	}

	public Ticket get(String ticketId, String outletId) {
		return get(new Ticket(ticketId, outletId));
	}

	private Ticket get(String ticketId, String outletId, Session session) {
		return get(new Ticket(ticketId, outletId), session);
	}
}