package com.floreantpos.model;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;

public class WorkingHours {
	public static final String SUNDAY = "Sunday"; //$NON-NLS-1$
	public static final String MONDAY = "Monday"; //$NON-NLS-1$
	public static final String TUESDAY = "Tuesday"; //$NON-NLS-1$
	public static final String WEDNESDAY = "Wednesday"; //$NON-NLS-1$
	public static final String THURSDAY = "Thursday"; //$NON-NLS-1$
	public static final String FRIDAY = "Friday"; //$NON-NLS-1$
	public static final String SATURDAY = "Saturday"; //$NON-NLS-1$

	public static String[] days = new String[] { "", SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY }; //$NON-NLS-1$

	public static final String OPEN = ".Open"; //$NON-NLS-1$
	public static final String START_TIME = ".StartTime"; //$NON-NLS-1$
	public static final String END_TIME = ".EndTime"; //$NON-NLS-1$
	public static final String TIME_ZONE = "time_zone"; //$NON-NLS-1$

	private Calendar day;
	private Calendar openHour;
	private Calendar endHour;
	private String timeZone;
	private transient PropertyContainer propertiesContainer;
	private int dayOfWeek;

	public WorkingHours(int dayOfWeek, PropertyContainer propertyContainer) {
		this.dayOfWeek = dayOfWeek;
		this.propertiesContainer = propertyContainer;
		loadStartAndEndTime(dayOfWeek);
	}

	private void loadStartAndEndTime(int dayOfWeek) {
		String dayString = WorkingHours.days[dayOfWeek];
		String openTime = propertiesContainer.getProperty(dayString + WorkingHours.START_TIME);
		String endTime = propertiesContainer.getProperty(dayString + WorkingHours.END_TIME);
		this.timeZone = propertiesContainer.getProperty(WorkingHours.TIME_ZONE);
		this.day = Calendar.getInstance();
		this.day.clear();
		this.day.set(Calendar.DAY_OF_WEEK, dayOfWeek);
		openHour = parseStringToCalendar(openTime);
		endHour = parseStringToCalendar(endTime);
	}

	private Calendar parseStringToCalendar(String timeString) {
		if (StringUtils.isEmpty(timeString)) {
			return null;
		}
		Calendar c = Calendar.getInstance();
		try {
			c.clear();
			SimpleDateFormat format = new SimpleDateFormat("hh:mm a"); //$NON-NLS-1$
			Date parseTime = format.parse(timeString);
			c.setTime(parseTime);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return c;
	}

	public String getClosesTimeString() {
		if (isNull()) {
			return ""; //$NON-NLS-1$
		}
		SimpleDateFormat format = new SimpleDateFormat("hh:mm a"); //$NON-NLS-1$
		Date gmtTime = DateUtil.convertDateToGmtTimeZone(endHour.getTime(), getTimezoneOffset());
		return format.format(DateUtil.convertServerTimeToBrowserTime(gmtTime)) + getTimeZoneOffsetString(); //$NON-NLS-1$
	}

	public String getDisplayTimeString() {
		if (isNull()) {
			return ""; //$NON-NLS-1$
		}
		SimpleDateFormat format = new SimpleDateFormat("hh:mm a"); //$NON-NLS-1$
		Date startGmtTime = DateUtil.convertDateToGmtTimeZone(openHour.getTime(), getTimezoneOffset());
		Date endGmtTime = DateUtil.convertDateToGmtTimeZone(endHour.getTime(), getTimezoneOffset());
		return format.format(DateUtil.convertServerTimeToBrowserTime(startGmtTime)) + " - " //$NON-NLS-1$
				+ format.format(DateUtil.convertServerTimeToBrowserTime(endGmtTime)) + getTimeZoneOffsetString();
	}

	private String getTimeZoneOffsetString() {
		try {
			if (StringUtils.isBlank(timeZone)) {
				return ""; //$NON-NLS-1$
			}
			TimeZone timeZ = TimeZone.getTimeZone(timeZone);
			if (timeZ == null) {
				return ""; //$NON-NLS-1$
			}
			int outletTimezoneOffset = getRawOffsetWithDSTSavings(timeZ);
			if (outletTimezoneOffset == DataProvider.get().getTimeZoneOffset()) {
				return ""; //$NON-NLS-1$
			}
			int offsetIntValue = outletTimezoneOffset / 3600000;
			return " (GMT " + offsetIntValue + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return ""; //$NON-NLS-1$
	}

	public boolean isClosed() {
		return !dateExists(new Date());
	}

	private boolean isNull() {
		if (openHour == null || endHour == null) {
			return true;
		}
		if (openHour.getTime() == null || endHour.getTime() == null) {
			return true;
		}
		return false;
	}

	public boolean dateExists(Date date) {
		if (isNull()) {
			return false;
		}
		Calendar c = Calendar.getInstance();
		c.setTime(DateUtil.convertDateByTimeZone(date, getTimezoneOffset()));

		boolean dateExist = dateExists(c, c.get(Calendar.DAY_OF_MONTH));
		if (!dateExist) {
			WorkingHours previousWorkingDay = getWorkingHours(getPreviousWorkingDay());
			if (previousWorkingDay != null) {
				dateExist = previousWorkingDay.dateExists(c, c.get(Calendar.DAY_OF_MONTH) - 1);
				if (dateExist) {
					this.endHour = previousWorkingDay.endHour;
				}
			}
			if (!dateExist) {
				WorkingHours nextWorkingDay = getWorkingHours(getNextWorkingDay());
				if (nextWorkingDay != null) {
					dateExist = nextWorkingDay.dateExists(c, c.get(Calendar.DAY_OF_MONTH) + 1);
				}
				if (dateExist) {
					this.endHour = previousWorkingDay.endHour;
				}
			}
		}
		return dateExist;
	}

	public boolean dateExists(Calendar c, int day) {
		if (openHour == null) {
			return false;
		}
		openHour.set(Calendar.YEAR, c.get(Calendar.YEAR));
		openHour.set(Calendar.MONTH, c.get(Calendar.MONTH));
		openHour.set(Calendar.DAY_OF_MONTH, day);

		endHour.set(Calendar.YEAR, c.get(Calendar.YEAR));
		endHour.set(Calendar.MONTH, c.get(Calendar.MONTH));
		endHour.set(Calendar.DAY_OF_MONTH, day);

		int openHourAmPm = openHour.get(Calendar.AM_PM);
		int closedHourAmPm = endHour.get(Calendar.AM_PM);
		if (openHourAmPm == Calendar.PM && closedHourAmPm == Calendar.AM) {
			endHour.set(Calendar.DAY_OF_MONTH, endHour.get(Calendar.DAY_OF_MONTH) + 1);
		}
		else if (openHourAmPm == closedHourAmPm && endHour.getTime().before(openHour.getTime())) {
			endHour.set(Calendar.DAY_OF_MONTH, endHour.get(Calendar.DAY_OF_MONTH) + 1);
		}
		return DateUtil.between(openHour.getTime(), endHour.getTime(), c.getTime());
	}

	private int getPreviousWorkingDay() {
		int previousDay = dayOfWeek - 1;
		if (previousDay < 0) {
			previousDay = 7;
		}
		return previousDay;
	}

	private int getNextWorkingDay() {
		int nextDay = dayOfWeek + 1;
		if (nextDay > 7) {
			nextDay = 0;
		}
		return nextDay;
	}

	private int getTimezoneOffset() {
		if (StringUtils.isBlank(timeZone)) {
			return DataProvider.get().getTimeZoneOffset();
		}
		TimeZone timeZ = TimeZone.getTimeZone(timeZone);
		if (timeZ == null) {
			return DataProvider.get().getTimeZoneOffset();
		}
		return getRawOffsetWithDSTSavings(timeZ);
	}

	private int getRawOffsetWithDSTSavings(TimeZone timeZ) {
		return timeZ.getRawOffset() + timeZ.getDSTSavings();
	}

	public WorkingHours getWorkingHours(int day) {
		if (!propertiesContainer.getBooleanProperty(WorkingHours.days[day] + WorkingHours.OPEN, true)) {
			return null;
		}
		return new WorkingHours(day, propertiesContainer);
	}
}
