/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.VirtualPrinter;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PrinterGroupDAO extends BasePrinterGroupDAO {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private static final String EMPTY_DOT_STRING = "."; //$NON-NLS-1$
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$
	private static final String EMPTY_SPACE_STRING = " "; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PrinterGroupDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PrinterGroup) {
			PrinterGroup printerGroup = (PrinterGroup) obj;
			List<MenuItem> menuItems = MenuItemDAO.getInstance().getMenuItemsByPrinterGroup(printerGroup, s);
			if (menuItems != null && !menuItems.isEmpty()) {
				String message = printerGroup.getName() + EMPTY_SPACE_STRING + Messages.getString("PrinterGroupDAO.0"); //$NON-NLS-1$
				String details = this.constructExceptionDetailsByMenuItems(printerGroup, menuItems);
				throw new PosException(message, details);
			}
			printerGroup.setDeleted(Boolean.TRUE);
			super.update(printerGroup, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	private String constructExceptionDetailsByMenuItems(PrinterGroup printerGroup, List<MenuItem> menuItems) {
		if (menuItems != null && !menuItems.isEmpty()) {
			StringBuilder builder = new StringBuilder(printerGroup.getName() + EMPTY_SPACE_STRING + Messages.getString("PrinterGroupDAO.1")); //$NON-NLS-1$
			for (int i = 0; i < menuItems.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + menuItems.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<PrinterGroup> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public void saveOrUpdatePrinterGroup(List<PrinterGroup> printerGroupList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (printerGroupList == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<PrinterGroup> iterator = printerGroupList.iterator(); iterator.hasNext();) {
				PrinterGroup item = (PrinterGroup) iterator.next();
				PrinterGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	//Deleted filter for this method is strictly prohibited
	public PrinterGroup findByName(String name, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(PrinterGroup.PROP_NAME, name));
		return (PrinterGroup) criteria.uniqueResult();
	}

	public PrinterGroup findByName(String name) {
		try (Session session = this.createNewSession()) {
			return findByName(name, session);
		}
	}

	@SuppressWarnings("unchecked")
	public PrinterGroup getDefaultPrinterGroup() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(PrinterGroup.PROP_IS_DEFAULT, Boolean.TRUE));
			List<PrinterGroup> list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return list.get(0);
			}
			return null;
		}
	}

	@SuppressWarnings("unchecked")
	public List<PrinterGroup> getPrinterGroupsByVirtualPrinter(VirtualPrinter virtualPrinter, Session session) {
		List<PrinterGroup> printerGroups = null;
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		List<PrinterGroup> list = criteria.list();
		if (list != null && !list.isEmpty()) {
			printerGroups = new ArrayList<>();
			for (PrinterGroup printerGroup : list) {
				if (printerGroup.getPrinterNames().contains(virtualPrinter.getName())) {
					printerGroups.add(printerGroup);
				}
			}
		}
		return printerGroups;
	}
}