package com.floreantpos.extension.stripe;

import java.text.DecimalFormat;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.extension.cardconnect.IppMessages;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.POSUtil;
import com.stripe.Stripe;
import com.stripe.exception.CardException;
import com.stripe.model.Charge;
import com.stripe.model.Refund;
import com.stripe.model.Token;

public class StripeProcessor implements CardProcessor {
	private static final String CHARGE = "charge"; //$NON-NLS-1$
	private static final String NAME = "name"; //$NON-NLS-1$
	private static final String ADDRESS_ZIP = "address_zip"; //$NON-NLS-1$
	private static final String ADDRESS_LINE1 = "address_line1"; //$NON-NLS-1$
	private static final String SUCCEEDED = "succeeded"; //$NON-NLS-1$
	private static final String DESCRIPTION = "description"; //$NON-NLS-1$
	private static final String SOURCE = "source"; //$NON-NLS-1$
	private static final String CARD = "card"; //$NON-NLS-1$
	private static final String EXP_YEAR = "exp_year"; //$NON-NLS-1$
	private static final String EXP_MONTH = "exp_month"; //$NON-NLS-1$
	private static final String CARD_NUMBER = "number"; //$NON-NLS-1$
	private static final String CARD_CVC = "cvc"; //$NON-NLS-1$
	private static final String CURRENCY = "currency"; //$NON-NLS-1$
	private static final String AMOUNT = "amount"; //$NON-NLS-1$

	private String apiKey;
	private String secretKey;

	public StripeProcessor() {
	}

	private void init(PosTransaction transaction) {
		Outlet outlet = transaction.getTicket().getOutlet();
		this.apiKey = outlet.getProperty(Store.PROP_ONLINE_ORDER_STRIPE_API_KEY);
		this.secretKey = outlet.getProperty(Store.PROP_ONLINE_ORDER_STRIPE_SECRET_KEY);
		Stripe.apiKey = apiKey;
	}

	@Override
	public void preAuth(PosTransaction transaction) throws Exception {
	}

	@Override
	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		init(transaction);
		preProcessingLog(transaction);

		Charge charge = Charge.retrieve(transaction.getCardTransactionId());
		if (charge.getCaptured()) {
			transaction.setCaptured(true);
			return;
		}
		charge = charge.capture();
		if (isNotSucceeded(charge)) {
			throw new PosException(charge.getFailureMessage());
		}
		transaction.setCaptured(true);
	}

	@Override
	public void chargeAmount(PosTransaction transaction) throws Exception {
		init(transaction);
		preProcessingLog(transaction);

		checkCardValidation(transaction);

		Ticket ticket = transaction.getTicket();
		Token token = createToken(transaction);

		Map<String, Object> chargeParams = new HashMap<>();
		chargeParams.put(AMOUNT, new DecimalFormat("000").format(transaction.getAmount() * 100)); //$NON-NLS-1$
		chargeParams.put(CURRENCY, DataProvider.get().getMainCurrency().getCode());
		chargeParams.put(SOURCE, token.getId());
		chargeParams.put(DESCRIPTION, "Check#" + ticket.getId() + " Total: " + ticket.getTotalAmountWithTips()); //$NON-NLS-1$ //$NON-NLS-2$

		Charge charge = Charge.create(chargeParams);
		if (isNotSucceeded(charge)) {
			PosLog.error(getClass(), charge.getFailureMessage());
			throw new RuntimeException();
		}
		transaction.setCaptured(charge.getCaptured());
		transaction.setAuthorizable(false);
		transaction.setCardTransactionId(charge.getId());
		if (StringUtils.isNotBlank(charge.getAuthorizationCode())) {
			transaction.setCardAuthCode(charge.getAuthorizationCode());
		}
		else {
			transaction.setCardAuthCode(charge.getId());
		}
		String accountNumber = transaction.getCardNumber();
		if (StringUtils.isNotEmpty(accountNumber)) {
			accountNumber = accountNumber.substring(accountNumber.length() - 4, accountNumber.length());
			transaction.setCardNumber(accountNumber);
		}
	}

	private void preProcessingLog(PosTransaction transaction) {
	}

	private boolean checkCardValidation(PosTransaction transaction) throws Exception {
		if (transaction.getCardReader().equals(CardReader.SWIPE.name()) && StringUtils.isNotBlank(transaction.getCardTrack())) {
			return Boolean.TRUE;
		}
		else if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			if (StringUtils.isBlank(transaction.getCardNumber())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.0")); //$NON-NLS-1$
			}
			if (!(StringUtils.isNotBlank(transaction.getCardExpYear()) && transaction.getCardExpYear().length() >= 2)) {
				throw new PosException(IppMessages.getString("SecuredProcessor.1")); //$NON-NLS-1$
			}
			if (StringUtils.isBlank(transaction.getCardExpMonth())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.2")); //$NON-NLS-1$
			}
		}
		return Boolean.TRUE;
	}

	private Token createToken(PosTransaction transaction) {
		String cardNumber = transaction.getCardNumber();
		int expMonth = POSUtil.parseInteger(transaction.getCardExpMonth());
		int expYear = POSUtil.parseInteger(transaction.getCardExpYear());
		String cvc = transaction.getCardCVV();

		Ticket ticket = transaction.getTicket();
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();
			Map<String, Object> cardParams = new HashMap<String, Object>();
			cardParams.put(CARD_NUMBER, cardNumber);
			cardParams.put(EXP_MONTH, expMonth);
			cardParams.put(EXP_YEAR, expYear);
			cardParams.put(CARD_CVC, cvc);
			cardParams.put(ADDRESS_LINE1, ticket.getDeliveryAddress());
			cardParams.put(ADDRESS_ZIP, ticket.getDeliveryZipCode());
			cardParams.put(NAME, ticket.getCustomerName());
			tokenParams.put(CARD, cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	private boolean isNotSucceeded(Charge charge) {
		return charge.getStatus() == null || !charge.getStatus().equals(SUCCEEDED) || StringUtils.isNotBlank(charge.getFailureMessage());
	}

	private boolean isNotSucceeded(Refund refund) {
		return refund.getStatus() == null || !refund.getStatus().equals(SUCCEEDED);
	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		refundTransaction(transaction, transaction.getAmount());
	}

	@Override
	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		init(transaction);
		this.preProcessingLog(transaction);

		Map<String, Object> params = new HashMap<>();
		params.put(CHARGE, transaction.getCardTransactionId());
		Refund refund = Refund.create(params);
		if (isNotSucceeded(refund)) {
			PosLog.error(getClass(), "Refund failed. Trans. ID: " + transaction.getCardTransactionId()); //$NON-NLS-1$
			throw new RuntimeException();
		}
		transaction.setRefunded(true);
	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	@Override
	public void cancelTransaction() {
	}

}