/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;
import java.util.List;
import java.util.Set;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Index;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.TermVector;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseMenuCategory;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.HibernateSearchUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "buttonColor", "textColor", "menuGroups", "discounts", "orderTypes", "departments", "terminalTypes",
		"menuShifts" })
@XmlRootElement(name = "menu-category")
@Entity
@Indexed
public class MenuCategory extends BaseMenuCategory implements TimedModel {
	private static final long serialVersionUID = 1L;
	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public MenuCategory() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuCategory(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public MenuCategory(java.lang.String id, java.lang.String name) {

		super(id, name);
	}

	/*[CONSTRUCTOR MARKER END]*/

	private Color buttonColor;
	private Color textColor;

	private List<String> departmentIds;
	private List<String> orderTypeIds;
	private List<String> menuShiftIds;

	@Override
	public Integer getSortOrder() {
		return super.getSortOrder() == null ? 9999 : super.getSortOrder();
	}

	@Override
	public Integer getButtonColorCode() {
		Integer buttonColorCode = super.getButtonColorCode();
		if (buttonColorCode == null || buttonColorCode == 0 || buttonColorCode == -1316371) {
			buttonColorCode = Color.WHITE.getRGB();
		}
		return buttonColorCode;
	}

	@XmlTransient
	public Color getButtonColor() {
		if (buttonColor != null) {
			return buttonColor;
		}

		if (getButtonColorCode() == null || getButtonColorCode() == 0) {
			return null;
		}

		return buttonColor = new Color(getButtonColorCode());
	}

	public void setButtonColor(Color buttonColor) {
		this.buttonColor = buttonColor;
	}

	@XmlTransient
	public Color getTextColor() {
		if (textColor != null) {
			return textColor;
		}

		if (getTextColorCode() == null || getTextColorCode() == 0) {
			return null;
		}

		return textColor = new Color(getTextColorCode());
	}

	public void setTextColor(Color textColor) {
		this.textColor = textColor;
	}

	@JsonIgnore
	public String getDisplayName() {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		if (terminal != null && terminal.isShowTranslatedName() && StringUtils.isNotEmpty(getTranslatedName())) {
			return getTranslatedName();
		}

		return super.getName();
	}

	@XmlTransient
	public Set<Department> getDepartments() {
		return super.getDepartments();
	}

	@Override
	public String toString() {
		return getDisplayName();
	}

	public String getUniqueId() {
		return ("menu_category_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public List<String> getDepartmentIds() {
		return departmentIds;
	}

	public void setDepartmentIds(List<String> departmentIds) {
		this.departmentIds = departmentIds;
	}

	public List<String> getOrderTypeIds() {
		return orderTypeIds;
	}

	public void setOrderTypeIds(List<String> orderTypeIds) {
		this.orderTypeIds = orderTypeIds;
	}

	public List<String> getMenuShiftIds() {
		return menuShiftIds;
	}

	public void setMenuShiftIds(List<String> menuShiftIds) {
		this.menuShiftIds = menuShiftIds;
	}

	public void setTicketDiscountApplicable(boolean value) {
		addProperty("tDiscApplicable", String.valueOf(value)); //NON-NLS-1$
	}

	public boolean isTicketDiscountApplicable() {
		return POSUtil.getBoolean(getProperty("tDiscApplicable"), true); //NON-NLS-1$
	}

	@Id
	@Override
	public String getId() {
		return super.getId();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.YES, index = Index.YES)
	@Override
	public String getName() {
		return super.getName();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@XmlTransient
	public String getStoreId() {
		if (StringUtils.isBlank(HibernateSearchUtil.getCurrentStoreId())) {
			return DataProvider.get().getStore().getUuid();
		}

		return HibernateSearchUtil.getCurrentStoreId();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isDeleted() {
		return super.isDeleted();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isOnlineOrderingEnable() {
		return super.isOnlineOrderingEnable();
	}
}