package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.PriceShift;

public class PriceShiftDAO extends BasePriceShiftDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PriceShiftDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		validateData((PriceShift) obj, s);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		validateData((PriceShift) obj, s);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		validateData((PriceShift) obj, s);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void validateData(PriceShift priceShift, Session session) {
		GenericDAO.getInstance().checkIdOrNameExists(priceShift.getId(), priceShift.getName(), PriceShift.class, session);
		if (priceShiftExists(priceShift.getId(), priceShift.getStartHour(), priceShift.getStartMin(), priceShift.getEndHour(), priceShift.getEndMin(),
				priceShift.getDaysOfWeek(), session)) {
			throw new DuplicateDataException(Messages.getString("PriceShiftDAO.0")); //$NON-NLS-1$
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		ShiftDAO.getInstance().delete(obj, session);
	}

	@Override
	public List<PriceShift> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public boolean exists(String shiftName) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(PriceShift.PROP_NAME, shiftName));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return true;
			}
			return false;
		} catch (Exception e) {
			throw new PosException("An error occured while trying to check price shift duplicacy", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	private boolean priceShiftExists(String existingId, Integer startHour, Integer startMin, Integer endHour, Integer endMin, String daysOfWeek,
			Session session) throws PosException {

		Criteria criteria = session.createCriteria(getReferenceClass());
		if (StringUtils.isNotEmpty(existingId)) {
			SimpleExpression existingIdExp = Restrictions.ne("id", existingId); //$NON-NLS-1$
			criteria.add(existingIdExp);
		}
		addDeletedFilter(criteria);
		SimpleExpression startHourExp = Restrictions.eq(PriceShift.PROP_START_HOUR, startHour);
		SimpleExpression startMinExp = Restrictions.eq(PriceShift.PROP_START_MIN, startMin);
		SimpleExpression endHourExp = Restrictions.eq(PriceShift.PROP_END_HOUR, endHour);
		SimpleExpression endMinExp = Restrictions.eq(PriceShift.PROP_END_MIN, endMin);
		SimpleExpression daysOfWeekExp = Restrictions.eq(PriceShift.PROP_DAYS_OF_WEEK, daysOfWeek);
		Conjunction shiftTimeExp = Restrictions.and(startHourExp, startMinExp, endHourExp, endMinExp, daysOfWeekExp);
		criteria.add(shiftTimeExp);

		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return true;
		}
		return false;
	}

	/*public PriceShift getCurrentPriceShift(Session session) throws Exception {
		Criteria criteria = session.createCriteria(getReferenceClass());
		Calendar calendar2 = Calendar.getInstance();
		calendar2.setTime(new Date());
		Calendar calendar = Calendar.getInstance();
		calendar.clear();
		calendar.set(Calendar.HOUR_OF_DAY, calendar2.get(Calendar.HOUR_OF_DAY));
		calendar.set(Calendar.MINUTE, calendar2.get(Calendar.MINUTE));
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		criteria.add(Restrictions.or(
				Restrictions.eq(PriceShift.PROP_ANY_DAY, Boolean.TRUE),
				Restrictions.and(
						Restrictions.eq(PriceShift.PROP_DAY_OF_WEEK, calendar.get(Calendar.DAY_OF_WEEK)),
						Restrictions.and(Restrictions.ge(PriceShift.PROP_START_TIME, format.parse(format.format(calendar.getTime()))),
								Restrictions.lt(PriceShift.PROP_END_TIME, format.parse(format.format(calendar.getTime())))))));
	
		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return (PriceShift) list.get(0);
		}
		return null;
	}*/

	public List<PriceShift> findAllActive() {
		Session session = null;
		try {
			session = createNewSession();

			return findAllActive(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<PriceShift> findAllActive(Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(PriceShift.PROP_ENABLE, Boolean.TRUE));
		criteria.addOrder(Order.asc(PriceShift.PROP_PRIORITY));
		return criteria.list();
	}

	public PriceShift get(String priceShiftId, String outletId) {
		return super.get(new PriceShift(priceShiftId, outletId));
	}

}