/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.swing.ImageIcon;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Index;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.TermVector;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.base.BaseMenuItem;
import com.floreantpos.model.dao.CourseDAO;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.MenuGroupDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.MenuItemSizeDAO;
import com.floreantpos.model.dao.MenuModifierDAO;
import com.floreantpos.model.dao.PizzaCrustDAO;
import com.floreantpos.model.dao.PriceTableItemDAO;
import com.floreantpos.model.ext.UnitBaseItemPrice;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.HibernateSearchUtil;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.swing.PosUIManager;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "buttonColor", "textColor", "image", "orderTypeList", "properties", "discounts", "terminals", "recepieItems",
		"units", "sizes", "variantName", "parentMenuItem", "explorerImage", "taxGroup", "reportGroup", "unit", "stockStatus", "availableUnit", "unitOnHand", "variants" })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@Entity
@Indexed
public class MenuItem extends BaseMenuItem implements TimedModel {

	public static final String JSON_PROP_SUFFIX_TAXGROUP = ".taxgroup";
	public static final String JSON_PROP_SUFFIX_FOR_HERE = ".for_here";
	public static final String JSON_PROP_SUFFIX_TO_GO = ".to_go";

	private static final long serialVersionUID = 1L;
	public static final String TRANSIENT_PROP_VENDOR_NAME = "vendorNames"; //$NON-NLS-1$
	public static final String PROP_UNIT_BASE_PRICE_KEY = "unit.{name}.price"; //$NON-NLS-1$
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private static final String JSON_PROP_ENABLE_QTY_SELECTION = "enable.qty.selection"; //$NON-NLS-1$
	private static final String JSON_PROP_MOD_PRICE_COMBO_ITEM = "menuitem.modifiable.price.on.comboItem"; //$NON-NLS-1$
	/**
	 * Vendor Names added to show in inventory items which contains vendors.
	 * This property has been set in MenuItemDAO and method name is loadInventoryItems 
	 * 
	 */
	private transient String vendorNames;

	/* [CONSTRUCTOR MARKER BEGIN] */
	public MenuItem() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuItem(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public MenuItem(java.lang.String id, java.lang.String name, java.lang.Double price) {

		super(id, name, price);
	}

	/* [CONSTRUCTOR MARKER END] */

	private MenuItem parentMenuItem;
	private MenuItemInventoryStatus stockStatus;

	public static final String TYPE_RAW_METARIAL = "Raw Material"; //$NON-NLS-1$
	public static final String TYPE_INVENTORY_ITEM = "Inventory Item"; //$NON-NLS-1$
	public static final String TYPE_MENU_ITEM = "Menu Item"; //$NON-NLS-1$
	private transient JSONObject properties;
	private String outletId;

	public void setImage(ImageIcon image) {
	}

	public ImageIcon getImage() {
		ImageResource imageResource = DataProvider.get().getImageResource(getImageId());
		if (imageResource != null) {
			return imageResource.getButtonImage();
		}
		return null;
	}

	/**
	 * Only to be used from BeanTableModel
	 * 
	 * @return
	 */
	public ImageIcon getExplorerImage() {
		ImageResource imageResource = DataProvider.get().getImageResource(getImageId());
		if (imageResource != null) {
			return ImageResource.getScaledImageIcon(imageResource.getImageIcon(), PosUIManager.getSize(30));
		}
		return null;
	}

	/**
	 * Only to be used from BeanTableModel
	 * 
	 * @return
	 */
	public void setExplorerImage(ImageIcon image) {

	}

	@Override
	public Integer getSortOrder() {
		return super.getSortOrder() == null ? 9999 : super.getSortOrder();
	}

	@Override
	public Integer getButtonColorCode() {
		Integer buttonColorCode = super.getButtonColorCode();
		if (buttonColorCode == null || buttonColorCode == 0 || buttonColorCode == -1316371) {
			buttonColorCode = Color.WHITE.getRGB();
		}
		return buttonColorCode;
	}

	@XmlTransient
	public Color getButtonColor() {
		Integer buttonColorCode = getButtonColorCode();
		if (buttonColorCode == null) {
			return null;
		}

		return new Color(getButtonColorCode());
	}

	public void setButtonColor(Color buttonColor) {
		if (buttonColor != null) {
			setTextColorCode(buttonColor.getRGB());
		}
	}

	@XmlTransient
	public Color getTextColor() {
		if (getTextColorCode() == null) {
			return null;
		}
		return new Color(getTextColorCode());
	}

	public void setTextColor(Color textColor) {
		if (textColor != null) {
			setTextColorCode(textColor.getRGB());
		}
	}

	@XmlTransient
	public String getDisplayName() {
		//		Terminal terminal = DataProvider.get().getCurrentTerminal();
		//		if (((terminal != null && terminal.isShowTranslatedName()) || isVariant()) && StringUtils.isNotEmpty(super.getTranslatedName())) {
		//			return super.getTranslatedName();
		//		}
		if (StringUtils.isNotEmpty(super.getTranslatedName())) {
			return super.getTranslatedName();
		}
		return super.getName();
	}

	public String createVariantName(String name) {
		List<Attribute> attributes = getAttributes();
		if (attributes == null) {
			return name;
		}
		String variantName = " ("; //$NON-NLS-1$
		for (Iterator<Attribute> iterator = attributes.iterator(); iterator.hasNext();) {
			Attribute attribute = (Attribute) iterator.next();
			variantName += attribute.getName();
			if (iterator.hasNext()) {
				variantName += ", "; //$NON-NLS-1$
			}
		}
		variantName += ")"; //$NON-NLS-1$

		if (StringUtils.isNotBlank(name) && (name.length() + variantName.length()) > 255) {
			name = name.substring(0, name.length() - (variantName.length() + 3)) + "..." + variantName;
		}
		else {
			name += variantName;
		}
		return name;
	}

	@XmlTransient
	public String getVariantName() {
		String name = super.getName();
		List<Attribute> attributes = getAttributes();
		if (attributes == null) {
			return name;
		}
		name += " ("; //$NON-NLS-1$
		for (Iterator iterator = attributes.iterator(); iterator.hasNext();) {
			Attribute attribute = (Attribute) iterator.next();
			name += attribute.getName();
			if (iterator.hasNext()) {
				name += ", "; //$NON-NLS-1$
			}
		}
		name += ")"; //$NON-NLS-1$
		return name;
	}

	public void setVariantName(String variantName) {

	}

	public void setDisplayName(String displayName) {

	}

	@Override
	public String toString() {
		return getDisplayName();
	}

	public String getUniqueId() {
		return ("menu_item_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	//	public TicketItem convertToTicketItem() {
	//		return convertToTicketItem(null, 0);
	//	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity) {
		return convertToTicketItem(ticket, itemQuantity, false);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, IUnit destinationUnit) {
		return convertToTicketItem(ticket, itemQuantity, false, destinationUnit, true);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, boolean hasComboModifiers) {
		return this.convertToTicketItem(ticket, itemQuantity, hasComboModifiers, null, true);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, boolean hasComboModifiers, IUnit destinationUnit, boolean checkEditablePrice) {
		TicketItem ticketItem = null;
		MenuItem menuItem = this;
		boolean isComboItem = isComboItem();
		if (this.isVariant() && this.getParentMenuItem() != null) {
			menuItem = this.getParentMenuItem();
			isComboItem = menuItem.isComboItem();
		}
		if (isComboItem) {
			ticketItem = new ComboTicketItem();
		}
		else if (menuItem.isPizzaType() || menuItem.hasModifiers() || hasComboModifiers) {
			ticketItem = new ModifiableTicketItem();
		}
		else {
			ticketItem = new TicketItem();
		}
		ticketItem.setTaxIncluded(Application.getInstance().isPriceIncludesTax());
		OrderType orderType = null;
		if (ticket != null) {
			orderType = ticket.getOrderType();
		}
		if (orderType != null && orderType.isEnableCourse()) {
			String courseId = menuItem.getCourseOrganizeId();
			ticketItem.setCourseId(courseId);
			if (StringUtils.isNotEmpty(courseId)) {
				Course course = DataProvider.get().getCourse(courseId);
				if (course != null) {
					ticketItem.setCourseName(course.getShortName());
				}
			}
		}
		ticketItem.setMenuItemId(this.getId());
		ticketItem.setMenuItem(this);
		ticketItem.setComboItem(menuItem.isComboItem());
		ticketItem.setPizzaType(isPizzaType());
		ticketItem.setFractionalUnit(menuItem.isFractionalUnit());
		ticketItem.setName(this.getDisplayName());
		ticketItem.setGroupId(menuItem.getMenuGroupId());
		ticketItem.setGroupName(menuItem.getMenuGroupName());
		ticketItem.setCategoryName(menuItem.getMenuCategoryName());
		ticketItem.setCategoryId(menuItem.getMenuCategoryId());
		ticketItem.setServiceChargeApplicable(menuItem.isServiceChargeApplicable());
		ticketItem.setServiceChargeRate(menuItem.getServiceCharge());
		ticketItem.setTaxOnServiceCharge(menuItem.isTaxOnServiceCharge());
		ticketItem.setUnitSelection(menuItem.isAllowUnitSelection());
		if (StringUtils.isNotEmpty(menuItem.getTranslatedName())) {
			ticketItem.addProperty(AppConstants.TRANSLATED_NAME, menuItem.getTranslatedName());
		}
		setTicketItemUnitPriceAndCost(ticketItem, menuItem, destinationUnit, ticket, checkEditablePrice);
		ticketItem.setQuantity(itemQuantity);
		ticketItem.setInventoryItem(menuItem.isInventoryItem());
		ticketItem.setTicket(ticket);
		ticketItem.setShouldPrintToKitchen(menuItem.isShouldPrintToKitchen());
		ticketItem.setBeverage(menuItem.isBeverage());
		ticketItem.setPrinterGroup(menuItem.getPrinterGroup());
		setItemTaxes(ticketItem, getTaxGroup(), ticket.getOrderType());
		// SETTING KITCHEN PRINT AND KITCHEN STICKER PRINT BEGIN
		ticketItem.setShouldPrintToKitchen(menuItem.isShouldPrintToKitchen());
		ticketItem.setPrintKitchenSticker(menuItem.isPrintKitchenSticker());
		// END
		List<Discount> discountList = menuItem.getDiscounts();
		if (discountList != null) {
			for (Discount discount : discountList) {
				if (discount.isAutoApply()) {
					TicketItemDiscount ticketItemDiscount = convertToTicketItemDiscount(discount, ticketItem);
					ticketItem.addTodiscounts(ticketItemDiscount);
				}
			}
		}
		return ticketItem;
	}

	public void setTicketItemUnitPriceAndCost(TicketItem ticketItem, MenuItem menuItem, IUnit destinationUnit, Ticket ticket) {
		setTicketItemUnitPriceAndCost(ticketItem, menuItem, destinationUnit, ticket, true);
	}

	public void setTicketItemUnitPriceAndCost(TicketItem ticketItem, MenuItem menuItem, IUnit destinationUnit, Ticket ticket, boolean checkEditablePrice) {
		if (isEditablePrice() && checkEditablePrice) {
			if (destinationUnit != null) {
				double unitQuantity = this.getUnitQuantity(this.getUnit(), destinationUnit);
				Double unitCost = menuItem.getItemCost() / unitQuantity;
				ticketItem.setUnitName(destinationUnit.getUniqueCode());
				ticketItem.setUnitCost(unitCost);
			}
			else {
				ticketItem.setUnitName(menuItem.getUnitName());
				ticketItem.setUnitCost(this.getItemCost());
			}
			ticketItem.setUnitPrice(this.getPrice());
		}
		else {
			Double priceRulePrice = getOnlinePrice(ticket.getOutletId());
			ticketItem.setUnitPrice(priceRulePrice);
			if (destinationUnit != null) {
				double unitQuantity = this.getUnitQuantity(this.getUnit(), destinationUnit);
				Double unitPrice = getDestinationUnitPrice(menuItem, destinationUnit, ticketItem.getUnitPrice(), unitQuantity);

				Double unitCost = menuItem.getItemCost() / unitQuantity;
				ticketItem.setUnitName(destinationUnit.getUniqueCode());
				ticketItem.setUnitCost(unitCost);
				ticketItem.setUnitPrice(unitPrice);
			}
			else {
				ticketItem.setUnitName(menuItem.getUnitName());
				ticketItem.setUnitCost(this.getItemCost());
			}
		}
	}

	public double getDestinationUnitPrice(MenuItem menuItem, IUnit destinationUnit, Double priceFromPriceRule, double unitQuantity) {
		double destinationUnitPrice = 0;
		if (destinationUnit != null) {
			destinationUnitPrice = priceFromPriceRule / unitQuantity;
			if (this.isAllowUnitSelection()) {
				String key = MenuItem.PROP_UNIT_BASE_PRICE_KEY;
				key = key.replace("{name}", destinationUnit.getUniqueCode()); //$NON-NLS-1$
				String priceProperty = this.getProperty(key);
				if (StringUtils.isNotEmpty(priceProperty)) {
					destinationUnitPrice = (priceFromPriceRule * NumberUtil.parseDouble(priceProperty)) / menuItem.getPrice();
				}
			}
		}
		return destinationUnitPrice;
	}

	public double getPriceFromPriceRule(Ticket ticket) {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		Department department = terminal == null ? null : terminal.getDepartment();

		SalesArea salesArea = null;
		OrderType orderType = null;
		String customerId = null;
		CustomerGroup customerGroup = null;

		if (ticket != null) {
			orderType = ticket.getOrderType();
			salesArea = ticket.getSalesArea();
			customerId = ticket.getCustomerId();
		}
		if (customerId != null) {
			Customer customer = DataProvider.get().getCustomer(customerId);
			if (customer != null) {
				customerGroup = customer.getCustomerGroup();
			}
		}

		double priceFromPriceRule = DataProvider.get().getPriceFromPriceRule(this, orderType, department, salesArea, customerGroup);
		if (isVariant()) {
			return priceFromPriceRule + getParentMenuItem().getPrice();
		}
		else {
			return priceFromPriceRule;
		}
	}

	/*
	 * This method is used to return default course id from cache manager if
	 * course id is null.
	 */
	public String getCourseOrganizeId() {
		String courseId = super.getCourseId();
		//		if (StringUtils.isEmpty(courseId)) {
		//			courseId = DataProvider.get().getDefaultCourseId();
		//		}
		return courseId;
	}

	public static TicketItemDiscount convertToTicketItemDiscount(Discount discount, TicketItem ticketItem) {
		TicketItemDiscount ticketItemDiscount = new TicketItemDiscount();
		ticketItemDiscount.setDiscountId(discount.getId());
		ticketItemDiscount.setAutoApply(discount.isAutoApply());
		ticketItemDiscount.setName(discount.getName());
		ticketItemDiscount.setType(discount.getType());
		ticketItemDiscount.setMinimumAmount(discount.getMinimumBuy());
		ticketItemDiscount.setValue(discount.getValue());
		ticketItemDiscount.setAmount(discount.getValue());
		ticketItemDiscount.setCouponQuantity(1.0);
		ticketItemDiscount.setTicketItem(ticketItem);
		return ticketItemDiscount;

	}

	public boolean hasModifiers() {
		return super.isHasModifiers();
	}

	public boolean hasMandatoryModifiers() {
		return super.isHasMandatoryModifiers();
	}

	public boolean hasAutoShowGroup() {
		List<MenuItemModifierSpec> modiferGroups = getMenuItemModiferSpecs();
		if (modiferGroups == null || modiferGroups.size() == 0) {
			return false;
		}

		for (MenuItemModifierSpec menuItemModifierGroup : modiferGroups) {
			if (menuItemModifierGroup.isAutoShow()) {
				return true;
			}
		}

		return false;
	}

	private static TaxGroup getOutletTaxGroup() {
		Outlet outlet = Application.getOutlet();
		if (outlet != null) {
			TaxGroup outletTaxGroup = outlet.getTaxGroup();
			return outletTaxGroup;
		}
		return null;
	}

	public static void setItemTaxes(TicketItem ticketItem, TaxGroup itemTaxGroup, OrderType orderType) {
		MenuItem menuItem = ticketItem.getMenuItem();
		if (menuItem != null && orderType != null) {
			TaxGroup orderBaseTaxGroup = getOrderBaseTaxGroup(menuItem, "", orderType.getId());
			if (orderBaseTaxGroup != null) {
				itemTaxGroup = orderBaseTaxGroup;
			}
		}
		if (ticketItem.getTicket() != null) {
			SubOrderType subOrderType = ticketItem.getTicket().getSubOrderType();
			if (subOrderType != null) {
				if (subOrderType == SubOrderType.FOR_HERE) {
					TaxGroup orderBaseTaxGroup = getOrderBaseTaxGroup(menuItem, orderType.getId(), JSON_PROP_SUFFIX_FOR_HERE);
					if (orderBaseTaxGroup != null) {
						itemTaxGroup = orderBaseTaxGroup;
					}
					else {
						itemTaxGroup = orderType.getForHereTaxGroup();
					}
				}
				else if (subOrderType == SubOrderType.TO_GO) {
					TaxGroup orderBaseTaxGroup = getOrderBaseTaxGroup(menuItem, orderType.getId(), JSON_PROP_SUFFIX_TO_GO);
					if (orderBaseTaxGroup != null) {
						itemTaxGroup = orderBaseTaxGroup;
					}
					else {
						itemTaxGroup = orderType.getToGoTaxGroup();
					}
				}
			}
		}
		if (itemTaxGroup == null) {
			TaxGroup orderTypeTaxGroup = orderType == null ? null : orderType.getDefaultTaxGroup();
			if (orderTypeTaxGroup != null) {
				itemTaxGroup = orderTypeTaxGroup;
			}
			else {
				itemTaxGroup = getOutletTaxGroup();
			}
		}
		if (itemTaxGroup == null) {
			return;
		}
		List<Tax> taxes = itemTaxGroup.getTaxes();
		if (taxes != null) {
			List<TicketItemTax> ticketItemTaxes = new ArrayList<>();
			for (Tax tax : taxes) {
				TicketItemTax ticketItemTax = new TicketItemTax();
				ticketItemTax.setId(tax.getId());
				ticketItemTax.setName(tax.getName());
				ticketItemTax.setRate(tax.getRate());
				ticketItemTaxes.add(ticketItemTax);
			}
			ticketItem.setTaxes(ticketItemTaxes);
		}
	}

	private static TaxGroup getOrderBaseTaxGroup(MenuItem menuItem, String orderTypeId, String subOrderType) {
		String orderTypeTaxGroupId = menuItem.getProperty(orderTypeId + subOrderType + JSON_PROP_SUFFIX_TAXGROUP);
		if (StringUtils.isBlank(orderTypeTaxGroupId)) {
			return null;
		}
		return DataProvider.get().getTaxGroupById(orderTypeTaxGroupId);
	}

	public String getStringWithUnderScore(String orderType, String additionalString) {

		orderType = orderType.replaceAll(" ", "_");//$NON-NLS-1$ //$NON-NLS-2$

		return orderType + additionalString;
	}

	@XmlTransient
	public Set<MenuItemSize> getSizes() {
		Set<MenuItemSize> sizes = new HashSet<MenuItemSize>();

		List<PizzaPrice> priceList = getPizzaPriceList();
		if (priceList != null) {
			for (PizzaPrice pizzaPrice : priceList) {
				sizes.add(pizzaPrice.getSize());
			}
		}

		return sizes;
	}

	public Set<PizzaCrust> getCrustsForSize(MenuItemSize size) {
		Set<PizzaCrust> crusts = new HashSet<PizzaCrust>();

		List<PizzaPrice> priceList = getPizzaPriceList();
		if (priceList != null) {
			for (PizzaPrice pizzaPrice : priceList) {
				if (size.equals(pizzaPrice.getSize())) {
					crusts.add(pizzaPrice.getCrust());
				}
			}
		}

		return crusts;
	}

	public Set<PizzaPrice> getAvailablePrices(MenuItemSize size) {
		Set<PizzaPrice> prices = new HashSet<PizzaPrice>();

		List<PizzaPrice> priceList = getPizzaPriceList();
		if (priceList != null) {
			for (PizzaPrice pizzaPrice : priceList) {
				if (size.equals(pizzaPrice.getSize())) {
					prices.add(pizzaPrice);
				}
			}
		}

		return prices;
	}

	public List<IUnit> getUnits() {
		return getUnits(false);
	}

	public List<IUnit> getUnits(boolean includeRecipeUnits) {
		List<IUnit> units = new ArrayList<>();
		InventoryUnit unit = getUnit();
		if (unit != null && unit.getUnitGroup() != null) {
			List<InventoryUnit> groupUnits = unit.getUnitGroup().getUnits();
			if (groupUnits != null && groupUnits.size() > 0) {
				for (Iterator<InventoryUnit> iterator = groupUnits.iterator(); iterator.hasNext();) {
					InventoryUnit groupUnit = (InventoryUnit) iterator.next();
					if (groupUnit.isDeleted()) {
						continue;
					}
					units.add(groupUnit);
				}
			}
		}
		if (getStockUnits() != null) {
			for (InventoryStockUnit stockUnit : getStockUnits()) {
				if (stockUnit.isRecipeUnit() && !includeRecipeUnits) {
					continue;
				}
				units.add(stockUnit.getPackagingUnit());
			}
		}
		return units;
	}

	public Double getCost(IUnit selectedUnit) {
		if (selectedUnit == null) {
			return getCost();
		}
		InventoryUnit unit = getUnit();
		if (unit != null && selectedUnit.getUniqueCode().equals(unit.getCode())) {
			return getCost();
		}
		return InventoryUnitConvertionUtil.calculateCost(getCost(), selectedUnit, this);
	}

	public Double getCost(String unitCode) {
		if (unitCode == null) {
			return getCost();
		}
		InventoryUnit unit = getUnit();
		if (unit != null) {
			InventoryUnitGroup unitGroup = unit.getUnitGroup();
			if (unitGroup != null && unitGroup.getUnits() != null) {
				for (InventoryUnit inventoryUnit : unitGroup.getUnits()) {
					if (inventoryUnit.getCode().equals(unitCode)) {
						return getCost(inventoryUnit);
					}
				}
			}
			List<InventoryStockUnit> stockUnits = getStockUnits();
			if (stockUnits != null && stockUnits.size() > 0) {
				for (InventoryStockUnit itemStockUnit : stockUnits) {
					if (unitCode.equals(itemStockUnit.getPackagingUnit().getCode())) {
						return getCost(itemStockUnit.getPackagingUnit());
					}
				}
			}
		}
		return getCost();
	}

	public double getUnitQuantity(IUnit sourceInventoryUnit, IUnit destinationInventoryUnit) {
		return InventoryUnitConvertionUtil.getUnitQuantity(sourceInventoryUnit, destinationInventoryUnit, this);
	}

	public double getBaseUnitQuantity(String unitCode) {
		return InventoryUnitConvertionUtil.getBaseUnitQuantity(unitCode, this);
	}

	@Override
	public Double getAverageUnitPurchasePrice() {
		Double avgprice = super.getAverageUnitPurchasePrice();
		if (avgprice == 0)
			return getCost();
		return avgprice;
	}

	@Override
	public String getSku() {
		String sku2 = super.getSku();
		if (StringUtils.isEmpty(sku2)) {
			return super.getBarcode();
		}
		return sku2;
	}

	@XmlTransient
	public Double getAvailableUnit() {
		if (stockStatus != null)
			return stockStatus.getAvailableUnit();
		stockStatus = getStockStatus();
		return stockStatus == null ? 0.0 : stockStatus.getAvailableUnit();
	}

	@XmlTransient
	public Double getUnitOnHand() {
		if (stockStatus != null)
			return stockStatus.getUnitOnHand();

		stockStatus = getStockStatus();
		return stockStatus == null ? 0.0 : stockStatus.getUnitOnHand();
	}

	public void setUnitOnHand(Double unitOnHand) {
		if (stockStatus == null) {
			stockStatus = new MenuItemInventoryStatus();
			stockStatus.setId(getId());
		}
		stockStatus.setUnitOnHand(unitOnHand);
	}

	public void setAvailableUnit(Double availableUnit) {
		if (stockStatus == null) {
			stockStatus = new MenuItemInventoryStatus();
			stockStatus.setId(getId());
		}
		stockStatus.setAvailableUnit(availableUnit);
	}

	public Double getRetailPrice() {
		return getPrice();
	}

	@JsonIgnoreProperties
	@XmlTransient
	@Deprecated
	public MenuGroup getParent() {
		String groupId = getMenuGroupId();
		if (StringUtils.isNotEmpty(groupId)) {
			return MenuGroupDAO.getInstance().get(groupId);
		}
		return null;
	}

	public void setParent(MenuGroup menuGroup) {
		setMenuGroup(menuGroup);
	}

	public void setMenuGroup(MenuGroup menuGroup) {
		if (menuGroup == null) {
			setMenuGroupId(null);
			setMenuGroupName(null);
			setMenuCategoryId(null);
			setMenuCategoryName(null);
			setTicketDiscountApplicable(true);
		}
		else {
			setMenuGroupId(menuGroup.getId());
			setMenuGroupName(menuGroup.getDisplayName());
			setMenuCategoryId(menuGroup.getMenuCategoryId());
			setMenuCategoryName(menuGroup.getMenuCategoryName());

			String categoryId = menuGroup.getMenuCategoryId();
			if (StringUtils.isNotBlank(categoryId)) {
				MenuCategory menuCategory = MenuCategoryDAO.getInstance().get(categoryId);
				if (menuCategory != null) {
					setTicketDiscountApplicable(menuCategory.isTicketDiscountApplicable());
				}
			}
		}
	}

	@XmlTransient
	public PrinterGroup getPrinterGroup() {
		return DataProvider.get().getPrinterGroupById(getPrinterGroupId());
	}

	public void setPrinterGroup(PrinterGroup printerGroup) {
		String printerGroupId = null;
		if (printerGroup != null) {
			printerGroupId = printerGroup.getId();
		}
		super.setPrinterGroupId(printerGroupId);
	}

	@XmlTransient
	public InventoryUnit getUnit() {
		return DataProvider.get().getInventoryUnitById(getUnitId());
	}

	public void setUnit(InventoryUnit unit) {
		String unitId = null;
		if (unit != null) {
			unitId = unit.getId();
			setUnitName(unit.getCode());
		}
		super.setUnitId(unitId);
	}

	@XmlTransient
	public ReportGroup getReportGroup() {
		return DataProvider.get().getReportGroupById(getReportGroupId());
	}

	public void setReportGroup(ReportGroup reportGroup) {
		String reportGroupId = null;
		if (reportGroup != null) {
			reportGroupId = reportGroup.getId();
		}
		super.setReportGroupId(reportGroupId);
	}

	@XmlTransient
	public TaxGroup getTaxGroup() {
		TaxGroup taxGroup = DataProvider.get().getTaxGroupById(getTaxGroupId());
		if (taxGroup == null && isVariant() && getParentMenuItemId() != null) {
			taxGroup = getParentMenuItem().getTaxGroup();
		}

		return taxGroup;
	}

	public TaxGroup getTaxGroup(Customer customer) {
		return getTaxGroup();
	}

	public void setTaxGroup(TaxGroup taxGroup) {
		String taxGroupId = null;
		if (taxGroup != null) {
			taxGroupId = taxGroup.getId();
		}
		super.setTaxGroupId(taxGroupId);
	}

	//TODO:
	public void setOrderTypeList(List checkedValues) {
	}

	@XmlTransient
	public MenuItem getParentMenuItem() {
		if (parentMenuItem != null) {
			return parentMenuItem;
		}
		String parentMenuItemId = getParentMenuItemId();
		if (StringUtils.isEmpty(parentMenuItemId))
			return null;
		parentMenuItem = MenuItemDAO.getInstance().get(parentMenuItemId);
		return parentMenuItem;
	}

	public void setParentMenuItem(MenuItem parentMenuItem) {
		this.parentMenuItem = parentMenuItem;

		if (StringUtils.isBlank(super.getParentMenuItemId()) && parentMenuItem != null && StringUtils.isNotBlank(parentMenuItem.getId())) {
			super.setParentMenuItemId(parentMenuItem.getId());
		}
	}

	@XmlTransient
	public MenuItemInventoryStatus getStockStatus() {
		//		if (!isInventoryItem()) {
		//			return null;
		//		}
		if (stockStatus != null || getId() == null)
			return stockStatus;
		stockStatus = DataProvider.get().getMenuItemStockStatus(this);
		return stockStatus;
	}

	public void setStockStatus(MenuItemInventoryStatus stockStatus) {
		this.stockStatus = stockStatus;
	}

	@Override
	public void setPropertiesJson(String propertiesJson) {
		super.setPropertiesJson(propertiesJson);
		if (StringUtils.isNotEmpty(propertiesJson)) {
			properties = new JSONObject(propertiesJson);
		}
		else {
			properties = new JSONObject();
		}
	}

	private void buildPropertis() {
		if (properties != null) {
			return;
		}
		String json = getPropertiesJson();
		if (StringUtils.isEmpty(json)) {
			properties = new JSONObject();
			return;
		}
		properties = new JSONObject(json);
	}

	public void addProperty(String key, String value) {
		buildPropertis();
		properties.put(key, value);
		setPropertiesJson(properties.toString());
	}

	public JSONObject getProperties() {
		return properties;
	}

	public String getProperty(String key) {
		buildPropertis();
		if (properties.has(key)) {
			return properties.getString(key);
		}
		return null;
	}

	public void removeProperty(String key) {
		buildPropertis();
		properties.remove(key);
		setPropertiesJson(properties.toString());
	}

	@Deprecated
	public Course getCourse() {
		String courseId = getCourseId();
		if (StringUtils.isNotEmpty(courseId)) {
			return CourseDAO.getInstance().get(courseId);
		}
		return null;
	}

	public void setCourse(Course course) {
		setCourseId(course == null ? null : course.getId());
	}

	public MenuItem clone() {
		MenuItem newMenuItem = new MenuItem();

		newMenuItem = (MenuItem) SerializationUtils.clone(this);
		String newName = doDuplicateName(getName());
		newMenuItem.setName(newName);
		newMenuItem.setId(null);

		List<InventoryStockUnit> stockUnits = newMenuItem.getStockUnits();
		if (stockUnits != null) {
			for (InventoryStockUnit stockUnit : stockUnits) {
				stockUnit.setId(null);
				stockUnit.setMenuItemId(newMenuItem.getId());
			}
		}
		List<ComboItem> comboItems = newMenuItem.getComboItems();
		newMenuItem.setComboItems(new ArrayList<>());
		if (comboItems != null) {
			for (ComboItem comboItem : comboItems) {
				comboItem.setId(null);
				newMenuItem.getComboItems().add(comboItem);
			}
		}
		List<ComboGroup> comboGroups = newMenuItem.getComboGroups();
		newMenuItem.setComboGroups(new ArrayList<>());
		if (comboGroups != null) {
			for (ComboGroup comboGroup : comboGroups) {
				comboGroup.setId(null);
				List<MenuItem> comboGroupItems = comboGroup.getItems();
				comboGroup.setItems(new ArrayList<>());
				if (comboGroupItems != null) {
					for (MenuItem comboGroupItem : comboGroupItems) {
						comboGroup.getItems().add(comboGroupItem);
					}
				}
				newMenuItem.getComboGroups().add(comboGroup);
			}
		}
		copyModifierSpecsToMenuItem(newMenuItem, newMenuItem.getMenuItemModiferSpecs());
		if (newMenuItem.isPizzaType()) {
			List<PizzaPrice> pizzaPriceList = newMenuItem.getPizzaPriceList();
			if (pizzaPriceList != null) {
				List<PizzaPrice> newPriceList = new ArrayList<>();
				for (PizzaPrice pizzaPrice : pizzaPriceList) {
					PizzaPrice newPrice = (PizzaPrice) SerializationUtils.clone(pizzaPrice);
					newPrice.setId(null);
					newPriceList.add(newPrice);
				}
				newMenuItem.setPizzaPriceList(newPriceList);
			}
		}
		newMenuItem.setVariants(null);
		List<MenuItem> variants = getVariants();
		for (MenuItem variantMenuItem : variants) {
			if (variantMenuItem.isDeleted()) {
				continue;
			}
			MenuItem newVariantMenuItem = (MenuItem) SerializationUtils.clone(variantMenuItem);
			newVariantMenuItem.setId(null);

			newVariantMenuItem.setMenuItemModiferSpecs(null);
			newVariantMenuItem.setComboGroups(null);
			newVariantMenuItem.setComboItems(null);
			newVariantMenuItem.setVariants(null);

			List<Attribute> newAttributies = new ArrayList<Attribute>();
			List<Attribute> attributes2 = variantMenuItem.getAttributes();
			for (Attribute attribute : attributes2) {
				Attribute newVariantAttribute = (Attribute) SerializationUtils.clone(attribute);
				newAttributies.add(newVariantAttribute);
			}
			newVariantMenuItem.setAttributes(newAttributies);

			newMenuItem.addTovariants(newVariantMenuItem);
		}

		return newMenuItem;
	}

	public static void copyModifierSpecsToMenuItem(MenuItem newMenuItem, List<MenuItemModifierSpec> menuItemModifierSpecs) {
		List<MenuItemModifierSpec> newSpecs = new ArrayList<>();
		if (menuItemModifierSpecs != null) {
			for (MenuItemModifierSpec menuModifierSpec : menuItemModifierSpecs) {
				menuModifierSpec.setId(null);
				Set<MenuItemModifierPage> specPages = menuModifierSpec.getModifierPages();
				Set<MenuItemModifierPage> newPages = new HashSet<>();
				if (specPages != null) {
					for (MenuItemModifierPage menuItemModifierPage : specPages) {
						List<MenuItemModifierPageItem> newPageItems = new ArrayList<>();
						menuItemModifierPage.setId(null);
						menuItemModifierPage.setModifierSpecId(menuModifierSpec.getId());
						List<MenuItemModifierPageItem> pageItems = menuItemModifierPage.getPageItems();
						if (pageItems != null) {
							for (MenuItemModifierPageItem menuItemModifierPageItem : pageItems) {
								menuItemModifierPageItem.setId(null);
								menuItemModifierPageItem.setParentPage(menuItemModifierPage);
								newPageItems.add(menuItemModifierPageItem);
							}
						}
						menuItemModifierPage.setPageItems(newPageItems);
						newPages.add(menuItemModifierPage);
					}
				}
				menuModifierSpec.setModifierPages(newPages);
				newSpecs.add(menuModifierSpec);
			}
		}
		newMenuItem.setMenuItemModiferSpecs(newSpecs);
	}

	private static String doDuplicateName(String existingName) {
		String newName = new String();
		int lastIndexOf = existingName.lastIndexOf(" "); //$NON-NLS-1$
		if (lastIndexOf == -1) {
			newName = existingName + " 1"; //$NON-NLS-1$
		}
		else {
			String itemCountProcessName = existingName.substring(lastIndexOf + 1, existingName.length()).trim();
			if (!StringUtils.isEmpty(itemCountProcessName) && StringUtils.isNumeric(itemCountProcessName)) {
				Integer count = Integer.valueOf(itemCountProcessName);
				count += 1;
				newName = existingName.replace(itemCountProcessName, String.valueOf(count));
			}
			else {
				newName = existingName.trim() + " 1"; //$NON-NLS-1$
			}
		}
		return newName;
	}

	public void setUnitSelection(boolean unitSelection) {
		addProperty(AppConstants.ALLOW_UNIT_SELECTION, String.valueOf(unitSelection));
	}

	public boolean isAllowUnitSelection() {
		String isAllowUnitSelection = getProperty(AppConstants.ALLOW_UNIT_SELECTION);
		if (StringUtils.isNotEmpty(isAllowUnitSelection))
			return Boolean.parseBoolean(isAllowUnitSelection);
		return false;
	}

	public String getVendorNames() {
		return vendorNames;
	}

	public void setVendorNames(String vendorName) {
		this.vendorNames = vendorName;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@XmlTransient
	public double getTotalTaxRate() {
		TaxGroup taxGroup = getTaxGroup();
		if (taxGroup != null) {
			return taxGroup.getTotalTaxRate();
		}
		return 0.0;
	}

	public void setTotalTaxRate(double totalTaxRate) {
	}

	@Override
	public void addTomenuItemModiferSpecs(MenuItemModifierSpec menuItemModifierSpec) {
		List<MenuItemModifierSpec> specs = getMenuItemModiferSpecs();
		if (specs == null) {
			specs = new ArrayList<>(1);
			setMenuItemModiferSpecs(specs);
		}
		for (MenuItemModifierSpec spec : specs) {
			if (spec.getModifierGroupId().equals(menuItemModifierSpec.getModifierGroupId())) {
				return;
			}
		}
		specs.add(menuItemModifierSpec);
	}

	@Override
	public String getUnitName() {
		String unitName = super.getUnitName();
		if (StringUtils.isNotEmpty(unitName)) {
			return unitName;
		}
		String unitId = getUnitId();
		if (StringUtils.isNotEmpty(unitId)) {
			InventoryUnit unit = DataProvider.get().getInventoryUnitById(unitId);
			if (unit != null) {
				return unit.getCode();
			}
		}
		return unitName;
	}

	@XmlTransient
	public void setEnableComboQuantitySelection(boolean isEnableQtySelection) {
		this.addProperty(JSON_PROP_ENABLE_QTY_SELECTION, String.valueOf(isEnableQtySelection));
	}

	public boolean isEnableComboQuantitySelection() {
		return POSUtil.getBoolean(getProperty(JSON_PROP_ENABLE_QTY_SELECTION));
	}

	public Double updateUnitBasePrice(String unitCode, Double newPrice) {
		String key = PROP_UNIT_BASE_PRICE_KEY.replace("{name}", unitCode); //$NON-NLS-1$
		addProperty(key, String.valueOf(newPrice));
		return newPrice;
	}

	public void removeUnitBasePrice(String unitCode) {
		String key = PROP_UNIT_BASE_PRICE_KEY.replace("{name}", unitCode); //$NON-NLS-1$
		removeProperty(key);
	}

	/**
	 * This method will return all variant which is not deleted.
	 * 
	 */
	@XmlTransient
	public List<MenuItem> getActiveVariants() {
		List<MenuItem> variants2 = new ArrayList<>();
		if (getVariants() != null) {
			for (Iterator<MenuItem> iterator = getVariants().iterator(); iterator.hasNext();) {
				MenuItem menuItem = iterator.next();
				if (!menuItem.isDeleted() && menuItem.getParentMenuItemId() != null) {
					variants2.add(menuItem);
				}
			}
		}
		return variants2;
	}

	public boolean attributesEqual(MenuItem other) {
		List<Attribute> currentAttributeList = this.getAttributes();
		List<Attribute> otherAttributeList = other.getAttributes();
		if (currentAttributeList == null || otherAttributeList == null) {
			return false;
		}

		if (currentAttributeList.size() != otherAttributeList.size()) {
			return false;
		}

		for (Attribute attribute : currentAttributeList) {
			if (!otherAttributeList.contains(attribute)) {
				return false;
			}
		}
		return true;
	}

	public boolean hasSimilerGroupSpec(ModifierGroup selectedGroup) {
		List<MenuItemModifierSpec> menuItemModiferSpecs = this.getMenuItemModiferSpecs();
		if (menuItemModiferSpecs == null || menuItemModiferSpecs.isEmpty())
			return false;
		for (Iterator<MenuItemModifierSpec> iterator = menuItemModiferSpecs.iterator(); iterator.hasNext();) {
			MenuItemModifierSpec menuItemModifierSpec = (MenuItemModifierSpec) iterator.next();
			if (menuItemModifierSpec.getModifierGroup().getName().equals(selectedGroup.getName())) {
				return true;
			}
		}
		return false;
	}

	public boolean hasSimilerItemInOtherComboGroup(ComboGroup selectedcomboGroup, List<MenuItem> menuItems) {
		List<ComboGroup> comboGroups = getComboGroups();
		if (comboGroups == null) {
			return false;
		}

		for (ComboGroup comboGroup : comboGroups) {
			if (selectedcomboGroup == comboGroup) {
				continue;
			}

			if (menuItems == null || menuItems.isEmpty()) {
				return false;
			}
			List<MenuItem> existedItems = comboGroup.getItems();
			if (existedItems == null || existedItems.isEmpty()) {
				return false;
			}
			for (MenuItem existedItem : existedItems) {
				if (existedItem == null || StringUtils.isEmpty(existedItem.getId())) {
					continue;
				}
				for (MenuItem menuItem : menuItems) {
					if (menuItem != null && StringUtils.isNotEmpty(menuItem.getId()) && existedItem.getId().equals(menuItem.getId())) {
						return true;
					}
				}
			}
		}
		return false;
	}

	public void clearNonExistingUnitBasePriceProperties(List<UnitBaseItemPrice> selectedUnitBasePriceList) {
		try {
			List<String> unitBasePriceKeys = getUnitBasePriceKeys();
			if (!isAllowUnitSelection()) {
				for (String priceKey : unitBasePriceKeys) {
					removeProperty(priceKey);
				}
				return;
			}
			List<String> addedUnitPriceKeys = new ArrayList<>();
			if (selectedUnitBasePriceList != null && !selectedUnitBasePriceList.isEmpty()) {
				final String PRICE_KEY = MenuItem.PROP_UNIT_BASE_PRICE_KEY;
				for (UnitBaseItemPrice unitBaseItemPrice : selectedUnitBasePriceList) {
					String key = PRICE_KEY.replace("{name}", unitBaseItemPrice.getUnitCode()); //$NON-NLS-1$
					addProperty(key, NumberUtil.formatNumber(unitBaseItemPrice.getPrice()));
					addedUnitPriceKeys.add(key);
				}
			}
			if (unitBasePriceKeys.size() > 0) {
				for (String priceKey : unitBasePriceKeys) {
					if (addedUnitPriceKeys.contains(priceKey)) {
						continue;
					}
					removeProperty(priceKey);
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	private List<String> getUnitBasePriceKeys() {
		List<String> unitBasePriceKeys = new ArrayList<>();
		try {
			if (properties != null && properties.length() > 0) {
				Set<String> keySet = properties.keySet();
				for (String key : keySet) {
					if (key.startsWith("unit.") && key.endsWith(".price")) { //$NON-NLS-1$  //$NON-NLS-2$
						unitBasePriceKeys.add(key);
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return unitBasePriceKeys;
	}

	private double getItemCost() {
		double cost = getCost();
		if (isVariant()) {
			if (cost == 0) {
				if (getParentMenuItem() != null) {
					cost = getParentMenuItem().getCost();
				}
			}
		}
		return cost;
	}

	@XmlTransient
	public Double getAdjustedPrice() {
		if (isVariant()) {
			MenuItem variantParent = getParentMenuItem();
			if (variantParent != null) {
				return variantParent.getPrice() + super.getPrice();
			}
		}

		return super.getPrice();
	}

	@XmlTransient
	public String getOnlineDisplayString() {
		String sym = CurrencyUtil.getCurrencySymbol();
		return getDisplayName() + "   " + sym + NumberUtil.formatNumber(getOnlinePrice(outletId));//$NON-NLS-1$
	}

	public void setAdjustedPrice(Double adjustedPrice) {

	}

	public Boolean isHasOnlinePrice() {
		PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online", this);
		if (priceTableItem != null) {
			return Boolean.TRUE;
		}
		return Boolean.FALSE;
	}

	//This method is used only for showing grid online price.
	@XmlTransient
	public Double getOnlinePrice() {
		return getOnlinePrice(outletId);
	}

	public Double getOnlinePrice(String outletId) {
		if (isVariant()) {
			PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online_" + outletId, this);
			if (priceTableItem != null) {
				return priceTableItem.getPrice();
			}
			return getParentMenuItem() == null ? 0.0 : getParentMenuItem().getOnlinePrice(outletId) + getPrice();
		}
		PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online_" + outletId, this);
		if (priceTableItem != null) {
			return priceTableItem.getPrice();
		}
		return getPrice();
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

	public void setOnlinePrice(Double onlinePrice) {
	}

	public void setPriceForPizzaItem(TicketItem newTicketItem) {
		List<MenuItemSize> sizeList = MenuItemSizeDAO.getInstance().findAll();
		List<PizzaCrust> crustList = PizzaCrustDAO.getInstance().findAll();
		TicketItemModifier sizeAndCrustModifer = newTicketItem.getSizeModifier();
		if (sizeAndCrustModifer != null) {
			String sizeAndCrustName = sizeAndCrustModifer.getName().trim();
			String sizeName = ""; //$NON-NLS-1$
			String crustName = ""; //$NON-NLS-1$
			for (MenuItemSize menuItemSize : sizeList) {
				if (sizeAndCrustName.contains(menuItemSize.getName())) {
					if (menuItemSize.getName().length() > sizeName.length()) {
						sizeName = menuItemSize.getName();
					}
				}
			}
			for (PizzaCrust pizzaCrust : crustList) {
				if (sizeAndCrustName.contains(pizzaCrust.getName())) {
					if (pizzaCrust.getName().length() > crustName.length()) {
						crustName = pizzaCrust.getName();
					}
				}
			}
			List<PizzaPrice> pizzaPrices = this.getPizzaPriceList();
			if (pizzaPrices != null) {
				for (PizzaPrice pizzaPrice : pizzaPrices) {
					if (pizzaPrice.getCrust().getName().equalsIgnoreCase(crustName) && pizzaPrice.getSize().getName().equalsIgnoreCase(sizeName)) {
						newTicketItem.setUnitPrice(pizzaPrice.getPrice(this.getDefaultSellPortion()));
						break;
					}
				}
			}
		}
	}

	public void setPriceForPizzaTicketItemModifier(TicketItem ticketItem, TicketItemModifier ticketItemModifier, MenuModifier menuModifier) {
		String multiplierName = ticketItemModifier.getMultiplierName();
		if (StringUtils.isEmpty(multiplierName)) {
			multiplierName = Multiplier.REGULAR;
		}
		List<Multiplier> multiplierList = DataProvider.get().getMultiplierList();
		Multiplier selectedMultiplier = null;
		for (Multiplier multiplier : multiplierList) {
			if (multiplier.getId().equals(multiplierName)) {
				selectedMultiplier = multiplier;
			}
		}
		String selectedSection = ticketItemModifier.getSectionName();
		TicketItemModifier sizeAndCrustModifer = ticketItem.getSizeModifier();
		MenuItemSize menuItemSize = null;
		if (sizeAndCrustModifer != null) {
			String sizeAndCrustName = sizeAndCrustModifer.getName().trim();
			String sizeName = ""; //$NON-NLS-1$
			List<MenuItemSize> sizeList = MenuItemSizeDAO.getInstance().findAll();
			if (sizeList != null) {
				for (MenuItemSize itemSize : sizeList) {
					if (sizeAndCrustName.contains(itemSize.getName())) {
						if (itemSize.getName().length() > sizeName.length()) {
							sizeName = itemSize.getName();
						}
					}
				}
			}

			List<PizzaPrice> pizzaPrices = ticketItem.getMenuItem().getPizzaPriceList();
			if (pizzaPrices != null) {
				for (PizzaPrice pizzaPrice : pizzaPrices) {
					if (pizzaPrice.getSize().getName().equalsIgnoreCase(sizeName)) {
						menuItemSize = pizzaPrice.getSize();
						break;
					}
				}
			}
		}
		MenuModifierDAO.getInstance().initialize(menuModifier);
		//ticketItemModifier.setTaxIncluded(Application.getInstance().isPriceIncludesTax());
		MenuItemModifierSpec menuItemModifierGroup = menuModifier.getMenuItemModifierGroup();
		if (menuItemModifierGroup != null) {
			ticketItemModifier.setGroupId(menuItemModifierGroup.getId());
		}

		double priceForSize = menuModifier.getPriceForSizeAndMultiplier(menuItemSize, false, selectedMultiplier);
		double costForSize = menuModifier.getCost() * (selectedMultiplier.getRate() / 100.0);
		ticketItemModifier.setUnitPrice(priceForSize);
		ticketItemModifier.setUnitCost(costForSize);

		if (!ticketItemModifier.isInfoOnly()) {
			double defaultSellPortion = ticketItem.getMenuItem().getDefaultSellPortion();
			ticketItemModifier.setUnitPrice(ticketItemModifier.getUnitPrice() * defaultSellPortion / 100);
		}

		ticketItemModifier.setShouldSectionWisePrice(menuModifier.isShouldSectionWisePrice());
		if (ticketItemModifier.isShouldSectionWisePrice()) {
			ticketItemModifier.setUnitPrice(getPriceForSection(selectedSection, ticketItemModifier.getUnitPrice()));
		}
	}

	private Double getPriceForSection(String sectionName, double unitPrice) {
		switch (sectionName) {
			case "WHOLE": //$NON-NLS-1$
				return unitPrice * 1;
			case "Quarter 1": //$NON-NLS-1$
				return unitPrice * 0.25;
			case "Quarter 2": //$NON-NLS-1$
				return unitPrice * 0.25;
			case "Quarter 3": //$NON-NLS-1$
				return unitPrice * 0.25;
			case "Quarter 4": //$NON-NLS-1$
				return unitPrice * 0.25;
			case "Half 1": //$NON-NLS-1$
				return unitPrice * 0.5;
			case "Half 2": //$NON-NLS-1$
				return unitPrice * 0.5;
		}
		return null;
	}

	@Id
	@Override
	public String getId() {
		return super.getId();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.YES, index = Index.YES)
	@Override
	public String getName() {
		return super.getName();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@Override
	public String getDescription() {
		return super.getDescription();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@XmlTransient
	public String getStoreId() {
		if (StringUtils.isBlank(HibernateSearchUtil.getCurrentStoreId())) {
			return DataProvider.get().getStore().getUuid();
		}

		return HibernateSearchUtil.getCurrentStoreId();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isEnableOnlineOrdering() {
		return super.isEnableOnlineOrdering();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isDeleted() {
		return super.isDeleted();
	}

	@XmlTransient
	public Double getVariantPrice() {
		if (isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getPrice() + super.getPrice();
			}
		}
		return super.getPrice();
	}

	public void setVariantPrice(Double price) {
		//no use
	}

	@XmlTransient
	public Double getVariantCost() {
		Double cost = super.getCost();
		if (cost == 0 && isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getCost();
			}
		}
		return cost;
	}

	public void setVariantCost(Double price) {
		//no use
	}

	@XmlTransient
	public double getProfitMargin() {
		double profit = getVariantPrice() - getVariantCost();
		double abs = Math.abs(getVariantPrice());
		if (abs > 0) {
			return NumberUtil.round(profit / abs * 100);
		}
		else {
			return 0d;
		}
	}

	public void setProfitMargin(double profitMargin) {
	}

	public void setModifiablePriceForComboItem(boolean isModifiable) {
		this.addProperty(JSON_PROP_MOD_PRICE_COMBO_ITEM, String.valueOf(isModifiable));
	}

	@XmlTransient
	public boolean isModifiablePriceForComboItem() {
		return POSUtil.getBoolean(getProperty(JSON_PROP_MOD_PRICE_COMBO_ITEM));
	}

	public void setBasePriceForComboItem(boolean value) {
		this.addProperty("base.priced.comboItem", String.valueOf(value)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isBasePriceForComboItem() {
		return POSUtil.getBoolean(getProperty("base.priced.comboItem")); //$NON-NLS-1$
	}

	public void setBaseAndGroupPriceForComboItem(boolean value) {
		this.addProperty("base_and_group.priced.comboItem", String.valueOf(value)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isBaseAndGroupPriceForComboItem() {
		return POSUtil.getBoolean(getProperty("base_and_group.priced.comboItem")); //$NON-NLS-1$
	}

	public void setHighestPricedComboItem(boolean highestPricedComboItem) {
		this.addProperty("highest.priced.comboItem", String.valueOf(highestPricedComboItem)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isHighestPricedComboItem() {
		return POSUtil.getBoolean(getProperty("highest.priced.comboItem")); //$NON-NLS-1$
	}

}