/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.sql.JoinType;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Department;
import com.floreantpos.model.Discount;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuPage;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.OnlineCategory;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.ShiftUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class MenuCategoryDAO extends BaseMenuCategoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public MenuCategoryDAO() {
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(MenuCategory.PROP_SORT_ORDER);
	}

	@Override
	public Serializable save(Object menuCategory, Session session) throws HibernateException {
		//save the menu category
		updateTime(menuCategory);
		Serializable id = super.save(menuCategory, session);

		//update dependent models
		updateDependentModels((MenuCategory) menuCategory, session);

		return id;
	}

	@Override
	public void update(Object menuCategory, Session session) throws HibernateException {
		//save the menu category
		updateTime(menuCategory);
		super.update(menuCategory, session);

		//update dependent models
		updateDependentModels((MenuCategory) menuCategory, session);
	}

	@Override
	public void saveOrUpdate(Object menuCategory, Session session) throws HibernateException {
		//save the menu category
		updateTime(menuCategory);
		session.saveOrUpdate(menuCategory);

		//update dependent models
		updateDependentModels((MenuCategory) menuCategory, session);
	}

	@Override
	public void delete(Object obj, Session s) throws HibernateException {
		MenuCategory menuCategory = (MenuCategory) obj;
		if (menuCategory == null) {
			throw new PosException(Messages.getString("MenuCategoryDAO.0")); //$NON-NLS-1$
		}
		checkIfCategoryCanbeDeleted(s, menuCategory);
		
		menuCategory.setDeleted(true);
		menuCategory.setBeverage(false);
		menuCategory.setDepartments(null);
		menuCategory.setMenuGroups(null);
		menuCategory.setMenuShifts(null);
		menuCategory.setOrderTypes(null);
		menuCategory.setTerminalTypes(null);
		removeFromDependentModels(menuCategory, s);
		update(menuCategory, s);
		removeFromDiscounts(menuCategory, s);
	}

	private void checkIfCategoryCanbeDeleted(Session s, MenuCategory menuCategory) {
		String details = ""; //$NON-NLS-1$;
		
		List<MenuGroup> menuGroupList = MenuGroupDAO.getInstance().findMenuGroupByCategoryId(menuCategory.getId());
		if (menuGroupList != null && !menuGroupList.isEmpty()) {
			details = "Category " + menuCategory.getName() + " is being used by the following menu group:";
			int count = 1;
			for (MenuGroup group : menuGroupList) {
				details += "\n" + count + ". " + group.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		
		List<MenuItem> menuItemList = MenuItemDAO.getInstance().findMenuItemByCategoryId(menuCategory.getId());
		if (menuItemList != null && !menuItemList.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n"; //$NON-NLS-1$
			}
			details += "Category " + menuCategory.getName() + " is being used by the following menu item:";
			int count = 1;
			for (MenuItem menuItem : menuItemList) {
				details += "\n" + count + ". " + menuItem.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		
		
		if (StringUtils.isNotBlank(details)) {
			throw new PosException(Messages.getString("MenuItemDAO.2"), details);//$NON-NLS-1$
		}
	}

	public List<MenuCategory> findAll() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass(), "category"); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	private void updateDependentModels(MenuCategory menuCategory, Session session) {
		//@formatter:off
		//update menu group's category id and name
		String hqlString = "update MenuGroup set %s=:categoryName, %s=:beverage where %s=:categoryId"; //$NON-NLS-1$
		hqlString = String.format(hqlString, MenuGroup.PROP_MENU_CATEGORY_NAME, MenuGroup.PROP_BEVERAGE, MenuGroup.PROP_MENU_CATEGORY_ID);
		Query query = session.createQuery(hqlString);
		query.setParameter("categoryName", menuCategory.getDisplayName()); //$NON-NLS-1$
		query.setParameter("beverage", menuCategory.isBeverage()); //$NON-NLS-1$
		query.setParameter("categoryId", menuCategory.getId()); //$NON-NLS-1$
		query.executeUpdate();

		//update menu item's category name and id
		hqlString = "update MenuItem set %s=:categoryName, %s=:beverage where %s=:categoryId"; //$NON-NLS-1$
		hqlString = String.format(hqlString, MenuItem.PROP_MENU_CATEGORY_NAME, MenuItem.PROP_BEVERAGE, MenuItem.PROP_MENU_CATEGORY_ID);
		//@formatter:on
		query = session.createQuery(hqlString);
		query.setParameter("categoryName", menuCategory.getDisplayName()); //$NON-NLS-1$
		query.setParameter("beverage", menuCategory.isBeverage()); //$NON-NLS-1$
		query.setParameter("categoryId", menuCategory.getId()); //$NON-NLS-1$
		query.executeUpdate();

		List<OnlineCategory> onlineCategories = OnlineCategoryDAO.getInstance().findByCategoryId(menuCategory.getId(), session);
		if (onlineCategories != null && onlineCategories.size() > 0) {
			for (OnlineCategory onlineCategory : onlineCategories) {
				if (menuCategory.isDeleted()) {
					OnlineCategoryDAO.getInstance().delete(onlineCategory, session);
				}
				else {
					onlineCategory.setName(menuCategory.getName());
					OnlineCategoryDAO.getInstance().update(onlineCategory, session);
				}
			}
		}
	}

	private void removeFromDiscounts(MenuCategory category, Session session) {
		DiscountDAO discountDAO = DiscountDAO.getInstance();
		List<Discount> discounts = discountDAO.getDiscountsByMenuCategory(category, session);
		if (discounts != null && !discounts.isEmpty()) {
			for (Discount discount : discounts) {
				List<MenuCategory> menuCategories = discount.getMenuCategories();
				menuCategories.remove(category);
				discountDAO.saveOrUpdate(discount, session);
			}
		}
	}

	private void removeFromDependentModels(MenuCategory menuCategory, Session session) {
		String menuCategoryId = menuCategory.getId();
		String hqlString = "update MenuItem set " + //$NON-NLS-1$
				MenuItem.PROP_MENU_CATEGORY_ID + "= null, " + //$NON-NLS-1$
				MenuItem.PROP_MENU_CATEGORY_NAME + "= null, %s=:beverage where %s=:categoryId"; //$NON-NLS-1$

		hqlString = String.format(hqlString, MenuItem.PROP_BEVERAGE, MenuItem.PROP_MENU_CATEGORY_ID);
		Query query = session.createQuery(hqlString);
		query.setParameter("beverage", false); //$NON-NLS-1$
		query.setParameter("categoryId", menuCategoryId); //$NON-NLS-1$
		query.executeUpdate();

		String hqlMenuGroup = "update MenuGroup set " + //$NON-NLS-1$
				MenuGroup.PROP_MENU_CATEGORY_ID + "= null, " + //$NON-NLS-1$
				MenuGroup.PROP_MENU_CATEGORY_NAME + "= null where %s=:categoryId"; //$NON-NLS-1$

		hqlMenuGroup = String.format(hqlMenuGroup, MenuGroup.PROP_MENU_CATEGORY_ID);
		Query menuGroupQuery = session.createQuery(hqlMenuGroup);
		menuGroupQuery.setParameter("categoryId", menuCategoryId); //$NON-NLS-1$
		menuGroupQuery.executeUpdate();
	}

	public List<MenuCategory> findAllEnable(Department department) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass(), "category"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));

			if (department != null) {
				criteria.createAlias("category.departments", "department"); //$NON-NLS-1$ //$NON-NLS-2$
				criteria.add(Restrictions.eq("department.id", department.getId())); //$NON-NLS-1$
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public List<MenuCategory> findOnlineCategories(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass(), "category"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));

			DetachedCriteria onlineCategoriesCriteria = DetachedCriteria.forClass(OnlineCategory.class);
			onlineCategoriesCriteria.setProjection(Property.forName(OnlineCategory.PROP_CATEGORY_ID));
			onlineCategoriesCriteria.add(Restrictions.eq(OnlineCategory.PROP_OUTLET_ID, outletId));
			criteria.add(Property.forName(MenuCategory.PROP_ID).in(onlineCategoriesCriteria));

			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			return criteria.list();
		}
	}

	public List<MenuCategory> findActiveCategories(OrderType orderType) {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		Department department = terminal.getDepartment();
		List<MenuShift> menuShift = ShiftUtil.getCurrentMenuShifts();
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = setRestrictions(orderType, terminal, department, menuShift, session);
			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<String> findActiveCategoryIds(OrderType orderType) {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		Department department = terminal.getDepartment();
		List<MenuShift> menuShift = ShiftUtil.getCurrentMenuShifts();
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = setRestrictions(orderType, terminal, department, menuShift, session);
			criteria.setProjection(Projections.property(MenuCategory.PROP_ID));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public void findCategoriesForCategoryView(PaginatedListModel listModel, OrderType orderType) {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		Department department = terminal.getDepartment();
		List<MenuShift> menuShifts = ShiftUtil.getCurrentMenuShifts();

		Session session = null;
		try {
			//@formatter:off
			session = createNewSession();
			Criteria criteria = setRestrictions(orderType, terminal, department, menuShifts, session);
			//do not load empty categories
			criteria.add(Restrictions.isNotEmpty("menuGroups")); //$NON-NLS-1$
			criteria.createAlias("menuGroups", "menuGroup"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.isNotEmpty("menuGroup.menuPages")); //$NON-NLS-1$
			if (orderType != null && orderType.getId() != null) {
				criteria.createAlias("menuGroup.menuPages", "menuPage"); //$NON-NLS-1$ //$NON-NLS-2$
				Criterion orderIdNull = Restrictions.isNull("menuPage." + MenuPage.PROP_ORDER_TYPE_ID); //$NON-NLS-1$
				Criterion orderIdEqual = Restrictions.eq("menuPage." + MenuPage.PROP_ORDER_TYPE_ID, orderType.getId()); //$NON-NLS-1$
				criteria.add(Restrictions.or(orderIdNull, orderIdEqual)); //NON-NLS-1$
			}
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			List list = criteria.list();
			listModel.setNumRows(list.size());
			listModel.setPageSize(listModel.getNumRows());
			listModel.setData(list);
		} finally {
			closeSession(session);
		}
	}

	private Criteria setRestrictions(OrderType orderType, Terminal terminal, Department department, List<MenuShift> menuShifts, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass(), "c"); //$NON-NLS-1$

		if (orderType != null) {
			criteria.createAlias("c.orderTypes", "orderType", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.or(Restrictions.isEmpty("c.orderTypes"), //$NON-NLS-1$
					Restrictions.in("orderType.id", Arrays.asList(orderType.getId())) //$NON-NLS-1$
			));
		}
		if (terminal.getTerminalType() != null) {
			criteria.createAlias("c.terminalTypes", "terminalType", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.or(Restrictions.isEmpty("c.terminalTypes"), //$NON-NLS-1$
					Restrictions.in("terminalType.id", Arrays.asList(terminal.getTerminalType().getId())) //$NON-NLS-1$
			));
		}
		if (department != null) {
			criteria.createAlias("c.departments", "department", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.or(Restrictions.isEmpty("c.departments"), //$NON-NLS-1$
					Restrictions.in("department.id", Arrays.asList(department.getId())) //$NON-NLS-1$
			));
		}
		if (menuShifts != null) {
			criteria.createAlias("c.menuShifts", "menuShifts", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
			if(menuShifts.size()>0) {
				List<String> shiftIds=new ArrayList();
				for (MenuShift menuShift : menuShifts) {
					shiftIds.add(menuShift.getId());
				}
				criteria.add(Restrictions.or(Restrictions.isEmpty("c.menuShifts"), //$NON-NLS-1$
						Restrictions.in("menuShifts.id", shiftIds) //$NON-NLS-1$
				));
			}
			else {
				criteria.add(Restrictions.isEmpty("c.menuShifts")); //$NON-NLS-1$
			}
		}
		//@formatter:on
		criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		return criteria;
	}

	//	public List<MenuCategory> findNonBevegares() {
	//		Session session = null;
	//		
	//		try {
	//			session = getSession();
	//			Criteria criteria = session.createCriteria(getReferenceClass());
	//			criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
	//			criteria.add(Restrictions.or(Restrictions.isNull(MenuCategory.PROP_BEVERAGE), Restrictions.eq(MenuCategory.PROP_BEVERAGE, Boolean.FALSE)));
	//			return criteria.list();
	//		} finally {
	//			closeSession(session);
	//		}
	//	}
	//	
	//	public List<MenuCategory> findBevegares() {
	//		Session session = null;
	//		
	//		try {
	//			session = getSession();
	//			Criteria criteria = session.createCriteria(getReferenceClass());
	//			criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
	//			criteria.add(Restrictions.eq(MenuCategory.PROP_BEVERAGE, Boolean.TRUE));
	//			return criteria.list();
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public MenuCategory initialize(MenuCategory menuCategory) {
		if (menuCategory == null || menuCategory.getId() == null)
			return menuCategory;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(menuCategory);

			Hibernate.initialize(menuCategory.getDepartments());
			Hibernate.initialize(menuCategory.getOrderTypes());
			Hibernate.initialize(menuCategory.getMenuShifts());
			Hibernate.initialize(menuCategory.getMenuGroups());

			return menuCategory;
		} finally {
			closeSession(session);
		}
	}

	public String getNameById(String id) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(MenuCategory.PROP_NAME));
			criteria.add(Restrictions.eq(MenuCategory.PROP_ID, id));
			criteria.setMaxResults(1);
			return (String) criteria.uniqueResult();
		}
	}

	public void performBatchSaveOrUpdate(MenuCategory menuCategory, List<TerminalType> terminalTypes) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			saveOrUpdate((Object) menuCategory, session);
			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void releaseParentAndDelete(MenuCategory category) {
		if (category == null) {
			return;
		}

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			String queryString = "delete from MENUCATEGORY_DISCOUNT where MENUCATEGORY_ID='%s'"; //$NON-NLS-1$
			queryString = String.format(queryString, category.getId());
			Query query = session.createSQLQuery(queryString);
			query.executeUpdate();

			String queryString2 = "delete from CAT_DEPT where CATEGORY_ID='%s'"; //$NON-NLS-1$
			queryString2 = String.format(queryString2, category.getId());
			Query query2 = session.createSQLQuery(queryString2);
			query2.executeUpdate();

			String queryString3 = "delete from CATEGORY_ORDER_TYPE where CATEGORY_ID='%s'"; //$NON-NLS-1$
			queryString3 = String.format(queryString3, category.getId());
			Query query3 = session.createSQLQuery(queryString3);
			query3.executeUpdate();

			String queryString4 = "delete from TERMINAL_TYPE_CATEGORY where MENU_CATEGORY_ID='%s'"; //$NON-NLS-1$
			queryString4 = String.format(queryString4, category.getId());
			Query query4 = session.createSQLQuery(queryString4);
			query4.executeUpdate();

			String queryString5 = "update MENU_GROUP set CATEGORY_ID=null where CATEGORY_ID='%s'"; //$NON-NLS-1$
			queryString5 = String.format(queryString5, category.getId());
			Query query5 = session.createSQLQuery(queryString5);
			query5.executeUpdate();

			session.delete(category);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}

	}

	public void findCategories(PaginationSupport model, String name) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(name)) {
				criteria.add(Restrictions.ilike(MenuCategory.PROP_NAME, name, MatchMode.ANYWHERE));
			}
			model.setNumRows(rowCount(criteria));

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuCategory.PROP_NAME).ignoreCase());
			model.setRows(criteria.list());
		}
	}

	public void findCategories(PaginationSupport paginationSupport, boolean includeHiddenItems, String... fields) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass(), "c"); //$NON-NLS-1$
			if (!includeHiddenItems) {
				criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			if (fields != null && fields.length > 0) {
				ProjectionList projectionList = Projections.projectionList();
				for (String field : fields) {
					projectionList.add(Projections.property(field), field);
				}
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(MenuCategory.class));
				paginationSupport.setRows(criteria.list());
			}
			else {
				paginationSupport.setRows(criteria.list());
			}
		} finally {
			closeSession(session);
		}
	}

	/**
	 * Find Categories By Exact Name 
	 */
	public MenuCategory findCategoriesByName(String name) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuCategory.PROP_NAME, name));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));

			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (MenuCategory) list.get(0);
			}
			else {
				return null;
			}
		} finally {
			closeSession(session);
		}
	}

	public List<MenuCategory> findAllUnSyncMenuCategory() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			Criterion falseCriteria = Restrictions.eq(MenuCategory.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(MenuCategory.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateMenuCategories(String outletId, List<MenuCategory> dataList, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator<MenuCategory> iterator = dataList.iterator(); iterator.hasNext();) {
				MenuCategory item = (MenuCategory) iterator.next();
				MenuCategory existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					existingItem = MenuCategoryDAO.getInstance().initialize(existingItem);
					List<MenuGroup> groups = existingItem.getMenuGroups();
					Set<TerminalType> terminalTypes = existingItem.getTerminalTypes();
					long version = existingItem.getVersion();

					loadMenuCategoryCollections(outletId, item, existingItem.getDepartments(), existingItem.getOrderTypes(), existingItem.getMenuShifts());
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					existingItem.setMenuGroups(groups);
					existingItem.setTerminalTypes(terminalTypes);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					loadMenuCategoryCollections(outletId, item, item.getDepartments(), item.getOrderTypes(), item.getMenuShifts());
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void loadMenuCategoryCollections(String outletId, MenuCategory sourceCategory, Set<Department> departments, Set<OrderType> orderTypes,
			Set<MenuShift> menuShifts) throws Exception {

		loadDepartments(sourceCategory, departments);
		loadOrderTypes(outletId, sourceCategory, orderTypes);
		loadMenuShifts(outletId, sourceCategory, menuShifts);
	}

	private void loadMenuShifts(String outletId, MenuCategory sourceCategory, Set<MenuShift> existingMenuShifts) {
		List<String> menuShiftIds = sourceCategory.getMenuShiftIds();
		if (menuShiftIds == null) {
			menuShiftIds = new ArrayList<>();
		}
		if (!isEmpty(existingMenuShifts)) {
			for (Iterator<MenuShift> iterator = existingMenuShifts.iterator(); iterator.hasNext();) {
				MenuShift existingMenuShift = (MenuShift) iterator.next();
				if (!existingMenuShift.getOutletId().equals(outletId)) {
					continue;
				}
				if (!menuShiftIds.contains(existingMenuShift.getId())) {
					iterator.remove();
				}
				else {
					menuShiftIds.remove(existingMenuShift.getId());
				}
			}
		}
		if (existingMenuShifts == null) {
			existingMenuShifts = new HashSet<>();
		}
		for (String menuShiftId : menuShiftIds) {
			existingMenuShifts.add(new MenuShift(menuShiftId, outletId));
		}
		sourceCategory.setMenuShifts(existingMenuShifts);
	}

	private void loadOrderTypes(String outletId, MenuCategory sourceCategory, Set<OrderType> existingOrderTypes) {
		List<String> orderTypeIds = sourceCategory.getOrderTypeIds();
		if (orderTypeIds == null) {
			orderTypeIds = new ArrayList<>();
		}
		if (!isEmpty(existingOrderTypes)) {
			for (Iterator<OrderType> iterator = existingOrderTypes.iterator(); iterator.hasNext();) {
				OrderType existingOrderType = (OrderType) iterator.next();
				if (!existingOrderType.getOutletId().equals(outletId)) {
					continue;
				}
				if (!orderTypeIds.contains(existingOrderType.getId())) {
					iterator.remove();
				}
				else {
					orderTypeIds.remove(existingOrderType.getId());
				}
			}
		}
		if (existingOrderTypes == null) {
			existingOrderTypes = new HashSet<>();
		}
		for (String orderTypeId : orderTypeIds) {
			existingOrderTypes.add(new OrderType(orderTypeId, outletId));
		}
		sourceCategory.setOrderTypes(existingOrderTypes);
	}

	private void loadDepartments(MenuCategory sourceCategory, Set<Department> departments) {
		List<String> departmentIds = sourceCategory.getDepartmentIds();
		if (isEmpty(departmentIds)) {
			sourceCategory.setDepartments(null);
			return;
		}
		if (!isEmpty(departments)) {
			for (Iterator<Department> iterator = departments.iterator(); iterator.hasNext();) {
				Department existingDept = (Department) iterator.next();
				if (!departmentIds.contains(existingDept.getId())) {
					iterator.remove();
				}
				else {
					departmentIds.remove(existingDept.getId());
				}
			}
		}
		if (departments == null) {
			departments = new HashSet<>();
		}
		for (String departmentId : departmentIds) {
			departments.add(new Department(departmentId));
		}
		sourceCategory.setDepartments(departments);
	}

	private boolean isEmpty(Collection<?> collection) {
		return collection == null || collection.size() == 0;
	}

	/**
	 * Search Categories By Name (MatchMode.ANYWHERE)
	 */
	public void findCategoriesByName(PaginationSupport paginationSupport, String itemName, boolean active) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (active) {
				criteria.add(Restrictions.eq(MenuCategory.PROP_VISIBLE, Boolean.TRUE));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuCategory.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));

			paginationSupport.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public List<MenuCategory> findByMenuShiftId(String shiftId, Session session) {
		if (StringUtils.isBlank(shiftId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("menuShifts", "menuShift"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("menuShift.id", Arrays.asList(new String[] { shiftId }))); //$NON-NLS-1$
		criteria.setProjection(Projections.alias(Projections.property(MenuCategory.PROP_NAME), MenuCategory.PROP_NAME));
		criteria.setResultTransformer(Transformers.aliasToBean(MenuCategory.class));
		return criteria.list();
	}

}