/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.DefaultMenuModifier;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierPage;
import com.floreantpos.model.MenuItemModifierPageItem;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PizzaModifierPrice;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginationSupport;

public class MenuModifierDAO extends BaseMenuModifierDAO {
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public MenuModifierDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private String constructExceptionDetailsByModifierGroup(MenuModifier menuModifier, List<ModifierGroup> modifierGroups) {
		if (modifierGroups != null && !modifierGroups.isEmpty()) {
			StringBuilder builder = new StringBuilder(menuModifier.getName() + " " + Messages.getString("MenuModifierDAO.1")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < modifierGroups.size(); i++) {
				String message = (i + 1) + "." + " " + modifierGroups.get(i).getName(); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByModifier(MenuModifier menuModifier, List<String> itemNames) {
		if (itemNames != null && !itemNames.isEmpty()) {
			StringBuilder builder = new StringBuilder(menuModifier.getName() + " " + Messages.getString("MenuModifierDAO.2")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < itemNames.size(); i++) {
				String message = (i + 1) + "." + " " + itemNames.get(i); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByDefaultMenuModifier(MenuModifier menuModifier, List<String> menuItemNames) {
		if (menuItemNames != null && !menuItemNames.isEmpty()) {
			StringBuilder builder = new StringBuilder(menuModifier.getName() + " " + Messages.getString("MenuModifierDAO.3")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < menuItemNames.size(); i++) {
				String message = (i + 1) + "." + " " + menuItemNames.get(i); //$NON-NLS-1$ //$NON-NLS-2$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	protected void delete(Object obj, Session session) {
		MenuModifier menuModifier = (MenuModifier) obj;
		if (menuModifier == null) {
			throw new PosException(Messages.getString("MenuModifierDAO.0")); //$NON-NLS-1$
		}
		session.refresh(obj);
		StringBuilder details = new StringBuilder();

		List<ModifierGroup> modifierGroups = getModifierGroupByModifier(menuModifier, session);
		if (modifierGroups != null && !modifierGroups.isEmpty()) {
			details.append(constructExceptionDetailsByModifierGroup(menuModifier, modifierGroups));
			details.append(EMPTY_NEWLINE_STRING);
			details.append(EMPTY_NEWLINE_STRING);
		}

		List<String> menuItemNames = getMenuItemNamesByModifier(menuModifier, session);
		if (menuItemNames != null && !menuItemNames.isEmpty()) {
			details.append(constructExceptionDetailsByModifier(menuModifier, menuItemNames));
			details.append(EMPTY_NEWLINE_STRING);
			details.append(EMPTY_NEWLINE_STRING);
		}

		List<String> defaultMenuModifiers = getMenuItemNamesByDefaultModifier(menuModifier, session);
		if (defaultMenuModifiers != null && !defaultMenuModifiers.isEmpty()) {
			details.append(constructExceptionDetailsByDefaultMenuModifier(menuModifier, defaultMenuModifiers));
			details.append(EMPTY_NEWLINE_STRING);
		}

		String message = "Selected menu modifier is being used by others and cannot be deleted. Please select details to see who is using this Menu Modifier."; //$NON-NLS-1$
		String detailsMessage = details.toString();
		if (StringUtils.isNotBlank(detailsMessage)) {
			throw new PosException(message, detailsMessage);
		}

		menuModifier.setDeleted(true);
		update(menuModifier, session);
	}

	@SuppressWarnings("unchecked")
	private List<ModifierGroup> getModifierGroupByModifier(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(ModifierGroup.class);
		this.addDeletedFilter(criteria);
		criteria.createAlias("modifiers", "modifier"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("modifier." + MenuModifier.PROP_ID, menuModifier.getId())); //$NON-NLS-1$
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	private List<String> getMenuItemNamesByModifier(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		this.addDeletedFilter(criteria, MenuItem.class);
		criteria.createAlias("menuItemModiferSpecs", "ms"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.createAlias("ms.modifierPages", "mp"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.createAlias("mp.pageItems", "i"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("i." + MenuItemModifierPageItem.PROP_MENU_MODIFIER_ID, menuModifier.getId())); //$NON-NLS-1$
		criteria.setProjection(Projections.distinct(Projections.property(MenuItem.PROP_NAME)));
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	private List<String> getMenuItemNamesByDefaultModifier(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		this.addDeletedFilter(criteria, MenuItem.class);
		Criteria specs = criteria.createCriteria("menuItemModiferSpecs"); //$NON-NLS-1$
		specs.createAlias("defaultModifierList", "dml"); //$NON-NLS-1$ //$NON-NLS-2$
		specs.add(Restrictions.in("dml." + DefaultMenuModifier.PROP_MODIFIER, menuModifier)); //$NON-NLS-1$
		criteria.setProjection(Projections.distinct(Projections.property(MenuItem.PROP_NAME)));
		return criteria.list();

	}

	private void deleteModifierFromModifierPageItem(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(MenuItemModifierPageItem.class);
		criteria.add(Restrictions.eq(MenuItemModifierPageItem.PROP_MENU_MODIFIER_ID, menuModifier.getId()));
		List<MenuItemModifierPageItem> pageItems = criteria.list();
		if (pageItems == null || pageItems.isEmpty()) {
			return;
		}
		for (MenuItemModifierPageItem pageItem : pageItems) {
			deletePageItemFromPage(pageItem, session);
		}
	}

	private void deletePageItemFromPage(MenuItemModifierPageItem pageItem, Session session) {
		Criteria criteria = session.createCriteria(MenuItemModifierPage.class);
		criteria.createAlias("pageItems", "pageItem"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("pageItem." + MenuItemModifierPage.PROP_ID, pageItem.getId())); //$NON-NLS-1$
		List<MenuItemModifierPage> pages = criteria.list();
		for (MenuItemModifierPage page : pages) {
			page.getPageItems().remove(pageItem);
			MenuItemModifierPageDAO.getInstance().update(page, session);
		}
	}

	private void deleteModifierFromDefaultModifierList(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(MenuItemModifierSpec.class);
		criteria.createAlias("defaultModifierList", "m"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.eq("m.modifier", menuModifier)); //$NON-NLS-1$
		List<MenuItemModifierSpec> modifierSpecs = criteria.list();
		if (modifierSpecs == null || modifierSpecs.isEmpty()) {
			return;
		}
		for (MenuItemModifierSpec itemModifierSpec : modifierSpecs) {
			List<DefaultMenuModifier> defaultModifiers = itemModifierSpec.getDefaultModifierList();
			if (defaultModifiers != null && defaultModifiers.size() > 0) {
				for (Iterator iterator = defaultModifiers.iterator(); iterator.hasNext();) {
					DefaultMenuModifier menuModifier2 = (DefaultMenuModifier) iterator.next();
					if (menuModifier2.getModifier().getId().equals(menuModifier.getId()))
						iterator.remove();
				}
			}
			MenuItemModifierSpecDAO.getInstance().update(itemModifierSpec, session);
		}
	}

	private void deleteModifierFromModifierGroup(MenuModifier menuModifier, Session session) {
		Criteria criteria = session.createCriteria(ModifierGroup.class);
		criteria.createAlias("modifiers", "modifier"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("modifier." + MenuModifier.PROP_ID, Arrays.asList(menuModifier.getId()))); //$NON-NLS-1$
		List<ModifierGroup> groups = criteria.list();
		if (groups == null || groups.isEmpty()) {
			return;
		}
		for (ModifierGroup modifierGroup : groups) {
			modifierGroup.getModifiers().remove(menuModifier);
			ModifierGroupDAO.getInstance().update(modifierGroup, session);
		}
	}

	//	public void initialize(MenuModifier modifier) {
	//		if (modifier == null || modifier.getId() == null) {
	//			return;
	//		}
	//		if (Hibernate.isInitialized(modifier.getPizzaModifierPriceList()) && Hibernate.isInitialized(modifier.getMultiplierPriceList())) {
	//			return;
	//		}
	//		Session session = null;
	//
	//		try {
	//			session = createNewSession();
	//			session.refresh(modifier);
	//			Hibernate.initialize(modifier.getPizzaModifierPriceList());
	//			Hibernate.initialize(modifier.getMultiplierPriceList());
	//		} finally {
	//			closeSession(session);
	//		}
	//	}
	public MenuModifier loadFullModifier(String menuModifierId) {
		MenuModifier menuModifier = get(menuModifierId);
		initialize(menuModifier);
		return menuModifier;
	}

	public void initialize(MenuModifier menuModifier) {
		if (menuModifier == null || menuModifier.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(menuModifier.getPizzaModifierPriceList()) && Hibernate.isInitialized(menuModifier.getMultiplierPriceList())
				&& Hibernate.isInitialized(menuModifier.getModifierGroups())) {
			return;
		}
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(menuModifier);
			Hibernate.initialize(menuModifier.getPizzaModifierPriceList());
			Hibernate.initialize(menuModifier.getMultiplierPriceList());
			Hibernate.initialize(menuModifier.getModifierGroups());
		} finally {
			closeSession(session);
		}
	}

	public List<MenuModifier> getModifierList(String tagName) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(tagName))
				criteria.add(Restrictions.eq(MenuModifier.PROP_TAG, tagName));

			return criteria.list();
		}
	}

	public List<MenuModifier> getMenuModifiers(String itemName) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(MenuModifier.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			criteria.add(Restrictions.ne(MenuModifier.PROP_PIZZA_MODIFIER, Boolean.TRUE));
			criteria.addOrder(Order.asc(MenuModifier.PROP_SORT_ORDER));
			return criteria.list();
		}
	}

	@Deprecated
	public int getRowCount(String searchString, boolean pizzaModifier) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			criteria.setProjection(Projections.rowCount());

			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, searchString, MatchMode.START));
			}
			if (pizzaModifier)
				criteria.add(Restrictions.eq(MenuModifier.PROP_PIZZA_MODIFIER, Boolean.TRUE));
			else
				criteria.add(Restrictions.ne(MenuModifier.PROP_PIZZA_MODIFIER, Boolean.TRUE));

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
		}
		return 0;
	}

	public void loadItems(String searchString, boolean includeInvisibleItems, PaginatedListModel listModel) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());

			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, searchString, MatchMode.START));
			}

			if (!includeInvisibleItems) {
				criteria.add(Restrictions.eq(MenuModifier.PROP_ENABLE, Boolean.TRUE));
			}

			listModel.setNumRows(rowCount(criteria));

			criteria.addOrder(Order.asc(MenuModifier.PROP_SORT_ORDER));
			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			listModel.setData(criteria.list());
		}
	}

	public void loadItems(String searchString, boolean includeInvisibleItems, boolean pizzaModifier, Pagination<MenuModifier> listModel) {
		loadItems(searchString, null, includeInvisibleItems, pizzaModifier, listModel);
	}

	public void loadItems(String searchString, ModifierGroup modifierGroup, boolean includeInvisibleItems, boolean pizzaModifier,
			Pagination<MenuModifier> listModel) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());

			criteria.setProjection(Projections.rowCount());
			updateSearchCriteria(searchString, modifierGroup, includeInvisibleItems, pizzaModifier, criteria);

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				listModel.setNumRows(rowCount.intValue());
			}

			criteria = session.createCriteria(getReferenceClass());
			updateSearchCriteria(searchString, modifierGroup, includeInvisibleItems, pizzaModifier, criteria);
			criteria.addOrder(Order.asc(MenuModifier.PROP_NAME));
			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			listModel.setRows(criteria.list());
		}
	}

	private void updateSearchCriteria(String searchString, ModifierGroup modifierGroup, boolean includeInvisibleItems, boolean pizzaModifier,
			Criteria criteria) {
		updateCriteria(searchString, modifierGroup, criteria);
		if (pizzaModifier) {
			criteria.add(Restrictions.eq(MenuModifier.PROP_PIZZA_MODIFIER, Boolean.TRUE));
		}
		else {
			criteria.add(Restrictions.ne(MenuModifier.PROP_PIZZA_MODIFIER, Boolean.TRUE));
		}
		if (!includeInvisibleItems) {
			criteria.add(Restrictions.eq(MenuModifier.PROP_ENABLE, Boolean.TRUE));
		}
	}

	@Deprecated
	public List<MenuModifier> getComboModifiers() {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(MenuModifier.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(MenuModifier.PROP_COMBO_MODIFIER, Boolean.TRUE));
			return criteria.list();
		}
	}

	public MenuModifier findName(String stringValue) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuModifier.PROP_NAME, stringValue));
			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (MenuModifier) list.get(0);
			}
			return null;
		}
	}

	/**
	 * Search Menu Modifier By Name (MatchMode.ANYWHERE) and modifierGroupId.
	 */
	@Deprecated
	public void loadMenuModifier(PaginationSupport paginationSupport, String itemName, ModifierGroup modifierGroup) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());

			addDeletedFilter(criteria);

			updateCriteria(itemName, modifierGroup, criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria = session.createCriteria(getReferenceClass());
			updateCriteria(itemName, modifierGroup, criteria);
			criteria.addOrder(Order.asc(MenuModifier.PROP_SORT_ORDER));

			paginationSupport.setRows(criteria.list());

		} finally {
			closeSession(session);
		}
	}

	private void updateCriteria(String itemName, ModifierGroup modifierGroup, Criteria criteria) {
		addDeletedFilter(criteria);
		if (StringUtils.isNotEmpty(itemName)) {
			criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
		}

		if (modifierGroup != null) {
			criteria.createAlias("modifierGroups", "groups"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.eq("groups." + ModifierGroup.PROP_ID, modifierGroup.getId())); //$NON-NLS-1$
		}
	}

	public void removePizzaModifierPrices(List<String> listOfModifierPriceId, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("pizzaModifierPriceList", "modifierPrice"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("modifierPrice.id", listOfModifierPriceId)); //$NON-NLS-1$
		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		List<MenuModifier> list = criteria.list();
		if (list != null && !list.isEmpty()) {
			for (MenuModifier menuModifier : list) {
				List<PizzaModifierPrice> pizzaPriceList = menuModifier.getPizzaModifierPriceList();
				for (Iterator<PizzaModifierPrice> iterator = pizzaPriceList.iterator(); iterator.hasNext();) {
					PizzaModifierPrice pizzaModifierPrice = iterator.next();
					if (listOfModifierPriceId.contains(pizzaModifierPrice.getId())) {
						iterator.remove();
					}
				}
				session.update(menuModifier);
			}
		}
	}

	public void saveMenuModifierFormData(MenuModifier menuModifier) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(menuModifier, session);

			//update menu modifier page item
			MenuItemModifierPageItemDAO menuPageItemDAO = MenuItemModifierPageItemDAO.getInstance();
			List<MenuItemModifierPageItem> menuItemModifierPageItems = menuPageItemDAO.getPageItemFor(menuModifier, session);
			if (menuItemModifierPageItems != null) {
				for (MenuItemModifierPageItem menuItemModifierPageItem : menuItemModifierPageItems) {
					menuItemModifierPageItem.setMenuModifier(menuModifier);
					menuPageItemDAO.saveOrUpdate(menuItemModifierPageItem, session);
				}
			}

			tx.commit();
		} finally {
			closeSession(session);
		}
	}

	public List<MenuModifier> getPizzaModifiers(String searchString) {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuModifier.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(MenuModifier.PROP_PIZZA_MODIFIER, true));
			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuModifier.PROP_NAME, searchString.trim(), MatchMode.ANYWHERE));
			}
			return criteria.list();
		} finally {

			session.close();
		}
	}

	public void checkDuplicateNameExist(String existingId, String name, boolean pizzaModifier) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(MenuModifier.PROP_NAME, name).ignoreCase());
			criteria.add(Restrictions.eq(MenuModifier.PROP_PIZZA_MODIFIER, pizzaModifier));
			if (StringUtils.isNotEmpty(existingId)) {
				criteria.add(Restrictions.ne(MenuModifier.PROP_ID, existingId));
			}
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && rowCount.intValue() > 0) {
				throw new DuplicateDataException(String.format(Messages.getString("GenericDAO.20") + " %s " + Messages.getString("GenericDAO.22"), name)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}

	}

	@SuppressWarnings("rawtypes")
	public List<String> getNonExistingModifierIds(Set<MenuModifier> modifiers) {
		if (modifiers == null || modifiers.isEmpty()) {
			return null;
		}
		List<String> modifierIds = new ArrayList<String>();
		for (MenuModifier modifier : modifiers) {
			modifierIds.add(modifier.getId());
		}
		List<String> nonExistingModifierIds = new ArrayList<>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(MenuModifier.PROP_ID));
			criteria.add(Restrictions.in(MenuModifier.PROP_ID, modifierIds));
			List existingModifierIds = criteria.list();
			if (existingModifierIds == null || existingModifierIds.isEmpty()) {
				return modifierIds;
			}
			for (String modifierId : modifierIds) {
				if (!existingModifierIds.contains(modifierId)) {
					nonExistingModifierIds.add(modifierId);
				}
			}
		}
		return nonExistingModifierIds;
	}

}
