package com.floreantpos.model;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;

public class WorkingHours {
	public static final String SUNDAY = "Sunday"; //$NON-NLS-1$
	public static final String MONDAY = "Monday"; //$NON-NLS-1$
	public static final String TUESDAY = "Tuesday"; //$NON-NLS-1$
	public static final String WEDNESDAY = "Wednesday"; //$NON-NLS-1$
	public static final String THURSDAY = "Thursday"; //$NON-NLS-1$
	public static final String FRIDAY = "Friday"; //$NON-NLS-1$
	public static final String SATURDAY = "Saturday"; //$NON-NLS-1$

	public static String[] days = new String[] { "", SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY }; //$NON-NLS-1$

	public static final String OPEN = ".Open"; //$NON-NLS-1$
	public static final String START_TIME = ".StartTime"; //$NON-NLS-1$
	public static final String END_TIME = ".EndTime"; //$NON-NLS-1$
	public static final String TIME_ZONE = "time_zone"; //$NON-NLS-1$

	private Calendar openHour;
	private Calendar endHour;
	private TimeZone timeZone;
	private transient PropertyContainer propertiesContainer;
	private int dayOfWeek;
	private SimpleDateFormat hourMinFormat = new SimpleDateFormat("hh:mm a");

	public WorkingHours(int dayOfWeek, PropertyContainer propertyContainer) {
		this.dayOfWeek = dayOfWeek;
		this.propertiesContainer = propertyContainer;
		loadStartAndEndTime(dayOfWeek);
	}

	private void loadStartAndEndTime(int dayOfWeek) {
		try {
			this.timeZone = TimeZone.getTimeZone(propertiesContainer.getProperty(WorkingHours.TIME_ZONE));
			if (this.timeZone == null) {
				this.timeZone = TimeZone.getDefault();
			}
			hourMinFormat.setTimeZone(this.timeZone);
			if (propertiesContainer.getBooleanProperty(WorkingHours.days[dayOfWeek] + WorkingHours.OPEN, true)) {
				String dayString = WorkingHours.days[dayOfWeek];
				String openTime = propertiesContainer.getProperty(dayString + WorkingHours.START_TIME);
				String endTime = propertiesContainer.getProperty(dayString + WorkingHours.END_TIME);
				openHour = parseStringToCalendar(openTime);
				endHour = parseStringToCalendar(endTime);
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public Calendar getOpenHour() {
		return openHour;
	}

	private Calendar parseStringToCalendar(String timeString) {
		if (StringUtils.isEmpty(timeString)) {
			return null;
		}
		Calendar c = Calendar.getInstance();
		c.clear();
		c.setTimeZone(timeZone);
		try {
			Date parseTime = hourMinFormat.parse(timeString);
			c.setTime(parseTime);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return c;
	}

	public String getClosesTimeString() {
		if (isNull()) {
			return ""; //$NON-NLS-1$
		}
		return hourMinFormat.format(endHour.getTime()) + getTimeZoneOffsetString(); //$NON-NLS-1$
	}

	public String getDisplayTimeString() {
		if (isNull()) {
			return ""; //$NON-NLS-1$
		}

		//Date startGmtTime = DateUtil.convertDateToGmtTimeZone(openHour.getTime(), getTimezoneOffset());
		//Date endGmtTime = DateUtil.convertDateToGmtTimeZone(endHour.getTime(), getTimezoneOffset());
		return hourMinFormat.format(openHour.getTime()) + " - " //$NON-NLS-1$
				+ hourMinFormat.format(endHour.getTime()) + getTimeZoneOffsetString();
	}

	public String getNextDayOpeningTimeString() {
		int currentWorkingDay = dayOfWeek;
		int nextDay = currentWorkingDay;

		Calendar currentDate = Calendar.getInstance();
		currentDate.setTimeZone(this.timeZone);

		WorkingHours nextWorkingDay = this;
		if (nextWorkingDay == null || nextWorkingDay.getOpenHour() == null || nextWorkingDay.openHour.getTime().before(currentDate.getTime())) {
			nextDay = currentWorkingDay == Calendar.SATURDAY ? nextDay = Calendar.SUNDAY : currentWorkingDay + 1;
			while (nextDay != currentWorkingDay) {
				WorkingHours workingDay = getWorkingHours(nextDay);
				if (workingDay != null && workingDay.getOpenHour() != null) {
					nextWorkingDay = workingDay;
					break;
				}
				nextDay++;
				if (nextDay > Calendar.SATURDAY) {
					nextDay = Calendar.SUNDAY;
				}
			}
		}
		if (nextWorkingDay == null || nextWorkingDay.getOpenHour() == null) {
			return ""; //$NON-NLS-1$
		}
		return days[nextDay] + ", " + hourMinFormat.format(nextWorkingDay.getOpenHour().getTime()) + getTimeZoneOffsetString(); //$NON-NLS-1$
	}

	private String getTimeZoneOffsetString() {
		try {
			if (timeZone == null) {
				return ""; //$NON-NLS-1$
			}
			int outletTimezoneOffset = getRawOffsetWithDSTSavings(timeZone);
			if (outletTimezoneOffset == DataProvider.get().getTimeZoneOffset()) {
				return ""; //$NON-NLS-1$
			}
			int offsetIntValue = outletTimezoneOffset / 3600000;
			return " (GMT " + offsetIntValue + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return ""; //$NON-NLS-1$
	}

	//	public boolean isClosed() {
	//		return isClosed(0);
	//	}
	//
	//	public boolean isClosed(int beforeMin) {
	//		Calendar c = Calendar.getInstance();
	//		c.setTimeZone(this.timeZone);
	//		return !isOpenOnDate(c.getTime(), beforeMin);
	//	}

	private boolean isNull() {
		if (openHour == null || endHour == null) {
			return true;
		}
		if (openHour.getTime() == null || endHour.getTime() == null) {
			return true;
		}
		return false;
	}

	public boolean isOpen() {
		Calendar c = Calendar.getInstance();
		c.setTimeZone(this.timeZone);
		return isOpen(c.getTime());
	}

	public boolean isOpen(Date date) {
		return isOpenOnDate(date, 0);
	}

	public boolean isOpenOnDate(Date date, int beforeMin) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.setTimeZone(this.timeZone);
		//c.setTime(DateUtil.convertDateByTimeZone(date, getTimezoneOffset()));

		boolean isOpenOnDay = isNull() ? false : isOpenOnDay(c, c.get(Calendar.DAY_OF_MONTH), beforeMin);
		if (!isOpenOnDay) {
			WorkingHours previousWorkingDay = getWorkingHours(getPreviousWorkingDay(c));
			if (previousWorkingDay != null) {
				isOpenOnDay = previousWorkingDay.isOpenOnDay(c, c.get(Calendar.DAY_OF_MONTH) - 1, beforeMin);
				if (isOpenOnDay) {
					this.endHour = previousWorkingDay.endHour;
				}
			}
			if (!isOpenOnDay) {
				WorkingHours nextWorkingDay = getWorkingHours(getNextWorkingDay(c));
				if (nextWorkingDay != null) {
					isOpenOnDay = nextWorkingDay.isOpenOnDay(c, c.get(Calendar.DAY_OF_MONTH) + 1, beforeMin);
				}
				if (isOpenOnDay) {
					this.endHour = previousWorkingDay.endHour;
				}
			}
		}
		return isOpenOnDay;
	}

	public boolean isOpenOnDay(Calendar c, int day, int beforeMin) {
		if (openHour == null) {
			return false;
		}

		openHour.set(Calendar.YEAR, c.get(Calendar.YEAR));
		openHour.set(Calendar.MONTH, c.get(Calendar.MONTH));
		openHour.set(Calendar.DAY_OF_MONTH, day);

		endHour.set(Calendar.YEAR, c.get(Calendar.YEAR));
		endHour.set(Calendar.MONTH, c.get(Calendar.MONTH));
		endHour.set(Calendar.DAY_OF_MONTH, day);
		if (beforeMin != 0) {
			endHour.set(Calendar.MINUTE, endHour.get(Calendar.MINUTE) - beforeMin);
		}

		int openHourAmPm = openHour.get(Calendar.AM_PM);
		int closedHourAmPm = endHour.get(Calendar.AM_PM);
		if (openHourAmPm == Calendar.PM && closedHourAmPm == Calendar.AM) {
			endHour.set(Calendar.DAY_OF_MONTH, endHour.get(Calendar.DAY_OF_MONTH) + 1);
		}
		else if (openHourAmPm == closedHourAmPm && endHour.getTime().before(openHour.getTime())) {
			endHour.set(Calendar.DAY_OF_MONTH, endHour.get(Calendar.DAY_OF_MONTH) + 1);
		}
		//		SimpleDateFormat dateFormat = new SimpleDateFormat("dd MMM yyyy hh:mm a Z");
		//		dateFormat.setTimeZone(timeZone);
		//		System.out.println(dateFormat.format(openHour.getTime()));
		//		System.out.println(dateFormat.format(endHour.getTime()));
		//		System.out.println(dateFormat.format(c.getTime()));
		return DateUtil.between(openHour.getTime(), endHour.getTime(), c.getTime());
	}

	private int getPreviousWorkingDay(Calendar c) {
		int previousDay = c.get(Calendar.DAY_OF_WEEK) - 1;
		if (previousDay < 0) {
			previousDay = 7;
		}
		return previousDay;
	}

	private int getNextWorkingDay(Calendar c) {
		int nextDay = c.get(Calendar.DAY_OF_WEEK) + 1;
		if (nextDay > 7) {
			nextDay = 0;
		}
		return nextDay;
	}

	private int getTimezoneOffset() {
		if (timeZone == null) {
			return DataProvider.get().getTimeZoneOffset();
		}
		return getRawOffsetWithDSTSavings(timeZone);
	}

	private int getRawOffsetWithDSTSavings(TimeZone timeZ) {
		return timeZ.getRawOffset() + timeZ.getDSTSavings();
	}

	public WorkingHours getWorkingHours(int day) {
		//		if (!propertiesContainer.getBooleanProperty(WorkingHours.days[day] + WorkingHours.OPEN, true)) {
		//			return null;
		//		}
		return new WorkingHours(day, propertiesContainer);
	}
}
