/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Discount;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuPage;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Terminal;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class MenuGroupDAO extends BaseMenuGroupDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public MenuGroupDAO() {
	}

	@Override
	public Serializable save(Object menuGroup, Session s) throws HibernateException {
		updateTime(menuGroup);
		Serializable serializable = super.save(menuGroup, s);
		updateDependentModels((MenuGroup) menuGroup, s);
		return serializable;
	}

	@Override
	public void update(Object menuGroup, Session s) throws HibernateException {
		updateTime(menuGroup);
		super.update(menuGroup, s);
		updateDependentModels((MenuGroup) menuGroup, s);
	}

	@Override
	public void saveOrUpdate(Object menuGroup, Session s) throws HibernateException {
		updateTime(menuGroup);
		super.saveOrUpdate(menuGroup, s);
		updateDependentModels((MenuGroup) menuGroup, s);
	}

	@Override
	public void delete(Object obj, Session s) throws HibernateException {
		MenuGroup menuGroup = (MenuGroup) obj;
		if (menuGroup == null) {
			throw new PosException(Messages.getString("MenuGroupDAO.0")); //$NON-NLS-1$
		}
		checkIfGroupCanbeDeleted(s, menuGroup);
		menuGroup.setMenuCategory(null);
		menuGroup.setDeleted(true);
		removeFromDependentModels(menuGroup, s);
		update(menuGroup, s);
		removeFromDiscounts(menuGroup, s);
	}

	private void checkIfGroupCanbeDeleted(Session s, MenuGroup menuGroup) {
		String details = ""; //$NON-NLS-1$
		List<MenuItem> menuItems = MenuItemDAO.getInstance().getMenuItemsByGroupId(menuGroup.getId(), s);
		if (menuItems != null && !menuItems.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n\n"; //$NON-NLS-1$
			}
			details += Messages.getString("DataImportAction.8") + " " + menuGroup.getName() + " " + Messages.getString("MenuGroupDAO.4"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			int count = 1;
			for (MenuItem menuItem : menuItems) {
				details += "\n" + count + ". " + menuItem.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		if (StringUtils.isNotBlank(details)) {                                     
			throw new PosException(Messages.getString("DataImportAction.8") + " <b>" + menuGroup.getName() + "</b> " + Messages.getString("MenuItemDAO.15"), details);//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
	}

	public List<MenuGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuGroup.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	private void updateDependentModels(MenuGroup menuGroup, Session session) {
		String hqlString = "update MenuItem set %s=:groupName, %s=:categoryId, %s=:categoryName, %s=:beverage where %s=:groupId"; //$NON-NLS-1$
		//@formatter:off
		hqlString = String.format(hqlString, MenuItem.PROP_MENU_GROUP_NAME, MenuItem.PROP_MENU_CATEGORY_ID, MenuItem.PROP_MENU_CATEGORY_NAME,
				MenuItem.PROP_BEVERAGE, MenuItem.PROP_MENU_GROUP_ID);
		//@formatter:on
		Query query = session.createQuery(hqlString);
		query.setParameter("groupName", menuGroup.getName()); //$NON-NLS-1$
		query.setParameter("categoryId", menuGroup.getMenuCategoryId()); //$NON-NLS-1$
		query.setParameter("categoryName", menuGroup.getMenuCategoryName()); //$NON-NLS-1$
		query.setParameter("beverage", menuGroup.isBeverage()); //$NON-NLS-1$
		query.setParameter("groupId", menuGroup.getId()); //$NON-NLS-1$
		query.executeUpdate();
		session.saveOrUpdate(menuGroup);
	}

	private void removeFromDependentModels(MenuGroup menuGroup, Session session) {
		String menuGroupId = menuGroup.getId();
		// Deleting relation from MENU ITEM
		//@formatter:off
		String hqlString = "update MenuItem set " +  //$NON-NLS-1$
				MenuItem.PROP_MENU_GROUP_NAME + "= null, "+ //$NON-NLS-1$
				MenuItem.PROP_MENU_GROUP_ID + "= null, "+ //$NON-NLS-1$
				MenuItem.PROP_MENU_CATEGORY_ID +"= null, "+ //$NON-NLS-1$
				MenuItem.PROP_MENU_CATEGORY_NAME +"= null, %s=:beverage where %s=:groupId"; //$NON-NLS-1$
		
		hqlString = String.format(hqlString, 
				MenuItem.PROP_BEVERAGE, 
				MenuItem.PROP_MENU_GROUP_ID);
		//@formatter:on
		Query query = session.createQuery(hqlString);
		query.setParameter("beverage", false); //$NON-NLS-1$
		query.setParameter("groupId", menuGroupId); //$NON-NLS-1$
		query.executeUpdate();

		MenuPageDAO menuPageDAO = MenuPageDAO.getInstance();
		List<MenuPage> menuPages = menuPageDAO.findByGroup(menuGroup);
		if (menuPages != null && !menuPages.isEmpty()) {
			menuPageDAO.deleteAll(menuPages, session);
		}
	}

	private void removeFromDiscounts(MenuGroup menuGroup, Session session) {
		DiscountDAO discountDAO = DiscountDAO.getInstance();
		List<Discount> discounts = discountDAO.getDiscountsByMenuGroup(menuGroup, session);
		if (discounts != null && !discounts.isEmpty()) {
			for (Discount discount : discounts) {
				List<MenuGroup> menuGroups = discount.getMenuGroups();
				menuGroups.remove(menuGroup);
				discountDAO.saveOrUpdate(discount, session);
			}
		}
	}

	public void initialize(MenuGroup menuGroup) {
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(menuGroup);
		} finally {
			closeSession(session);
		}
	}

	public void loadGroupsForGroupView(OrderType orderType, MenuCategory parentCategory, PaginatedListModel listModel) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (orderType != null && orderType.getId() != null) {
				criteria.createAlias(MenuGroup.PROP_MENU_PAGES, "menuPage"); //NON-NLS-1$ //$NON-NLS-1$
				Criterion orderIdNull = Restrictions.isNull("menuPage." + MenuPage.PROP_ORDER_TYPE_ID); //$NON-NLS-1$
				Criterion orderIdEqual = Restrictions.eq("menuPage." + MenuPage.PROP_ORDER_TYPE_ID, orderType.getId()); //$NON-NLS-1$
				criteria.add(Restrictions.or(orderIdNull, orderIdEqual)); //NON-NLS-1$
			}

			criteria.add(Restrictions.eq(MenuGroup.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (parentCategory != null) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, parentCategory.getId()));
			}
			criteria.add(Restrictions.isNotEmpty(MenuGroup.PROP_MENU_PAGES));

			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

			List<MenuGroup> list = criteria.list();
			listModel.setNumRows(list.size());
			listModel.setPageSize(listModel.getNumRows());
			listModel.setData(list);
		} finally {
			closeSession(session);
		}
	}

	public List<MenuGroup> loadActiveGroupsByOrderType(OrderType orderType) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuGroup.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			List<MenuCategory> activeCategories = MenuCategoryDAO.getInstance().findActiveCategories(orderType);
			List<String> activeCatIds = new ArrayList<String>();
			for (MenuCategory menuCategory : activeCategories) {
				activeCatIds.add(menuCategory.getId());
			}
			if (!activeCatIds.isEmpty()) {
				criteria.add(Restrictions.in(MenuGroup.PROP_MENU_CATEGORY_ID, activeCatIds));
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));

			return criteria.list();
		}
	}

	public List<MenuGroup> findEnabledByParent(MenuCategory menuCategory) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuGroup.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));

			List<MenuGroup> list = criteria.list();
			return list;
		}
	}

	public List<String> findEnabledGroupsIdsByParent(MenuCategory menuCategory) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(MenuGroup.PROP_ID));
			criteria.add(Restrictions.eq(MenuGroup.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));
			return criteria.list();
		}
	}

	public List<MenuGroup> findByParent(MenuCategory menuCategory) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));

			List<MenuGroup> list = criteria.list();
			return list;
		}
	}

	public boolean hasChildren(Terminal terminal, MenuGroup group, OrderType orderType) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			//criteria.add(Restrictions.eq(MenuItem.PROP_PARENT, group));
			criteria.add(Restrictions.eqOrIsNull(MenuItem.PROP_DELETED, Boolean.FALSE));
			criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			//			if(terminal!=null) {
			//				criteria.add(Restrictions.eq(MenuItem., criteria))
			//			}
			criteria.setProjection(Projections.rowCount());

			//criteria.add(Restrictions.isEmpty("orderTypeList"));
			//criteria.createAlias("orderTypeList", "type", JoinType.LEFT_OUTER_JOIN);
			//criteria.add(Restrictions.or(Restrictions.isEmpty("orderTypeList"), Restrictions.eq("type.id", orderType.getId())));

			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount.intValue() > 0;
		} catch (Exception e) {
			PosLog.error(getClass(), e);
			throw new PosException(""); //$NON-NLS-1$
		}
	}

	public void releaseParent(List<MenuGroup> menuGroupList) {
		if (menuGroupList == null) {
			return;
		}

		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (MenuGroup menuGroup : menuGroupList) {
				menuGroup.setMenuCategoryId(null);
				session.saveOrUpdate(menuGroup);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		}
	}

	public void releaseParentAndDelete(MenuGroup group) {
		if (group == null) {
			return;
		}
		initialize(group);
		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			String queryString = "delete from MENUGROUP_DISCOUNT where MENUGROUP_ID='%s'"; //$NON-NLS-1$
			queryString = String.format(queryString, group.getId());
			Query query = session.createSQLQuery(queryString);
			query.executeUpdate();

			String queryString2 = "update MENU_ITEM set GROUP_ID=null where GROUP_ID='%s'"; //$NON-NLS-1$
			queryString2 = String.format(queryString2, group.getId());
			Query query2 = session.createSQLQuery(queryString2);
			query2.executeUpdate();

			session.delete(group);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ShopTableDAO.class).error(e);
			throw new RuntimeException(e);
		}
	}

	public boolean existsMenuGroups(MenuCategory menuCategory) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (menuCategory != null)
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));

			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	public void loadMenuGroups(PaginationSupport paginationSupport, String itemName, MenuCategory menuCategory, String... fields) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuGroup.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuGroup.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuGroup.PROP_NAME));

			if (fields != null && fields.length > 0) {
				ProjectionList projectionList = Projections.projectionList();
				for (String field : fields) {
					projectionList.add(Projections.property(field), field);
				}
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(MenuGroup.class));
				paginationSupport.setRows(criteria.list());
			}
			else {
				paginationSupport.setRows(criteria.list());
			}

		} finally {
			closeSession(session);
		}
	}

	public List<MenuGroup> findGroupsWithInventoryItems() throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			criteria.setProjection(Projections.property(MenuItem.PROP_MENU_GROUP_ID));
			List<String> groupIdList = criteria.list();

			criteria = session.createCriteria(MenuGroup.class);
			if (!groupIdList.isEmpty()) {
				criteria.add(Restrictions.in(MenuGroup.PROP_ID, groupIdList));
			}
			return criteria.list();
		}
	}

	public MenuGroup findMenuGroupByName(String name) {
		return findMenuGroupByName(name, Boolean.TRUE);
	}

	public MenuGroup findMenuGroupByName(String name, boolean addDeletedFilter) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuGroup.PROP_NAME, name).ignoreCase());
			if (addDeletedFilter) {
				criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			}

			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (MenuGroup) list.get(0);
			}
			return null;
		}
	}

	public List<MenuGroup> findMenuGroupByCategoryId(String categoryId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuGroup.class);
			if (!StringUtils.isBlank(categoryId)) {
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, categoryId));
			}
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}
	 public int getMenuGroupCountByCategoryId(String categoryId) {
		 if(categoryId==null) {
				return 0;
			}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuGroup.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, categoryId));
			criteria.setProjection(Projections.rowCount());
			Number rowCount =(Number) criteria.uniqueResult();
			if(rowCount!=null) {
				return rowCount.intValue();
			}
			return 0;
		}
	}
	public List<MenuGroup> findAllUnSyncMenuGroup() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			Criterion falseCriteria = Restrictions.eq(MenuGroup.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(MenuGroup.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		}
	}

	public void saveOrUpdateMenuGroups(List<MenuGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				MenuGroup item = (MenuGroup) iterator.next();
				MenuGroup existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					//TODO: local menu page not replaced
					List<MenuPage> menuPages = existingItem.getMenuPages();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setMenuPages(menuPages);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public void findGroups(PaginationSupport model, String gruopName, MenuCategory menuCategory) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuGroup.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(gruopName))
				criteria.add(Restrictions.ilike(MenuGroup.PROP_NAME, gruopName, MatchMode.ANYWHERE));
			if (menuCategory != null)
				criteria.add(Restrictions.eq(MenuGroup.PROP_MENU_CATEGORY_ID, menuCategory.getId()));

			model.setNumRows(rowCount(criteria));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.addOrder(Order.asc(MenuGroup.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuGroup.PROP_NAME).ignoreCase());
			model.setRows(criteria.list());
		}
	}
}