package com.orocube.rest.service.server;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.SequenceNumber;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.TicketItemSeat;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.GratuityDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.SequenceNumberDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.dao.TicketItemModifierDAO;
import com.floreantpos.model.dao.TicketItemSeatDAO;
import com.floreantpos.util.POSUtil;

public class TicketDataServiceDao {

	private static TicketDataServiceDao instance;

	private TicketDataServiceDao() {
	}

	public static TicketDataServiceDao getInstance() {
		if (instance == null) {
			instance = new TicketDataServiceDao();
		}
		return instance;
	}

	public static TicketDataServiceDao get() {
		return getInstance();
	}
	
	public void saveOrUpdateTicket(Ticket ticket, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		Map<String, Object> existingItemMap = new HashMap<>();
		TicketDAO ticketDAO = TicketDAO.getInstance();
		Ticket existingTicket = null;
		
		try (Session session = ticketDAO.createNewSession()) {
			existingTicket = ticketDAO.loadFullTicket(ticket.getId(), ticket.getOutletId());

			if (existingTicket != null) {
				ExistingItemMapBuilder.buildExistingItemMap(session, existingItemMap, ticket, existingTicket);
				ticket.mergeTicket(existingTicket);
			}
		}

		Session session = null;
		Transaction tx = null;
		try {
			session = ticketDAO.createNewSession();
			tx = session.beginTransaction();

			Gratuity gratuity = ticket.getGratuity();
			List<TicketItem> ticketItemList = POSUtil.copyList(ticket.getTicketItems());
			Set<PosTransaction> transactions = POSUtil.copySet(ticket.getTransactions());

			ticket.setUpdateLastUpdateTime(updateLastUpdateTime);
			ticket.setUpdateSyncTime(updateSyncTime);
			ticket.setGratuity(null);

			POSUtil.clear(ticket.getTicketItems());
			POSUtil.clear(ticket.getTransactions());

			saveOrUpdateTicket(session, ticket, existingTicket);
			saveOrUpdateTicketItems(session, ticket, ticketItemList, existingItemMap);
			saveOrUpdatePosTransactions(session, ticket, transactions, existingItemMap);
			saveOrUpdateGratuity(session, ticket, gratuity, existingItemMap);

			if (ticket.getTicketItems() != null) {
				ticket.getTicketItems().addAll(ticketItemList);
			}
			if (ticket.getTransactions() != null) {
				ticket.getTransactions().addAll(transactions);
			}
			
			ticket.setGratuity(gratuity);

			ticket.setShouldUpdateTableStatus(existingTicket == null);
			saveCustomerIfNotExists(session, ticket);

			Date currentTime = new Date();
			if (updateLastUpdateTime) {
				ticket.setLastUpdateTime(currentTime);
			}
			if (updateSyncTime) {
				ticket.setLastSyncTime(currentTime);
			}

			session.saveOrUpdate(ticket);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			PosLog.error(getClass(), "Error saving ticket: " + ticket.getId(), e);
			throw e;
		} finally {
			ticketDAO.closeSession(session);
		}
	}

	private void saveOrUpdateTicket(Session session, Ticket ticket, Ticket existingTicket) {
		if (existingTicket == null) {
			if (StringUtils.isEmpty(ticket.getShortId())) {
				ticket.setShortId(RandomStringUtils.randomNumeric(7));
			}
			if (ticket.getTokenNo() == 0) {
				ticket.setTokenNo(SequenceNumberDAO.getInstance().getNextSequenceNumber(SequenceNumber.TICKET_TOKEN, session));
			}

			TicketDAO.getInstance().save(ticket, session);
		}
		else {
			ticket.setVersion(existingTicket.getVersion());
		}
	}

	private void saveOrUpdateTicketItems(Session session, Ticket ticket, List<TicketItem> ticketItemList, Map<String, Object> existingItemMap) {
		for (TicketItem ticketItem : ticketItemList) {
			ticketItem.setTicket(ticket);
			saveOrUpdateTicketItem(session, ticket, ticketItem, existingItemMap);
		}
	}

	private void saveOrUpdatePosTransactions(Session session, Ticket ticket, Set<PosTransaction> transactions, Map<String, Object> existingItemMap) {
		PosTransactionDAO dao = PosTransactionDAO.getInstance();
		for (PosTransaction transaction : transactions) {
			transaction.setTicket(ticket);

			PosTransaction existingPosTransaction = (PosTransaction) existingItemMap.get(transaction.getId());
			if (existingPosTransaction == null) {
				dao.save(transaction, session);
			}
			else {
				transaction.setVersion(existingPosTransaction.getVersion());
			}
		}
	}

	private void saveCustomerIfNotExists(Session session, Ticket ticket) {
		if (StringUtils.isEmpty(ticket.getCustomerId())) {
			return;
		}
		Customer existingCustomer = CustomerDAO.getInstance().get(ticket.getCustomerId(), session);
		if (existingCustomer == null) {
			Customer customer = ticket.getCustomer();
			if (customer != null) {
				CustomerDAO.getInstance().save(customer, session);
			}
		}
	}

	private void saveOrUpdateTicketItem(Session session, Ticket ticket, TicketItem ticketItem, Map<String, Object> existingItemMap) {
		TicketItemDAO ticketItemDAO = TicketItemDAO.getInstance();

		TicketItemModifier sizeModifier = ticketItem.getSizeModifier();
		TicketItemSeat ticketItemSeat = ticketItem.getSeat();
		List<TicketItemModifier> ticketItemModifierList = POSUtil.copyList(ticketItem.getTicketItemModifiers());
		List<TicketItem> comboItemsList = POSUtil.copyList(ticketItem.getComboItems());

		
		ticketItem.setSizeModifier(null);
		ticketItem.setSeat(null);
		POSUtil.clear(ticketItem.getTicketItemModifiers());
		POSUtil.clear(ticketItem.getComboItems());

		TicketItem existingTicketItem = (TicketItem) existingItemMap.get(ticketItem.getId());
		if (existingTicketItem == null) {
			ticketItemDAO.save(ticketItem, session);
		}
		else {
			ticketItem.setVersion(existingTicketItem.getVersion());
		}

		if (!ticketItemModifierList.isEmpty()) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifierList) {
				ticketItemModifier.setTicketItem(ticketItem);
				saveTicketItemModifier(session, ticketItemModifier, existingItemMap);
			}
			ticketItem.getTicketItemModifiers().addAll(ticketItemModifierList);
		}
		

		saveOrUpdateSizeModifier(session, ticketItem, sizeModifier, existingItemMap);
		saveOrSetVersionTicketItemSeat(session, ticketItem, ticketItemSeat, existingItemMap);

		//Combo Item
		if (!comboItemsList.isEmpty()) {
			for (TicketItem comboTicketItem : comboItemsList) {
				comboTicketItem.setTicket(null);
				comboTicketItem.setParentTicketItem(ticketItem);
				saveOrUpdateTicketItem(session, ticket, comboTicketItem, existingItemMap);
			}
			ticketItem.getComboItems().addAll(comboItemsList);
		}
	}

	private void saveOrSetVersionTicketItemSeat(Session session, TicketItem ticketItem, TicketItemSeat ticketItemSeat, Map<String, Object> existingItemMap) {
		if (ticketItemSeat == null) {
			return;
		}
		TicketItemSeat existingSeat = (TicketItemSeat) existingItemMap.get(ticketItemSeat.getId());
		if (existingSeat == null) {
			TicketItemSeatDAO.getInstance().save(ticketItemSeat, session);
		}
		else {
			ticketItemSeat.setVersion(existingSeat.getVersion());
		}
		
		ticketItem.setSeat(ticketItemSeat);
	}

	private void saveOrUpdateSizeModifier(Session session, TicketItem ticketItem, TicketItemModifier sItemModifier, Map<String, Object> existingItemMap) {
		if (sItemModifier != null) {
			TicketItemModifier existingSizeModifier = (TicketItemModifier) existingItemMap.get(sItemModifier.getId());
			if (existingSizeModifier == null) {
				TicketItemModifierDAO.getInstance().save(sItemModifier, session);
			}
			else {
				sItemModifier.setVersion(existingSizeModifier.getVersion());
			}
			sItemModifier.setTicketItem(ticketItem);
			ticketItem.setSizeModifier(sItemModifier);
		}
	}

	private void saveTicketItemModifier(Session session, TicketItemModifier ticketItemModifier, Map<String, Object> existingItemMap) {
		TicketItemModifier existingTicketItemModifier = (TicketItemModifier) existingItemMap.get(ticketItemModifier.getId());
		if (existingTicketItemModifier == null) {
			TicketItemModifierDAO.getInstance().save(ticketItemModifier, session);
		}
		else {
			ticketItemModifier.setVersion(existingTicketItemModifier.getVersion());
		}
	}

	private void saveOrUpdateGratuity(Session session, Ticket ticket, Gratuity gratuity, Map<String, Object> existingTicketItemMap) {
		if (gratuity == null || StringUtils.isEmpty(gratuity.getId()))
			return;

		gratuity.setTicketId(ticket.getId());
		gratuity.setUpdateLastUpdateTime(false);
		gratuity.setUpdateSyncTime(false);

		Gratuity existingItem = (Gratuity) existingTicketItemMap.get(gratuity.getClass() + gratuity.getId());
		if (existingItem == null) {
			GratuityDAO.getInstance().save(gratuity, session);
		}
		else {
			gratuity.setVersion(existingItem.getVersion());
		}
	}
	
	static class ExistingItemMapBuilder {
		private static void buildExistingItemMap(Session session, Map<String, Object> existingItemMap, Ticket newTicket, Ticket existingTicket) {
			List<TicketItem> existingTicketItems = existingTicket.getTicketItems();
			if (existingTicketItems != null) {
				for (TicketItem ticketItem : existingTicketItems) {
					buildExistingItemMap(existingItemMap, ticketItem);
				}
			}

			Set<PosTransaction> transactions = existingTicket.getTransactions();
			for (PosTransaction posTransaction : transactions) {
				existingItemMap.put(posTransaction.getId(), posTransaction);
			}

			Gratuity gratuity = existingTicket.getGratuity();
			if (gratuity != null) {
				existingItemMap.put(gratuity.getClass() + gratuity.getId(), gratuity);
			}
			
			verifyAllItemsPopulated(session, existingItemMap, newTicket);
			verifyAllTransactionsPopulated(session, newTicket, existingItemMap);
		}

		private static void verifyAllItemsPopulated(Session session, Map<String, Object> existingItemMap, Ticket newTicket) {
			List<TicketItem> ticketItems = newTicket.getTicketItems();
			if (ticketItems != null) {
				for (TicketItem ticketItem : ticketItems) {
					verifyItemsPopulated(session, existingItemMap, ticketItem);
				}
			}
		}

		private static void verifyAllTransactionsPopulated(Session session, Ticket newTicket, Map<String, Object> existingItemMap) {
			Set<PosTransaction> newTransactions = newTicket.getTransactions();
			if (newTransactions != null) {
				for (PosTransaction posTransaction : newTransactions) {
					PosTransaction posTransaction2 = (PosTransaction) existingItemMap.get(posTransaction.getId());
					if (posTransaction2 == null) {
						posTransaction2 = PosTransactionDAO.getInstance().get(posTransaction.getId(), session);
						if (posTransaction2 != null) {
							existingItemMap.put(posTransaction.getId(), posTransaction2);
						}
					}
				}
			}
		}

		/**
		 * Verify if all items of cloud ticket is actually loaded in item map. This is actually needed only for very old database.
		 * 
		 * @param session
		 * @param existingItemMap
		 * @param ticketItem
		 */
		private static void verifyItemsPopulated(Session session, Map<String, Object> existingItemMap, TicketItem ticketItem) {
			TicketItem ticketItem2 = (TicketItem) existingItemMap.get(ticketItem.getId());
			if (ticketItem2 == null) {
				ticketItem2 = TicketItemDAO.getInstance().get(ticketItem.getId(), session);
				if (ticketItem2 != null) {
					existingItemMap.put(ticketItem.getId(), ticketItem2);
				}
			}
			
			TicketItemModifier sizeModifier = ticketItem.getSizeModifier();
			if (sizeModifier != null) {
				TicketItemModifier sizeModifier2 = (TicketItemModifier) existingItemMap.get(sizeModifier.getId());
				if (sizeModifier2 == null) {
					sizeModifier2 = TicketItemModifierDAO.getInstance().get(sizeModifier.getId(), session);
					if (sizeModifier2 != null) {
						existingItemMap.put(sizeModifier.getId(), sizeModifier2);
					}
				}
			}

			List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
					TicketItemModifier ticketItemModifier2 = (TicketItemModifier) existingItemMap.get(ticketItemModifier.getId());
					if (ticketItemModifier2 == null) {
						ticketItemModifier2 = TicketItemModifierDAO.getInstance().get(ticketItemModifier.getId(), session);
						if (ticketItemModifier2 != null) {
							existingItemMap.put(ticketItemModifier.getId(), ticketItemModifier2);
						}
					}
				}
			}

			List<TicketItem> comboItems = ticketItem.getComboItems();
			if (comboItems != null) {
				for (TicketItem comboItem : comboItems) {
					verifyItemsPopulated(session, existingItemMap, comboItem);
				}
			}
		}

		private static void buildExistingItemMap(Map<String, Object> existingItemMap, TicketItem existingTicketItem) {
			existingItemMap.put(existingTicketItem.getId(), existingTicketItem);

			TicketItemModifier sizeModifier = existingTicketItem.getSizeModifier();
			if (sizeModifier != null) {
				existingItemMap.put(sizeModifier.getId(), sizeModifier);
			}

			TicketItemSeat ticketItemSeat = existingTicketItem.getSeat();
			if (ticketItemSeat != null) {
				existingItemMap.put(ticketItemSeat.getId(), ticketItemSeat);
			}

			List<TicketItemModifier> ticketItemModifiers = existingTicketItem.getTicketItemModifiers();
			if (ticketItemModifiers != null) {
				for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
					existingItemMap.put(ticketItemModifier.getId(), ticketItemModifier);
				}
			}

			List<TicketItem> comboItems = existingTicketItem.getComboItems();
			if (comboItems != null) {
				for (TicketItem comboItem : comboItems) {
					buildExistingItemMap(existingItemMap, comboItem);
				}
			}
		}
	}
}
