package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.model.base.BaseShopTableStatus;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "userName", "tableStatus", "ticketNumbers", "ticketId" })
@XmlRootElement
public class ShopTableStatus extends BaseShopTableStatus implements TimedModel {
	private static final long serialVersionUID = 1L;

	public static final String SEAT_TIME = "seat.time"; //$NON-NLS-1$

	private transient JsonObject propertiesContainer;
	private boolean shouldPublishMqtt = true;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public ShopTableStatus() {
	}

	/**
	 * Constructor for primary key
	 */
	public ShopTableStatus(java.lang.Integer id, java.lang.String outletId) {
		super(id, outletId);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@XmlTransient
	public TableStatus getTableStatus() {
		Integer tableStatus = super.getTableStatusNum();
		return TableStatus.get(tableStatus);
	}

	public void setTableStatus(TableStatus tableStatus) {
		super.setTableStatusNum(tableStatus.getValue());
	}

	@XmlTransient
	public List<ShopTableTicket> getTicketNumbers() {
		String ticketInformations = getTicketInformations();
		if (StringUtils.isEmpty(ticketInformations)) {
			return null;
		}
		Gson gson = new Gson();
		List<ShopTableTicket> fromJson = gson.fromJson(ticketInformations, new TypeToken<List<ShopTableTicket>>() {
		}.getType());
		return fromJson;
	}

	public void setTicketNumbers(java.util.List<ShopTableTicket> ticketNumbers) {
		if (ticketNumbers == null || ticketNumbers.size() == 0) {
			setTicketInformations(null);
			return;
		}
		com.google.gson.JsonArray jsonArray = new com.google.gson.JsonArray();
		for (ShopTableTicket shopTableTicket : ticketNumbers) {
			jsonArray.add(shopTableTicket.toJson());
		}
		setTicketInformations(jsonArray.toString());
	}

	@XmlTransient
	public String getTicketId() {
		List<String> ticketNumbers = getListOfTicketNumbers();
		if (ticketNumbers != null && ticketNumbers.size() > 0)
			return ticketNumbers.get(0);
		return null;
	}

	public List<String> getListOfTicketNumbers() {
		List<ShopTableTicket> shopTableTickets = getTicketNumbers();
		List<String> listOfTicketNumbers = new ArrayList<>();
		if (shopTableTickets != null) {
			for (ShopTableTicket shopTableTicket : shopTableTickets) {
				listOfTicketNumbers.add(shopTableTicket.getTicketId());
			}
		}
		return listOfTicketNumbers;
	}

	public boolean hasMultipleTickets() {
		List<ShopTableTicket> ticketNumbers = getTicketNumbers();
		if (ticketNumbers != null && ticketNumbers.size() > 0)
			return true;
		return false;
	}

	public void setTicketId(String ticketId) {
		setTableTicket(ticketId, null, null, null);
	}

	public void setTableTicket(String ticketId, Integer tokenNo, String userId, String userFirstName) {
		setTableTicket(ticketId, tokenNo, userId, userFirstName, null);
	}

	public void setTableTicket(String ticketId, Integer tokenNo, String userId, String userFirstName, Date ticketCreateTime) {
		if (ticketId == null) {
			setTableStatus(TableStatus.Available);
			setTicketNumbers(null);
		}
		else {
			ShopTableTicket shopTableTicket = null;
			List<ShopTableTicket> shopTableTickets = getTicketNumbers();
			if (shopTableTickets != null && !shopTableTickets.isEmpty()) {
				for (ShopTableTicket shopT : shopTableTickets) {
					if (shopT.getTicketId().equals(ticketId))
						shopTableTicket = shopT;
				}
			}
			else {
				shopTableTickets = new ArrayList<ShopTableTicket>();
			}
			if (shopTableTicket == null) {
				shopTableTicket = new ShopTableTicket();
				shopTableTickets.add(shopTableTicket);
			}
			shopTableTicket.setTicketId(ticketId);
			shopTableTicket.setTokenNo(tokenNo);
			shopTableTicket.setUserId(userId);
			shopTableTicket.setUserName(userFirstName);
			shopTableTicket.setTicketCreateTime(ticketCreateTime);
			setTicketNumbers(shopTableTickets);
		}
	}

	public void addToTableTickets(List<Ticket> tickets) {
		if (tickets == null)
			return;
		List<String> existingTicketIds = new ArrayList<>();
		List<ShopTableTicket> shopTableTickets = getTicketNumbers();
		if (shopTableTickets == null) {
			shopTableTickets = new ArrayList<>();
		}
		for (ShopTableTicket shopTableTicket : shopTableTickets) {
			String ticketId = shopTableTicket.getTicketId();
			if (ticketId != null)
				existingTicketIds.add(ticketId);
		}
		for (Ticket ticket : tickets) {
			if (existingTicketIds.contains(ticket.getId()))
				continue;
			ShopTableTicket shopTableTicket = new ShopTableTicket();
			shopTableTicket.setTicketId(ticket.getId());
			shopTableTicket.setTokenNo(ticket.getTokenNo());
			shopTableTicket.setUserId(ticket.getOwner().getId());
			shopTableTicket.setUserName(ticket.getOwner().getFirstName());
			shopTableTicket.setTicketCreateTime(ticket.getCreateDate());
			shopTableTickets.add(shopTableTicket);
		}
		setTicketNumbers(shopTableTickets);
	}

	public String getUserId() {
		List<ShopTableTicket> shopTableTickets = getTicketNumbers();
		if (shopTableTickets == null || shopTableTickets.isEmpty())
			return null;
		return shopTableTickets.get(0).getUserId();
	}

	public String getUserName() {
		List<ShopTableTicket> shopTableTickets = getTicketNumbers();
		if (shopTableTickets == null || shopTableTickets.isEmpty())
			return ""; //$NON-NLS-1$
		int size = shopTableTickets.size();
		if (size > 1) {
			List<String> userIds = new ArrayList<>();
			for (Iterator iterator = shopTableTickets.iterator(); iterator.hasNext();) {
				ShopTableTicket shopTableTicket = (ShopTableTicket) iterator.next();
				if (userIds.contains(shopTableTicket.getUserId()))
					continue;
				userIds.add(shopTableTicket.getUserId());
			}
			if (userIds.size() > 1)
				return Messages.getString("ShopTableStatus.2"); //$NON-NLS-1$
		}
		return shopTableTickets.get(0).getUserName();
	}

	public String getTokenNo() {
		List<ShopTableTicket> shopTableTickets = getTicketNumbers();
		if (shopTableTickets == null || shopTableTickets.isEmpty())
			return ""; //$NON-NLS-1$
		int size = shopTableTickets.size();
		if (size == 1) {
			return String.valueOf(shopTableTickets.get(0).getTokenNo());
		}
		String displayString = ""; //$NON-NLS-1$
		int count = 1;
		for (Iterator iterator = shopTableTickets.iterator(); iterator.hasNext();) {
			ShopTableTicket shopTableTicket = (ShopTableTicket) iterator.next();
			displayString += String.valueOf(shopTableTicket.getTokenNo());
			if (count == 4)
				break;
			count++;
			if (iterator.hasNext()) {
				displayString += ","; //$NON-NLS-1$
			}
		}
		return displayString;
	}

	public Date getLastTicketCreateTime() {
		String ticketInformations = getTicketInformations();
		if (StringUtils.isEmpty(ticketInformations)) {
			return null;
		}
		Gson gson = new Gson();
		List<ShopTableTicket> fromJson = gson.fromJson(ticketInformations, new TypeToken<List<ShopTableTicket>>() {
		}.getType());

		if (fromJson == null || fromJson.isEmpty()) {
			return null;
		}

		Collections.sort(fromJson, new Comparator<ShopTableTicket>() {
			public int compare(ShopTableTicket o1, ShopTableTicket o2) {
				if (o1 == null || o2 == null) {
					return 1;
				}
				if (o1.getTicketCreateTime() == null || o2.getTicketCreateTime() == null) {
					return 1;
				}
				return o1.getTicketCreateTime().compareTo(o2.getTicketCreateTime());
			}
		});
		return fromJson.get(0).getTicketCreateTime();
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		if (propertiesContainer != null) {
			return propertiesContainer.has(key);
		}
		return false;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public boolean isShouldPublishMqtt() {
		return shouldPublishMqtt;
	}

	public void setShouldPublishMqtt(boolean shouldPublishMqtt) {
		this.shouldPublishMqtt = shouldPublishMqtt;
	}
}