/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import com.floreantpos.model.base.BasePosTransaction;
import com.floreantpos.model.dao.CashDrawerDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@XmlSeeAlso({ CashTransaction.class, CustomPaymentTransaction.class, GiftCertificateTransaction.class, CreditCardTransaction.class, DebitCardTransaction.class,
		CashDropTransaction.class, RefundTransaction.class, PayOutTransaction.class, VoidTransaction.class, CustomerAccountTransaction.class,
		ReversalTransaction.class, CashInTransaction.class })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@JsonIgnoreProperties(ignoreUnknown = true, value = { "ticketItems", "terminal", "properties", "cashDrawer" })
@JsonTypeInfo(use = Id.CLASS, include = JsonTypeInfo.As.PROPERTY, property = "classType")
@JsonSubTypes({ @Type(value = CashTransaction.class), @Type(value = CustomPaymentTransaction.class), @Type(value = GiftCertificateTransaction.class),
		@Type(value = CreditCardTransaction.class), @Type(value = DebitCardTransaction.class), @Type(value = CashDropTransaction.class),
		@Type(value = RefundTransaction.class), @Type(value = PayOutTransaction.class), @Type(value = VoidTransaction.class),
		@Type(value = CustomerAccountTransaction.class), @Type(value = ReversalTransaction.class), @Type(value = CashInTransaction.class) })
public class PosTransaction extends BasePosTransaction implements TimedModel {
	public static final String JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO = "gift_card_balance_add_info"; //$NON-NLS-1$

	public static final String JSON_PROP_GIFT_CARD_ADDED_AMOUNT = "addedAmount"; //$NON-NLS-1$

	public static final String JSON_PROP_GIFT_CARD_GIFT_CARD_NO = "giftCardNo"; //$NON-NLS-1$

	public static final String JSON_PROP_GIFT_CARD_TICKET_ITEM_ID = "ticketItemId"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_NAME = "CUSTOMER_NAME"; //$NON-NLS-1$
	private static final long serialVersionUID = 1L;

	private transient boolean refunded = false;
	private transient boolean syncEdited;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private CashDrawer cashDrawer;
	private String userName;
	private String serverName;
	private transient com.google.gson.JsonObject propertiesContainer;

	public PosTransaction() {
		setDataVersion(2);
	}

	/**
	 * Constructor for primary key
	 */
	public PosTransaction(java.lang.String id) {
		super(id);
		setDataVersion(2);
	}

	/**
	 * Constructor for required fields
	 */
	public PosTransaction(java.lang.String id, java.lang.String transactionType, java.lang.String paymentTypeString) {

		super(id, transactionType, paymentTypeString);
		setDataVersion(2);
	}

	private String cardTrack;
	private String cardNo;
	private String cardExpYear;
	private String cardExpMonth;
	private String cardCVV;

	public final static String CASH = "CASH"; //$NON-NLS-1$
	public final static String GIFT_CERT = "GIFT_CERT"; //$NON-NLS-1$
	public final static String CREDIT_CARD = "CREDIT_CARD"; //$NON-NLS-1$
	public final static String DEBIT_CARD = "DEBIT_CARD"; //$NON-NLS-1$
	public final static String CASH_DROP = "CASH_DROP"; //$NON-NLS-1$
	public final static String REFUND = "REFUND"; //$NON-NLS-1$
	public final static String PAY_OUT = "PAY_OUT"; //$NON-NLS-1$
	public final static String VOID_TRANS = "VOID_TRANS"; //$NON-NLS-1$
	public final static String GIFT_CARD_BALANCE_ADD = "GIFT_CARD_BALANCE_ADD"; //$NON-NLS-1$
	public static final String PROP_CASH_BACK_POSTFIX = "_CASH_BACK"; //$NON-NLS-1$
	public static final String PROP_TENDERED_POSTFIX = "_TENDERED"; //$NON-NLS-1$
	public static final String PROP_PAID_POSTFIX = "_PAID"; //$NON-NLS-1$
	public static final String JSON_PROP_REFUNDED_AMOUNT = "REFUNDED_AMOUNT"; //$NON-NLS-1$

	public static final String TRANSACTION_FEE = "transactionFee"; //$NON-NLS-1$

	private static final String JSON_PROP_GIFT_CARD_NUMBER = "gift_card_number"; //$NON-NLS-1$
	private static final String JSON_PROP_GIFT_CARD_BALANCE_AMOUNT = "gift_card_balance_amount"; //$NON-NLS-1$

	public static final String CURRENCY_ID = "currency.id"; //$NON-NLS-1$
	public static final String CURRENCY_NAME = "currency.name"; //$NON-NLS-1$
	public static final String CURRENCY_CODE = "currency.code"; //$NON-NLS-1$
	public static final String CURRENCY_SYMBOL = "currency.symbol"; //$NON-NLS-1$
	public static final String CURRENCY_RATE = "currency.rate"; //$NON-NLS-1$
	public static final String CURRENCY_TENDERED = "currency.tendered"; //$NON-NLS-1$
	public static final String CURRENCY_CHANGE = "currency.change"; //$NON-NLS-1$
	public static final String CURRENCY_PAID_AMOUNT = "currency.paidAmount"; //$NON-NLS-1$
	public static final String PAYMENT_CURRENCIES = "paymentCurrencies"; //$NON-NLS-1$

	private transient String ticketId;

	private String classType;

	@Override
	public String getTransactionType() {
		String type = super.getTransactionType();

		if (StringUtils.isEmpty(type)) {
			return TransactionType.CREDIT.name();
		}

		return type;
	}

	public void updateTerminalBalance() {
		Terminal terminal = getTerminal();
		if (terminal == null) {
			return;
		}

		Double amount = getAmount();
		if (amount == null || amount == 0) {
			return;
		}

		/*double terminalBalance = terminal.getCurrentBalance();
		
		TransactionType transactionType = TransactionType.valueOf(getTransactionType());
		switch (transactionType) {
			case CREDIT:
				terminalBalance += amount;
				break;
		
			case DEBIT:
				terminalBalance -= amount;
		}
		
		terminal.setCurrentBalance(terminalBalance);*/
	}

	public boolean isCard() {
		return (this instanceof CreditCardTransaction) || (this instanceof DebitCardTransaction);
	}

	public void addProperty(String name, String value) {
		addExtraProperty(name, value);
	}

	public boolean hasProperty(String key) {
		return getExtraProperty(key) != null;
	}

	public String getProperty(String key) {
		return getExtraProperty(key);
	}

	public String getProperty(String key, String defaultValue) {
		if (getExtraProperties() == null) {
			return defaultValue;
		}

		String string = getExtraProperty(key);
		if (StringUtils.isEmpty(string)) {
			return defaultValue;
		}

		return string;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getExtraProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public Double calculateTotalAmount() {
		return getAmount() + getTipsAmount();
	}

	public Double calculateAuthorizeAmount() {
		double advanceTipsPercentage = 0;//CardConfig.getAdvanceTipsPercentage();
		return getTenderAmount() + getTenderAmount() * (advanceTipsPercentage / 100);
	}

	@XmlTransient
	public String getCardTrack() {
		return cardTrack;
	}

	public void setCardTrack(String cardTrack) {
		this.cardTrack = cardTrack;
	}

	public String getCardNo() {
		return cardNo;
	}

	public void setCardNo(String cardNo) {
		this.cardNo = cardNo;
	}

	public String getCardExpYear() {
		return cardExpYear;
	}

	public void setCardExpYear(String expYear) {
		this.cardExpYear = expYear;
	}

	public String getCardExpMonth() {
		return cardExpMonth;
	}

	public void setCardExpMonth(String expMonth) {
		this.cardExpMonth = expMonth;
	}

	public String getCardCVV() {
		return cardCVV;
	}

	public void setCardCVV(String cardCVV) {
		this.cardCVV = cardCVV;
	}

	public String getGlobalId() {
		return super.getId();
	}

	public String getTicketId() {
		if (StringUtils.isNotEmpty(ticketId)) {
			return ticketId;
		}
		Ticket ticket = getTicket();
		if (ticket == null) {
			return ""; //$NON-NLS-1$
		}
		return ticket.getId();
	}

	public void setTicketId(String ticketId) {
		this.ticketId = ticketId;
	}

	public void calculateTaxAmount() {
		if (getAmount() <= 0)
			return;
		Ticket ticket = getTicket();
		if (ticket == null || ticket.getTotalAmount() <= 0 || isVoided() || this instanceof RefundTransaction)
			return;
		setTaxAmount(((getAmountWithTolerance() - getTipsAmount()) * ticket.getTaxAmount() / ticket.getTotalAmount()));
	}

	public void calculateServiceChargeAmount() {
		if (getAmount() <= 0)
			return;
		Ticket ticket = getTicket();
		if (ticket == null || ticket.getServiceCharge() <= 0 || isVoided() || this instanceof RefundTransaction)
			return;
		setServiceChargeAmount(((getAmountWithTolerance() - getTipsAmount()) * ticket.getServiceCharge() / ticket.getTotalAmount()));
	}

	public boolean isRefunded() {
		return refunded;
	}

	public void setRefunded(boolean refunded) {
		this.refunded = refunded;
	}

	@XmlTransient
	public Department getDepartment() {
		return DataProvider.get().getDepartmentById(getDepartmentId());
	}

	public void setDepartment(Department department) {
		String departmentId = null;
		if (department != null) {
			departmentId = department.getId();
		}
		super.setDepartmentId(departmentId);
	}

	@XmlTransient
	public Terminal getTerminal() {
		return DataProvider.get().getTerminalById(getTerminalId(), getOutletId());
	}

	public void setTerminal(Terminal terminal) {
		Integer terminalId = null;
		if (terminal != null) {
			terminalId = terminal.getId();
		}
		super.setTerminalId(terminalId);
	}

	@XmlTransient
	public User getUser() {
		return DataProvider.get().getUserById(getUserId(), getOutletId());
	}

	public void setUser(User user) {
		String userId = null;
		if (user != null) {
			userId = user.getId();
		}
		super.setUserId(userId);
	}

	@XmlTransient
	public String getUserName() {
		if (userName != null) {
			return userName;
		}
		User user = getUser();
		return userName = user == null ? null : user.getFullName();
	}

	public void setUserName(String userName) {
		this.userName = userName;
	}

	public String getServerName() {
		if (serverName != null) {
			return serverName;
		}
		User server = getServer();
		return serverName = server == null ? null : server.getFullName();
	}

	public void setServerName(String serverName) {
		this.serverName = serverName;
	}

	@XmlTransient
	public User getServer() {
		return DataProvider.get().getUserById(getServerId(), getOutletId());
	}

	public void setServer(User server) {
		if (server == null) {
			setServerId(null);
		}
		else {
			setServerId(server.getId());
		}
	}

	public void setCashDrawer(CashDrawer cashDrawer) {
		this.cashDrawer = cashDrawer;
		if (cashDrawer != null)
			super.setCashDrawerId(cashDrawer.getId());
	}

	@XmlTransient
	public CashDrawer getCashDrawer() {
		if (cashDrawer != null)
			return cashDrawer;
		return CashDrawerDAO.getInstance().get(getCashDrawerId());
	}

	public boolean isSyncEdited() {
		return syncEdited;
	}

	public void setSyncEdited(boolean syncEdited) {
		this.syncEdited = syncEdited;
	}

	public void setPaymentType(PaymentType paymentType) {
		if (paymentType != null) {
			setPaymentTypeString(paymentType.name());
		}
		else {
			setPaymentTypeString(null);
		}
	}

	@XmlTransient
	public PaymentType getPaymentType() {
		String paymentTypeString = super.getPaymentTypeString();
		if (paymentTypeString == null) {
			return null;
		}
		try {
			return PaymentType.valueOf(paymentTypeString);
		} catch (Exception e) {
			return PaymentType.fromDisplayString(paymentTypeString);
		}
	}

	public double getAmountWithTolerance() {
		return super.getAmount() - super.getToleranceAmount();
	}

	@Override
	public String getExtraProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getExtraProperties();
		if (StringUtils.isBlank(properties)) {
			propertiesContainer = new JsonObject();
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		return properties;
	}

	@Override
	public void setExtraProperties(String properties) {
		super.setExtraProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
	}

	public void addExtraProperty(String key, String value) {
		if (StringUtils.isEmpty(value)) {
			return;
		}
		if (propertiesContainer == null) {
			propertiesContainer = new com.google.gson.JsonObject();
		}
		propertiesContainer.addProperty(key, value);
		if (TRANSACTION_FEE == key) {
			setFeeAmount(POSUtil.getDoubleAmount(value));
		}
	}

	public String getExtraProperty(String key) {
		if (propertiesContainer == null) {
			getExtraProperties();
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public String getClassType() {
		if (classType == null) {
			return classType = getClass().getName();
		}
		return classType;
	}

	public void setClassType(String classType) {
		this.classType = classType;
	}

	public double getGiftCardBalanceId() {
		return POSUtil.getDoubleAmount(getProperty(JSON_PROP_GIFT_CARD_NUMBER));
	}

	public void setGiftCardBalanceNumber(String giftCardId) {
		addProperty(JSON_PROP_GIFT_CARD_NUMBER, String.valueOf(giftCardId));
	}

	public double getGiftCardBalanceAmount() {
		return POSUtil.parseDouble(getProperty(JSON_PROP_GIFT_CARD_BALANCE_AMOUNT));
	}

	public void setGiftCardBalanceAmount(double amount) {
		addProperty(JSON_PROP_GIFT_CARD_BALANCE_AMOUNT, String.valueOf(amount));
	}

	public void addGiftCardBalanceAddInfo(String ticketItemId, String giftCardNo, double addedAmount) {
		JSONObject jsonObject = new JSONObject();
		jsonObject.put(JSON_PROP_GIFT_CARD_TICKET_ITEM_ID, ticketItemId);
		jsonObject.put(JSON_PROP_GIFT_CARD_GIFT_CARD_NO, giftCardNo);
		jsonObject.put(JSON_PROP_GIFT_CARD_ADDED_AMOUNT, addedAmount);

		String giftCardInfo = getGiftCardBalanceAddInfo();
		if (StringUtils.isBlank(giftCardInfo)) {
			JSONArray jsonArray = new JSONArray();
			jsonArray.put(jsonObject);
			addProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO, jsonArray.toString());
		}
		else {
			JSONArray jsonArray = new JSONArray(giftCardInfo);
			jsonArray.put(jsonObject);
			addProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO, jsonArray.toString());
		}
	}

	public String getGiftCardBalanceAddInfo() {
		return getProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO);
	}

	public Double getRefundedAmount() {
		String refundedAmountText = getProperty(JSON_PROP_REFUNDED_AMOUNT);
		if (StringUtils.isNotEmpty(refundedAmountText)) {
			try {
				return NumberUtil.parseDouble(refundedAmountText);
			} catch (Exception e2) {
			}
		}

		return 0.0;
	}

	public Double getRefundableAmount() {
		return getAmount() - getRefundedAmount();
	}

	@XmlTransient
	public Customer getCustomer() {
		String customerId = getCustomerId();
		if (StringUtils.isEmpty(customerId)) {
			return null;
		}
		return DataProvider.get().getCustomer(customerId);
	}

	public void setCustomer(Customer customer) {
		if (customer == null) {
			removeProperty(JSON_PROP_CUSTOMER_NAME);
			return;
		}
		setCustomerId(customer.getId());
		setCustomerName(customer.getName());
	}

	@XmlTransient
	public String getCustomerName() {
		if (getCustomer() != null) {
			return getCustomer().getName();
		}
		return getProperty(JSON_PROP_CUSTOMER_NAME, ""); //$NON-NLS-1$
	}

	public void setCustomerName(String customerName) {
		addProperty(JSON_PROP_CUSTOMER_NAME, customerName);
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String toString() {
		//return ToStringBuilder.reflectionToString(this);
		return getTransactionType() + ", " + getCardAuthCode();
	}
}