package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.CustomPayment;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CustomPaymentDAO extends BaseCustomPaymentDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CustomPaymentDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		CustomPayment bean = (CustomPayment) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("CustomPaymentDAO.0")); //$NON-NLS-1$
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<CustomPayment> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<CustomPayment> findAllActive() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(CustomPayment.PROP_ENABLE, true));
			criteria.addOrder(Order.asc(CustomPayment.PROP_NAME));
			return criteria.list();
		}
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(CustomPayment.PROP_ID);
	}

	public CustomPayment getByName(String name) {
		Session session = null;
		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.like(CustomPayment.PROP_NAME, name));
			addDeletedFilter(criteria);
			return (CustomPayment) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateCustomPaymentList(List<CustomPayment> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<CustomPayment> iterator = dataList.iterator(); iterator.hasNext();) {
				CustomPayment item = (CustomPayment) iterator.next();
				CustomPayment existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public CustomPayment get(String id, String outletId) {
		return get(new CustomPayment(id, outletId));
	}

	public boolean nameExists(String name, String id) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne(CustomPayment.PROP_ID, id));
			}
			criteria.add(Restrictions.eq(CustomPayment.PROP_NAME, name).ignoreCase());
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			closeSession(session);
		}
	}

}