/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import com.floreantpos.payment.PaymentPlugin;

public enum PaymentType {
	//@formatter:off
	CASH("CASH"), //$NON-NLS-1$
	CUSTOM_PAYMENT("CUSTOM PAYMENT"), //$NON-NLS-1$
	CREDIT_CARD("CREDIT CARD"), //$NON-NLS-1$ //$NON-NLS-2$
	DEBIT_CARD("DEBIT CARD"), //$NON-NLS-1$ //$NON-NLS-2$
	//DEBIT_VISA("VISA", "visa_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	//DEBIT_MASTER_CARD("MASTER CARD", "master_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	//CREDIT_VISA("VISA", "visa_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	//CREDIT_MASTER_CARD("MASTER CARD", "master_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	//CREDIT_AMEX("AMEX", "am_ex_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	//CREDIT_DISCOVERY("DISCOVER", "discover_card.png"), //$NON-NLS-1$ //$NON-NLS-2$
	GIFT_CERTIFICATE("GIFT CERTIFICATE"), //$NON-NLS-1$ //$NON-NLS-2$
	MEMBER_ACCOUNT("CUSTOMER ACCOUNT"),//$NON-NLS-1$ //$NON-NLS-2$
	PROMOTION("PROMOTION"), //$NON-NLS-1$ 
	CARD("CARD") //$NON-NLS-1$ 
	; 
	//@formatter:on

	private String displayString;
	private String imageFile;

	private PaymentType(String display) {
		this.displayString = display;
	}

	private PaymentType(String display, String image) {
		this.displayString = display;
		this.imageFile = image;
	}

	@Override
	public String toString() {
		return displayString;
	}

	public String getDisplayString() {
		return displayString;
	}

	public void setDisplayString(String displayString) {
		this.displayString = displayString;
	}

	public String getImageFile() {
		return imageFile;
	}

	public void setImageFile(String imageFile) {
		this.imageFile = imageFile;
	};

	public boolean isSupported() {
		switch (this) {
			case CASH:
				return true;

			default:
				return true;
		}
	}

	public static PaymentType fromDisplayString(String displayString) {
		PaymentType[] paymentTypes = values();
		for (PaymentType paymentType : paymentTypes) {
			if (paymentType.getDisplayString().equals(displayString)) {
				return paymentType;
			}
		}
		return null;
	}
	
	public static PaymentType fromNameString(String displayString) {
		PaymentType[] paymentTypes = values();
		for (PaymentType paymentType : paymentTypes) {
			if (paymentType.name().equals(displayString)) {
				return paymentType;
			}
		}
		return null;
	}

	public PosTransaction createTransaction() {
		PosTransaction transaction = null;
		switch (this) {
			case CREDIT_CARD:
				//			case CREDIT_VISA:
				//			case CREDIT_AMEX:
				//			case CREDIT_DISCOVERY:
				//			case CREDIT_MASTER_CARD:
				transaction = new CreditCardTransaction();
				break;

			case DEBIT_CARD:
				//			case DEBIT_VISA:
				transaction = new DebitCardTransaction();
				break;

			case GIFT_CERTIFICATE:
				transaction = new GiftCertificateTransaction();
				break;

			case CUSTOM_PAYMENT:
				transaction = new CustomPaymentTransaction();
				break;

			case MEMBER_ACCOUNT:
				transaction = new CustomerAccountTransaction();
				break;

			default:
				transaction = new CashTransaction();
				break;
		}

		transaction.setPaymentType(this);
		return transaction;
	}

	public PosTransaction createTransaction(Ticket ticket, double tenderAmount) {
		PosTransaction transaction = null;
		switch (this) {
			case CREDIT_CARD:
				//			case CREDIT_VISA:
				//			case CREDIT_AMEX:
				//			case CREDIT_DISCOVERY:
				//			case CREDIT_MASTER_CARD:
				transaction = new CreditCardTransaction();
				transaction.setAuthorizable(true);
				break;

			case DEBIT_CARD:
				//			case DEBIT_VISA:
				transaction = new DebitCardTransaction();
				transaction.setAuthorizable(true);
				break;

			case GIFT_CERTIFICATE:
				transaction = new GiftCertificateTransaction();
				break;

			case CUSTOM_PAYMENT:
				transaction = new CustomPaymentTransaction();
				break;

			case MEMBER_ACCOUNT:
				transaction = new CustomerAccountTransaction();
				break;

			default:
				transaction = new CashTransaction();
				break;
		}

		transaction.setTicket(ticket);
		transaction.setTenderAmount(tenderAmount);
		if (ticket != null && tenderAmount >= ticket.getDueAmount()) {
			transaction.setAmount(ticket.getDueAmount());
		}
		else {
			transaction.setAmount(tenderAmount);
		}

		transaction.setPaymentType(this);
		return transaction;
	}

	public PaymentPlugin getPaymentPlugin() {
		switch (this) {
			//			case GIFT_CERTIFICATE:
			//				return (PaymentPlugin) ExtensionManager.getPlugin(GiftCardPaymentPlugin.class);
			//				
			//			case MEMBER_ACCOUNT:
			//				return new MemberPaymentPlugin();

			default:
				return null;
		}
	}
}
