package com.floreantpos.model.util;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.ext.UnitBaseItemPrice;
import com.floreantpos.util.NumberUtil;

public class InventoryUnitConvertionUtil {

	public static double calculateCost(Double unitCost, IUnit selectedUnit, MenuItem menuItem) {
		return calculateCost(unitCost, menuItem.getUnit(), selectedUnit, menuItem);
	}

	public static double calculateCost(Double unitCost, IUnit sourceUnit, IUnit targetUnit, MenuItem menuItem) {
		if (sourceUnit == null || targetUnit == null) {
			return unitCost;
		}
		if (targetUnit instanceof PackagingUnit) {
			PackagingUnit packagingUnit = (PackagingUnit) targetUnit;
			if (menuItem.getStockUnits() != null && !menuItem.getStockUnits().isEmpty()) {
				for (InventoryStockUnit stockUnit : menuItem.getStockUnits()) {
					if (stockUnit.getPackagingUnit().getId().equals(packagingUnit.getId())) {
						return getInventoryUnitCost(unitCost, (InventoryUnit) sourceUnit, stockUnit.getUnit()) * stockUnit.getConversionValue();
					}
				}
			}
		}
		else if (sourceUnit instanceof PackagingUnit) {
			PackagingUnit packagingUnit = (PackagingUnit) sourceUnit;
			if (menuItem.getStockUnits() != null && menuItem.getStockUnits().size() > 0) {
				for (InventoryStockUnit itemStockUnit : menuItem.getStockUnits()) {
					if (packagingUnit.getCode().equals(itemStockUnit.getPackagingUnit().getCode())) {
						return getInventoryUnitCost(unitCost, itemStockUnit.getUnit(), (InventoryUnit) targetUnit) / itemStockUnit.getConversionValue();
					}
				}
			}
		}
		else {
			InventoryUnit targetInventoryUnit = (InventoryUnit) targetUnit;
			if (!sourceUnit.getUniqueCode().equals(targetInventoryUnit.getUniqueCode())) {
				return getInventoryUnitCost(unitCost, (InventoryUnit) sourceUnit, targetInventoryUnit);
			}
		}
		return unitCost == null ? 0D : unitCost;
	}

	private static double getInventoryUnitCost(Double unitCost, InventoryUnit sourceUnit, InventoryUnit targetUnit) {
		double newUnitCost = unitCost;
		InventoryUnit inventoryUnit = targetUnit;
		String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE, InventoryUnitConversionRule.DIVISION.getName());
		if (!sourceUnit.isBaseUnit() && sourceUnit.getConversionRate() != 1) {
			newUnitCost = unitCost * sourceUnit.getConversionRate();
		}
		InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
		if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
			return newUnitCost * 1D / inventoryUnit.getConversionRate();
		}
		return newUnitCost / inventoryUnit.getConversionRate();
	}

	public static double calculatePrice(Double unitPrice, IUnit selectedUnit, MenuItem menuItem) {
		return calculatePrice(unitPrice, menuItem.getUnit(), selectedUnit, menuItem);
	}

	public static double calculatePrice(Double unitPrice, IUnit sourceUnit, IUnit targetUnit, MenuItem menuItem) {
		if (sourceUnit == null || targetUnit == null) {
			return unitPrice;
		}
		if (targetUnit instanceof PackagingUnit) {
			PackagingUnit packagingUnit = (PackagingUnit) targetUnit;
			if (menuItem.getStockUnits() != null && !menuItem.getStockUnits().isEmpty()) {
				for (InventoryStockUnit stockUnit : menuItem.getStockUnits()) {
					if (stockUnit.getPackagingUnit().getId().equals(packagingUnit.getId())) {
						return getInventoryUnitPrice(unitPrice, (InventoryUnit) sourceUnit, stockUnit.getUnit()) * stockUnit.getConversionValue();
					}
				}
			}
		}
		else {
			InventoryUnit targetInventoryUnit = (InventoryUnit) targetUnit;
			if (!sourceUnit.getUniqueCode().equals(targetInventoryUnit.getUniqueCode())) {
				return getInventoryUnitPrice(unitPrice, (InventoryUnit) sourceUnit, targetInventoryUnit);
			}
		}
		return unitPrice == null ? 0D : unitPrice;
	}

	private static double getInventoryUnitPrice(Double unitPrice, InventoryUnit sourceUnit, InventoryUnit targetUnit) {
		double newUnitPrice = unitPrice;
		InventoryUnit inventoryUnit = targetUnit;
		String inventoryUnitConversionRuleName = inventoryUnit.getProperty(InventoryUnit.PROP_CONVERSION_RULE, InventoryUnitConversionRule.DIVISION.getName());
		if (!sourceUnit.isBaseUnit() && sourceUnit.getConversionRate() != 1) {
			newUnitPrice = unitPrice * sourceUnit.getConversionRate();
		}
		InventoryUnitConversionRule inventoryUnitConversionRule = InventoryUnitConversionRule.fromName(inventoryUnitConversionRuleName);
		if (inventoryUnitConversionRule == InventoryUnitConversionRule.MULTIPLICATION) {
			return newUnitPrice * 1D / inventoryUnit.getConversionRate();
		}
		else {
			return newUnitPrice / inventoryUnit.getConversionRate();
		}
	}

	public static double getUnitQuantity(IUnit sourceInventoryUnit, IUnit destinationInventoryUnit, MenuItem menuItem) {
		if (destinationInventoryUnit instanceof PackagingUnit) {
			PackagingUnit stockUnit = (PackagingUnit) destinationInventoryUnit;
			if (menuItem.getStockUnits() != null && menuItem.getStockUnits().size() > 0) {
				for (InventoryStockUnit itemStockUnit : menuItem.getStockUnits()) {
					if (stockUnit.getCode().equals(itemStockUnit.getPackagingUnit().getCode())) {
						return 1.0 / (itemStockUnit.getConversionValue() * getBaseUnitQuantity(itemStockUnit.getUnit().getUniqueCode(), menuItem));
					}
				}
			}
		}
		if (sourceInventoryUnit instanceof PackagingUnit) {
			PackagingUnit stockUnit = (PackagingUnit) sourceInventoryUnit;
			if (menuItem.getStockUnits() != null && menuItem.getStockUnits().size() > 0) {
				for (InventoryStockUnit itemStockUnit : menuItem.getStockUnits()) {
					if (stockUnit.getCode().equals(itemStockUnit.getPackagingUnit().getCode())) {
						return destinationInventoryUnit.getConversionRate() * itemStockUnit.getBaseUnitValue();
					}
				}
			}
		}
		if (sourceInventoryUnit != null && destinationInventoryUnit != null) {
			return sourceInventoryUnit.getConversionRate() == 0 ? destinationInventoryUnit.getConversionRate()
					: destinationInventoryUnit.getConversionRate() / sourceInventoryUnit.getConversionRate();
		}
		return 1;
	}

	public static double getBaseUnitQuantity(String unitCode, MenuItem menuItem) {
		InventoryUnit myUnit = menuItem.getUnit();
		if (myUnit == null || myUnit.getCode().equals(unitCode)) {
			return 1;
		}
		if (menuItem.getStockUnits() != null) {
			for (InventoryStockUnit packUnit : menuItem.getStockUnits()) {
				if (packUnit.getPackagingUnit().getCode().equals(unitCode)) {
					if (!menuItem.getUnit().isBaseUnit()) {
						return menuItem.getUnit().getConversionRate() * packUnit.getBaseUnitValue();
					}
					return packUnit.getBaseUnitValue();
				}
			}
		}
		if (myUnit.getUnitGroup() != null && myUnit.getUnitGroup().getUnits() != null) {
			for (InventoryUnit groupUnit : myUnit.getUnitGroup().getUnits()) {
				if (groupUnit.getCode().equals(unitCode)) {
					return myUnit.getConversionRate() / groupUnit.getConversionRate();
				}
			}
		}
		return 1;
	}

	public static List<UnitBaseItemPrice> buildUnitBasePriceList(MenuItem menuItem, boolean regeneratePrice) {
		List<UnitBaseItemPrice> unitBasePrices = new ArrayList<>();
		for (IUnit unit : menuItem.getUnits()) {
			InventoryUnit inventoryUnit = menuItem.getUnit();
			if (unit.equals(inventoryUnit)) {
				continue;
			}
			Double unitBaseItemPrice = calculateUnitBaseItemPrice(menuItem, unit, regeneratePrice);
			UnitBaseItemPrice itemPrice = new UnitBaseItemPrice(unit.getName(), unit.getUniqueCode(), unitBaseItemPrice);
			unitBasePrices.add(itemPrice);
		}
		return unitBasePrices;
	}

	public static Double calculateUnitBaseItemPrice(MenuItem menuItem, IUnit unit, boolean regeneratePrice) {
		String key = MenuItem.PROP_UNIT_BASE_PRICE_KEY.replace("{name}", unit.getUniqueCode()); //$NON-NLS-1$
		String value = menuItem.getProperty(key);
		Double price = 0D;
		if (StringUtils.isBlank(value) || regeneratePrice) {
			price = calculatePrice(menuItem.getPrice(), menuItem.getUnit(), unit, menuItem);
		}
		else {
			price = NumberUtil.parseOrGetZero(value).doubleValue();
		}
		menuItem.addProperty(key, String.valueOf(price));
		return price;
	}
}
