/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.util.ArrayList;
import java.util.LinkedHashMap;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Currency;
import com.floreantpos.model.ITicketItem;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.dao.CurrencyDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.ui.ticket.TicketItemRowCreator;

public class TicketDataSource extends AbstractReportDataSource {

	private Store store;
	private String modifierTextColor;
	private String seatTextColor;
	private Ticket ticket;
	private String currencySymbol = "";

	public TicketDataSource() {
		super(new String[] { "itemName", "itemQty", "itemSubtotal", "colorCode" }); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		store = DataProvider.get().getStore();
	}

	public TicketDataSource(Ticket ticket) {
		this(ticket, false);
	}

	public TicketDataSource(Ticket ticket, boolean includeSeat) {
		super(new String[] { "itemName", "itemQty", "itemSubtotal", "colorCode" }); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		this.ticket = ticket;
		store = DataProvider.get().getStore();
		setTicket(ticket, includeSeat);
	}

	private void setTicket(Ticket ticket, boolean includeSeat) {
		ArrayList<ITicketItem> rows = new ArrayList<ITicketItem>();

		boolean suppressZerocostModifiers = Boolean.parseBoolean(DataProvider.get().getStore().getProperty(Store.STORE_PROP_SUPPRESS_ZERO_COST_MODIFIER));
		LinkedHashMap<String, ITicketItem> tableRows = new LinkedHashMap<String, ITicketItem>();
		TicketItemRowCreator.calculateTicketRows(ticket, tableRows, true, false, true, includeSeat, suppressZerocostModifiers);

		rows.addAll(tableRows.values());
		setRows(rows);
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		ITicketItem item = (ITicketItem) rows.get(rowIndex);
		if (item instanceof TicketItem) {
			TicketItem ticketItem = (TicketItem) item;
			ticketItem.setIncludeVoidQuantity(true);
		}
		if (StringUtils.isBlank(currencySymbol)) {
			if (StringUtils.isNotBlank(this.ticket.getOutletId())) {
				Currency currency = CurrencyDAO.getInstance().getMainCurrencyByOutletId(this.ticket.getOutletId());
				if (currency != null) {
					currencySymbol = currency.getSymbol();
				}
			}
		}

		switch (columnIndex) {
			case 0:
				String nameDisplay = item.getNameDisplay();//$NON-NLS-1$//$NON-NLS-2$
				nameDisplay = nameDisplay.replaceAll("<br/>", "&#10;");//$NON-NLS-1$//$NON-NLS-2$
				return "<html>" + nameDisplay + "</html>";//$NON-NLS-1$//$NON-NLS-2$;

			case 1:
				return ""; //$NON-NLS-1$
			//return item.getItemQuantityDisplay();

			case 2:
				if (item.getSubTotalAmountDisplay() == null) {
					return ""; //$NON-NLS-1$
				}
				if (item.getSubtotalAmount() < 0 && item.getSubTotalAmountDisplay().contains("-")) { //$NON-NLS-1$
					return item.getSubTotalAmountDisplay().replace("-", "-" + currencySymbol); //$NON-NLS-1$ //$NON-NLS-2$
				}
				return currencySymbol + item.getSubTotalAmountDisplay();
			case 3:
				return getColorCode(store, item);
		}

		return null;
	}

	public String getColorCode(Store store, ITicketItem item) {
		if (item instanceof TicketItemModifier) {
			if (modifierTextColor != null) {
				return modifierTextColor;
			}
			TicketItemModifier ticketItemModifier = (TicketItemModifier) item;
			Ticket ticket = ticketItemModifier.getTicketItem().getTicket();
			String orderTypeId = getOrderTypeId(ticket); //$NON-NLS-1$
			modifierTextColor = store.getProperty(orderTypeId + AppConstants.PROP_TICKET_MODIFIER_COLOR);
			return modifierTextColor;
		}
		if (item instanceof TicketItem) {
			TicketItem ticketItem = (TicketItem) item;
			if (!ticketItem.isTreatAsSeat()) {
				return Messages.getString("KitchenTicketDataSource.7");//$NON-NLS-1$
			}
			if (seatTextColor != null) {
				return seatTextColor;
			}
			Ticket ticket = ticketItem.getTicket();
			String orderTypeId = getOrderTypeId(ticket);
			seatTextColor = store.getProperty(orderTypeId + AppConstants.PROP_TICKET_SEAT_COLOR);
			return seatTextColor;
		}
		return Messages.getString("KitchenTicketDataSource.7"); //$NON-NLS-1$
	}

	private String getOrderTypeId(Ticket ticket) {
		String orderTypeId = "";
		if (ticket != null && ticket.getOrderType() != null) {
			orderTypeId = ticket.getOrderType().getId();
			if (StringUtils.isNotEmpty(orderTypeId)) {
				orderTypeId += "."; //$NON-NLS-1$
			}
		}
		return orderTypeId;
	}
}
