/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.StockCountItem;
import com.floreantpos.model.Terminal;
import com.floreantpos.util.DefaultDataInserter;
import com.orocube.rest.service.server.BaseDataServiceDao;

@SuppressWarnings({ "unchecked" })
public class InventoryLocationDAO extends BaseInventoryLocationDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryLocationDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session session) {
		InventoryLocation location = (InventoryLocation) obj;
		if (location == null) {
			throw new PosException(Messages.getString("InventoryLocationDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(location, session);
		location.setParentLocation(null);
		location.setDeleted(true);
		update(location, session);
	}

	private void checkForeignRelation(InventoryLocation bean, Session session) throws PosException {
		String id = bean.getId();
		StringBuilder ref = new StringBuilder();
		StringBuilder details = new StringBuilder();
		GenericDAO dao = GenericDAO.getInstance();

		List<InventoryLocation> childrenList = bean.getChildren();
		if (childrenList != null && !childrenList.isEmpty()) {
			ref.append(Messages.getString("InventoryLocationDAO.1") + ", "); //$NON-NLS-1$ //$NON-NLS-2$
			details.append(constructExceptionDetailsByInventoryLocations(bean, childrenList));
			details.append("\n \n"); //$NON-NLS-1$
		}

		append(dao.getForeignDataList(session, InventoryStock.class, InventoryStock.PROP_LOCATION_ID, id, InventoryStock.PROP_ITEM_NAME),
				Messages.getString("InventoryLocationDAO.3"), //$NON-NLS-1$
				ref, details);

		List<StockCountItem> stockCountItems = StockCountItemDAO.getInstance().getStockCountItemByInvLocation(bean, session);
		if (stockCountItems != null && !stockCountItems.isEmpty()) {
			ref.append(Messages.getString("InventoryLocationDAO.5") + ","); //$NON-NLS-1$//$NON-NLS-2$
			details.append(constructExceptionDetailsByStockCountItems(bean, stockCountItems));
			details.append("\n \n"); //$NON-NLS-1$
		}

		List<PurchaseOrder> purchaseOrders = PurchaseOrderDAO.getInstance().getPurchaseOrderByInvLocation(bean, session);
		if (purchaseOrders != null && !purchaseOrders.isEmpty()) {
			ref.append(Messages.getString("InventoryLocationDAO.6")); //$NON-NLS-1$
			details.append(constructExceptionDetailsByPurchaseOrders(bean, purchaseOrders));
			details.append("\n"); //$NON-NLS-1$
		}

		String foreignItemDetails = details.toString();
		if (StringUtils.isNotBlank(foreignItemDetails)) {
			String foreignReferences = ref.toString();
			if (foreignReferences.endsWith(", ")) { //$NON-NLS-1$
				foreignReferences = foreignReferences.substring(0, foreignReferences.length() - 2);
			}
			StringBuilder refHeading = new StringBuilder();
			String messageHeading = refHeading.append(Messages.getString("InventoryLocationDAO.7")).append(foreignReferences).append(". ").append("\n") //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
					.append(Messages.getString("InventoryLocationDAO.8")).toString(); //$NON-NLS-1$
			throw new PosException(messageHeading, foreignItemDetails);
		}
	}

	private void append(List<String> foreignDataListNames, String ref, StringBuilder refDetails, StringBuilder details) {
		if (foreignDataListNames == null || foreignDataListNames.isEmpty()) {
			return;
		}
		details.append(Messages.getString("InventoryLocationDAO.9") + ref + ":"); //$NON-NLS-1$//$NON-NLS-2$
		refDetails.append(ref + ", "); //$NON-NLS-1$
		int count = 1;
		for (String itemName : foreignDataListNames) {
			details.append("\n" + count + ". " + itemName); //$NON-NLS-1$ //$NON-NLS-2$
			count++;
		}
		details.append("\n \n"); //$NON-NLS-1$
	}

	private String constructExceptionDetailsByStockCountItems(InventoryLocation inventoryLocation, List<StockCountItem> stockCountItems) {
		if (stockCountItems != null && !stockCountItems.isEmpty()) {
			StringBuilder builder = new StringBuilder(Messages.getString("InventoryLocationDAO.9") + Messages.getString("InventoryLocationDAO.11")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < stockCountItems.size(); i++) {
				String message = (i + 1) + ". " + stockCountItems.get(i).getName(); //$NON-NLS-1$
				builder.append("\n ").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByPurchaseOrders(InventoryLocation inventoryLocation, List<PurchaseOrder> purchaseOrders) {
		if (purchaseOrders != null && !purchaseOrders.isEmpty()) {
			StringBuilder builder = new StringBuilder(Messages.getString("InventoryLocationDAO.9") + Messages.getString("InventoryLocationDAO.13")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < purchaseOrders.size(); i++) {
				String message = (i + 1) + ". " + purchaseOrders.get(i).getId(); //$NON-NLS-1$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	private String constructExceptionDetailsByInventoryLocations(InventoryLocation inventoryLocation, List<InventoryLocation> purchaseOrders) {
		if (purchaseOrders != null && !purchaseOrders.isEmpty()) {
			StringBuilder builder = new StringBuilder(Messages.getString("InventoryLocationDAO.14") + Messages.getString("InventoryLocationDAO.15")); //$NON-NLS-1$ //$NON-NLS-2$
			for (int i = 0; i < purchaseOrders.size(); i++) {
				String message = (i + 1) + ". " + purchaseOrders.get(i).getName(); //$NON-NLS-1$
				builder.append("\n").append(message); //$NON-NLS-1$
			}
			return builder.toString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	public List<InventoryLocation> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(Order.asc(InventoryLocation.PROP_NAME));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public InventoryLocation initialize(InventoryLocation inventoryLocation) {
		if (inventoryLocation == null || inventoryLocation.getId() == null)
			return inventoryLocation;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(inventoryLocation);

			Hibernate.initialize(InventoryLocation.class);
			//Hibernate.initialize(inventoryLocation.getShifts());

			return inventoryLocation;
		} finally {
			closeSession(session);
		}
	}

	public InventoryLocation getRootLocation() {

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryLocation.PROP_ROOT, true));

			List list = criteria.list();

			if (list != null && list.size() > 0) {
				return (InventoryLocation) list.get(0);
			}

			return null;

		}
	}

	public List<InventoryLocation> getRootLocations() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryLocation.PROP_ROOT, true));
			return criteria.list();
		}
	}

	public InventoryLocation getDefaultInInventoryLocation() {
		return getDefaultInInventoryLocation(null, null);
	}

	public InventoryLocation getDefaultInInventoryLocation(String outletId, Terminal terminal) {
		if (terminal != null) {
			String currentTerminalInventoryInLoc = terminal.getInventoryInLocationId();
			if (StringUtils.isNotBlank(currentTerminalInventoryInLoc)) {
				InventoryLocation location = get(new InventoryLocation(currentTerminalInventoryInLoc, outletId));
				if (location != null && !location.isDeleted()) {
					return location;
				}
			}
		}
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(InventoryLocation.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryLocation.PROP_DEFAULT_IN_LOCATION, true));
			if (outletId != null) {
				criteria.add(Restrictions.eq(InventoryLocation.PROP_OUTLET_ID, outletId));
			}
			List<InventoryLocation> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return new DefaultDataInserter().createDefaultInventoryLocation(outletId);
			}
			return (InventoryLocation) result.get(0);

		} catch (Exception e) {
			PosLog.info(getClass(), "" + e); //$NON-NLS-1$
		}
		return null;
	}

	public InventoryLocation getDefaultOutInventoryLocation() {
		//TODO:
		return getDefaultOutInventoryLocation(null, null);
	}

	public InventoryLocation getDefaultOutInventoryLocation(String outletId, Terminal terminal) {
		if (terminal != null) {
			String currentTerminalInventoryOutLoc = terminal.getInventoryOutLocationId();
			if (StringUtils.isNotBlank(currentTerminalInventoryOutLoc)) {
				InventoryLocation location = get(new InventoryLocation(currentTerminalInventoryOutLoc, outletId));
				if (location != null && !location.isDeleted()) {
					return location;
				}
			}
		}
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(InventoryLocation.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryLocation.PROP_DEFAULT_OUT_LOCATION, true));
			if (outletId != null) {
				criteria.add(Restrictions.eq(InventoryLocation.PROP_OUTLET_ID, outletId));
			}
			List<InventoryLocation> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return new DefaultDataInserter().createDefaultInventoryLocation(outletId);
			}
			return (InventoryLocation) result.get(0);

		} catch (Exception e) {
			PosLog.info(getClass(), "" + e); //$NON-NLS-1$
		}
		return null;
	}

	public InventoryLocation getDefaultOutInventoryLocation(Session session) {
		Criteria criteria = session.createCriteria(InventoryLocation.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(InventoryLocation.PROP_DEFAULT_OUT_LOCATION, true));
		List<InventoryLocation> result = criteria.list();
		if (result == null || result.isEmpty()) {
			criteria = session.createCriteria(InventoryLocation.class);
			this.addDeletedFilter(criteria);
			criteria.setMaxResults(1);
			return (InventoryLocation) criteria.uniqueResult();
		}
		return (InventoryLocation) result.get(0);
	}

	public InventoryLocation getDefaultInInventoryLocation(Session session) {
		Criteria criteria = session.createCriteria(InventoryLocation.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(InventoryLocation.PROP_DEFAULT_IN_LOCATION, true));
		List<InventoryLocation> result = criteria.list();
		if (result == null || result.isEmpty()) {
			criteria = session.createCriteria(InventoryLocation.class);
			this.addDeletedFilter(criteria);
			criteria.setMaxResults(1);
			return (InventoryLocation) criteria.uniqueResult();
		}
		return (InventoryLocation) result.get(0);
	}

	public void saveOrUpdateInventoryLocation(List<InventoryLocation> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null) {
			return;
		}

		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			for (Iterator<InventoryLocation> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryLocation item = (InventoryLocation) iterator.next();
				InventoryLocation existingItem = get(new InventoryLocation(item.getId(), item.getOutletId()));
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					List<InventoryLocation> existingChildLocations = existingItem.getChildren();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setChildren(existingChildLocations);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public InventoryLocation get(String id, String outletId) {
		return get(new InventoryLocation(id, outletId));
	}

	public InventoryLocation get(String id, String outletId, Session session) {
		return get(new InventoryLocation(id, outletId), session);
	}
}