package com.orocube.workspace.subscription;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.joda.time.Period;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.util.StoreUtil;
import com.orocube.siiopa.common.model.SubscriptionStatus;
import com.stripe.Stripe;
import com.stripe.exception.CardException;
import com.stripe.exception.StripeException;
import com.stripe.model.Card;
import com.stripe.model.Coupon;
import com.stripe.model.Customer;
import com.stripe.model.PaymentSource;
import com.stripe.model.Plan;
import com.stripe.model.Price;
import com.stripe.model.PriceCollection;
import com.stripe.model.Product;
import com.stripe.model.ProductCollection;
import com.stripe.model.PromotionCode;
import com.stripe.model.PromotionCodeCollection;
import com.stripe.model.Source;
import com.stripe.model.Subscription;
import com.stripe.model.SubscriptionCollection;
import com.stripe.model.SubscriptionItem;
import com.stripe.model.Token;
import com.stripe.param.PriceListParams;
import com.stripe.param.ProductListParams;
import com.stripe.param.SubscriptionCancelParams;
import com.stripe.param.SubscriptionCreateParams;
import com.stripe.param.SubscriptionCreateParams.CollectionMethod;
import com.stripe.param.SubscriptionCreateParams.PaymentBehavior;
import com.stripe.param.SubscriptionCreateParams.ProrationBehavior;
import com.stripe.param.SubscriptionListParams;

public class StripeUtil {
	private static final String STRIPE_CARD_HOLDER_NAME = "name"; //$NON-NLS-1$
	private static final String STRIPE_METADATA = "metadata"; //$NON-NLS-1$
	private static final String STRIPE_CARD_EXP_YEAR = "exp_year"; //$NON-NLS-1$
	private static final String STRIPE_CARD_EXP_MONTH = "exp_month"; //$NON-NLS-1$
	private static final String STRIPE_CARD_NUMBER = "number"; //$NON-NLS-1$
	private static final String STRIPE_CARD_CVC = "cvc"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_CARD = "card"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_SOURCE = "source"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_ID = "id"; //$NON-NLS-1$
	private static final String STRIPE_CUSTOMER_EMAIL = "email"; //$NON-NLS-1$

	public static List<Product> getProducts(final String apiKey) {
		Stripe.apiKey = apiKey;
		try {
			ProductCollection collection = Product.list(ProductListParams.builder().setLimit(100L).setActive(true).build());
			List<Product> products = collection.getData();
			while (collection.getHasMore()) {
				products.addAll((collection = Product.list(ProductListParams.builder().setLimit(100L)
						.setStartingAfter(collection.getData().get(collection.getData().size() - 1).getId()).setActive(true).build())).getData());
			}
			return products;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static List<Price> getPrices(String apiKey, String productId) {
		Stripe.apiKey = apiKey;
		try {
			PriceCollection collection = Price.list(PriceListParams.builder().setLimit(100L).setProduct(productId).build());
			List<Price> plans = collection.getData();
			while (collection.getHasMore()) {
				plans.addAll((collection = Price.list(PriceListParams.builder().setLimit(100L).setProduct(productId)
						.setStartingAfter(collection.getData().get(collection.getData().size() - 1).getId()).build())).getData());
			}
			return plans;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static List<Card> getCards(String apiKey, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer stripeCustomer = Customer.retrieve(customerId);
			if (stripeCustomer == null) {
				return null;
			}
			List<Card> cards = new ArrayList<>();
			List<PaymentSource> accounts = stripeCustomer.getSources().getData();
			if (!accounts.isEmpty()) {
				for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
					PaymentSource account = (PaymentSource) iterator.next();
					if (account instanceof Card) {
						Card creditCard = (Card) account;
						cards.add(creditCard);
					}
				}
			}
			return cards;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription getSubscriptionById(String apiKey, String subscriptionId) {
		Stripe.apiKey = apiKey;
		try {
			return Subscription.retrieve(subscriptionId);
		} catch (Exception e) {
			PosLog.error(StripeUtil.class, e.getMessage());
		}
		return null;
	}

	public static String getSubscriptionStatus(String apiKey, String customerId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = getSubscription(apiKey, customerId, productId);
			return getSubscriptionStatus(subscription);
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static String getSubscriptionStatus(Subscription subscription) {
		if (subscription == null) {
			return null;
		}
		if (subscription.getTrialEnd() != null && !subscription.getStatus().equals("active")) {
			java.util.Date currentDate = new java.util.Date();
			java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
			Period p = new Period(currentDate.getTime(), expiredDate.getTime());
			long days = p.getDays();
			long hours = p.getHours();
			long minutes = p.getMinutes();
			long seconds = p.getSeconds();
			if (days < 0 || hours < 0 || minutes < 0 || seconds < 0) {
				return "trail_end"; //$NON-NLS-1$
			}
		}
		return subscription.getStatus();
	}

	public static boolean hasActiveTrialPeriod(Subscription subscription) {
		if (subscription == null) {
			return false;
		}
		if (subscription.getTrialEnd() == null || subscription.getStatus().equals("active")) { //$NON-NLS-1$
			return false;
		}
		java.util.Date currentDate = new java.util.Date();
		java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);

		Period p = new Period(currentDate.getTime(), expiredDate.getTime());
		long days = p.getDays();
		long hours = p.getHours();
		long minutes = p.getMinutes();
		long seconds = p.getSeconds();
		if (days > 0 || hours > 0 || minutes > 0 || seconds > 0) {
			return true;
		}
		return false;
	}

	public static String getSubscriptionTrialStatus(String stripeApiKey, Subscription subscription, SubscriptionProduct product) {
		if (subscription == null) {
			subscription = getLastSubscription(stripeApiKey, product);
			if (subscription == null) {
				return null;
			}
		}
		if (subscription.getTrialEnd() == null || subscription.getStatus().equals("active")) { //$NON-NLS-1$
			return null;
		}
		java.util.Date currentDate = new java.util.Date();
		java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
		Period p = new Period(currentDate.getTime(), expiredDate.getTime());

		long days = p.getDays();
		long hours = p.getHours();
		long minutes = p.getMinutes();
		long seconds = p.getSeconds();
		if (days < 0 || hours < 0 || minutes < 0 || seconds < 0) {
			return Messages.getString("StripeUtil.1") + " " + product.getLabel() + " " + Messages.getString("StripeUtil.2"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}

		String trialMsg = Messages.getString("StripeUtil.3"); //$NON-NLS-1$
		if (days > 0) {
			return trialMsg + " " + days + " " + (days > 1 ? Messages.getString("StripeUtil.4") : Messages.getString("StripeUtil.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (hours > 0) {
			return trialMsg + " " + hours + " " + (hours > 1 ? Messages.getString("StripeUtil.6") : Messages.getString("StripeUtil.7")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (minutes > 0) {
			return trialMsg + " " + minutes + " " + (minutes > 1 ? Messages.getString("StripeUtil.8") : Messages.getString("StripeUtil.9")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (seconds > 0) {
			return trialMsg + " " + seconds + " " + (seconds > 1 ? Messages.getString("StripeUtil.10") : Messages.getString("StripeUtil.11")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		return null;
	}

	public static Subscription getCurrentSubscription(String stripeApiKey, String customerId, SubscriptionProduct product) {
		Subscription subscription = getSubscription(stripeApiKey, customerId, product.getId());
		if (subscription != null) {
			return subscription;
		}
		return getLastSubscription(stripeApiKey, product);
	}

	public static Subscription getLastSubscription(String stripeApiKey, SubscriptionProduct product) {
		String lastSubscriptionId = GlobalConfigDAO.getInstance().getProperty(product.getSubscriptionKey() + ".id"); //$NON-NLS-1$
		if (StringUtils.isBlank(lastSubscriptionId)) {
			return null;
		}
		return getSubscriptionById(stripeApiKey, lastSubscriptionId);
	}

	public static Integer getSubscriptionTrailEndDayCount(String apiKey, String customerId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = getSubscription(apiKey, customerId, productId);
			if (subscription == null) {
				return null;
			}
			if (subscription.getTrialEnd() != null && !subscription.getStatus().equals("active")) { //$NON-NLS-1$
				java.util.Date currentDate = new java.util.Date();
				java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
				return Days.daysBetween(new LocalDate(currentDate.getTime()), new LocalDate(expiredDate.getTime())).getDays();
			}
			return null;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription getActiveSubscription(String apiKey, String customerId, String productId) {
		List<Subscription> subscriptions = getSubscriptions(apiKey, customerId, productId);
		if (subscriptions == null || subscriptions.isEmpty()) {
			return null;
		}
		for (Subscription subscription : subscriptions) {
			if (subscription.getItems() == null || subscription.getItems().getData() == null) {
				continue;
			}
			for (SubscriptionItem item : subscription.getItems().getData()) {
				if (item.getPrice().getProduct().equals(productId)) {
					if (subscription.getTrialEnd() != null || !isCanceled(subscription)) {
						return subscription;
					}
				}
			}
		}
		return null;
	}

	public static Subscription getSubscription(String apiKey, String customerId, String productId) {
		List<Subscription> subscriptions = getSubscriptions(apiKey, customerId, productId);
		if (subscriptions != null && !subscriptions.isEmpty()) {
			for (Subscription subscription : subscriptions) {
				if (subscription.getStatus() != null && subscription.getStatus().equals("active")) { //$NON-NLS-1$
					return subscription;
				}
			}
			return subscriptions.get(0);
		}
		return null;
	}

	public static boolean isCanceled(Subscription subscription) {
		return subscription.getStatus() == null || !subscription.getStatus().equals(SubscriptionStatus.ACTIVE.getStatusName());
	}

	public static List<Subscription> getSubscriptions(String apiKey, String customerId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			if (StringUtils.isBlank(customerId) || StringUtils.isBlank(productId)) {
				return null;
			}
			List<Subscription> validSubscriptions = new ArrayList<>();
			SubscriptionCollection subscriptionCollection = Subscription.list(SubscriptionListParams.builder().setCustomer(customerId).build());
			List<Subscription> subscriptions = subscriptionCollection.getData();
			for (Subscription s : subscriptions) {
				Plan p = s.getItems().getData().get(0).getPlan();
				if (p.getProduct().equals(productId)) {
					validSubscriptions.add(s);
				}
			}
			Collections.sort(validSubscriptions, new Comparator<Subscription>() {

				@Override
				public int compare(Subscription o1, Subscription o2) {
					if (o1.getStatus() == null || o2.getStatus() == null) {
						return 0;
					}
					return o1.getStatus().compareTo(o2.getStatus());
				}
			});
			return validSubscriptions;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static void doCancelNonActiveSubscriptionIfNeeded(String stripeApiKey, String customerId, SubscriptionProduct product) {
		try {
			List<Subscription> subscriptions = getSubscriptions(stripeApiKey, customerId, product.getId());
			doCancelNonActiveSubscriptionIfNeeded(stripeApiKey, product, subscriptions);
		} catch (Exception e) {
			PosLog.error(StripeUtil.class, e);
		}
	}

	public static void doCancelNonActiveSubscriptionIfNeeded(String stripeApiKey, SubscriptionProduct product, List<Subscription> subscriptions) {
		if (subscriptions == null || subscriptions.isEmpty()) {
			return;
		}
		for (Subscription oldSubscription : subscriptions) {
			if (oldSubscription.getStatus() == null || !oldSubscription.getStatus().equals("active")) { //$NON-NLS-1$
				List<SubscriptionItem> items = oldSubscription.getItems().getData();
				if (items == null || items.size() == 0) {
					continue;
				}
				Price price = items.get(0).getPrice();
				if (price.getProduct().equals(product.getId())) {
					doCancelSubscription(stripeApiKey, oldSubscription);
				}
			}
		}
	}

	public static Subscription doCancelSubscription(String apiKey, Subscription subscription) {
		Stripe.apiKey = apiKey;
		try {
			subscription = Subscription.retrieve(subscription.getId());
			if (subscription.getStatus().equals("canceled")) { //$NON-NLS-1$
				return subscription;
			}
			subscription = subscription.cancel(SubscriptionCancelParams.builder().setInvoiceNow(true).setProrate(true).build());
			return subscription;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Customer getCustomerById(String apiKey, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = Customer.retrieve(customerId);
			return customer;
		} catch (Exception e0) {
			PosLog.error(StoreUtil.class, e0.getMessage());
		}
		return null;
	}

	public static Customer createCustomer(String apiKey, String customerEmail, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = getCustomerById(apiKey, customerId);
			if (customer == null) {
				customer = createCustomer(customerEmail, customerId);
			}
			return customer;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	private static Customer createCustomer(String customerEmail, String customerId) throws StripeException {
		Map<String, Object> customerParams = new HashMap<>();
		customerParams.put(STRIPE_CUSTOMER_EMAIL, customerEmail);
		if (StringUtils.isNotEmpty(customerId)) {
			customerParams.put(STRIPE_MODEL_ID, customerId);
		}
		return Customer.create(customerParams);
	}

	public static Token createToken(String apiKey, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc, String outletId) {
		return createToken(apiKey, null, cardNumber, cardExpMonth, cardExpYear, cvc, outletId);
	}

	public static Token createToken(String apiKey, String cardHolderName, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc,
			String outletId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();

			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			cardMetaData.put(outletId, true);

			Map<String, Object> cardParams = new HashMap<String, Object>();
			if (cardHolderName != null) {
				cardParams.put("name", cardHolderName); //$NON-NLS-1$
			}
			cardParams.put(STRIPE_CARD_NUMBER, cardNumber);
			cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
			cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
			cardParams.put(STRIPE_CARD_CVC, cvc);
			cardParams.put(STRIPE_METADATA, cardMetaData);

			tokenParams.put(STRIPE_MODEL_CARD, cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static Card createOrUpdateCard(String apiKey, String customerId, String stripeCardId, String cardHolderName, String cardNumber, Integer cardExpMonth,
			Integer cardExpYear, String cvc, String outletId) {
		Customer customer = getCustomerById(apiKey, customerId);
		Card card = null;
		if (StringUtils.isNotBlank(stripeCardId)) {
			card = getCardById(customer, stripeCardId);
		}
		return createOrUpdateCard(apiKey, customer, card, cardHolderName, cardNumber, cardExpMonth, cardExpYear, cvc, outletId);
	}

	public static Card createOrUpdateCard(String apiKey, Customer customer, Card stripeCard, String cardHolderName, String cardNumber, Integer cardExpMonth,
			Integer cardExpYear, String cvc, String outletId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			if (outletId != null) {
				cardMetaData.put(outletId, true);
			}
			if (stripeCard == null) {
				Map<String, Object> tokenParams = new HashMap<String, Object>();
				Map<String, Object> cardParams = new HashMap<String, Object>();
				if (StringUtils.isNotBlank(cardHolderName)) {
					cardParams.put(STRIPE_CARD_HOLDER_NAME, cardHolderName);
				}
				cardParams.put(STRIPE_CARD_NUMBER, cardNumber);
				cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
				cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
				cardParams.put(STRIPE_CARD_CVC, cvc);
				cardParams.put(STRIPE_METADATA, cardMetaData);

				tokenParams.put(STRIPE_MODEL_CARD, cardParams);
				Token token = Token.create(tokenParams);

				Map<String, Object> sourceParams = new HashMap<>();
				sourceParams.put(STRIPE_MODEL_SOURCE, token.getId());
				return (Card) customer.getSources().create(sourceParams);
			}
			else {
				Map<String, Object> cardParams = new HashMap<String, Object>();
				if (StringUtils.isNotBlank(cardHolderName)) {
					cardParams.put(STRIPE_CARD_HOLDER_NAME, cardHolderName);
				}
				cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
				cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
				cardParams.put(STRIPE_METADATA, cardMetaData);
				return (Card) stripeCard.update(cardParams);
			}
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static Subscription createSubscription(String apiKey, String customerId, String stripePriceId, String lookupKey, String cardId, String promoCode,
			String discountId, String productId, Long planQuantity) {
		Stripe.apiKey = apiKey;
		try {
			PromotionCode promotion = getPromotionByCode(apiKey, promoCode, productId);
			Coupon defaultCoupon = promotion != null ? null : getCoupon(apiKey, discountId, productId);

			//@formatter:off
			SubscriptionCreateParams params =
					SubscriptionCreateParams.builder()
			    .setCancelAtPeriodEnd(false)
			    .setCustomer(customerId)
			    .setDefaultPaymentMethod(cardId)
			    .setDefaultSource(cardId)
			    .setPaymentBehavior(PaymentBehavior.ERROR_IF_INCOMPLETE)
			    .setCollectionMethod(CollectionMethod.CHARGE_AUTOMATICALLY)
			    .setProrationBehavior(ProrationBehavior.ALWAYS_INVOICE)
			    .setPromotionCode(promotion==null?null:promotion.getId())
			    .setCoupon(defaultCoupon==null?null:defaultCoupon.getId())
			    .addItem(
			    		SubscriptionCreateParams.Item.builder()
			        .setQuantity(planQuantity)
			        .setPrice(stripePriceId)
			        .build())
			    .build();
			//@formatter:on
			return Subscription.create(params);
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription createTrialSubscription(String apiKey, String storeId, String customerId, String customerEmail, String productId,
			String planId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = getCustomerById(apiKey, customerId);
			if (customer == null) {
				createCustomer(customerEmail, customerId);
			}
			else {
				Subscription subscription = getSubscription(apiKey, customerId, productId);
				if (subscription != null) {
					return subscription;
				}
			}
			String priceId = null;
			List<Price> stripePricePlans = getPrices(apiKey, productId);
			if (stripePricePlans != null && stripePricePlans.size() > 0) {
				for (Price price : stripePricePlans) {
					if (price.getLookupKey() != null && price.getLookupKey().equals(planId)) {
						priceId = price.getId();
					}
				}
			}
			if (priceId == null) {
				return null;
			}
			Calendar calendar = Calendar.getInstance();
			if (apiKey.startsWith("sk_test")) { //$NON-NLS-1$
				calendar.add(Calendar.DAY_OF_MONTH, 1);
			}
			else {
				calendar.add(Calendar.DAY_OF_MONTH, 30);
			}
			long timeInMillis = calendar.getTimeInMillis();
			//@formatter:off
			SubscriptionCreateParams params =
					SubscriptionCreateParams.builder()
			    .setCancelAtPeriodEnd(true)
			    .setCustomer(customerId)
			    .setPaymentBehavior(PaymentBehavior.ERROR_IF_INCOMPLETE)
			    .setTrialEnd(timeInMillis / 1000L)
			    .addItem(
			    		SubscriptionCreateParams.Item.builder()
			        .setQuantity(1L)
			        .setPrice(priceId)
			        .build())
			    .build();
			//@formatter:on
			return Subscription.create(params);
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Card getCardById(String apiKey, String customerId, String cardId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = Customer.retrieve(customerId);
			PaymentSource account = null;
			Card card = null;
			try {
				account = customer.getSources().retrieve(cardId);
				card = (Card) account;
			} catch (Exception e1) {
				if (account != null) {
					Source source = Source.retrieve(cardId);
					source.detach();
				}
			}
			return card;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Card getCardById(Customer stripeCustomer, String cardId) {
		if (stripeCustomer == null) {
			return null;
		}
		List<PaymentSource> accounts = stripeCustomer.getSources().getData();
		if (!accounts.isEmpty()) {
			for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
				PaymentSource account = (PaymentSource) iterator.next();
				if (account instanceof Card) {
					Card creditCard = (Card) account;
					if (creditCard.getId().equals(cardId)) {
						return creditCard;
					}
				}
			}
		}
		return null;
	}

	public static Card removeStripeCard(String apiKey, Card stripeCard) {
		Stripe.apiKey = apiKey;
		try {
			return stripeCard.delete();
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static PromotionCode getPromotionByCode(String apiKey, String promoCode, String productId) {
		if (StringUtils.isBlank(promoCode)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> params = new HashMap<>();
			params.put("code", promoCode); //$NON-NLS-1$  
			params.put("limit", 1); //$NON-NLS-1$  
			PromotionCodeCollection promotionCodes = PromotionCode.list(params);
			if (promotionCodes == null || promotionCodes.getData() == null || promotionCodes.getData().isEmpty()) {
				return null;
			}
			PromotionCode promotion = promotionCodes.getData().get(0);
			if (promotion == null) {
				return null;
			}
			if (!promotion.getActive()) {
				if (promotion.getExpiresAt() != null) {
					java.util.Date currentDate = new java.util.Date();
					java.util.Date expiredDate = new java.util.Date(promotion.getExpiresAt() * 1000L);
					if (currentDate.after(expiredDate)) {
						throw new PosException(Messages.getString("StripeUtil.12")); //$NON-NLS-1$
					}
				}
				return null;
			}
			if (getCoupon(apiKey, promotion.getCoupon().getId(), productId) == null) {
				return null;
			}
			return promotion;
		} catch (PosException e0) {
			throw e0;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0.getMessage());
		}
		return null;
	}

	public static Coupon getCoupon(String apiKey, String couponId, String productId) {
		if (StringUtils.isBlank(couponId)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			List<String> expandList = new ArrayList<>();
			expandList.add("applies_to"); //$NON-NLS-1$  

			Map<String, Object> params = new HashMap<String, Object>();
			params.put("expand", expandList); //$NON-NLS-1$  
			Coupon coupon = Coupon.retrieve(couponId, params, null);
			if (!isValidCoupon(coupon, productId)) {
				return null;
			}
			return coupon;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0.getMessage());
		}
		return null;
	}

	public static boolean isValidCoupon(Coupon coupon, String productId) {
		if (coupon == null || !coupon.getValid()) {
			return false;
		}
		if (StringUtils.isNotBlank(productId) && coupon.getAppliesTo() != null) {
			List<String> products = coupon.getAppliesTo().getProducts();
			if (products != null && products.size() > 0) {
				if (!products.contains(productId)) {
					return false;
				}
			}
		}
		return true;
	}

	public static void setSubscriptionPropertiesToGlobalConfig(Subscription subscription, String productId) {
		SubscriptionProduct product = SubscriptionProduct.getById(productId);
		if (product != null) {
			String active = subscription != null && subscription.getStatus().equals("active") ? "true" : "false"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String subscriptionId = subscription == null ? "" : subscription.getId(); //$NON-NLS-1$
			GlobalConfigDAO instance = GlobalConfigDAO.getInstance();
			instance.addProperty(product.getSubscriptionKey(), active);
			instance.addProperty(product.getSubscriptionKey() + ".id", subscriptionId); //$NON-NLS-1$
			List<SubscriptionItem> data = subscription.getItems().getData();
			if (data != null && data.size() > 0) {
				instance.addProperty(product.getSubscriptionKey() + ".plan", data.get(0).getPrice().getLookupKey());
			}
		}
	}

	public static boolean isTrialingOrActive(Subscription subscription) {
		return subscription.getStatus().equals("active") || subscription.getStatus().equals("trialing"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static boolean isMenugreateStarterPlan() {
		String planId = GlobalConfigDAO.getInstance().getProperty(SubscriptionProduct.Menugreat.getSubscriptionKey() + ".plan"); //$NON-NLS-1$
		SubscriptionPlan subscriptionPlan = SubscriptionPlan.getById(planId);
		if (subscriptionPlan == null) {
			return true;
		}
		if (subscriptionPlan == SubscriptionPlan.StandardMonthly || subscriptionPlan == SubscriptionPlan.StandardYearly
				|| subscriptionPlan == SubscriptionPlan.ProMonthly || subscriptionPlan == SubscriptionPlan.ProYearly) {
			return false;
		}
		return true;
	}

	public static boolean hasPosLiveSubscription(String stripeApiKey, String storeId, String storeOwnerEmail) {
		Customer stripeCustomer = getCustomerById(stripeApiKey, storeId);
		String oldSubscriptionId = GlobalConfigDAO.getInstance().getProperty(SubscriptionProduct.OroposLive.getSubscriptionKey() + ".id"); //$NON-NLS-1$
		if (stripeCustomer == null || StringUtils.isBlank(oldSubscriptionId)) {
			return false;
		}
		String subscriptionStatus = getSubscriptionStatus(stripeApiKey, storeId, SubscriptionProduct.OroposLive.getId());
		if (subscriptionStatus == null) {
			if (StringUtils.isNotBlank(oldSubscriptionId)) {
				Subscription oldSubscription = getSubscriptionById(stripeApiKey, oldSubscriptionId);
				subscriptionStatus = StripeUtil.getSubscriptionStatus(oldSubscription);
				if (subscriptionStatus != null) {
					if (subscriptionStatus.equals("trail_end")) { //$NON-NLS-1$
						throw new PosException(Messages.getString("SubsTrailEndMsg")); //$NON-NLS-1$
					}
					if (subscriptionStatus.equals("canceled")) { //$NON-NLS-1$
						throw new PosException(Messages.getString("SubsCanceledMsg")); //$NON-NLS-1$
					}
				}
			}
			throw new PosException(Messages.getString("SubsActiveMsg")); //$NON-NLS-1$
		}
		if (subscriptionStatus.equals("active")) { //$NON-NLS-1$
			return true;
		}
		if (subscriptionStatus.equals("trialing")) { //$NON-NLS-1$
			return true;
		}
		if (subscriptionStatus.equals("trail_end")) { //$NON-NLS-1$
			throw new PosException(Messages.getString("SubsTrailEndMsg")); //$NON-NLS-1$
		}
		if (subscriptionStatus.equals("canceled") || subscriptionStatus.equals("invalid")) { //$NON-NLS-1$ //$NON-NLS-2$
			throw new PosException(Messages.getString("SubsCanceledMsg")); //$NON-NLS-1$
		}
		throw new PosException(Messages.getString("SubsPaymentFailedMsg")); //$NON-NLS-1$
	}

	public static void createPosLiveTrialSubscription(String stripeApiKey, String storeId, String storeOwnerEmail) {
		if (storeOwnerEmail == null) {
			return;
		}
		Subscription trialSubscription = createTrialSubscription(stripeApiKey, storeId, storeId, storeOwnerEmail, SubscriptionProduct.OroposLive.getId(),
				SubscriptionPlan.OroposLiveMonthly.getId());
		setSubscriptionPropertiesToGlobalConfig(trialSubscription, SubscriptionProduct.OroposLive.getId());
	}
}
