package com.floreantpos.model;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Index;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.TermVector;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseOnlineStore;
import com.floreantpos.model.dao.SiiopaCustomerDAO;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@Entity
@Indexed
public class OnlineStore extends BaseOnlineStore implements TimedModel, PropertyContainer, AddressContainer {
	private static final long serialVersionUID = 1L;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private transient JsonObject propertiesContainer;
	private List<WorkingHours> workingHours;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public OnlineStore() {
	}

	/**
	 * Constructor for primary key
	 */
	public OnlineStore(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	@Override
	public java.lang.String getFriendlyUrl() {
		if (StringUtils.isBlank(super.getFriendlyUrl())) {
			return getStoreId() + "___" + getOutletId(); //$NON-NLS-1$
		}
		return super.getFriendlyUrl();
	}

	public boolean isNotFriendlyUrl() {
		if (StringUtils.isBlank(super.getStoreId()) || StringUtils.isBlank(super.getOutletId())) {
			return true;
		}
		return getFriendlyUrl().equals(getStoreId() + "___" + getOutletId()); //$NON-NLS-1$
	}

	public String buildFriendlyUrl() {
		String friendlyUrl = ""; //$NON-NLS-1$
		if (StringUtils.isNotBlank(getStoreName())) {
			friendlyUrl = getStoreName().toLowerCase();
		}
		if (StringUtils.isNotBlank(getCity())) {
			friendlyUrl += "_" + getCity().toLowerCase(); //$NON-NLS-1$
		}
		if (StringUtils.isNotBlank(getZipCode())) {
			friendlyUrl += "_" + getZipCode(); //$NON-NLS-1$
		}
		friendlyUrl = friendlyUrl.replaceAll("\\W", ""); //$NON-NLS-1$ //$NON-NLS-2$
		return friendlyUrl;
	}

	@XmlTransient
	public List<WorkingHours> getWorkingHours() {
		if (workingHours == null) {
			workingHours = new ArrayList<>();
			for (int i = 1; i <= 7; i++) {
				WorkingHours workingHour = getWorkingHours(i);
				if (workingHour == null) {
					continue;
				}
				workingHours.add(workingHour);
			}
		}
		return workingHours;
	}

	public WorkingHours getWorkingHours(int day) {
		if (!getBooleanProperty(WorkingHours.days[day] + WorkingHours.OPEN, true)) {
			return null;
		}
		return new WorkingHours(day, this);
	}

	public void setWorkingHours(List<WorkingHours> workingHours) {
		this.workingHours = workingHours;
	}

	@Id
	@Override
	public String getId() {
		return super.getId();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.YES, index = Index.YES)
	@Override
	public String getStoreName() {
		return super.getStoreName();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@Override
	public String getOutletName() {
		return super.getOutletName();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isLive() {
		return super.isLive();
	}

	public void setOnlineDiscountIds(List<String> onlineDiscountIds) {
		JsonArray discountIdsJsonArray = new JsonArray();
		if (onlineDiscountIds != null && onlineDiscountIds.size() > 0) {
			for (String discountId : onlineDiscountIds) {
				discountIdsJsonArray.add(discountId);
			}
		}
		getPropertyStore().add("onlineDiscountIds", discountIdsJsonArray);//$NON-NLS-1$
	}

	public List<String> getOnlineDiscountIds() {
		JsonArray onlineDiscountJsonArray = getPropertyStore().getAsJsonArray("onlineDiscountIds");//$NON-NLS-1$
		if (onlineDiscountJsonArray == null) {
			return null;
		}
		List<String> discountIds = new ArrayList<>();
		for (int i = 0; i < onlineDiscountJsonArray.size(); i++) {
			JsonElement jsonElement = onlineDiscountJsonArray.get(i);
			if (jsonElement == null) {
				continue;
			}
			String discountId = jsonElement.getAsString();
			if (discountId == null) {
				continue;
			}
			discountIds.add(discountId);
		}
		return discountIds;
	}

	@JsonIgnore
	public String getOwnerEmail() {
		String storeId = getStoreId();
		List<SiiopaCustomer> siiopaCustomers = SiiopaCustomerDAO.getInstance().findByStoreId(storeId);
		if (siiopaCustomers.size() > 1) {
			PosLog.error(getClass(), "There is more than one siiopa customer for store: " + storeId);
		}
		if (siiopaCustomers.isEmpty()) {
			PosLog.error(getClass(), "No siiopa customer found for store: " + storeId);
			return null;
		}
		return siiopaCustomers.get(0).getEmail();
	}
}