/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;
import java.awt.Dimension;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.IconFactory;
import com.floreantpos.model.base.BaseShopFloor;
import com.floreantpos.model.util.DataProvider;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@JsonIgnoreProperties(ignoreUnknown = true, value = { "foregroundColor", "backgroundColor", "floorSize", "image", "orderTypes" })
public class ShopFloor extends BaseShopFloor implements TimedModel, PropertyContainer {
	public static final String JSON_PROP_ID = "id"; //$NON-NLS-1$

	public static final String JSON_PROP_ORDER_TYPES = "orderTypes"; //$NON-NLS-1$

	private static final long serialVersionUID = 1L;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private transient JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public ShopFloor() {
	}

	/**
	 * Constructor for primary key
	 */
	public ShopFloor(java.lang.String id, java.lang.String outletId) {

		super(id, outletId);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@XmlTransient
	public ImageIcon getImage() {
		return IconFactory.getIconFromImageResource(getImageId());
	}

	@Override
	public String toString() {
		return getName();
	}

	//	@XmlTransient
	public boolean hasTableWithNumber(String number) {
		Set<ShopTable> tables = getTables();
		if (tables == null) {
			return false;
		}

		for (ShopTable shopTable : tables) {
			if (shopTable.getTableNumber().equals(number)) {
				return true;
			}
		}

		return false;
	}

	@XmlTransient
	public Color getForegroundColor() {
		if (getForegroundColorCode() == null) {
			return null;
		}
		return new Color(getForegroundColorCode());
	}

	public void setForegroundColor(Color foregroundColor) {
		if (foregroundColor != null) {
			setForegroundColorCode(foregroundColor.getRGB());
		}
	}

	@XmlTransient
	public Color getBackgroundColor() {
		if (getBackgroundColorCode() == null) {
			return null;
		}
		return new Color(getBackgroundColorCode());
	}

	public void setBackgroundColor(Color backgroundColor) {
		if (backgroundColor != null) {
			setBackgroundColorCode(backgroundColor.getRGB());
		}
	}

	@XmlTransient
	public Dimension getFloorSize() {
		Integer height = getHeight();
		Integer width = getWidth();
		if (width <= 0) {
			width = 850;
		}
		if (height <= 0) {
			height = 500;
		}
		return new Dimension(width, height);
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	public void setOrderTypes(List<OrderType> orderTypes) {
		if (orderTypes == null || orderTypes.isEmpty()) {
			if (propertiesContainer != null) {
				propertiesContainer.remove(JSON_PROP_ORDER_TYPES);
			}
			return;
		}
		if (propertiesContainer == null) {
			propertiesContainer = getPropertyStore();
		}
		JsonArray orderTypesArray = new JsonArray();
		for (OrderType orderType : orderTypes) {
			JsonObject orderTypeObject = new JsonObject();
			orderTypeObject.addProperty(OrderType.PROP_ID, orderType.getId());
			orderTypesArray.add(orderTypeObject);
		}
		propertiesContainer.add(JSON_PROP_ORDER_TYPES, orderTypesArray);
	}

	@XmlTransient
	public List<OrderType> getOrderTypes() {
		List<OrderType> orderTypes = new ArrayList<OrderType>();
		if (propertiesContainer == null) {
			return orderTypes;
		}
		JsonArray orderTypesArray = propertiesContainer.getAsJsonArray(JSON_PROP_ORDER_TYPES);

		if (orderTypesArray == null || orderTypesArray.isJsonNull()) {
			return orderTypes;
		}
		for (JsonElement jsonElement : orderTypesArray) {
			JsonObject jsonObject = jsonElement.getAsJsonObject();
			if (jsonObject == null) {
				continue;
			}
			JsonElement id = jsonObject.get(JSON_PROP_ID);
			if (id == null) {
				continue;
			}
			OrderType orderType = DataProvider.get().getOrderType(id.getAsString(), getOutletId());
			if (orderType != null) {
				orderTypes.add(orderType);
			}

		}

		return orderTypes;

	}

}
