package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.OnlineCategory;
import com.floreantpos.model.Outlet;
import com.floreantpos.swing.PaginationSupport;

public class OnlineCategoryDAO extends BaseOnlineCategoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OnlineCategoryDAO() {
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public void saveOrUpdateOnlineCategories(List<MenuCategory> categoris, Outlet outlet) {
		if (categoris == null || categoris.isEmpty()) {
			return;
		}
		List<String> ids = new ArrayList<>();
		for (MenuCategory menuCategory : categoris) {
			ids.add(menuCategory.getId());
		}
		Map<String, OnlineCategory> existingDataMap = new HashMap<>();
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.in(OnlineCategory.PROP_CATEGORY_ID, ids));
			criteria.add(Restrictions.eq(OnlineCategory.PROP_OUTLET_ID, outlet.getId()));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				for (Iterator<OnlineCategory> iterator = list.iterator(); iterator.hasNext();) {
					OnlineCategory onlineCategory = (OnlineCategory) iterator.next();
					existingDataMap.put(onlineCategory.getCategoryId(), onlineCategory);
				}
			}
			Transaction tx = session.beginTransaction();
			for (MenuCategory menuCategory : categoris) {
				OnlineCategory onlineCategory = existingDataMap.get(menuCategory.getId());
				if (onlineCategory == null) {
					onlineCategory = new OnlineCategory();
					onlineCategory.setCategoryId(menuCategory.getId());
				}
				onlineCategory.setOutletId(outlet.getId());
				onlineCategory.setName(menuCategory.getName());
				saveOrUpdate(onlineCategory, session);
			}
			tx.commit();
		}
	}

	public void removeOnlineCategory(MenuCategory menuCategory, Outlet outlet) {
		OnlineCategory onlineCategory = findByCategoryId(menuCategory.getId(), outlet);
		if (onlineCategory == null) {
			return;
		}
		delete(onlineCategory);
	}

	public OnlineCategory findByCategoryId(String categoryId, Outlet outlet) {
		try (Session session = createNewSession()) {
			return findByCategoryId(categoryId, outlet, session);
		}
	}

	public OnlineCategory findByCategoryId(String categoryId, Outlet outlet, Session session) {
		if (StringUtils.isBlank(categoryId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OnlineCategory.PROP_CATEGORY_ID, categoryId));
		criteria.add(Restrictions.eq(OnlineCategory.PROP_OUTLET_ID, outlet.getId()));
		criteria.setMaxResults(1);
		return (OnlineCategory) criteria.uniqueResult();
	}

	public List<OnlineCategory> findByCategoryId(String categoryId, Session session) {
		if (StringUtils.isBlank(categoryId)) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OnlineCategory.PROP_CATEGORY_ID, categoryId));
		return criteria.list();
	}

	public void findOnlineCategoriesByName(PaginationSupport paginationSupport, String itemName, Outlet outlet, Boolean isOnlineCategory) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(OnlineCategory.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(OnlineCategory.PROP_OUTLET_ID, outlet.getId()));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			List onlineCategories = criteria.list();
			if (onlineCategories == null || onlineCategories.isEmpty()) {
				paginationSupport.setRows(new ArrayList<>());
				return;
			}
			List<String> categoryIds = new ArrayList<>();
			for (Iterator iterator = onlineCategories.iterator(); iterator.hasNext();) {
				OnlineCategory menuCategory = (OnlineCategory) iterator.next();
				categoryIds.add(menuCategory.getCategoryId());
			}
			criteria = session.createCriteria(MenuCategory.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.in(MenuCategory.PROP_ID, categoryIds));
			criteria.addOrder(Order.asc(MenuCategory.PROP_SORT_ORDER));
			paginationSupport.setRows(criteria.list());
		}
	}

	@SuppressWarnings("unchecked")
	public List<MenuCategory> getOnlineCategories(Outlet outlet) {
		List<String> onlineCategoryIds = null;
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(OnlineCategory.PROP_CATEGORY_ID));
			criteria.add(Restrictions.eq(OnlineCategory.PROP_OUTLET_ID, outlet.getId()));
			onlineCategoryIds = criteria.list();
			if (onlineCategoryIds == null || onlineCategoryIds.isEmpty()) {
				return null;
			}
			criteria = session.createCriteria(MenuCategory.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.in(MenuCategory.PROP_ID, onlineCategoryIds));
			return criteria.list();
		}
	}
}