/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
/*
 * MessageDialog.java
 *
 * Created on August 23, 2006, 10:45 PM
 */

package com.floreantpos.ui.dialog;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import org.apache.commons.lang.StringUtils;
import org.hibernate.JDBCException;
import org.hibernate.exception.ConstraintViolationException;
import org.hibernate.exception.GenericJDBCException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.floreantpos.IconFactory;
import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.main.Application;
import com.floreantpos.swing.PosUIManager;
import com.floreantpos.ui.RefreshableView;
import com.floreantpos.util.POSUtil;

/**
 * 
 * @author MShahriar
 */
public class POSMessageDialog {

	private static Logger logger = LoggerFactory.getLogger(Application.class);

	private static void showDialog(Component parent, String message, int messageType, int optionType) {
		showDialog(parent, message, messageType, optionType, null, null);
	}

	private static boolean showDialog(Component parent, String message, int messageType, int optionType, String option, final Throwable x) {
		if (StringUtils.isEmpty(message) || message.startsWith("Batch update")) { //$NON-NLS-1$
			message = Messages.getString("POSMessageDialog.0"); //$NON-NLS-1$ //An unexpected error has occurred. Please check log for detail.
		}
		if ((StringUtils.isNotBlank(message) && message.startsWith("Unable to acquire JDBC Connection"))) { //$NON-NLS-1$
			message = Messages.getString("POSMessageDialog.6"); //$NON-NLS-1$
		}
		String submitErrorText = Messages.getString("POSMessageDialog.1"); //$NON-NLS-1$ //Submit error
		JOptionPane optionPane = null;
		List<String> optionList = new ArrayList<>(3);
		if (option != null) {
			optionList.add(option);
		}
		if (messageType == JOptionPane.ERROR_MESSAGE && x != null && !(x instanceof PosException) && !(x instanceof JDBCException && !(x instanceof ConstraintViolationException))
				&& !(x instanceof GenericJDBCException)) {
			optionList.add(submitErrorText);
			logger.error(message, x);
		}
		optionList.add(POSConstants.OK);
		optionPane = new JOptionPane(message, messageType, optionType, null, optionList.toArray(new String[0]));

		customizeOption(optionPane, submitErrorText, x);

		String title = "";//$NON-NLS-1$
		if (messageType == JOptionPane.ERROR_MESSAGE) {
			title = Messages.getString("MessageDialog.0");//$NON-NLS-1$
		}
		else if (messageType == JOptionPane.INFORMATION_MESSAGE) {
			title = Messages.getString("POSMessageDialog.5"); //$NON-NLS-1$
		}
		else {
			title = "";
		}
		JDialog dialog = optionPane.createDialog(parent, title);
		dialog.setModal(true);
		dialog.setVisible(true);
		if (option == null)
			return false;

		Object selectedValue = optionPane.getValue();
		if (selectedValue != null && selectedValue.equals(option))
			return true;

		return false;
	}

	public static void showMessage(String message) {
		showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.INFORMATION_MESSAGE, JOptionPane.DEFAULT_OPTION);
	}

	public static void showMessage(Component parent, String message) {
		showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.INFORMATION_MESSAGE, JOptionPane.DEFAULT_OPTION);
	}

	public static void showError(String message) {
		showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.ERROR_MESSAGE, JOptionPane.DEFAULT_OPTION);
	}

	public static void showError(Component parent, String message) {
		showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.ERROR_MESSAGE, JOptionPane.DEFAULT_OPTION);
	}

	public static boolean showErrorWithOption(Component parent, String message, String option) {
		return showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.ERROR_MESSAGE, JOptionPane.DEFAULT_OPTION, option, null);
	}

	public static void showError(Component parent, String message, Throwable x) {
		showDialog(POSUtil.getFocusedWindow(), message, JOptionPane.ERROR_MESSAGE, JOptionPane.DEFAULT_OPTION, null, x);
	}

	public static int showYesNoQuestionDialog(String message) {
		return showYesNoQuestionDialog(POSUtil.getFocusedWindow(), message, POSConstants.CONFIRM, null, null);
	}

	public static int showYesNoQuestionDialog(String message, String title) {
		return showYesNoQuestionDialog(POSUtil.getFocusedWindow(), message, title, null, null);
	}

	public static int showYesNoQuestionDialog(Component parent, String message, String title) {
		return showYesNoQuestionDialog(parent, message, title, null, null);
	}

	public static int showYesNoQuestionDialog(Component parent, String message, String title, String yesButtonText, String noButtonText) {
		JOptionPane optionPane = null;
		if (yesButtonText != null && noButtonText != null) {
			optionPane = new JOptionPane(message, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_CANCEL_OPTION, null,
					new String[] { yesButtonText, noButtonText });

		}
		else {
			optionPane = new JOptionPane(message, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_OPTION);
		}
		Object[] options = optionPane.getComponents();
		for (Object object : options) {
			if (object instanceof JPanel) {
				JPanel panel = (JPanel) object;
				Component[] components = panel.getComponents();
				for (Component component : components) {
					if (component instanceof JButton) {
						component.setPreferredSize(new Dimension(component.getPreferredSize().width, 60));
					}
				}
			}
		}

		JDialog dialog = optionPane.createDialog(parent, title);
		dialog.setVisible(true);

		Object selectedValue = optionPane.getValue();
		if (selectedValue == null)
			return JOptionPane.CLOSED_OPTION;

		if (selectedValue instanceof String) {
			return selectedValue.equals(noButtonText) ? JOptionPane.NO_OPTION : JOptionPane.YES_OPTION;
		}
		return ((Integer) selectedValue).intValue();
	}

	public static String showOptionDialog(Component parent, String message, String title, String... options) {
		JOptionPane optionPane = null;
		if (options != null) {
			optionPane = new JOptionPane(message, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_CANCEL_OPTION, null, options);

		}
		else {
			optionPane = new JOptionPane(message, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_OPTION);
		}
		Object[] optionComponents = optionPane.getComponents();
		for (Object object : optionComponents) {
			if (object instanceof JPanel) {
				JPanel panel = (JPanel) object;
				Component[] components = panel.getComponents();
				for (Component component : components) {
					if (component instanceof JButton) {
						component.setPreferredSize(new Dimension(component.getPreferredSize().width, 60));
					}
				}
			}
		}

		JDialog dialog = optionPane.createDialog(parent, title);
		dialog.setVisible(true);

		Object selectedValue = optionPane.getValue();
		if (selectedValue != null)
			return selectedValue.toString();
		return null;
	}

	public static boolean showMessageAndPromtToPrint(String msg) {
		return showMessageAndPromtToPrint(null, msg);
	}

	public static boolean showMessageAndPromtToPrint(Component parent, String msg) {
		JOptionPane promtPane = new JOptionPane(msg, JOptionPane.INFORMATION_MESSAGE, JOptionPane.YES_NO_CANCEL_OPTION, null,
				new String[] { POSConstants.PRINT, POSConstants.OK.toUpperCase() });

		Object[] optionValues = promtPane.getComponents();
		for (Object object : optionValues) {
			if (object instanceof JPanel) {
				JPanel panel = (JPanel) object;
				Component[] components = panel.getComponents();

				boolean printIcon = true;
				for (Component component : components) {
					if (component instanceof JButton) {
						component.setPreferredSize(PosUIManager.getSize(100, 50));
						JButton button = (JButton) component;
						if (printIcon) {
							button.setIcon(IconFactory.getIcon("/ui_icons/", "print_32.png")); //$NON-NLS-1$ //$NON-NLS-2$
							button.setText(Messages.getString("POSMessageDialog.2")); //$NON-NLS-1$
							printIcon = false;
							button.setFocusable(false);
						}
					}
				}
			}
		}
		JDialog dialog = promtPane.createDialog(parent == null ? POSUtil.getFocusedWindow() : parent, "Message"); //$NON-NLS-1$
		dialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		dialog.setVisible(true);
		Object selectedValue = (String) promtPane.getValue();

		if (selectedValue.equals(POSConstants.PRINT)) {
			return true;
		}
		return false;
	}

	public static void showMessageDialogWithReloadButton(Component parent, RefreshableView refreshView) {
		showMessageDialogWithReloadButton(parent, refreshView, Messages.getString("POSMessageDialog.3")); //$NON-NLS-1$
	}

	public static void showMessageDialogWithReloadButton(Component parent, RefreshableView refreshView, String msg) {
		// @formatter:off
		JOptionPane reloadPane = new JOptionPane(msg, 
				JOptionPane.ERROR_MESSAGE, 
				JOptionPane.YES_NO_OPTION, null, 
				new String[] { Messages.getString("POSMessageDialog.4")}); //$NON-NLS-1$ //Reload
		// @formatter:on

		Object[] options = reloadPane.getComponents();
		for (Object object : options) {
			if (object instanceof JPanel) {
				JPanel panel = (JPanel) object;
				Component[] components = panel.getComponents();
				for (Component component : components) {
					if (component instanceof JButton) {
						component.setPreferredSize(new Dimension(component.getPreferredSize().width, 60));
					}
				}
			}
		}

		JDialog dialog = reloadPane.createDialog(parent == null ? POSUtil.getFocusedWindow() : parent, Messages.getString("MessageDialog.0"));//$NON-NLS-1$
		dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		dialog.setVisible(true);
		if (refreshView != null) {
			refreshView.refresh();
		}
	}

	private static void customizeOption(JOptionPane optionPane, String submitErrorText, final Throwable x) {
		Object[] options = optionPane.getComponents();
		for (Object object : options) {
			if (object instanceof JPanel) {
				JPanel panel = (JPanel) object;
				Component[] components = panel.getComponents();
				for (Component component : components) {
					if (component instanceof JButton) {
						JButton button = (JButton) component;
						component.setPreferredSize(new Dimension(component.getPreferredSize().width, 60));
						if (submitErrorText.equals(button.getText())) {
							createSubmitErrorAction(button, x);
						}
					}
				}
			}
		}
	}

	private static void createSubmitErrorAction(JButton button, final Throwable x) {
		button.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0) {
				
			}
		});
	}

	public static void showError(String message, Throwable e2) {
		showError(POSUtil.getFocusedWindow(), message, e2);
	}
}
