package com.floreantpos.extension.cardconnect;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.DatatypeConverter;

import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.extension.PaymentMethod;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.Currency;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

public class SecuredProcessor implements CardProcessor {
	private static final String STR_NEWLINE = "\n"; //$NON-NLS-1$
	private static final String GET = "GET"; //$NON-NLS-1$
	private static final String PUT = "PUT"; //$NON-NLS-1$
	private static final String DELETE = "DELETE"; //$NON-NLS-1$
	private static final String POST = "POST"; //$NON-NLS-1$
	private String username;
	private String password;
	private String merchantId;
	private String authString;
	private String encodedAuthString;
	private GsonBuilder gsonBuilder;

	public SecuredProcessor() {
		Outlet outlet = DataProvider.get().getOutlet();
		this.merchantId = outlet.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_MERCHANT_ID);
		this.username = outlet.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_USERNAME);
		this.password = outlet.getProperty(SecuredConstants.ONLINEORDER_CARDCONNECT_PASSWORD);

		this.authString = this.username + ":" + this.password; //$NON-NLS-1$
		this.encodedAuthString = Base64.getEncoder().encodeToString(this.authString.getBytes());
		this.gsonBuilder = new GsonBuilder();
	}

	public SecuredProcessor(String username2, String password2, String merchantId2) {
		this.username = username2;
		this.merchantId = merchantId2;
		this.password = password2;

		this.authString = this.username + ":" + this.password; //$NON-NLS-1$
		this.encodedAuthString = Base64.getEncoder().encodeToString(this.authString.getBytes());
		this.gsonBuilder = new GsonBuilder();
	}

	@Override
	public Map<String, Object> saveCreditCardInfo(String customerId, String customerEmail, String cardHolderName, String cardNumber, String expMonth,
			String expYear, String cvv) throws Exception {
		String expiry = expMonth + (expYear.length() > 3 ? expYear.substring(2) : expYear);

		Map<String, Object> requestParams = new HashMap<String, Object>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put("defaultacct", "Y");
		requestParams.put("account", cardNumber);
		requestParams.put("expiry", expiry);
		requestParams.put("name", cardHolderName);
		requestParams.put("email", customerEmail);
		if (StringUtils.isNotBlank(customerId)) {
			requestParams.put("profile", customerId);
		}

		String responseJSON = makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_CARD, requestParams, null);
		JSONObject response = new JSONObject(responseJSON);
		Map<String, Object> properties = response.toMap();
		return properties;
	}

	@Override
	public List<PaymentMethod> getPaymentMethods(com.floreantpos.model.Customer customer) throws Exception {
		String customerProfileId = customer.getProperty(getProfileKey());
		if (StringUtils.isBlank(customerProfileId)) {
			return null;
		}
		JSONArray profiles = getProfiles(customerProfileId, merchantId);
		List<PaymentMethod> paymentMethods = new ArrayList<>();
		if (profiles != null && profiles.length() > 0) {
			for (int i = 0; i < profiles.length(); i++) {
				JSONObject profile = profiles.getJSONObject(i);
				if (profile != null) {
					if (!profile.has(SecuredConstants.PROP_IPP320_TOKEN)) {
						continue;
					}
					String token = profile.getString(SecuredConstants.PROP_IPP320_TOKEN);
					if (token != null && token.length() > 4) {
						token = token.substring(token.length() - 4);
					}
					String cardType = convertCardTypeAsFullString(profile.getString(SecuredConstants.PROP_IPP320_ACCTTYPE));
					PaymentMethod paymentMethod = new PaymentMethod(profile.getString(SecuredConstants.PROP_IPP320_PROFILEID),
							profile.getString(SecuredConstants.PROP_IPP320_ACCTID), token, cardType, profile.getString("defaultacct").equals("Y"));
					paymentMethod.setAccountHolderName(profile.getString("name"));
					String expiry = profile.getString("expiry");
					if (expiry != null && expiry.length() > 2) {
						paymentMethod.setExpMonth(expiry.substring(0, 2));
						paymentMethod.setExpYear(expiry.substring(2, 4));
					}
					paymentMethods.add(paymentMethod);
				}
			}
		}
		return paymentMethods;
	}

	public String convertCardTypeAsFullString(String shortCardType) {
		switch (shortCardType) {
			case "MC":
				return "MasterCard";
			case "VISA":
				return "Visa";
			case "DISC":
				return "Discover";
			case "DC":
				return "Diners Club";
			case "AMEX":
				return "American Express";
			default:
				break;
		}
		return shortCardType;
	}

	public JSONArray getProfiles(String profileid, String merchid) throws Exception {
		if (isEmpty(profileid))
			throw new IllegalArgumentException("Missing required parameter: profileid");
		String requestPath = SecuredConstants.IPP320_URL_CARD + "/" + profileid + "/" + "" + "/" + merchid;
		return (JSONArray) send(requestPath, GET, null);
	}

	@Override
	public void removeCard(PaymentMethod paymentMethod, com.floreantpos.model.Customer customer) throws Exception {
		if (paymentMethod.isDefault()) {
			List<PaymentMethod> paymentMethods = getPaymentMethods(customer);
			if (paymentMethods != null && paymentMethods.size() > 0) {
				for (PaymentMethod pM : paymentMethods) {
					if (!pM.getAccountId().equals(paymentMethod.getAccountId())) {
						Map<String, Object> requestParams = new HashMap<String, Object>();
						requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
						requestParams.put("defaultacct", "Y");
						requestParams.put("profileupdate", "Y");
						requestParams.put("profile", pM.getId() + "/" + pM.getAccountId());
						makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_CARD, requestParams, null);
					}
				}
			}
		}

		String profileid = paymentMethod.getId();
		String accountid = paymentMethod.getAccountId();
		if (isEmpty(profileid))
			throw new IllegalArgumentException("Missing required parameter: profileid");
		if (accountid == null)
			accountid = "";

		String requestPath = SecuredConstants.IPP320_URL_CARD + "/" + profileid + "/" + accountid + "/" + merchantId;
		JSONObject response = (JSONObject) send(requestPath, DELETE, null);
		if (response == null || response.getString(SecuredConstants.PROP_IPP320_RESPSTAT) == null
				|| !response.getString(SecuredConstants.PROP_IPP320_RESPSTAT).equals("A")) {
			if (response != null) {
				PosLog.error(getClass(),
						response.getString(SecuredConstants.PROP_IPP320_RESPCODE) + " " + response.getString(SecuredConstants.PROP_IPP320_RESPTEXT));
				if (response.getString(SecuredConstants.PROP_IPP320_RESPSTAT).equals("C")) {
					throw new PosException(response.getString(SecuredConstants.PROP_IPP320_RESPTEXT));
				}
			}
			throw new PosException("Failed to delete card.");
		}
	}

	public void preAuth(PosTransaction transaction) throws Exception {
		preAuth(transaction, null);
	}

	@Override
	public void preAuth(PosTransaction transaction, PaymentMethod paymentMethod) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeAuthRequest(transaction, paymentMethod, false, false);
		transaction.setAuthorizable(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeCaptureRequest(transaction);
		transaction.setCaptured(true);
		this.saveTransaction(transaction, responseJSON);
	}

	public void chargeAmount(PosTransaction transaction) throws Exception {
		chargeAmount(transaction, null, false);
	}

	@Override
	public void chargeAmount(PosTransaction transaction, PaymentMethod paymentMethod, Boolean saveCard) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeAuthRequest(transaction, paymentMethod, true, saveCard);
		transaction.setAuthorizable(false);
		transaction.setCaptured(true);
		this.saveTransaction(transaction, responseJSON);
	}

	private void createPaymentMethod(PosTransaction transaction) throws Exception {
		try {
			String cardHolderName = transaction.getCardHolderName();
			String cardNumber = transaction.getCardNumber();
			String cardExpYear = transaction.getCardExpYear();
			String cardExpMonth = transaction.getCardExpMonth();
			String cvv = transaction.getCardCVV();
			String customerId = transaction.getProperty(getProfileKey());
			String customerEmail = transaction.getTicket().getCustomerEmail();
			Map<String, Object> response = saveCreditCardInfo(customerId, customerEmail, cardHolderName, cardNumber, cardExpMonth, cardExpYear, cvv);
			transaction.setCardType(convertCardTypeAsFullString((String) response.get(SecuredConstants.PROP_IPP320_ACCTTYPE)));
			transaction.addProperty(getProfileKey(), (String) response.get(SecuredConstants.PROP_IPP320_PROFILEID));
		} catch (Exception e) {
			PosLog.error(getClass(), "Failed to save credit card info: " + e.getMessage());
		}
	}

	@Override
	public String getProfileKey() {
		return SecuredConstants.PROFILEID + "." + merchantId;
	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeVoidRequest(transaction.getCardTransactionId(), transaction);
		transaction.setVoided(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		this.preProcessingLog(transaction);
		String responseJSON = this.makeRefundRequest(transaction.getCardTransactionId(), refundAmount, transaction);
		transaction.setRefunded(true);
		this.saveTransaction(transaction, responseJSON);
	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	@Override
	public void cancelTransaction() {
	}

	private void saveTransaction(PosTransaction transaction, String responseJSON) throws Exception {
		JSONObject jsonObject = new JSONObject(responseJSON);
		Map<String, Object> properties = jsonObject.toMap();
		String retref = (String) properties.get(SecuredConstants.PROP_IPP320_RETREF);
		String authCode = (String) properties.get(SecuredConstants.PROP_IPP320_AUTHCODE);
		String accountNumber = (String) properties.get(SecuredConstants.PROP_IPP320_ACCOUNT);
		transaction.setCardTransactionId(retref);
		if (StringUtils.isNotEmpty(authCode)) {
			transaction.setCardAuthCode(authCode);
		}
		if (StringUtils.isNotEmpty(accountNumber)) {
			accountNumber = accountNumber.substring(accountNumber.length() - 4, accountNumber.length());
			transaction.setCardNumber(accountNumber);
		}
		this.setTransactionProperties(transaction, properties);
	}

	private void setTransactionProperties(PosTransaction transaction, Map<String, Object> properties) throws Exception {
		if (transaction == null || properties == null) {
			return;
		}
		List<Entry<String, Object>> entries = new ArrayList<>(properties.entrySet());
		for (Entry<String, Object> entry : entries) {
			transaction.addExtraProperty(entry.getKey(), String.valueOf(entry.getValue()));
		}
	}

	private boolean isCardInfoValid(PosTransaction transaction) throws Exception {
		if (transaction.getCardReader().equals(CardReader.SWIPE.name()) && StringUtils.isNotBlank(transaction.getCardTrack())) {
			return Boolean.TRUE;
		}
		else if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			if (StringUtils.isBlank(transaction.getCardNumber())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.0")); //$NON-NLS-1$
			}
			if (!(StringUtils.isNotBlank(transaction.getCardExpYear()) && transaction.getCardExpYear().length() >= 2)) {
				throw new PosException(IppMessages.getString("SecuredProcessor.1")); //$NON-NLS-1$
			}
			if (StringUtils.isBlank(transaction.getCardExpMonth())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.2")); //$NON-NLS-1$
			}
			//			if (StringUtils.isBlank(transaction.getCardCVV())) {
			//				return Boolean.FALSE;
			//			}
		}
		return Boolean.TRUE;
	}

	private String makeAuthRequest(PosTransaction transaction, PaymentMethod paymentMethod, boolean capture, Boolean saveCard) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_ORDERID, transaction.getTicketId());
		requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, transaction.getAmount());
		Currency mainCurrency = DataProvider.get().getMainCurrency();
		requestParams.put(SecuredConstants.PROP_IPP320_CURRENCY, mainCurrency != null ? mainCurrency.getCode() : "$");
		requestParams.put(SecuredConstants.PROP_IPP320_CAPTURE, capture ? "Y" : "N"); //$NON-NLS-1$ //$NON-NLS-2$
		requestParams.put(SecuredConstants.PROP_IPP320_TOKENIZE, "Y"); //$NON-NLS-1$

		String cardTokenId = null;
		if (paymentMethod != null) {
			cardTokenId = paymentMethod.getId();
			if (StringUtils.isNotBlank(paymentMethod.getAccountId())) {
				cardTokenId += "/" + paymentMethod.getAccountId();
			}
			transaction.setCardType(paymentMethod.getCardType());
			transaction.setCardHolderName(paymentMethod.getAccountHolderName());
			transaction.setCardExpMonth(paymentMethod.getExpMonth());
			transaction.setCardExpYear(paymentMethod.getExpYear());
		}
		if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			if (StringUtils.isNotBlank(cardTokenId)) {
				requestParams.put("profile", cardTokenId);
			}
			else {
				this.isCardInfoValid(transaction);
				requestParams.put(SecuredConstants.PROP_IPP320_ACCOUNT, transaction.getCardNumber());
				String expiry = transaction.getCardExpMonth() + transaction.getCardExpYear().substring(transaction.getCardExpYear().length() - 2);
				requestParams.put(SecuredConstants.PROP_IPP320_EXPIRY, expiry);
				requestParams.put(SecuredConstants.PROP_IPP320_CVV2, transaction.getCardCVV());
				requestParams.put(SecuredConstants.PROP_IPP320_NAME, transaction.getCardHolderName());
			}
		}
		else if (transaction.getCardReader().equals(CardReader.SWIPE.name())) {
			requestParams.put(SecuredConstants.PROP_IPP320_TRACK, transaction.getCardTrack());
		}
		String response = this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_AUTH, requestParams, transaction);
		if (saveCard) {
			createPaymentMethod(transaction);
		}
		return response;
	}

	private String makeCaptureRequest(PosTransaction posTransaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, posTransaction.getCardTransactionId());
		requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, posTransaction.getAmount());

		String responseJSON = this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_CAPTURE, requestParams, posTransaction);
		//		Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
		//		Object amountStr = responseObjects.get(SecuredConstants.PROP_IPP320_AMOUNT);
		//		if (amountStr != null) {
		//			Double amount = Double.parseDouble((String) amountStr);
		//			posTransaction.setAmount(amount);
		//		}

		return responseJSON;
	}

	private String makeVoidRequest(String retref, PosTransaction transaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, retref);
		if (transaction.getAmount() != null) {
			requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, transaction.getAmount());
		}
		return this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_VOID, requestParams, transaction);
	}

	private String makeRefundRequest(String retref, Double amount, PosTransaction transaction) throws Exception {
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		requestParams.put(SecuredConstants.PROP_IPP320_RETREF, retref);
		if (amount != null) {
			requestParams.put(SecuredConstants.PROP_IPP320_AMOUNT, amount);
		}
		return this.makeRequest(SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_REFUND, requestParams, transaction);
	}

	@SuppressWarnings("resource")
	private String makeRequest(String requestURL, Map<String, Object> requestParams, PosTransaction posTransaction) throws Exception {
		HttpClient httpClient = HttpClientBuilder.create().build();
		HttpPut putRequest = new HttpPut(requestURL);
		Gson gson = this.gsonBuilder.create();
		String paramsJSON = gson.toJson(requestParams);
		StringEntity requestParamsEntity = new StringEntity(paramsJSON);
		putRequest.addHeader("content-type", "application/json"); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setHeader("Authorization", "Basic " + this.encodedAuthString); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setEntity(requestParamsEntity);
		HttpResponse response = httpClient.execute(putRequest);
		int statusCode = response.getStatusLine().getStatusCode();
		String responseJSON = ""; //$NON-NLS-1$

		if (statusCode == 200) {
			HttpEntity entity = response.getEntity();
			responseJSON = EntityUtils.toString(entity);
			this.postProcessingLog(statusCode, responseJSON);
			Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
			SecuredResponseStatus responseStatus = SecuredResponseStatus.fromStatus((String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPSTAT));
			if (responseStatus != SecuredResponseStatus.APPROVED) {
				Object responseCode = responseObjects.get(SecuredConstants.PROP_IPP320_RESPCODE);
				String resMsg = (String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPTEXT);
				if (responseCode != null) {
					if (responseCode.equals("14") && resMsg.equalsIgnoreCase("Non-numeric CVV")) { //$NON-NLS-1$ //$NON-NLS-2$
						throw new PosException("CVV code is not valid."); //$NON-NLS-1$
					}
					if (responseCode.equals("32") && resMsg.equalsIgnoreCase("Wrong currency for merch")) { //$NON-NLS-1$ //$NON-NLS-2$
						throw new PosException("Payment gateway does not support this currency. Please use USD as default currency."); //$NON-NLS-1$
					}
				}
				throw new PosException(resMsg);
				//				if (responseStatus == SecuredResponseStatus.RETRY) {
				//					throw new SecuredException(IppMessages.getString("IPP320Processor.0")); //$NON-NLS-1$
				//				}
				//				else {
				//					throw new PosException(jsonObject.getString("resptext")); //$NON-NLS-1$
				//				}
			}
		}
		else if (statusCode == 401) {
			throw new PosException(IppMessages.getString("SecuredProcessor.3")); //$NON-NLS-1$
		}
		else {
			this.postProcessingLog(statusCode, responseJSON);
			throw new SecuredException(IppMessages.getString("IPP320Processor.1")); //$NON-NLS-1$
		}
		return responseJSON;
	}

	@SuppressWarnings("resource")
	public boolean makeTestRequest() throws Exception {
		HttpClient httpClient = HttpClientBuilder.create().build();
		String requestURL = SecuredConfig.getHostURL() + SecuredConstants.IPP320_URL_TEST_CONNECTION;
		HttpPut putRequest = new HttpPut(requestURL);
		Map<String, Object> requestParams = new HashMap<>();
		requestParams.put(SecuredConstants.PROP_IPP320_MERCHID, this.merchantId);
		Gson gson = this.gsonBuilder.create();
		String paramsJSON = gson.toJson(requestParams);
		StringEntity requestParamsEntity = new StringEntity(paramsJSON);
		putRequest.addHeader("content-type", "application/json"); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setHeader("Authorization", "Basic " + this.encodedAuthString); //$NON-NLS-1$ //$NON-NLS-2$
		putRequest.setEntity(requestParamsEntity);
		HttpResponse response = httpClient.execute(putRequest);
		int statusCode = response.getStatusLine().getStatusCode();
		if (statusCode == 200) {
			HttpEntity entity = response.getEntity();
			String responseHTML = EntityUtils.toString(entity);
			String successMessage = "CardConnect REST Servlet"; //$NON-NLS-1$
			return responseHTML.contains(successMessage);
		}
		return false;
	}

	private void postProcessingLog(int statusCode, String responseJSON) {
		StringBuilder logBuilder = new StringBuilder().append(STR_NEWLINE);
		logBuilder.append(IppMessages.getString("SecuredProcessor.5")).append(STR_NEWLINE); //$NON-NLS-1$
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.6"), statusCode)).append(STR_NEWLINE); //$NON-NLS-1$
		if (StringUtils.isNotBlank(responseJSON)) {
			Map<String, Object> responseObjects = new JSONObject(responseJSON).toMap();
			SecuredResponseStatus responseStatus = SecuredResponseStatus.fromStatus((String) responseObjects.get(SecuredConstants.PROP_IPP320_RESPSTAT));
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.7"), responseStatus.name())).append(STR_NEWLINE); //$NON-NLS-1$
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.8"), responseObjects.get(SecuredConstants.PROP_IPP320_AMOUNT))) //$NON-NLS-1$
					.append(STR_NEWLINE);
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.9"), responseObjects.get(SecuredConstants.PROP_IPP320_AUTHCODE))) //$NON-NLS-1$
					.append(STR_NEWLINE);
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.10"), responseJSON)).append(STR_NEWLINE); //$NON-NLS-1$
		}
	}

	private void preProcessingLog(PosTransaction transaction) {
		StringBuilder logBuilder = new StringBuilder().append(STR_NEWLINE);
		logBuilder.append(IppMessages.getString("SecuredProcessor.11")).append(STR_NEWLINE); //$NON-NLS-1$
		try {
			logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.12"), transaction.getCardReader())).append(STR_NEWLINE); //$NON-NLS-1$
			if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.13"), transaction.getCardHolderName())).append(STR_NEWLINE); //$NON-NLS-1$
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.14"), //$NON-NLS-1$
						transaction.getCardNumber().substring(transaction.getCardNo().length() - 4))).append(STR_NEWLINE);
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.15"), transaction.getCardExpMonth())).append(STR_NEWLINE); //$NON-NLS-1$
				logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.16"), transaction.getCardExpYear())).append(STR_NEWLINE); //$NON-NLS-1$
			}
		} catch (Exception e) {
		}
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.17"), transaction.getTicketId())).append(STR_NEWLINE); //$NON-NLS-1$
		logBuilder.append(String.format(IppMessages.getString("SecuredProcessor.18"), transaction.getAmount())).append(STR_NEWLINE); //$NON-NLS-1$
	}

	private boolean isEmpty(String str) {
		if (str == null)
			return true;
		if (str.length() <= 0)
			return true;
		if ("".equals(str))
			return true;
		return false;
	}

	private Object send(String endpoint, String operation, JSONObject request) throws Exception {
		Object response = null;
		String resp = null;
		String url = SecuredConfig.getHostURL() + endpoint;
		try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
			HttpClientContext context = HttpClientContext.create();
			Header[] headers = getHeaders();

			StringEntity entity = null;
			if (request != null) {
				entity = new StringEntity(request.toString(), "UTF-8");
				entity.setContentType("application/json");
			}

			ResponseHandler<String> rh = getResponseHandler();
			switch (operation) {
				case PUT: {
					HttpPut put = new HttpPut(url);
					put.setHeaders(headers);
					put.setEntity(entity);
					resp = httpclient.execute(put, rh, context);
					break;
				}
				case POST: {
					HttpPost post = new HttpPost(url);
					post.setHeaders(headers);
					post.setEntity(entity);
					resp = httpclient.execute(post, rh, context);
					break;
				}
				case GET: {
					HttpGet get = new HttpGet(url);
					get.setHeaders(headers);
					resp = httpclient.execute(get, rh, context);
					if (resp != null) {
						response = new JSONArray(resp);
					}
					break;
				}
				case DELETE: {
					HttpDelete delete = new HttpDelete(url);
					delete.setHeaders(headers);
					resp = httpclient.execute(delete, rh, context);
					if (resp != null) {
						response = new JSONObject(resp);
					}
					break;
				}
			}
		}
		return response;
	}

	// Creates a response handler for the HTTP responses
	private ResponseHandler<String> getResponseHandler() {
		return new ResponseHandler<String>() {
			public String handleResponse(final HttpResponse response) throws IOException {
				StatusLine statusLine = response.getStatusLine();
				HttpEntity entity = response.getEntity();
				if (statusLine.getStatusCode() >= 300) {
					throw new HttpResponseException(statusLine.getStatusCode(), statusLine.getReasonPhrase());
				}
				if (entity == null) {
					throw new ClientProtocolException("Response contains no content");
				}

				InputStream instream = entity.getContent();
				StringBuilder sb = new StringBuilder();
				BufferedReader r = new BufferedReader(new InputStreamReader(instream), 2000);
				for (String line = r.readLine(); line != null; line = r.readLine()) {
					sb.append(line);
				}
				instream.close();
				return sb.toString();
			}
		};
	}

	// Generaters headers for the HTTP calls - for Authorization and JSON
	private Header[] getHeaders() {
		Header[] headers = new BasicHeader[3];
		headers[0] = new BasicHeader("Authorization", "Basic " + DatatypeConverter.printBase64Binary(this.authString.getBytes()));
		headers[1] = new BasicHeader("Accept", "application/json");
		headers[2] = new BasicHeader("Content-Type", "application/json");

		return headers;
	}
}