/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.model.base.BasePurchaseOrder;

@XmlRootElement
@JsonIgnoreProperties(ignoreUnknown = true, value = { "statusDisplay", "orderStatusDisplay" })
public class PurchaseOrder extends BasePurchaseOrder implements TimedModel {
	private static final long serialVersionUID = 1L;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public PurchaseOrder() {
	}

	/**
	 * Constructor for primary key
	 */
	public PurchaseOrder(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public static final int ORDER_PENDING = 0;
	public static final int ORDER_VERIFIED = 1;
	public static final int ORDER_FULLY_INVOICED = 2;
	public static final int ORDER_PARTIALLY_INVOICED = 3;
	public static final int ORDER_FULLY_RECEIVED = 4;
	public static final int ORDER_PARTIALLY_RECEIVED = 5;
	public static final int ORDER_PARTIALLY_RECEIVED_AND_INVOICED = 6;
	public static final int ORDER_CLOSED = 7;
	public static final int ORDER_CANCELLED = 8;

	public static final String[] ORDER_STATUS = { Messages.getString("PurchaseOrder.0"), Messages.getString("PurchaseOrder.1"), //$NON-NLS-1$//$NON-NLS-2$
			Messages.getString("PurchaseOrder.2"), Messages.getString("PurchaseOrder.3"), Messages.getString("PurchaseOrder.4"), //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			Messages.getString("PurchaseOrder.5"), //$NON-NLS-1$
			Messages.getString("PurchaseOrder.6"), Messages.getString("PurchaseOrder.7"), Messages.getString("PurchaseOrder.8") }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

	public static final String PO_VERIFIED_BY = Messages.getString("PurchaseOrder.9"); //$NON-NLS-1$
	public static final String PO_SENT_BY = Messages.getString("PurchaseOrder.10"); //$NON-NLS-1$
	public static final String PO_RECEIVED_BY = Messages.getString("PurchaseOrder.11"); //$NON-NLS-1$
	public static final String PO_INVOICED_BY = Messages.getString("PurchaseOrder.12"); //$NON-NLS-1$
	public static final String PO_CLOSED_BY = Messages.getString("PurchaseOrder.13"); //$NON-NLS-1$

	public String orderStatusDisplay;

	public static final String DEBIT = Messages.getString("PurchaseOrder.14"); //$NON-NLS-1$
	public static final String CREDIT = Messages.getString("PurchaseOrder.15"); //$NON-NLS-1$
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	public String statusDisplay;

	@XmlTransient
	public String getOrderStatusDisplay() {
		return ORDER_STATUS[getStatus()];
	}

	public void setOrderStatusDisplay(String orderStatusDisplay) {
		this.orderStatusDisplay = ORDER_STATUS[getStatus()];
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public List<PurchaseOrderItem> getOrderItems() {
		List<PurchaseOrderItem> items = super.getOrderItems();

		if (items == null) {
			items = new ArrayList<PurchaseOrderItem>();
			super.setOrderItems(items);
		}
		return items;
	}

	public void calculatePrice() {
		List<PurchaseOrderItem> items = getOrderItems();
		if (items == null) {
			return;
		}

		double subtotalAmount = 0;
		double discountAmount = 0;
		double taxAmount = 0;

		for (PurchaseOrderItem item : items) {
			item.calculatePrice();
			subtotalAmount += item.getSubtotalAmount();
			discountAmount += item.getDiscountAmount();
			taxAmount += item.getTaxAmount();
		}

		setSubtotalAmount(subtotalAmount);
		setDiscountAmount(discountAmount);
		setTaxAmount(taxAmount);

		double totalAmount = subtotalAmount - discountAmount + taxAmount;

		totalAmount = fixInvalidAmount(totalAmount);
		setTotalAmount((totalAmount));

		double dueAmount = totalAmount - getPaidAmount();
		setDueAmount((dueAmount));
	}

	private double fixInvalidAmount(double tax) {
		if (tax < 0 || Double.isNaN(tax)) {
			tax = 0;
		}
		return tax;
	}

	public void addProperty(String name, String value) {
		if (getProperties() == null) {
			setProperties(new HashMap<String, String>());
		}

		getProperties().put(name, value);
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public String getProperty(String key) {
		if (getProperties() == null) {
			return null;
		}

		return getProperties().get(key);
	}

	public String getProperty(String key, String defaultValue) {
		if (getProperties() == null) {
			return null;
		}

		String string = getProperties().get(key);
		if (StringUtils.isEmpty(string)) {
			return defaultValue;
		}

		return string;
	}

	public void removeProperty(String propertyName) {
		Map<String, String> properties = getProperties();
		if (properties == null) {
			return;
		}

		properties.remove(propertyName);
	}

	@XmlTransient
	public String getStatusDisplay() {
		int status = getStatus();
		if (status == PurchaseOrder.ORDER_PENDING)
			return Messages.getString("PurchaseOrder.16"); //$NON-NLS-1$
		else if (status == PurchaseOrder.ORDER_VERIFIED || status == PurchaseOrder.ORDER_PARTIALLY_RECEIVED)
			return Messages.getString("PurchaseOrder.17"); //$NON-NLS-1$
		else
			return ""; //$NON-NLS-1$
	}

	public void setStatusDisplay(String statusDisplay) {
		this.statusDisplay = statusDisplay;
	}
}