package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import javax.sql.rowset.serial.SerialBlob;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.sql.JoinType;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Department;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.model.Ticket;
import com.floreantpos.util.NameBaseIdGenerator;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class OrderTypeDAO extends BaseOrderTypeDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OrderTypeDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime((OrderType) obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime((OrderType) obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime((OrderType) obj);
		super.saveOrUpdate(obj, s);
	}

	protected void updateTime(OrderType model) {
		if (model.getId() == null) {
			model.setId(NameBaseIdGenerator.generateId(model.getName()));
		}
		super.updateTime(model);
	}

	@Override
	protected void delete(Object obj, Session session) {
		OrderType orderType = (OrderType) obj;
		if (orderType == null) {
			throw new PosException(Messages.getString("OrderTypeDAO.0")); //$NON-NLS-1$
		}
		orderType.setDeleted(true);
		orderType.setCategories(null);
		orderType.setDefaultTaxGroup(null);
		orderType.setDefaultTaxGroupId(null);
		orderType.setDepartments(null);
		orderType.setEnableCourse(false);
		orderType.setEnabled(false);
		orderType.setForHereTaxGroup(null);
		orderType.setForHereTaxGroupId(null);
		orderType.setTerminalTypes(null);
		update(orderType, session);
	}

	public List<OrderType> findAll() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(OrderType.PROP_SORT_ORDER));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<OrderType> findEnabledOrderTypesForTerminal(Terminal terminal) {
		Session session = null;

		try {
			TerminalType terminalType = terminal.getTerminalType();
			Department department = terminal.getDepartment();
			SalesArea salesArea = terminal.getSalesArea();

			//@formatter:off
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass(), "c"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			
			if (terminalType != null) {
				criteria.createAlias("c.terminalTypes", "terminalType", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
				criteria.add(
						Restrictions.or(
								Restrictions.isEmpty("c.terminalTypes"), //$NON-NLS-1$
								Restrictions.in("terminalType.id", Arrays.asList(terminalType.getId())) //$NON-NLS-1$
								)
						);
			}
			if (department != null) {
				criteria.createAlias("c.departments", "department", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
				criteria.add(
						Restrictions.or(
							Restrictions.isEmpty("c.departments"), //$NON-NLS-1$
							Restrictions.in("department.id", Arrays.asList(department.getId())) //$NON-NLS-1$
						)
				);
			}
			if (salesArea != null) {
				  //!!!Warning, please do not remove or clean this block!!!
                 //OR-1250 need to fix in future if sales area is mapped by order type
				//criteria.add(Restrictions.eqOrIsNull(OrderType.PROP_SALES_AREA_ID, salesArea.getId()));
			}
			//@formatter:on
			criteria.addOrder(Order.asc(OrderType.PROP_SORT_ORDER));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			return criteria.list();

			//			TerminalTypeDAO dao = TerminalTypeDAO.getInstance();
			//			TerminalType terminalType = dao.getTerminalType(null);
			//
			//			List<OrderType> orderTypes = null;
			//			if (terminalType != null) {
			//				dao.initialize(terminalType);
			//				orderTypes = terminalType.getOrderTypes();
			//			}
			//			Department thisTerminalDepartment = Application.getInstance().getTerminal().getDepartment();
			//			if (thisTerminalDepartment == null && orderTypes != null && !orderTypes.isEmpty())
			//				return orderTypes;
			//
			//			if (orderTypes == null || orderTypes.isEmpty()) {
			//				session = createNewSession();
			//				Criteria criteria = session.createCriteria(getReferenceClass());
			//				criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
			//
			//				orderTypes = criteria.list();
			//			}
			//			if (thisTerminalDepartment != null) {
			//				List<OrderType> departmentOrderTypes = thisTerminalDepartment.getOrderTypes();
			//				if (orderTypes != null && departmentOrderTypes != null && !departmentOrderTypes.isEmpty()) {
			//					for (Iterator iterator = orderTypes.iterator(); iterator.hasNext();) {
			//						OrderType orderType = (OrderType) iterator.next();
			//						if (!DepartmentDAO.getInstance().existsDepartment(orderType))
			//							continue;
			//						if (!departmentOrderTypes.contains(orderType)) {
			//							iterator.remove();
			//						}
			//					}
			//				}
			//			}
		} finally {
			closeSession(session);
		}
	}

	//	public List<OrderType> findEnabledOrderTypes() {
	//		Terminal terminal = Application.getInstance().getTerminal();
	//		Department department = terminal.getDepartment();
	//
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			Criteria criteria = session.createCriteria(getReferenceClass(), "o"); //$NON-NLS-1$
//			//@formatter:off
//			if (terminal.getTerminalType() != null) {
//				criteria.createAlias("o.terminalTypes", "terminalType", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
//				criteria.add(
//						Restrictions.or(
//								Restrictions.isEmpty("o.terminalTypes"), //$NON-NLS-1$
//								Restrictions.in("terminalType.id", Arrays.asList(terminal.getTerminalType().getId())) //$NON-NLS-1$
//								)
//						);
//			}
//			if (department != null) {
//				criteria.createAlias("o.departments", "department", JoinType.LEFT_OUTER_JOIN); //$NON-NLS-1$ //$NON-NLS-2$
//				criteria.add(
//						Restrictions.or(
//							Restrictions.isEmpty("o.departments"), //$NON-NLS-1$
//							Restrictions.in("department.id", Arrays.asList(department.getId())) //$NON-NLS-1$
//						)
//				);
//			}
//			//@formatter:on
	//			criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, Boolean.TRUE));
	//			return criteria.list();
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public List<OrderType> findLoginScreenViewOrderTypes() {
		Session session = null;
		try {
			session = createNewSession();

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
			criteria.add(Restrictions.eq(OrderType.PROP_SHOW_IN_LOGIN_SCREEN, true));

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	@Deprecated
	public OrderType findByName(String orderType) {
		Session session = null;
		try {
			session = createNewSession();
			return findByName(orderType, session);
		} finally {
			closeSession(session);
		}
	}

	@Deprecated
	public OrderType findByName(String orderType, Session session) {
		if (session == null)
			session = getSession();

		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(OrderType.PROP_NAME, orderType));
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

		return (OrderType) criteria.uniqueResult();
	}

	/*
	 * 
	 * Checking menu item order types contain order type object or name.
	 * 
	 */
	@Deprecated
	public boolean containsOrderTypeObj() {
		Session session = null;
		try {
			session = createNewSession();
			Query query = session.createSQLQuery("select count(s.MENU_ITEM_ID), count(s.ORDER_TYPE_ID) from ITEM_ORDER_TYPE s"); //$NON-NLS-1$
			List result = query.list();
			Object[] object = (Object[]) result.get(0);
			Integer menuItemCount = getInt(object, 0);
			Integer orderTypeCount = getInt(object, 1);

			if (menuItemCount < 1) {
				return true;
			}
			return orderTypeCount > 0;
		} catch (Exception e) {
			//PosLog.error(getClass(), e);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
		return false;
	}

	/*
	 * This method is used for updating menu item order type name to order type object. 
	 * 
	 */
	@Deprecated
	public void updateMenuItemOrderType() {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			Query query = session.createSQLQuery("Update ITEM_ORDER_TYPE t SET t.ORDER_TYPE_ID=(Select o.id from ORDER_TYPE o where o.NAME=t.ORDER_TYPE)"); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	private Integer getInt(Object[] array, int index) {
		if (array.length < (index + 1))
			return null;

		if (array[index] instanceof Number) {
			return ((Number) array[index]).intValue();
		}

		return null;
	}

	public OrderType getFirstOrderType() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setFirstResult(0);
			criteria.setMaxResults(1);
			List result = criteria.list();
			if (result.size() > 0) {
				return (OrderType) result.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}
	}

	public OrderType getOrderType(boolean dineIn) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
			criteria.add(Restrictions.eq(OrderType.PROP_SHOW_TABLE_SELECTION, dineIn));
			if (!dineIn) {
				criteria.add(Restrictions.eq(OrderType.PROP_DELIVERY, Boolean.FALSE));
			}
			List result = criteria.list();
			if (result.size() > 0) {
				return (OrderType) result.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}
	}

	public List<String> getHomeDeliveryOrderTypeIds() {
		try (Session session = this.createNewSession()) {
			return this.getHomeDeliveryOrderTypeIds(session);
		}
	}

	public List<String> getHomeDeliveryOrderTypeIds(Session session) {
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
		criteria.add(Restrictions.eq(OrderType.PROP_DELIVERY, Boolean.TRUE));
		criteria.setProjection(Projections.property(OrderType.PROP_ID));
		return criteria.list();
	}

	public OrderType getHomeDeliveryOrderType() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(OrderType.PROP_ENABLED, true));
			criteria.add(Restrictions.eq(OrderType.PROP_DELIVERY, Boolean.TRUE));
			List result = criteria.list();
			if (result.size() > 0) {
				return (OrderType) result.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateOrderType(OrderType ordersType, List<Department> checkedValues) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			session.saveOrUpdate(ordersType);
			for (Department object : checkedValues) {
				session.saveOrUpdate(object);
			}
			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public OrderType initialize(OrderType orderType) {
		if (orderType == null || orderType.getId() == null)
			return orderType;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(orderType);

			Hibernate.initialize(orderType.getDepartments());
			Hibernate.initialize(orderType.getTerminalTypes());
			Hibernate.initialize(orderType.getCategories());

			return orderType;
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateOrderTypes(List<OrderType> orderTypes, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (orderTypes == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (OrderType orderType : orderTypes) {
				String orderTypeId = orderType.getId();
				OrderType existingOrderType = null;
				if (StringUtils.isNotBlank(orderTypeId)) {
					existingOrderType = get(orderTypeId, orderType.getOutletId());
				}
				byte[] imageBytes = orderType.getImageBytes();
				if (existingOrderType != null) {
					if (!BaseDataServiceDao.get().shouldSave(orderType.getLastUpdateTime(), existingOrderType.getLastUpdateTime())) {
						PosLog.info(getClass(), orderType.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingOrderType.getId();
					long version = existingOrderType.getVersion();
					PropertyUtils.copyProperties(existingOrderType, orderType);
					existingOrderType.setId(id);
					existingOrderType.setVersion(version);
					existingOrderType.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingOrderType.setUpdateSyncTime(updateSyncTime);
					if (imageBytes != null) {
						existingOrderType.setImageData(new SerialBlob(imageBytes));
					}
					update(existingOrderType, session);
				}
				else {
					orderType.setUpdateLastUpdateTime(updateLastUpdateTime);
					orderType.setUpdateSyncTime(updateSyncTime);
					orderType.setVersion(0);
					if (imageBytes != null) {
						orderType.setImageData(new SerialBlob(imageBytes));
					}

					save(orderType, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	public boolean canDelete(OrderType orderType) {
		if (orderType == null || orderType.getId() == null)
			return false;

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
			criteria.setMaxResults(1);
			return criteria.list().size() == 0;
		}
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public List<OrderType> getOrderTypesByTerminalType(TerminalType terminalType, Session session) {
		String sql = "SELECT DISTINCT ORDER_TYPE_ID FROM TERMINAL_TYPE_ORDER_TYPE WHERE TERMINAL_TYPE_ID = :tt_id";
		Query query = session.createSQLQuery(sql);
		query.setParameter("tt_id", terminalType.getId());
		List orderTypeIds = query.list();
		if (orderTypeIds != null && !orderTypeIds.isEmpty()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			criteria.add(Restrictions.in(OrderType.PROP_ID, orderTypeIds));
			criteria.setProjection(Projections.alias(Projections.property(OrderType.PROP_NAME), OrderType.PROP_NAME));
			return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
		}
		return null;
	}

	//cloud sync
	public void saveOrUpdateOrderTypeCollections(String outletId, List<OrderType> orderTypesWithCollections) {
		Transaction tx = null;
		try (Session session = OrderTypeDAO.getInstance().createNewSession()) {
			tx = session.beginTransaction();
			for (OrderType orderType : orderTypesWithCollections) {
				OrderType existingOrderType = OrderTypeDAO.getInstance().get(orderType.getId(), orderType.getOutletId(), session);
				if (existingOrderType == null) {
					continue;
				}
				Hibernate.initialize(existingOrderType.getDepartments());
				Hibernate.initialize(existingOrderType.getTerminalTypes());
				Hibernate.initialize(existingOrderType.getCategories());

				if (existingOrderType.getTerminalTypes() != null) {
					existingOrderType.getTerminalTypes().clear();
				}
				Set<TerminalType> terminalTypes = orderType.getTerminalTypes();
				if (terminalTypes != null && terminalTypes.size() > 0) {
					for (TerminalType terminalType : terminalTypes) {
						terminalType.setOutletId(outletId);
						existingOrderType.addToterminalTypes(terminalType);
					}
				}
				if (existingOrderType.getCategories() != null) {
					existingOrderType.getCategories().clear();
				}
				Set<MenuCategory> categories = orderType.getCategories();
				if (categories != null && categories.size() > 0) {
					for (MenuCategory menuCategory : categories) {
						existingOrderType.addTocategories(menuCategory);
					}
				}
				if (existingOrderType.getDepartments() != null) {
					existingOrderType.getDepartments().clear();
				}
				Set<Department> departments = orderType.getDepartments();
				if (departments != null && departments.size() > 0) {
					for (Department department : departments) {
						existingOrderType.addTodepartments(department);
					}
				}
				existingOrderType.setUpdateLastUpdateTime(false);
				OrderTypeDAO.getInstance().saveOrUpdate(existingOrderType, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	private OrderType get(String id, String outletId, Session session) {
		return get(new OrderType(id, outletId), session);
	}

	public OrderType get(String orderTypeId, String outletId) {
		return get(new OrderType(orderTypeId, outletId));
	}

	@SuppressWarnings("unchecked")
	public List<OrderType> findAll(Outlet outlet) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (outlet != null) {
				criteria.add(Restrictions.eq(OrderType.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.addOrder(Order.asc(OrderType.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public OrderType getFirstOrderTypeById(String deliveryOrderTypeId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(OrderType.PROP_ID, deliveryOrderTypeId));
			criteria.addOrder(Order.asc(OrderType.PROP_SORT_ORDER));
			criteria.setMaxResults(1);
			return (OrderType) criteria.uniqueResult();
		}
	}
}