/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import static com.floreantpos.util.NumberUtil.convertToBigDecimal;
import static com.floreantpos.util.NumberUtil.round;

import java.io.StringReader;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.base.BaseTicketItemModifier;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.OrgJsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "itemQuantityDisplay" })
public class TicketItemModifier extends BaseTicketItemModifier implements ITicketItem, PropertyContainer {
	private static final long serialVersionUID = 1L;

	public final static int NORMAL_MODIFIER = 1;
	public final static int EXTRA_MODIFIER = 3;
	public final static int CRUST = 5;
	public final static int SEPERATOR = 6;

	public final static String TRANSIENT_PROP_TICKET_ITEM_QUANTITY = "ticketItemQuantity"; //$NON-NLS-1$

	public MenuModifier menuModifier;
	private boolean selected;
	private int tableRowNum;
	private double ticketItemQuantity;
	private List<TicketItemTax> taxes;
	private transient com.google.gson.JsonObject propertiesContainer;

	public TicketItemModifier() {
	}

	/**
	 * Constructor for primary key
	 */
	public TicketItemModifier(java.lang.String id) {
		super(id);
	}

	public TicketItemModifier(TicketItem ticketItem, String name, double unitPrice, double quantity) {
		setTicketItem(ticketItem);
		setName(name);
		setUnitPrice(unitPrice);
		setItemQuantity(quantity);
	}

	public TicketItemModifier(String id, TicketItem ticketItem, String name, double unitPrice, double quantity, TicketItemTax tax) {
		setId(id);
		setTicketItem(ticketItem);
		setName(name);
		setUnitPrice(unitPrice);
		setItemQuantity(quantity);
		addTotaxes(tax);
	}

	public int getTableRowNum() {
		return tableRowNum;
	}

	public void setTableRowNum(int tableRowNum) {
		this.tableRowNum = tableRowNum;
	}

	@Override
	public String toString() {
		return getName();
	}

	public boolean canAddCookingInstruction() {
		return false;
	}

	public void calculatePrice() {
		if (getTicketItem() != null && getTicketItem().getDataVersion() == 2) {
			calculatePriceV2();
			return;
		}

		if (isInfoOnly()) {
			return;
		}

		BigDecimal unitPrice = convertToBigDecimal(getUnitPrice());
		BigDecimal subTotalAmount = calculateSubTotal(unitPrice);
		BigDecimal discountAmount = calculateDiscount(subTotalAmount);
		BigDecimal taxAmount = calculateTax(subTotalAmount, discountAmount);
		BigDecimal serviceCharge = BigDecimal.ZERO;
		BigDecimal totalAmount = convertToBigDecimal(0);
		double totalCost = calculateTotalCost(getUnitCost());

		TicketItem ticketItem = getTicketItem();
		if (ticketItem != null) {
			serviceCharge = ticketItem.calculateServiceCharge(subTotalAmount);
			if (ticketItem.isTaxOnServiceCharge()) {
				taxAmount = calculateTax(subTotalAmount.add(serviceCharge), discountAmount);
			}
		}

		if (isTaxIncluded()) {
			totalAmount = subTotalAmount.subtract(discountAmount);
		}
		else {
			totalAmount = subTotalAmount.subtract(discountAmount).add(taxAmount);
		}

		totalAmount = totalAmount.add(serviceCharge);

		setTotalCost(totalCost);
		setSubTotalAmount(subTotalAmount.doubleValue());
		setDiscountAmount(discountAmount.doubleValue());
		setTaxAmount(taxAmount.doubleValue());
		setServiceCharge(serviceCharge.doubleValue());
		setTotalAmount(totalAmount.doubleValue());

		//adjusted price
		setAdjustedUnitPrice(getUnitPrice());
		setAdjustedDiscount(discountAmount.doubleValue());
		setAdjustedSubtotal(getSubTotalAmount());
		setAdjustedTax(getTaxAmount());
		setAdjustedTotal(getTotalAmount());
	}

	@Deprecated
	public void calculateAdjustedPrice() {
		Ticket ticket = getTicketItem().getTicket();
		if (ticket == null) {
			return;
		}

		BigDecimal subTotalAmount = convertToBigDecimal(getSubTotalAmount());
		BigDecimal discountAmount = convertToBigDecimal(getDiscountAmount());
		BigDecimal subtotalAfterDiscount = round(subTotalAmount.subtract(discountAmount));

		BigDecimal ticketDiscountAmount = round(convertToBigDecimal(ticket.getTicketDiscountAmount()));
		BigDecimal ticketSubtotalAmount = round(convertToBigDecimal(ticket.getSubtotalAmountWithVoidItems()));
		BigDecimal totalItemDiscountAmount = round(convertToBigDecimal(ticket.getItemDiscountAmount()));
		BigDecimal ticketSubtotalAfterItemDiscount = round(ticketSubtotalAmount.subtract(totalItemDiscountAmount));
		BigDecimal ticketSubtotalAfterAllDiscount = round(ticketSubtotalAfterItemDiscount.subtract(ticketDiscountAmount));

		BigDecimal adjustedSubtotal = subtotalAfterDiscount.multiply(ticketSubtotalAfterAllDiscount).divide(ticketSubtotalAfterItemDiscount, 4,
				RoundingMode.HALF_UP);
		BigDecimal adjustedDiscount = round(subTotalAmount.subtract(adjustedSubtotal));
		BigDecimal adjustedTax = round(calculateTax(adjustedSubtotal, new BigDecimal("0"))); //$NON-NLS-1$
		BigDecimal adjustedTotal = convertToBigDecimal(0);

		if (adjustedTax.doubleValue() < 0) {
			adjustedTax = BigDecimal.ZERO;
		}

		if (isTaxIncluded()) {
			adjustedTotal = adjustedSubtotal;
		}
		else {
			adjustedTotal = round(adjustedSubtotal.add(adjustedTax));
		}
		adjustedTotal = adjustedTotal.add(convertToBigDecimal(getServiceCharge()));

		setAdjustedUnitPrice(adjustedSubtotal.divide(convertToBigDecimal(getTicketItem().getQuantity()), 4, RoundingMode.HALF_UP).doubleValue());
		setAdjustedDiscount(adjustedDiscount.doubleValue());
		setAdjustedSubtotal(adjustedSubtotal.doubleValue());
		setAdjustedTax(adjustedTax.doubleValue());
		setAdjustedTotal(adjustedTotal.doubleValue());
	}

	public void calculatePriceV2() {
		if (isInfoOnly()) {
			return;
		}

		double unitPrice = getUnitPrice();
		double subTotalAmount = round(calculateSubTotalV2(unitPrice));
		double discountAmount = round(calculateDiscount(subTotalAmount));
		double totalCost = round(calculateTotalCost(getUnitCost()));
		double taxAmount = 0;
		double serviceCharge = 0;
		double totalAmount = 0;

		TicketItem ticketItem = getTicketItem();
		Ticket ticket = ticketItem == null ? null : ticketItem.getTicket();
		if (ticketItem != null && ticket != null) {
			if (ticket.isDiscountOnSerivceCharge() || ticket.hasRepriceDiscount()) {
				serviceCharge = round(ticketItem.calculateServiceChargeV2(subTotalAmount - discountAmount));
			}
			else {
				serviceCharge = round(ticketItem.calculateServiceChargeV2(subTotalAmount));
			}
		}
		if (ticketItem != null && ticketItem.isTaxOnServiceCharge()) {
			taxAmount = round(calculateTaxV2(subTotalAmount + serviceCharge, discountAmount));
		}
		else {
			taxAmount = round(calculateTaxV2(subTotalAmount, discountAmount));
		}

		if (isTaxIncluded()) {
			totalAmount = round(subTotalAmount - discountAmount + serviceCharge);
		}
		else {
			totalAmount = round(subTotalAmount - discountAmount + taxAmount + serviceCharge);
		}

		setTotalCost(totalCost);

		setSubTotalAmount(subTotalAmount);
		setDiscountAmount(discountAmount);
		setTaxAmount(taxAmount);
		setServiceCharge(serviceCharge);
		setTotalAmount(totalAmount);

		//adjusted price
		setAdjustedUnitPrice(getUnitPrice());
		setAdjustedDiscount(discountAmount);
		setAdjustedSubtotal(subTotalAmount);
		setAdjustedTax(taxAmount);
		setAdjustedTotal(totalAmount);
	}

	public void calculateAdjustedPriceV2() {
		TicketItem ticketItem = getTicketItem();
		Ticket ticket = null;
		if (ticketItem != null) {
			ticket = getTicketItem().getTicket();
		}

		double serviceCharge = getServiceCharge();
		double adjustedTax = 0;
		double itemQuantity = getItemQuantity();
		double ticketDiscountPercentage = ticket.getDiscountPercentageRate();
		double unitPriceAfterDiscount = (getSubtotalAmount() - getDiscountAmount()) / itemQuantity;
		double adjustedUnitPrice = unitPriceAfterDiscount - (unitPriceAfterDiscount * ticketDiscountPercentage);
		if (getUnitPrice() == 0) {
			adjustedUnitPrice = 0;
		}

		double adjustedSubtotal = round(adjustedUnitPrice * itemQuantity);
		double adjustedDiscount = round(getSubtotalAmount() - adjustedSubtotal);

		if (ticket != null && ticket.isDiscountOnSerivceCharge() || ticket.hasRepriceDiscount()) {
			serviceCharge = round(ticketItem.calculateServiceChargeV2(adjustedSubtotal));
		}

		if (ticketItem != null && ticketItem.isTaxOnServiceCharge()) {
			adjustedTax = round(calculateTaxV2(adjustedSubtotal + serviceCharge, 0));
		}
		else {
			adjustedTax = round(calculateTaxV2(adjustedSubtotal, 0));
		}

		double adjustedTotal = 0.0;
		if (isTaxIncluded()) {
			adjustedTotal = round(adjustedSubtotal + serviceCharge);
		}
		else {
			adjustedTotal = round(adjustedSubtotal + adjustedTax + serviceCharge);
		}

		setAdjustedUnitPrice(adjustedUnitPrice);
		setAdjustedDiscount(adjustedDiscount);
		setAdjustedSubtotal(adjustedSubtotal);
		setAdjustedTax(adjustedTax);
		setServiceCharge(serviceCharge);
		setAdjustedTotal(adjustedTotal);
	}

	public double calculateDiscount(final double subtotalAmount) {
		double discount = 0;
		JSONArray jsonArray = new JSONArray();
		TicketItem ticketItem = getTicketItem();
		if (ticketItem == null) {
			return 0;
		}
		List<TicketItemDiscount> discounts = ticketItem.getDiscounts();
		if (discounts != null) {
			for (TicketItemDiscount ticketItemDiscount : discounts) {
				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
					continue;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					TicketItemDiscount modifierDiscount = (TicketItemDiscount) SerializationUtils.clone(ticketItemDiscount);
					double newModifierDiscount = modifierDiscount.calculateDiscount(subtotalAmount - discount);
					jsonArray.put(modifierDiscount.toJson());
					ticketItemDiscount.setAmount(ticketItemDiscount.getAmount() + newModifierDiscount);
					discount += newModifierDiscount;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
					//					double discountPercentage = (ticketItemDiscount.getValue() * 100.0) / getTicketItem().getSubtotalAmount();
					//					TicketItemDiscount percentDiscount = new TicketItemDiscount();
					//					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
					//					percentDiscount.setValue(discountPercentage);
					//					percentDiscount.setTicketItem(getTicketItem());
					//					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
					//					discount += percentDiscount.calculateDiscount(subtotalAmount);
				}
			}

			addProperty("discountDetail", jsonArray.toString()); //$NON-NLS-1$
		}
		if (discount > Math.abs(subtotalAmount))
			return subtotalAmount;

		return discount;
	}

	public List<TicketItemDiscount> getDiscounts() {
		List<TicketItemDiscount> discounts = new ArrayList<>();

		String property = getProperty("discountDetail"); //$NON-NLS-1$
		if (StringUtils.isNotEmpty(property)) {
			JSONArray jsonArray = new JSONArray(property);
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject jsonObject = jsonArray.getJSONObject(i);
				TicketItemDiscount tid = new TicketItemDiscount();
				tid.setTicketItem(this.getTicketItem());
				tid.setDiscountId(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_DISCOUNT_ID));
				tid.setName(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_NAME));
				tid.setType(OrgJsonUtil.getInt(jsonObject, TicketItemDiscount.PROP_TYPE));
				tid.setAutoApply(OrgJsonUtil.getBoolean(jsonObject, TicketItemDiscount.PROP_AUTO_APPLY));
				tid.setCouponQuantity(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_COUPON_QUANTITY));
				tid.setMinimumAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_MINIMUM_AMOUNT));
				tid.setValue(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_VALUE));
				tid.setAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_AMOUNT));
				discounts.add(tid);
			}
		}
		return discounts;
	}

	public BigDecimal calculateDiscount(BigDecimal subtotalAmount) {
		BigDecimal discount = convertToBigDecimal(0);
		List<TicketItemDiscount> discounts = getTicketItem().getDiscounts();
		if (discounts != null) {
			for (TicketItemDiscount ticketItemDiscount : discounts) {
				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
					continue;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					discount = discount.add(ticketItemDiscount.calculateDiscount(subtotalAmount));
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
					double discountPercentage = (ticketItemDiscount.getValue() * 100.0) / getTicketItem().getSubtotalAmount();
					TicketItemDiscount percentDiscount = new TicketItemDiscount();
					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
					percentDiscount.setValue(discountPercentage);
					percentDiscount.setTicketItem(getTicketItem());
					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
					discount = discount.add(percentDiscount.calculateDiscount(subtotalAmount));
				}
			}
		}
		if (discount.compareTo(subtotalAmount.abs()) > 0)
			return subtotalAmount;

		return round(discount);
	}

	public void merge(TicketItemModifier otherItem) {
		setItemQuantity(getItemQuantity() + otherItem.getItemQuantity());
		return;
	}

	private BigDecimal calculateTax(BigDecimal subtotalAmount, BigDecimal discountAmount) {
		Ticket ticket = getTicketItem().getTicket();
		if (ticket != null && ticket.isTaxExempt()) {
			return BigDecimal.ZERO;
		}
		BigDecimal totalTaxAmount = convertToBigDecimal("0"); //$NON-NLS-1$
		subtotalAmount = subtotalAmount.subtract(discountAmount);
		List<TicketItemTax> taxList = getTaxes();
		if (taxList != null) {
			if (isTaxIncluded()) {
				BigDecimal taxRatePercentage = convertToBigDecimal(0);
				for (TicketItemTax ticketItemModifierTax : taxList) {
					taxRatePercentage = taxRatePercentage
							.add(convertToBigDecimal(ticketItemModifierTax.getRate()).divide(convertToBigDecimal("100"), 4, RoundingMode.HALF_UP)); //$NON-NLS-1$
				}

				if (taxRatePercentage.compareTo(convertToBigDecimal("0")) == 0) { //$NON-NLS-1$
					return totalTaxAmount;
				}
				else {
					BigDecimal totalTaxRatePercentage = taxRatePercentage.add(convertToBigDecimal(1d));
					double actualPrice = subtotalAmount.doubleValue() / totalTaxRatePercentage.doubleValue();

					for (TicketItemTax ticketItemModifierTax : taxList) {
						taxRatePercentage = convertToBigDecimal(ticketItemModifierTax.getRate()).divide(convertToBigDecimal("100"), 4, RoundingMode.HALF_UP); //$NON-NLS-1$
						BigDecimal tax = convertToBigDecimal(actualPrice).multiply(taxRatePercentage);
						ticketItemModifierTax.setTaxAmount(tax.doubleValue());
						totalTaxAmount = totalTaxAmount.add(tax);
					}
				}
			}
			else {
				for (TicketItemTax ticketItemModifierTax : taxList) {
					BigDecimal taxRatePercentage = convertToBigDecimal(ticketItemModifierTax.getRate()).divide(convertToBigDecimal("100"), 4, //$NON-NLS-1$
							RoundingMode.HALF_UP);
					if (taxRatePercentage.compareTo(convertToBigDecimal("0")) == 0) { //$NON-NLS-1$
						return totalTaxAmount;
					}
					else {
						BigDecimal tax = subtotalAmount.multiply(taxRatePercentage);
						ticketItemModifierTax.setTaxAmount(tax.doubleValue());
						totalTaxAmount = totalTaxAmount.add(tax);
					}
				}
			}
		}
		if (totalTaxAmount.doubleValue() < 0) {
			totalTaxAmount = BigDecimal.ZERO;
		}
		buildTaxes();
		return Tax.applyFloridaTaxRule(subtotalAmount, totalTaxAmount);

	}

	private double calculateTaxV2(double subtotalAmount, double discountAmount) {
		Ticket ticket = getTicketItem().getTicket();
		double totalTaxAmount = 0; //$NON-NLS-1$
		subtotalAmount = subtotalAmount - discountAmount;
		List<TicketItemTax> taxList = getTaxes();
		if (taxList != null) {
			if (isTaxIncluded()) {
				double taxRatePercentage = 0;
				for (TicketItemTax ticketItemModifierTax : taxList) {
					taxRatePercentage = taxRatePercentage + ticketItemModifierTax.getRate() / 100.0;
				}

				if (taxRatePercentage == 0) { //$NON-NLS-1$
					return totalTaxAmount;
				}
				else {
					double totalTaxRatePercentage = taxRatePercentage + 1;
					double actualPrice = subtotalAmount / totalTaxRatePercentage;

					for (TicketItemTax ticketItemModifierTax : taxList) {
						taxRatePercentage = ticketItemModifierTax.getRate() / 100.0;
						double tax = actualPrice * taxRatePercentage;
						ticketItemModifierTax.setTaxAmount(tax);
						totalTaxAmount = totalTaxAmount + tax;
					}
				}
			}
			else {
				for (TicketItemTax ticketItemModifierTax : taxList) {
					double taxRatePercentage = ticketItemModifierTax.getRate() / 100.0;
					if (taxRatePercentage == 0) { //$NON-NLS-1$
						return totalTaxAmount;
					}
					else {
						double tax = subtotalAmount * taxRatePercentage;
						ticketItemModifierTax.setTaxAmount(tax);
						totalTaxAmount = totalTaxAmount + tax;
					}
				}
			}
		}
		totalTaxAmount = Tax.applyFloridaTaxRuleV2(getTicketItem().getTicket(), subtotalAmount, totalTaxAmount);
		buildTaxes();
		setTaxExemptAmount(0.0);
		if (ticket != null && ticket.isTaxExempt()) {
			setTaxExemptAmount(round(totalTaxAmount));
			return 0.0;
		}

		return totalTaxAmount;

	}

	public double getTotalTaxRate() {
		List<TicketItemTax> ticketItemTaxes = getTaxes();
		if (ticketItemTaxes == null || ticketItemTaxes.isEmpty())
			return 0;

		double totalTaxRate = 0;
		for (TicketItemTax tax : ticketItemTaxes) {
			totalTaxRate += tax.getRate();
		}
		return totalTaxRate;

	}

	private BigDecimal calculateSubTotal(BigDecimal unitPrice) {
		Double modifierQty = getItemQuantity();
		if (getTicketItem() != null) {
			modifierQty = modifierQty * (Math.abs(getTicketItem().getQuantity()));
		}
		return unitPrice.multiply(convertToBigDecimal(modifierQty));
	}

	private double calculateTotalCost(double unitCost) {
		Double modifierQty = getItemQuantity();
		if (getTicketItem() != null) {
			modifierQty = modifierQty * (Math.abs(getTicketItem().getQuantity()));
		}
		return unitCost * modifierQty;
	}

	private double calculateSubTotalV2(double unitPrice) {
		Double modifierQty = getItemQuantity();
		if (getTicketItem() != null) {
			modifierQty = modifierQty * (Math.abs(getTicketItem().getQuantity()));
		}
		return unitPrice * modifierQty;
	}

	@Override
	public String getNameDisplay() {
		return getItemQuantityDisplay() + " " + getNameDisplay(false); //$NON-NLS-1$
	}

	public String getNameDisplay(boolean kitchenReceipt) {
		if (isInfoOnly()) {
			return "&nbsp;&nbsp;&nbsp;" + getName().trim();
		}
		double itemQuantity = Math.abs(getItemQuantity());
		String itemQuantityDisplay = ""; //$NON-NLS-1$
		if (itemQuantity > 1) {
			itemQuantityDisplay = NumberUtil.trimDecilamIfNotNeeded(itemQuantity) + "x "; //$NON-NLS-1$
		}
		String display;
		if (itemQuantity > 1) {
			display = itemQuantityDisplay + getName(); //$NON-NLS-1$
		}
		else {
			display = getName().trim(); //$NON-NLS-1$
		}
		if (getModifierType() == NORMAL_MODIFIER && !kitchenReceipt) {
			display += "*"; //$NON-NLS-1$
		}
		Store store = DataProvider.get().getStore();
		boolean isShowModifierPrice = false;
		if (store != null) {
			isShowModifierPrice = store.getProperty(AppConstants.PROP_SHOW_MODIFIER_PRICE) == null ? false
					: Boolean.valueOf(store.getProperty(AppConstants.PROP_SHOW_MODIFIER_PRICE));
		}

		if (isShowModifierPrice && getUnitPrice() > 0 && !kitchenReceipt) {
			display = itemQuantityDisplay + getName() + " @" + NumberUtil.formatNumber(getUnitPrice()); //$NON-NLS-1$
		}
		if (kitchenReceipt) {
			return "&nbsp;&nbsp;&nbsp;" + "-- " + display;
		}
		return "&nbsp;&nbsp;&nbsp;" + " -- " + display; //$NON-NLS-1$
	}

	@Override
	public String getUnitPriceDisplay() {
		if (isInfoOnly()) {
			return null;
		}
		return NumberUtil.formatNumberAcceptNegative(getUnitPrice());
	}

	@Override
	public String getItemQuantityDisplay() {
		/*if (isInfoOnly()) {
			return "";
		}
		Double quantity = getItemQuantity();
		if (quantity <= 1)
			return "";
		return NumberUtil.trimDecilamIfNotNeeded(quantity);*/
		return ""; //$NON-NLS-1$
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getTaxAmount());
	}

	@Override
	public String getTotalAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getTotalAmount());
	}

	@Override
	public String getSubTotalAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getSubTotalAmount());
	}

	@Override
	public String getItemCode() {
		return ""; //$NON-NLS-1$
	}

	public boolean isSelected() {
		return selected;
	}

	public void setSelected(boolean selected) {
		this.selected = selected;
	}

	@Override
	public Double getItemQuantity() {
		Double itemQuantity = super.getItemQuantity();
		if (itemQuantity == 0) {
			itemQuantity = (double) super.getItemCount();
		}

		return itemQuantity;
	}

	@Override
	public boolean canAddDiscount() {
		return false;
	}

	@Override
	public boolean canVoid() {
		return false;
	}

	@Override
	public boolean canAddAdOn() {
		return false;
	}

	public KitchenStatus getKitchenStatusValue() {
		return KitchenStatus.fromString(super.getStatus());
	}

	public void setKitchenStatusValue(KitchenStatus kitchenStatus) {
		super.setStatus(kitchenStatus.name());
	}

	@Override
	public Double getSubtotalAmount() {
		return super.getSubTotalAmount();
	}

	@Override
	public String getSubTotalAmountDisplay() {
		return null;
	}

	@XmlTransient
	public MenuModifier getMenuModifier() {
		return menuModifier;
	}

	public void setMenuModifier(MenuModifier menuModifier) {
		this.menuModifier = menuModifier;
	}

	@Override
	public boolean isSaved() {
		return getId() != null;
	}

	public void setTaxes(List<TicketItemTax> taxes) {
		this.taxes = taxes;
		if (this.taxes == null) {
			this.taxes = new ArrayList<>();
		}
	}

	public List<TicketItemTax> getTaxes() {
		if (taxes == null) {
			taxes = new ArrayList<>();

			String property = super.getTaxesProperty();
			if (StringUtils.isNotEmpty(property)) {
				JsonReader jsonParser = Json.createReader(new StringReader(property));
				JsonArray jsonArray = jsonParser.readArray();
				jsonParser.close();
				for (int i = 0; i < jsonArray.size(); i++) {
					JsonObject tObj = (JsonObject) jsonArray.get(i);
					TicketItemTax tit = new TicketItemTax();
					tit.setName(tObj.getString(TicketItemTax.PROP_NAME));
					tit.setRate(POSUtil.parseDouble("" + tObj.get(TicketItemTax.PROP_RATE))); //$NON-NLS-1$
					tit.setTaxAmount(POSUtil.parseDouble("" + tObj.get(TicketItemTax.PROP_TAX_AMOUNT))); //$NON-NLS-1$
					taxes.add(tit);
				}
			}
		}
		return taxes;
	}

	public void addTotaxes(TicketItemTax tax) {
		if (taxes == null) {
			taxes = new ArrayList<TicketItemTax>();
		}
		taxes = getTaxes();
		taxes.add(tax);
	}

	public void buildTaxes() {
		if (taxes == null || taxes.isEmpty()) {
			setTaxesProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemTax ticketItemTax : taxes) {
			jsonArray.put(ticketItemTax.toJson());
		}
		setTaxesProperty(jsonArray.toString());
	}

	public void setTicketItemQuantity(double ticketItemQuantity) {
		this.ticketItemQuantity = ticketItemQuantity;
	}

	public double getTicketItemQuantity() {
		if (getTicketItem() != null) {
			return getTicketItem().getQuantity();
		}
		return ticketItemQuantity;
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			String properties = super.getProperties();
			if (StringUtils.isEmpty(properties)) {
				return null;
			}
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		try {
			propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		} catch (Exception e) {
		}
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new com.google.gson.JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new com.google.gson.JsonObject();
		}
		return propertiesContainer;
	}

	public Double getMultiplierPrice(Double groupPrice) {
		if (StringUtils.isEmpty(super.getMultiplierName())) {
			return groupPrice;
		}
		Multiplier multiplier = DataProvider.get().getMultiplierById(super.getMultiplierName());
		if (multiplier == null || multiplier.isMain()) {
			return groupPrice;
		}
		return groupPrice * multiplier.getRate() / 100;
	}
}