package com.floreantpos.services.report;

import java.util.Iterator;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.util.POSUtil;

public class CashDrawerReportService {
	private CashDrawer cashDrawer;
	private Session session;

	public CashDrawerReportService(CashDrawer cashDrawer) {
		super();
		this.cashDrawer = cashDrawer;
	}

	public void populateReport() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();
			populateReport(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void populateSummaryOfCashdrawer() {
		GenericDAO dao = new GenericDAO();
		try {
			session = dao.createNewSession();
			populateSummaryOfCashdrawer(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void populateSummaryOfCashdrawer(Session session) {
		this.session = session;
		calculateNetSales();
		calculateRefundAmount();
		calculateDrawerBleed();
		calculateCashTips();
		calculateChargedTips();

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() - cashDrawer.getDrawerBleedAmount() + cashDrawer.getSalesTax()
				+ cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue((totalRev));
		cashDrawer.setGrossReceipts((cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));
	}

	public void populateReport(Session session) {
		this.session = session;

		calculateNetSales();
		calculateRefundAmount();
		calculateDrawerBleed();
		calculateCashTips();
		calculateChargedTips();

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() - cashDrawer.getDrawerBleedAmount() + cashDrawer.getSalesTax()
				+ cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue((totalRev));
		cashDrawer.setGrossReceipts((cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));

		calculateCashReceipt();
		calculateCreditReceipt();
		calculateDebitReceipt();
		calculateMemberPayment();

		calculateCustomPaymentWithoutPromotion();
		calculatePromotionAmount();
		calculateGiftCertReceipts();
		calculateVoidAmount();
		calculateCashPayout();
		calculateCashBack();

		calculateTipsPaid();
		calculateReceiptDiff();
		calculateTipsDiff();

		calculateDrawerAccountable();
		calculateTicketCount();
	}

	private void calculateTicketCount() {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.setProjection(Projections.countDistinct(PosTransaction.PROP_TICKET));
		Number rowCount = (Number) criteria.uniqueResult();
		int ticketCount = 0;
		if (rowCount != null) {
			ticketCount = rowCount.intValue();
		}
		cashDrawer.setTicketCount(ticketCount);
	}

	private void calculateMemberPayment() {
		Criteria criteria = getCriteriaForTransaction(CustomerAccountTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCustomerPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerAccountable() {
		double drawerAccountableAmnt = cashDrawer.getCashReceiptAmount() - cashDrawer.getTipsPaid() - cashDrawer.getPayOutAmount() - cashDrawer.getCashBack()
				+ cashDrawer.getBeginCash() - cashDrawer.getDrawerBleedAmount();
		cashDrawer.setDrawerAccountable((drawerAccountableAmnt));
	}

	private void calculateTipsDiff() {
		double tipsDiff = cashDrawer.getCashTips() + cashDrawer.getChargedTips() - cashDrawer.getTipsPaid();
		cashDrawer.setTipsDifferential((tipsDiff));
	}

	private void calculateReceiptDiff() {
		double receiptDiff = cashDrawer.getGrossReceipts() - cashDrawer.getCashReceiptAmount() - cashDrawer.getCreditCardReceiptAmount()
				- cashDrawer.getDebitCardReceiptAmount() - cashDrawer.getCustomerPaymentAmount() - cashDrawer.getCustomPaymentAmount()
				- cashDrawer.getPromotionAmount() - cashDrawer.getGiftCertChangeAmount() - cashDrawer.getGiftCertReturnAmount() + cashDrawer.getTotalVoid()
				+ cashDrawer.getRefundAmount();
		cashDrawer.setReceiptDifferential((receiptDiff));
	}

	private Criteria getCriteriaForTransaction(Class transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass);
		if (transactionClass.equals(RefundTransaction.class) || transactionClass.equals(PayOutTransaction.class))
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		else
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		return criteria;
	}

	private void calculateNetSales() {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TAX_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_SERVICE_CHARGE_AMOUNT));
		criteria.setProjection(projectionList);

		List<Object[]> list = criteria.list();
		if (list == null || list.isEmpty()) {
			return;
		}
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			Object[] objects = (Object[]) iterator.next();
			if (objects[0] == null) {
				return;
			}
			double salesAmnt = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double taxAmnt = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;
			double tipsAmnt = (objects[2] != null) ? ((Number) objects[2]).doubleValue() : 0;
			double serviceChrgAmnt = (objects[3] != null) ? ((Number) objects[3]).doubleValue() : 0;

			double netSales = salesAmnt - taxAmnt - tipsAmnt - serviceChrgAmnt;
			cashDrawer.setNetSales((netSales));
			cashDrawer.setSalesTax((taxAmnt));
			cashDrawer.setServiceCharge((serviceChrgAmnt));
		}
	}

	private void calculateCashTips() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setCashTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateChargedTips() {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setChargedTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashReceipt() {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCashReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCreditReceipt() {
		Criteria criteria = getCriteriaForTransaction(CreditCardTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

		cashDrawer.setCreditCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDebitReceipt() {
		Criteria criteria = getCriteriaForTransaction(DebitCardTransaction.class);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		cashDrawer.setDebitCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	//	private void calculateCustomPayment() {
	//		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
	//		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
	//		cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	//	}

	private void calculateCustomPaymentWithoutPromotion() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculatePromotionAmount() {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setPromotionAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateGiftCertReceipts() {
		Criteria criteria = getCriteriaForTransaction(GiftCertificateTransaction.class);
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
		cashDrawer.setGiftCertChangeAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateRefundAmount() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setRefundAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateVoidAmount() {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.TRUE));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCardVoidAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashPayout() {
		Criteria criteria = getCriteriaForTransaction(PayOutTransaction.class);
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setPayOutAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerBleed() {
		Criteria criteria = getCriteriaForTransaction(CashDropTransaction.class);
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setDrawerBleedAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashBack() {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCashBack(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateTipsPaid() {
		Criteria criteria = session.createCriteria(GratuityPaymentHistory.class);
		criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, cashDrawer));
		criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
		cashDrawer.setTipsPaid(POSUtil.getDoubleAmount(criteria.uniqueResult()));

	}

	//	private void calculateGrossReceipts() {
	//		Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
	//		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
	//		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	//	}
	//	

}
