package com.floreantpos.util;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.floreantpos.model.Discount;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemDiscount;

public class DiscountUtil {
	public static Double calculateDiscountAmount(TicketItemDiscount ticketItemDiscount, final double subtotalAmount) {
		TicketItem ticketItem = ticketItemDiscount.getTicketItem();
		//		Ticket ticket = ticketItem.getTicket();
		if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE)
			return ticketItemDiscount.getValue();

		if (ticketItemDiscount.getMinimumAmount() > 1) {
			double minQuantity = ticketItemDiscount.getMinimumAmount();
			double itemCount = ticketItem.getQuantity(); //ticket.countItem(ticketItem);

			switch (ticketItemDiscount.getType()) {
				case Discount.DISCOUNT_TYPE_AMOUNT:
					return Math.floor(itemCount / minQuantity) * ticketItemDiscount.getValue();

				case Discount.DISCOUNT_TYPE_PERCENTAGE:
					return Math.floor(itemCount / minQuantity) * ((subtotalAmount / itemCount) * ticketItemDiscount.getValue() / 100);
			}
		}
		switch (ticketItemDiscount.getType()) {
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return ticketItemDiscount.getValue();

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return (subtotalAmount * ticketItemDiscount.getValue()) / 100.0;
		}

		return 0.0;
	}

	public static BigDecimal calculateDiscountAmount(TicketItemDiscount ticketItemDiscount, BigDecimal subtotalAmount) {
		//@formatter:off
		TicketItem ticketItem = ticketItemDiscount.getTicketItem();
		BigDecimal discountValue = NumberUtil.convertToBigDecimal(ticketItemDiscount.getValue());
		BigDecimal percentage = discountValue.divide(NumberUtil.convertToBigDecimal("100.0"), 4, RoundingMode.FLOOR); //$NON-NLS-1$
		
		if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE)
			return discountValue;

		if (ticketItemDiscount.getMinimumAmount() > 1) {
			BigDecimal minQuantity = NumberUtil.convertToBigDecimal(ticketItemDiscount.getMinimumAmount());
			BigDecimal itemCount = NumberUtil.convertToBigDecimal(ticketItem.getQuantity()); //ticket.countItem(ticketItem);

			switch (ticketItemDiscount.getType()) {
				case Discount.DISCOUNT_TYPE_AMOUNT:
					return itemCount
							.divide(minQuantity, 4, RoundingMode.FLOOR)
							.multiply(discountValue);

				case Discount.DISCOUNT_TYPE_PERCENTAGE:
					BigDecimal itemCountDivideQuantity = itemCount.divide(minQuantity, 4, RoundingMode.FLOOR);
					BigDecimal subtotalDivideItemCount = subtotalAmount.divide(itemCount, 4, RoundingMode.FLOOR);
					
					return itemCountDivideQuantity.multiply(subtotalDivideItemCount)
							.multiply(percentage);
			}
		}
		switch (ticketItemDiscount.getType()) {
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return discountValue;

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return subtotalAmount.multiply(percentage);
		}

		return new BigDecimal("0"); //$NON-NLS-1$
		//@formatter:on
	}

	public static Double calculateDiscountAmount(double price, TicketDiscount discount) {

		switch (discount.getType()) {
			case Discount.DISCOUNT_TYPE_REPRICE:
				return price == 0.0 ? 0.0 : discount.getValue();
				
			case Discount.DISCOUNT_TYPE_AMOUNT:
				return discount.getValue();

			case Discount.DISCOUNT_TYPE_PERCENTAGE:
				return price * ((discount.getValue() * discount.getCouponQuantity()) / 100.0);
		}

		return (price * discount.getValue()) / 100.0;
	}

	public static TicketItemDiscount getMaxDiscount(List<TicketItemDiscount> discounts) {
		if (discounts == null || discounts.isEmpty()) {
			return null;
		}

		TicketItemDiscount maxDiscount = Collections.max(discounts, new Comparator<TicketItemDiscount>() {
			@Override
			public int compare(TicketItemDiscount o1, TicketItemDiscount o2) {
				return (int) (o1.getAmount() - o2.getAmount());
			}
		});

		return maxDiscount;
	}

	public static TicketDiscount getMaxDiscount(List<TicketDiscount> discounts, final double price) {
		if (discounts == null || discounts.isEmpty()) {
			return null;
		}

		TicketDiscount maxDiscount = Collections.max(discounts, new Comparator<TicketDiscount>() {
			@Override
			public int compare(TicketDiscount o1, TicketDiscount o2) {
				return (int) (calculateDiscountAmount(price, o1) - calculateDiscountAmount(price, o2));
			}
		});

		return maxDiscount;
	}

	public static Double calculateRepriceDiscount(Ticket ticket, double repriceValue) {
		Double discountValue;
		Double ticketTotalAmount = ticket.getTotalAmount();
		Double ticketTaxAmount = ticket.getTaxAmount();
		if (ticketTotalAmount == 0) {
			return 0.0;
		}

		double repricedTaxAmount = (repriceValue * ticketTaxAmount) / ticketTotalAmount;
		if (ticket.isTaxIncluded()) {
			repricedTaxAmount = 0.0;
		}
		discountValue = (ticket.getSubtotalAmount() - ticket.getItemDiscountAmount()) - (repriceValue - repricedTaxAmount);
		return discountValue;
	}
	
	public static Double calculatePercentDiscount(double amount, double percentage) {
		return (amount * percentage);
	}

	public static double calculateDiscountPercentageFromAmount(double subtotalAmount, double discountAmount) {
		double d = (subtotalAmount - discountAmount) / subtotalAmount;
		double d2 = d * 100;

		return (100 - d2);
		
//		return (1 - (discountAmount / subtotalAmount)) * 100.0;
	}

	public static double calculatePercentageForTotal(double expectedTotal, double currentTotal) {
		double d = (expectedTotal / currentTotal) * 100.0;
		d = 100.0 - d;
		return d;
	}
}
